/**
 * tp_event_collect_ctrl.c
 *
 * Event collection controller, triggers an actor when one of its
 * ISensors asserts an event covered by its mask and the
 * condition for this sensor is true
 * 
 * (c) 2005 Peppercon AG, 19/07/2005, miba@peppercon.de
 */

#include <pp/bmc/debug.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_actor.h>
#include <pp/bmc/tp_cond.h>
#include <pp/bmc/tp_ipmi_sens.h>
#include <pp/bmc/tp_sens_subscriber.h>
#include <pp/bmc/tp_actor.h>
#include <pp/bmc/tp_event_collect_ctrl.h>

typedef struct {
    pp_tp_event_collect_ctrl_t* this;
    
    pp_tp_ipmi_sens_t *isens;
    pp_tp_ipmi_sens_subscriber_t sens_subs;

    pp_tp_cond_t* cond;
    pp_tp_sensdev_subscriber_t cond_subs;
    int cond_val;

    int signals_pending;
    int mask;
} collect_sens_t;

#define INITIAL_COLLECT_COUNT	20

static void collect_sens_del_func(void* arg);
static void sens_recv_reading(pp_tp_ipmi_sens_subscriber_t* subscriber, 
			      pp_tp_ipmi_sens_t* source, int reading);
static void cond_recv_reading(pp_tp_sensdev_subscriber_t* subscriber, 
			      pp_tp_sensdev_t* source, int reading);
static void update_output(pp_tp_event_collect_ctrl_t* this);

pp_tp_obj_t*
pp_tp_event_collect_ctrl_ctor(const char* id, vector_t* args)
{
    const char* errmsg = "[EventCollectCtrl] %s: '%s' failed (%s)";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;  
    pp_tp_event_collect_ctrl_t* instance = NULL;
    vector_t *sensvec = NULL;
    pp_tp_gpio_act_t *act;
    collect_sens_t *sens;
    int count = 0;
    int arg_idx = 0;

    if (pp_tp_arg_scanf(args, arg_idx++, &err, "o<ag>", &act) != 1) {
	pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
	goto bail;
    }

    sensvec = vector_new(NULL, INITIAL_COLLECT_COUNT, collect_sens_del_func);
    do {
	pp_tp_ipmi_sens_t *isens;
	pp_tp_cond_t* cond;
	int mask;

	count = pp_tp_arg_scanf(args, arg_idx, &err, "o<p>do<sc>", &isens, &mask, &cond);
	if (count == 0) {
	    break;
	}
	if (count != 3) {
	    pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
	    goto bail;
	}

	// got the expected parameters per ISensor
	sens = malloc(sizeof(collect_sens_t));

	sens->isens = pp_tp_ipmi_sens_duplicate(isens);
	sens->cond  = pp_tp_cond_duplicate(cond);
	sens->mask  = mask;
	
	vector_add(sensvec, sens);

	arg_idx += 3;
    } while(count != 0);


    if (vector_size(sensvec) != 0) {
	// got at least one sensor
	instance = malloc(sizeof(pp_tp_event_collect_ctrl_t));
	
	if (PP_FAILED(pp_tp_event_collect_ctrl_init(instance, PP_TP_CTRL, id,
						 pp_tp_event_collect_ctrl_dtor,
						 act, sensvec))) {
	    pp_bmc_log_error(errmsg, ___F, id, "init");
	    free(instance);
	    instance = NULL;
	}
    }
    
 bail:
    pp_strstream_free(&err);
    if (instance == NULL) {
	vector_delete(sensvec);
    }

    return (pp_tp_obj_t*)instance;
}

void
pp_tp_event_collect_ctrl_dtor(pp_tp_obj_t* this)
{
    pp_tp_event_collect_ctrl_cleanup((pp_tp_event_collect_ctrl_t*)this);
    free(this);    
}

int
pp_tp_event_collect_ctrl_init(pp_tp_event_collect_ctrl_t* this,
			   pp_tp_obj_type_t type,
			   const char* id, pp_tp_obj_dtor_func_t dtor,
			   pp_tp_gpio_act_t *act, vector_t *sensvec)
{
    if (PP_SUCCED(pp_tp_ctrl_init(&this->base, type, id, dtor))) {
	size_t count, i;
	collect_sens_t* sens;

	this->act = pp_tp_gpio_act_duplicate(act);
	this->sensvec = sensvec;
	this->signals_pending = 0;
	this->output = -1;
	
	count =	vector_size(this->sensvec);
	for (i = 0; i < count; i++) {
	    sens = (collect_sens_t*) vector_get(this->sensvec, i);
	    sens->this = this;
	    sens->signals_pending = 0;
	    sens->cond_val = -1;
	    sens->sens_subs.recv_reading = sens_recv_reading;
	    sens->cond_subs.recv_reading = cond_recv_reading;
	    
	    pp_tp_ipmi_sens_subscribe(sens->isens, &sens->sens_subs);
	    pp_tp_cond_subscribe(sens->cond, &sens->cond_subs);
	}
	
	pp_bmc_log_debug("[EventCollectCtrl] '%s' init", 
			 pp_tp_obj_to_string((pp_tp_obj_t*)this));
	
	return PP_SUC;
    }
    
    return PP_ERR;
}

void
pp_tp_event_collect_ctrl_cleanup(pp_tp_event_collect_ctrl_t* this)
{
    size_t count, i;
    
    count = vector_size(this->sensvec);
    for (i = 0; i < count; i++) {
	collect_sens_t* sens = (collect_sens_t*) vector_get(this->sensvec, i);
	pp_tp_ipmi_sens_unsubscribe(sens->isens, &sens->sens_subs);
	pp_tp_cond_unsubscribe(sens->cond, &sens->cond_subs);
    }
    
    vector_delete(this->sensvec);

    pp_bmc_log_debug("[EventCollectCtrl] '%s' cleanup", 
		     pp_tp_obj_to_string((pp_tp_obj_t*)this));

    pp_tp_ctrl_cleanup(&this->base);        
}

static void
collect_sens_del_func(void* arg)
{
    collect_sens_t *collect_sens = (collect_sens_t*) arg;

    pp_tp_ipmi_sens_release(collect_sens->isens);
    pp_tp_cond_release(collect_sens->cond);
    free(collect_sens);
}

static void
sens_recv_reading(pp_tp_ipmi_sens_subscriber_t* subscriber, 
		  pp_tp_ipmi_sens_t* source UNUSED, int reading)
{
    collect_sens_t *sens =
	PP_TP_INTF_2_OBJ_CAST(subscriber, collect_sens_t, sens_subs);
    pp_tp_event_collect_ctrl_t* this = sens->this;

   
    if ((reading & sens->mask) == 0) {
	pp_bmc_log_debug("[EventCollectCtrl] '%s' reading %08x from '%s' assert %d pending %d IGNORED",
			 pp_tp_obj_to_string((pp_tp_obj_t*)this),
			 reading,
			 ((pp_tp_obj_t*)sens->isens)->id,
			 PP_ISENS_NOTIFY_IS_ASSERTION(reading),
			 sens->signals_pending);
	// ignore this event
	return;
    }
    
    if (PP_ISENS_NOTIFY_IS_ASSERTION(reading)) {
	sens->signals_pending++;
    } else {
	sens->signals_pending--;
    }
    
    pp_bmc_log_debug("[EventCollectCtrl] '%s' reading %08x from '%s' assert %d pending %d",
		     pp_tp_obj_to_string((pp_tp_obj_t*)this),
		     reading,
		     ((pp_tp_obj_t*)sens->isens)->id,
		     PP_ISENS_NOTIFY_IS_ASSERTION(reading),
		     sens->signals_pending);
    
    update_output(this);
}

static void
cond_recv_reading(pp_tp_sensdev_subscriber_t* subscriber, 
		  pp_tp_sensdev_t* source UNUSED, int reading)
{
    collect_sens_t *sens =
	PP_TP_INTF_2_OBJ_CAST(subscriber, collect_sens_t, cond_subs);
    pp_tp_event_collect_ctrl_t* this = sens->this;

    sens->cond_val = reading;

    pp_bmc_log_debug("[EventCollectCtrl] '%s' cond '%s' changed to %d",
		     pp_tp_obj_to_string((pp_tp_obj_t*)this),
		     ((pp_tp_obj_t*)sens->cond)->id,
		     sens->cond_val);
    
    update_output(this);
}

static void
update_output(pp_tp_event_collect_ctrl_t* this)
{
    int count, i, signals_pending = 0;
    
    count = vector_size(this->sensvec);
    for (i = 0; i < count; i++) {
	collect_sens_t* sens = (collect_sens_t*) vector_get(this->sensvec, i);

	if (sens->cond_val <= 0) continue;
	
	if (sens->signals_pending) {
	    signals_pending++;
	}
    }

    pp_bmc_log_debug("[EventCollectCtrl] '%s', signals pending %d",
		     pp_tp_obj_to_string((pp_tp_obj_t*)this),
		     signals_pending);  
    
    if (signals_pending) {
	if (this->output != 1) {
	    if (PP_SUCCED(this->act->set_gpio(this->act, 1))) {
		this->output = 1;
	    }
	}
    } else {
	if (this->output != 0) {
	    if (PP_SUCCED(this->act->set_gpio(this->act, 0))) {
		this->output = 0;
	    }
	}
    }
}
