/**
 * tp_gpio_act.c
 * 
 * A gpio actor that can set an output signal.
 * 
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include <pp/bmc/tp_gpio_act.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/debug.h>

/* internal prototypes */
static int get_gpio(pp_tp_gpio_act_t* o);
static int set_gpio(pp_tp_gpio_act_t* o, unsigned char value);

pp_tp_obj_t* pp_tp_gpio_act_ctor(const char* id UNUSED, vector_t* args) {
    pp_tp_gpio_act_t* o;
    pp_tp_gpio_dev_t* gpio_dev;
    int gpio_no;
    int sig_ass;
    int sig_rel;
    int initial_ass;
    
    int argno;
    
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;

    argno = pp_tp_arg_scanf(args, 0, &err, "o<g>dddb", &gpio_dev, &gpio_no,
                    &sig_ass, &sig_rel, &initial_ass); 
    if ((argno < 4) || (argno > 5)) {
        pp_bmc_log_debug("[GPIO_out] gpio_act_ctor failed: %s", pp_strstream_buf(&err));
        return NULL;
    }
    if (argno == 4) {
        initial_ass = 2;
    }

    if (sig_ass == 2)
        sig_ass = 'z';
    if (sig_rel == 2)
        sig_rel = 'z';
    if ((sig_ass != 0) && (sig_ass != 1) && (sig_ass != 'z')) {
        pp_bmc_log_debug("[GPIO_out] signal_asserted parameter must be 0,1 or 'z'");
	pp_strstream_free(&err);
        return NULL;
    }
    if ((sig_rel != 0) && (sig_rel != 1) && (sig_rel != 'z')) {
        pp_bmc_log_debug("[GPIO_out] signal_released parameter must be 0,1 or 'z'");
	pp_strstream_free(&err);
        return NULL;
    }

    o = malloc(sizeof(pp_tp_gpio_act_t));
    pp_tp_gpio_act_init(o, id, gpio_dev, gpio_no, sig_ass, sig_rel, initial_ass);

    pp_strstream_free(&err);
    return (pp_tp_obj_t*)o;
}

void pp_tp_gpio_act_init(pp_tp_gpio_act_t* o,
			 const char* id,
			 pp_tp_gpio_dev_t* gpio_dev,
			 unsigned char gpio_no,
			 unsigned char sig_asserted,
			 unsigned char sig_released,
			 unsigned char initial_ass)
{
    assert(PP_TP_OBJ_IS_TYPE(PP_TP_GPIO_DEV, (pp_tp_obj_t*)gpio_dev));
    
    pp_tp_obj_init((pp_tp_obj_t*)o, PP_TP_GPIO_ACT, id,
                   (void*)(pp_tp_gpio_act_cleanup));

    o->gpio_dev = pp_tp_gpio_dev_duplicate(gpio_dev);  
    o->gpio_no = gpio_no;
    o->get_gpio = get_gpio;
    o->set_gpio = set_gpio;
    o->sig_asserted = sig_asserted;
    o->sig_released = sig_released;
    o->value = -1;

    pp_tp_gpio_dev_init_gpio(o->gpio_dev, 1 << o->gpio_no);
    
    if (initial_ass != 2) {
        set_gpio(o, initial_ass);
    }
    
    pp_bmc_log_debug("[GPIO_out] init '%s' (%s), sig=%d", id, ((pp_tp_obj_t*)gpio_dev)->id, o->gpio_no);
}

void pp_tp_gpio_act_cleanup(pp_tp_gpio_act_t* o) {
    assert(PP_TP_OBJ_IS_TYPE(PP_TP_GPIO_ACT, (pp_tp_obj_t*)o));
    assert(PP_TP_OBJ_IS_TYPE(PP_TP_GPIO_DEV, (pp_tp_obj_t*)(o->gpio_dev)));
    
    pp_tp_obj_release((pp_tp_obj_t*)(o->gpio_dev));

    pp_bmc_log_debug("[GPIO_out] '%s' cleanup", ((pp_tp_obj_t*)o)->id);

    pp_tp_obj_cleanup((pp_tp_obj_t*)o);
}

static int get_gpio(pp_tp_gpio_act_t* o) {
    assert(PP_TP_OBJ_IS_TYPE(PP_TP_GPIO_ACT, (pp_tp_obj_t*)o));
    pp_tp_gpio_dev_t * gpio_dev = (pp_tp_gpio_dev_t*)(o->gpio_dev);
    int ret = o->value;
    
    /* if the gpio device is not present return -1 */
    if (gpio_dev->get_gpio(gpio_dev, o->gpio_no) < 0) {
	if (errno == ENODEV) ret = -1;
    }
    return ret;
}

static int set_gpio(pp_tp_gpio_act_t* o, unsigned char in_value) {
    assert(PP_TP_OBJ_IS_TYPE(PP_TP_GPIO_ACT, (pp_tp_obj_t*)o));
    pp_tp_gpio_dev_t * gpio_dev = (pp_tp_gpio_dev_t*)(o->gpio_dev);
    
    unsigned char value;

    o->value = in_value;

    if (in_value == 1)
        value = (o->sig_asserted);
    else
        value = (o->sig_released);

/*     pp_bmc_log_debug("[GPIO_out] setting gpio '%s' to %c", */
/* 		     ((pp_tp_obj_t*)o)->id, */
/*                      value == 'z' ? 'z' : value ? '1' : '0'); */
    return gpio_dev->set_gpio(gpio_dev, o->gpio_no, value);
}
