/**
 * tp_gpio_dev.c
 *
 * An abstract gpio device.
 * This file is quite empty as the gpio_dev is abstract and contains
 * merely no functionality.
 * 
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include <pp/bmc/tp_gpio_dev.h>
#include <pp/bmc/debug.h>


static int init_gpio_base(struct pp_tp_gpio_dev_s* s UNUSED, unsigned long mask UNUSED) {
    /* we don't need to do anything here */
    return 0;
}


static int get_gpio(struct pp_tp_gpio_dev_s* s, int gpio_num) {
    int value;
    
    if ((value = s->get_val(s, 1 << gpio_num)) >= 0) {
	value =  value >> gpio_num;
    }
    return value;
}

static int set_gpio(struct pp_tp_gpio_dev_s* s, int gpio_num,
		    unsigned char value) {
    assert(((1 << gpio_num) & s->valid_mask) != 0);
    return s->set_val(s, 1 << gpio_num,
		      value == 1 ? (1 << gpio_num) : 0,
		      value == 'z' ? (1 << gpio_num) : 0);
}

void pp_tp_gpio_dev_register_subscriber(struct pp_tp_gpio_dev_s* d,
					pp_tp_gpio_subscriber_t* subscriber) {
    assert(PP_TP_OBJ_IS_TYPE(PP_TP_GPIO_DEV, d));

    if (subscriber->get_gpio_mask(subscriber) & ~d->valid_mask) {
        pp_bmc_log_error("[GPIO_dev] failed to subscribe client with "
			 "mask %x (only supporting %x)",
                         subscriber->get_gpio_mask(subscriber), d->valid_mask);
    } else {
	vector_add(d->gpio_subscribers, subscriber);
	if (d->subscribers_changed) d->subscribers_changed(d);
    }
}

int pp_tp_gpio_dev_unregister_subscriber(struct pp_tp_gpio_dev_s* d,
					 pp_tp_gpio_subscriber_t* subscriber) {
    unsigned int i;

    assert(PP_TP_OBJ_IS_TYPE(PP_TP_GPIO_DEV, d));

    for (i = 0; i < vector_size(d->gpio_subscribers); i++) {
        if (subscriber == vector_get(d->gpio_subscribers, i)) {
            vector_remove(d->gpio_subscribers, i);
            if (d->subscribers_changed) d->subscribers_changed(d);
            return PP_SUC;
        }
    }

    return PP_ERR;
}

void pp_tp_gpio_dev_notify_subscribers(struct pp_tp_gpio_dev_s* s, unsigned long value,
				       unsigned long change_mask, int valid) {
    unsigned int i;
    for (i=0; i<vector_size(s->gpio_subscribers); i++) {
        pp_tp_gpio_subscriber_t* sub = vector_get(s->gpio_subscribers, i);
        if (!valid || (change_mask & sub->get_gpio_mask(sub)) != 0) {
            sub->gpio_changed(sub, value, valid);
        }
    }
}

void pp_tp_gpio_dev_init(pp_tp_gpio_dev_t* d,
                         pp_tp_obj_type_t type,
                         const char* id,
                         unsigned long valid_mask,
                         void (*dtor)(pp_tp_obj_t*),
                         int (*init_gpio)(struct pp_tp_gpio_dev_s* s,
					  unsigned long mask),
                         int (*get_val)(struct pp_tp_gpio_dev_s* s,
					unsigned long mask),
                         int (*set_val)(struct pp_tp_gpio_dev_s* s,
					unsigned long mask,
					unsigned long val, unsigned long open),
                         void (*subscribers_changed)(struct pp_tp_gpio_dev_s* s))
{
    pp_tp_obj_init((pp_tp_obj_t*)d, type, id, dtor);

    d->gpio_subscribers = vector_new(NULL, 0, NULL);

    d->valid_mask = valid_mask;

    /* set inherited class implemented methods */
    d->get_val = get_val;
    d->set_val = set_val;
    d->subscribers_changed = subscribers_changed;

    /* set base class implemented methods */
    d->get_gpio = get_gpio;
    d->set_gpio = set_gpio;

    /* if specified, set gpio init routine,
       otherwise use or own dummy */
    if (init_gpio) {
	d->init_gpio = init_gpio;
    } else {
	d->init_gpio = init_gpio_base;
    }
}

void pp_tp_gpio_dev_cleanup(pp_tp_gpio_dev_t* d)
{
    vector_delete(d->gpio_subscribers);
    pp_tp_obj_cleanup((pp_tp_obj_t*)d);
}
