/**
 * tp_gpio_loop.c
 *
 * Loop actor, connects an input gpio to an output gpio.
 * 
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include <pp/bmc/debug.h>
#include <pp/bmc/tp_gpio_loop.h>
#include <pp/bmc/tp_sens_subscriber.h>

/* internal prototypes */
static void gpio_loop_dtor(pp_tp_obj_t* d);
static void gpio_loop_receive_sensor_reading(pp_tp_sensdev_subscriber_t* o,
					     pp_tp_sensdev_t* source, int reading);

/**
 * initializer
 */
void pp_tp_gpio_loop_init(pp_tp_gpio_loop_t* d,
                          const char* id,
                          pp_tp_gpio_sens_t* gpio_in,
                          pp_tp_gpio_act_t* gpio_out)
{
    pp_tp_ctrl_init(&d->base, PP_TP_GPIO_LOOP, id, gpio_loop_dtor);
    d->subscriber.recv_reading = gpio_loop_receive_sensor_reading;
    d->in = pp_tp_gpio_sens_duplicate(gpio_in);
    d->out = pp_tp_gpio_act_duplicate(gpio_out);
    
    // subscribe at in
    pp_bmc_tp_sensdev_subscribe(&d->in->base, &d->subscriber);

    pp_bmc_log_debug("[GPIO_loop]  '%s' init", id);
}

static void
gpio_loop_receive_sensor_reading(pp_tp_sensdev_subscriber_t* subscriber,
				 pp_tp_sensdev_t* source UNUSED, int reading)
{
    pp_tp_gpio_loop_t* this =
	PP_TP_INTF_2_OBJ_CAST(subscriber, pp_tp_gpio_loop_t, subscriber);
    
    if (reading >= 0) {
	pp_tp_gpio_act_set_gpio(this->out, reading);
    }
}

static void gpio_loop_dtor(pp_tp_obj_t* d) {
    pp_tp_gpio_loop_t* this = (pp_tp_gpio_loop_t*)d;
    if (this != NULL) {
	// unsubscribe
	pp_bmc_tp_sensdev_unsubscribe(&this->in->base, &this->subscriber);
	
	// release in/out objects
	pp_tp_obj_release((pp_tp_obj_t*)this->in);
	pp_tp_obj_release((pp_tp_obj_t*)this->out);
	
	pp_tp_ctrl_cleanup(&this->base);
	free(this);
	pp_bmc_log_debug("[GPIO_loop]  '%s' cleaned up", d->id);
    }
}

pp_tp_obj_t* pp_tp_gpio_loop_ctor(const char* id, vector_t* args) {
    pp_tp_gpio_loop_t* gpio_loop;
    pp_tp_gpio_sens_t* gpio_in;
    pp_tp_gpio_act_t*  gpio_out;

    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;

    if (pp_tp_arg_scanf(args, 0, &err, "o<sg>o<ag>", &gpio_in, &gpio_out)
	!= 2) {
        pp_bmc_log_perror("[GPIO_loop] %s c'tor: %s", id,
			  pp_strstream_buf(&err));
        return NULL;
    }

    gpio_loop = malloc(sizeof(pp_tp_gpio_loop_t));
    
    pp_tp_gpio_loop_init(gpio_loop, id, gpio_in, gpio_out);
    
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)gpio_loop;
}

