/**
 * tp_gpio_sens.c
 *
 * A sensor for one concrete gpio input signal
 * 
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/tp_gpio_sens.h>
#include <pp/bmc/debug.h>
#include <pp/selector.h>

static void gpio_apply_new_value(pp_tp_gpio_sens_t* this, int value);
static void gpio_changed(pp_tp_gpio_subscriber_t* subscriber, unsigned long value, int valid);
static int gpio_debounce_handler(int item_id UNUSED, void* ctx);
static u_long get_gpio_mask(pp_tp_gpio_subscriber_t* subscriber);
static void gpio_sens_dtor(pp_tp_obj_t* this);

pp_tp_obj_t*
pp_tp_gpio_sens_ctor(const char* id, vector_t* args)
{
    const char* errmsg = "[GPIO_in] %s: '%s' failed (%s)";
    pp_tp_gpio_sens_t* gpio_sens = NULL;
    pp_tp_gpio_dev_t* gpio_dev;
    int gpio_num;
    int gpio_sensvert;
    int debounce;
    u_char sens_type = IPMI_SENSOR_TYPE_BUTTON;
    u_char reading_type = IPMI_EVENT_READING_TYPE_DISCRETE_STATE;
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;

    if (pp_tp_arg_scanf(args, 0, &err, "o<g>dbd", &gpio_dev, &gpio_num, 
			&gpio_sensvert, &debounce) != 4) {
	pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
	goto bail;
    }  else if (vector_size(args) > 4 &&
		pp_tp_arg_scanf(args, 4, &err, "d<c>d<c>", &sens_type,
				&reading_type) != 2) {
        pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
    }
    
    gpio_sens = calloc(1, sizeof(pp_tp_gpio_sens_t));
    if (PP_FAILED(pp_tp_gpio_sens_init(gpio_sens, PP_TP_GPIO_SENS, id,
				       gpio_sens_dtor, NULL,
				       gpio_dev, gpio_num, gpio_sensvert,
				       debounce))) {
	pp_bmc_log_error(errmsg, ___F, id, "init");
	free(gpio_sens);
	gpio_sens = NULL;
    }

 bail:
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)gpio_sens;
}

static void gpio_sens_dtor(pp_tp_obj_t* this) {
    pp_tp_gpio_sens_cleanup((pp_tp_gpio_sens_t *)this);
    free(this);
}

int
pp_tp_gpio_sens_init(pp_tp_gpio_sens_t* this, pp_tp_obj_type_t type,
		     const char* id, pp_tp_obj_dtor_func_t dtor,
		     pp_tp_sensdev_default_sdr_func_t default_sdr,
		     pp_tp_gpio_dev_t* gpio_dev, u_char gpio_no,
		     u_char invert_signal, int debounce)
{
    pp_bmc_log_debug("[GPIO_in]  init '%s' (%s), gpio_no %d, "
		     "sensor type 0x%02x, reading type 0x%02x", 
		     id, gpio_dev->base.id, gpio_no);
    
    if (PP_SUCCED(pp_tp_sensdev_init(&this->base, type, id, dtor,
				     default_sdr))) {
	this->invert = invert_signal;
	this->debounce = debounce;
	this->debounce_hdl = 0;
	this->gpio_dev = pp_tp_gpio_dev_duplicate(gpio_dev);
	this->gpio_no = gpio_no;
	this->gpio_subscriber_interface.gpio_changed = gpio_changed;
	this->gpio_subscriber_interface.get_gpio_mask = get_gpio_mask;

	pp_tp_gpio_dev_init_gpio(this->gpio_dev, 1 << this->gpio_no);

	/* set gpio to tristate/input */
	((pp_tp_gpio_dev_t*)(this->gpio_dev))->set_gpio((pp_tp_gpio_dev_t*)(this->gpio_dev),
							this->gpio_no, 'z');
	
	// subscribe as gpio listener
	pp_tp_gpio_dev_register_subscriber(this->gpio_dev,
					   &this->gpio_subscriber_interface);
	return PP_SUC;
    }
    return PP_ERR;
}

void
pp_tp_gpio_sens_cleanup(pp_tp_gpio_sens_t* this)
{
    // unregister from the gpio bank
    pp_tp_gpio_dev_unregister_subscriber(this->gpio_dev,
					 &this->gpio_subscriber_interface);
    // and release it
    pp_tp_gpio_dev_release(this->gpio_dev);

    pp_bmc_log_debug("[GPIO_in]  '%s' cleanup", ((pp_tp_obj_t*)this)->id);

    pp_tp_sensdev_cleanup(&this->base);
}

static void
gpio_apply_new_value(pp_tp_gpio_sens_t* this, int value)
{
    

    // always notify subscribers (not only if when value change detected), because:
    // value could has been changed back again in the meanwhile
    // but some events must not been missed (e.g. RESET)
    pp_bmc_log_debug("[GPIO_in]  GPIO '%s' has new reading %d", 
                     this->base.base.id, value);
    this->base.reading = value;
    pp_tp_gpio_sens_notify_subscribers(this, value);
}

/**
 * The gpio_changed function for the gpio_subscriber interface
 */
static void
gpio_changed(pp_tp_gpio_subscriber_t* s, unsigned long value, int valid)
{
    int new_reading;
    pp_tp_gpio_sens_t* this;
    this = PP_TP_INTF_2_OBJ_CAST(s, pp_tp_gpio_sens_t,
				 gpio_subscriber_interface);
    
    if (!valid) {
	new_reading = -1;
    } else {
	int bitval = (value >> this->gpio_no) & 0x1;
	
	new_reading = this->invert ? pp_tp_gpio_sens_invert(bitval) : bitval;
    }

    if (this->debounce == 0) {
	/* use the new value directly */
	gpio_apply_new_value(this, new_reading);
    } else {
	/* setup timer but don't change value yet */
	this->value_pending = new_reading;

	if (this->debounce_hdl != 0) pp_select_remove_to(this->debounce_hdl);
	this->debounce_hdl = pp_select_add_to(this->debounce, 0,
					      gpio_debounce_handler, (void*)this);
    }
}

/**
 * Timeout handler for debouncing the signal
 */
static int
gpio_debounce_handler(int item_id UNUSED, void* ctx)
{
    pp_tp_gpio_sens_t* this = (pp_tp_gpio_sens_t*)ctx;

    gpio_apply_new_value(this, this->value_pending);
    this->debounce_hdl = 0;
    
    return PP_SUC;
}


/**
 * The get_gpio_mask function for the gpio_subscriber interface
 */
static u_long
get_gpio_mask(pp_tp_gpio_subscriber_t* subscriber)
{
    pp_tp_gpio_sens_t* this;
    this = PP_TP_INTF_2_OBJ_CAST(subscriber, pp_tp_gpio_sens_t,
				 gpio_subscriber_interface);
    return 1 << this->gpio_no;
}


