/**
 * tp_i2c_alive_cond.c
 *
 * Tests whether i2c address is alive on the given bus
 * 
 * (c) 2005 Peppercon AG, 2005/11/4, tbr@peppecon.de
 */

#include <pp/base.h>
#include <pp/selector.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_i2c_comdev.h>
#include <pp/bmc/tp_i2c_alive_cond.h>
#include "sensor_scanner.h"

typedef struct {
    pp_tp_cond_t base;
    pp_tp_i2c_comdev_t* i2cdev;
    unsigned char i2caddr;
    
    /* regarding scannable, refer to docu in tp_scan_sensdev.h */
    pp_sensor_scannable_t scannable;
    pthread_mutex_t mtx;
    int update_sf_id;
} pp_tp_i2c_alive_cond_t;

static void i2c_alive_cond_dtor(pp_tp_obj_t* o);
static void i2c_alive_cond_update(pp_sensor_scannable_t* s);
static int  i2c_alive_cond_notify(void* ctx);

pp_tp_obj_t* pp_tp_i2c_alive_cond_ctor(const char* id, vector_t* args) {
    const char* errmsg = "%s(): '%s' failed: %s";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_tp_i2c_alive_cond_t* this = NULL;
    pp_tp_i2c_comdev_t* i2cdev;
    int i2caddr;

    if (pp_tp_arg_scanf(args, 0, &err, "o<i>d", &i2cdev, &i2caddr) != 2) {
	pp_bmc_log_perror(errmsg, ___F, id, pp_strstream_buf(&err));
    } else {
	this = malloc(sizeof(pp_tp_i2c_alive_cond_t));
	pp_tp_cond_init(&this->base, PP_TP_COND, id, i2c_alive_cond_dtor);
	this->i2cdev = pp_tp_i2c_comdev_duplicate(i2cdev);
	this->i2caddr = i2caddr;
	this->scannable.update = i2c_alive_cond_update;
	this->update_sf_id = 0;
	MUTEX_CREATE_ERRORCHECKING(&this->mtx);
	pp_sensor_scanner_add(&this->scannable, 0);
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

void i2c_alive_cond_dtor(pp_tp_obj_t* o) {
    pp_tp_i2c_alive_cond_t* this = (pp_tp_i2c_alive_cond_t*)o;
    assert(this != NULL);
    pp_sensor_scanner_remove(&this->scannable);
    if (this->update_sf_id != 0) {
	pp_select_remove_sf(this->update_sf_id);
    }
    pp_tp_i2c_comdev_release(this->i2cdev);
    pp_tp_cond_cleanup(&this->base);
    free(this);
}

void i2c_alive_cond_update(pp_sensor_scannable_t* s) {
    pp_tp_i2c_alive_cond_t* this =
	PP_TP_INTF_2_OBJ_CAST(s, pp_tp_i2c_alive_cond_t, scannable);
    pp_tp_i2c_comdev_t* i2cdev = this->i2cdev;
    unsigned char i2caddr = this->i2caddr;
    unsigned char data;
    int reading = 0;
    
    
    if (PP_SUC == pp_tp_i2c_comdev_pre_com(i2cdev, i2caddr)) {
	if (PP_SUC == pp_tp_i2c_comdev_rx_byte(i2cdev, i2caddr, &data)) {
	    reading = 1;
	}
	pp_tp_i2c_comdev_post_com(i2cdev, i2caddr);
    }

    MUTEX_LOCK(&this->mtx);
    if (this->update_sf_id == 0) {
	if (this->base.base.reading != reading) {
	    this->base.base.reading = reading;
	    this->update_sf_id = pp_select_add_sf(i2c_alive_cond_notify, this);
	}
    }
    MUTEX_UNLOCK(&this->mtx);
}

static int i2c_alive_cond_notify(void* ctx) {
    pp_tp_i2c_alive_cond_t* this = (pp_tp_i2c_alive_cond_t*)ctx;
    MUTEX_LOCK(&this->mtx);
    pp_bmc_tp_sensdev_notify_subscribers(&this->base.base,
					 this->base.base.reading);
    this->update_sf_id = 0;
    MUTEX_UNLOCK(&this->mtx);
    return 0;
}
