/**
 * tp_i2c_chip
 *
 * Defines an abstract i2c communication endpoint.
 * It is quite equivalent to an i2c communication device
 * except that the i2c address is known and fix.
 *
 * Main intention for introducing this class is to have an
 * object instance for Multi-Sensor-Chips that need
 * power-on-initialization, for instance setting up registers and enabling it
 * 
 * (c) 2005 Peppercon AG, 2005/10/27, tbr@peppecon.de
 */

#include <pp/base.h>
#include <pp/bmc/tp_i2c_chip.h>
#include <pp/bmc/topo_factory.h>

#if !defined(NDEBUG)
#define DBG(fmt, x...)	pp_log("tp_i2c_chip.c: " fmt, ##x)
#else /* NDEBUG */
#define DBG(fmt, x...)	((void)0)
#endif /* NDEBUG */

static void
power_cond_recv_reading(pp_tp_sensdev_subscriber_t* subscriber,
		        pp_tp_sensdev_t* source UNUSED, int reading) {
    pp_tp_i2c_chip_t* this;
    this = PP_TP_INTF_2_OBJ_CAST(subscriber, pp_tp_i2c_chip_t, cond_subs);
    if (reading > 0) {
	if (this->power_up_init) this->power_up_init(this);
	this->initialized = 1;
    } else if (reading == 0) {
	this->initialized = 0;
        if (this->power_down_cleanup) this->power_down_cleanup(this);
    }
}

static void i2c_chip_dtor(pp_tp_obj_t* o) {
    pp_tp_i2c_chip_t* this = (pp_tp_i2c_chip_t*)o;
    assert(this != NULL);
    pp_tp_i2c_chip_cleanup(this);
    free(this);
}

pp_tp_obj_t* pp_tp_i2c_chip_ctor(const char* id, vector_t* args) {
    const char* errmsg = "[I2CChip] %s c'tor: %s";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_tp_i2c_chip_t* this = NULL;
    pp_tp_i2c_comdev_t* i2cdev;
    int i2caddr;

    if (pp_tp_arg_scanf(args, 0, &err, "o<i>d", &i2cdev, &i2caddr) != 2) {
	pp_bmc_log_perror(errmsg, id, pp_strstream_buf(&err));
    } else {
	this = malloc(sizeof(pp_tp_i2c_chip_t));
	pp_tp_i2c_chip_init(this, PP_TP_I2C_CHIP, id, i2c_chip_dtor,
			    PP_TP_I2C_CHIP_GENERIC, i2cdev, i2caddr,
			    NULL, NULL, NULL);
    }
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)this;
}

void pp_tp_i2c_chip_init(pp_tp_i2c_chip_t* this, pp_tp_obj_type_t type,
			 const char* id, void (*dtor)(pp_tp_obj_t*),
			 const char* model, pp_tp_i2c_comdev_t* i2cdev,
			 unsigned char i2caddr, pp_tp_cond_t* init_cond,
			 void (*power_up_init)(pp_tp_i2c_chip_t*),
                         void (*power_down_cleanup)(pp_tp_i2c_chip_t*))
{
    pp_tp_obj_init((pp_tp_obj_t*)this, type, id, dtor);
    this->model = model;
    this->i2cdev = pp_tp_i2c_comdev_duplicate(i2cdev);
    this->i2caddr = i2caddr;    
    if(init_cond != NULL){
    	this->init_cond = pp_tp_cond_duplicate(init_cond);
    }
    this->initialized = 0;

    this->power_up_init = power_up_init;

    this->power_down_cleanup = power_down_cleanup;


    /* register for init condition to call init code */
    if(init_cond != NULL){
    	this->cond_subs.recv_reading = power_cond_recv_reading;
    	pp_tp_cond_subscribe(init_cond, &this->cond_subs);
    }
}

void pp_tp_i2c_chip_cleanup(pp_tp_i2c_chip_t* this) {
    if(this->init_cond != NULL) {			
    	pp_tp_cond_unsubscribe(this->init_cond, &this->cond_subs);
    	pp_tp_cond_release(this->init_cond);
    }
    pp_tp_i2c_comdev_release(this->i2cdev);
    pp_tp_obj_cleanup((pp_tp_obj_t*)this);
}

