/**
 * tp_i2c_mux.c
 *
 * An abstraction for i2c multiplexer devices
 * 
 * (c) 2005 Peppercon AG, 6/14/2005, miba@peppercon.de
 */

#include <pp/base.h>
#include <pp/bmc/tp_i2c_mux.h>
#include <pp/bmc/debug.h>

static int i2c_mux_root_handle(pp_tp_i2c_comdev_t* d) {
    pp_tp_i2c_mux_t *dev = (pp_tp_i2c_mux_t*) d;
    assert(dev->parent);
    return pp_tp_i2c_comdev_root_handle(dev->parent);
}

static int i2c_mux_pre_com(pp_tp_i2c_comdev_t* d,
			   unsigned char i2caddr)
{
    pp_tp_i2c_mux_t *dev = (pp_tp_i2c_mux_t*) d;

    assert(dev);
    assert(dev->parent);
    assert(dev->parent->pre_com);

    if (PP_FAILED(dev->parent->pre_com((pp_tp_i2c_comdev_t*)dev->parent,
				       i2caddr))) {
	return PP_ERR;
    }
    
    if (PP_FAILED(dev->switch_channel(dev))) {
	dev->parent->post_com((pp_tp_i2c_comdev_t*)dev->parent, i2caddr);
	return PP_ERR;
    }

    return PP_SUC;
}


static int i2c_mux_post_com(pp_tp_i2c_comdev_t* d,
			    unsigned char i2caddr)
{
    pp_tp_i2c_mux_t *dev = (pp_tp_i2c_mux_t*) d;

    if (dev->disable_channels && PP_FAILED(dev->disable_channels(dev))) {
	dev->parent->post_com((pp_tp_i2c_comdev_t*)dev->parent, i2caddr);
	return PP_ERR;
    }

    return dev->parent->post_com((pp_tp_i2c_comdev_t*)dev->parent, i2caddr);
}

static int i2c_mux_rx_byte(pp_tp_i2c_comdev_t* d,
			   unsigned char i2caddr,
			   unsigned char* data)
{
    pp_tp_i2c_mux_t *dev = (pp_tp_i2c_mux_t*) d;

    return dev->parent->rx_byte((pp_tp_i2c_comdev_t*)dev->parent, i2caddr, data);
}

static int i2c_mux_tx_byte(pp_tp_i2c_comdev_t* d,
			   unsigned char i2caddr,
			   unsigned char data) 
{
    pp_tp_i2c_mux_t *dev = (pp_tp_i2c_mux_t*) d;

    return dev->parent->tx_byte((pp_tp_i2c_comdev_t*)dev->parent, i2caddr, data);
}

static int i2c_mux_rx_byte_data(pp_tp_i2c_comdev_t* d,
				unsigned char i2caddr,
				unsigned char reg, unsigned char* data)
{
    pp_tp_i2c_mux_t *dev = (pp_tp_i2c_mux_t*) d;

    return dev->parent->rx_byte_data((pp_tp_i2c_comdev_t*)dev->parent, i2caddr, reg, data);
}

static int i2c_mux_tx_byte_data(pp_tp_i2c_comdev_t* d,
				unsigned char i2caddr,
				unsigned char reg, unsigned char data) 
{
    pp_tp_i2c_mux_t *dev = (pp_tp_i2c_mux_t*) d;

    return dev->parent->tx_byte_data((pp_tp_i2c_comdev_t*)dev->parent, i2caddr, reg, data);
}

static int i2c_mux_rx_word_data(pp_tp_i2c_comdev_t* d,
				unsigned char i2caddr,
				unsigned char reg, unsigned short* data)
{
    pp_tp_i2c_mux_t *dev = (pp_tp_i2c_mux_t*) d;

    return dev->parent->rx_word_data((pp_tp_i2c_comdev_t*)dev->parent, i2caddr, reg, data);
}

static int i2c_mux_tx_word_data(pp_tp_i2c_comdev_t* d,
				unsigned char i2caddr,
				unsigned char reg, unsigned short data) 
{
    pp_tp_i2c_mux_t *dev = (pp_tp_i2c_mux_t*) d;

    return dev->parent->tx_word_data((pp_tp_i2c_comdev_t*)dev->parent, i2caddr, reg, data);
}
void pp_tp_i2c_mux_init(pp_tp_i2c_mux_t* d,
			pp_tp_obj_type_t type,
			const char* id,
			void (*dtor)(pp_tp_obj_t*),
			int (*switch_channel)(struct pp_tp_i2c_mux_s*),
			int (*disable_channels)(struct pp_tp_i2c_mux_s*),
			pp_tp_i2c_comdev_t *parent,
			u_char i2caddr,
			u_char channel)
{
    pp_bmc_log_debug("[PP_I2C_MUX] init %s channel %d", id, channel);
    
    /* call super */
    pp_tp_i2c_comdev_init((pp_tp_i2c_comdev_t*)d, type, id,
                          dtor,
			  i2c_mux_root_handle,
                          i2c_mux_pre_com,
                          i2c_mux_post_com,
			  i2c_mux_rx_byte,
                          i2c_mux_tx_byte,
                          i2c_mux_rx_byte_data,
                          i2c_mux_tx_byte_data,
                          i2c_mux_rx_word_data,
                          i2c_mux_tx_word_data);

    d->switch_channel = switch_channel;
    d->disable_channels = disable_channels;
    d->parent = (pp_tp_i2c_comdev_t*)pp_tp_obj_duplicate((pp_tp_obj_t*)parent);
    d->channel = channel;
    d->i2caddr = i2caddr;
}

void pp_tp_i2c_mux_cleanup(pp_tp_i2c_mux_t* d)
{
    pp_bmc_log_debug("[PP_I2C_MUX] cleanup %s channel %d",
		     ((pp_tp_obj_t*)d)->id, d->channel);
    
    pp_tp_obj_release((pp_tp_obj_t*)(d->parent));
    
    pp_tp_i2c_comdev_cleanup((pp_tp_i2c_comdev_t*)d);
}
