/**
 * tp_ipmi_sens.c
 *
 * A generic IPMI sensor object based on a sensdev object.
 * 
 * (c) 2005 Peppercon AG, 3/9/2005, tbr@peppecon.de
 */

#include <pp/vector.h>
#include <pp/cfg.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/bmc_event.h>
#include <pp/bmc/ipmi_chan.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_cmd_sensor.h>

#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/topo_classes.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_ipmi_sens.h>
#include <pp/bmc/tp_cond.h>

/* internal prototypes */
static pp_tp_ipmi_sens_t*
thresh_ipmi_sens_create(const char* id, pp_tp_sensdev_t* sensor,
			unsigned char snum, ipmi_sdr_header_t* defsdr, 
			pp_tp_cond_t* presence);
static int thresh_isen_set_sdr(pp_tp_ipmi_sens_t* s,ipmi_sdr_header_t* sdr);
static int thresh_isen_get_reading(pp_tp_ipmi_sens_t *s,
				   unsigned char* resp_code,
				   ipmi_sensor_get_reading_rs_t* resp_data);
static int thresh_isen_set_hysteresis(pp_tp_ipmi_sens_t* s,
				      ipmi_set_sensor_hysteresis_rq_t* hyst);
static int thresh_isen_get_hysteresis(pp_tp_ipmi_sens_t* s,
				      ipmi_get_sensor_hysteresis_rs_t* hyst);
static int thresh_isen_set_thresholds(pp_tp_ipmi_sens_t* s,
				      ipmi_set_sensor_thresholds_rq_t* thresh);
static int thresh_isen_get_thresholds(pp_tp_ipmi_sens_t* s,
				      ipmi_get_sensor_thresholds_rs_t* thresh);
static int thresh_isen_set_event_enable(pp_tp_ipmi_sens_t* s,
					ipmi_set_sensor_event_enable_rq_t* rq);
static int thresh_isen_get_event_enable(pp_tp_ipmi_sens_t* s,
					ipmi_get_sensor_event_enable_rs_t* rs);
static int thresh_isen_rearm_events(pp_tp_ipmi_sens_t* s,
				    ipmi_rearm_sensor_events_rq_t* rq);
static int thresh_isen_get_event_status(pp_tp_ipmi_sens_t* s,
					ipmi_sensor_get_event_status_rs_t* rs);
static int thresh_isen_get_reading_factors(pp_tp_ipmi_sens_t* s,
					   unsigned char reading,
					   ipmi_get_sensor_reading_factors_rs_t* rs);
static signed char thresh_isen_get_reading_offset(pp_tp_ipmi_sens_t* s);
static int thresh_isen_set_reading_offset(pp_tp_ipmi_sens_t* s,
					  signed char offset);

static void thresh_isen_dtor(pp_tp_obj_t* o);
static void thresh_isen_setup_for_sdr(pp_tp_thresh_ipmi_sens_t* isens,
				      ipmi_sdr_header_t* sdr);
static void thresh_isen_check_thresh(pp_tp_thresh_ipmi_sens_t* o, 
				     unsigned char reading, 
				     unsigned char threshold,
				     unsigned short hi_event, 
				     unsigned short lo_event,
				     unsigned char sensor_type, 
				     unsigned char sensor_no);
static void thresh_isen_fire_assertion_event(pp_tp_thresh_ipmi_sens_t* o, 
					     unsigned short event,
					     unsigned char reading, 
					     unsigned char threshold,
					     unsigned char sensor_type, 
					     unsigned char sensor_no);
static void thresh_isen_fire_deassertion_event(pp_tp_thresh_ipmi_sens_t* o,
					       unsigned short event,
					       unsigned char reading, 
					       unsigned char threshold,
					       unsigned char sensor_type, 
					       unsigned char sensor_no);
static void thresh_isen_recv_reading(pp_tp_sensdev_subscriber_t * subs,
				     pp_tp_sensdev_t* source, int reading);
static void thresh_isen_recv_presence_reading(pp_tp_sensdev_subscriber_t * subs,
					      pp_tp_sensdev_t* source, int reading);
static void thresh_isen_load_sensor_offset(pp_tp_thresh_ipmi_sens_t* this);
static void thresh_isen_save_sensor_offset(pp_tp_thresh_ipmi_sens_t* this);
static void thresh_isen_check_events(pp_tp_thresh_ipmi_sens_t* this, int reading);
static unsigned char sdr_sens_type(ipmi_sdr_header_t* sdr);
static unsigned char sdr_event_type(ipmi_sdr_header_t* sdr);

static pp_tp_ipmi_sens_vtable_t thresh_isen_vt = {
    .set_sdr                 = thresh_isen_set_sdr,
    .get_reading             = thresh_isen_get_reading,
    .set_hysteresis          = thresh_isen_set_hysteresis,
    .get_hysteresis          = thresh_isen_get_hysteresis,
    .set_thresholds          = thresh_isen_set_thresholds,
    .get_thresholds          = thresh_isen_get_thresholds,
    .set_event_enable        = thresh_isen_set_event_enable,
    .get_event_enable        = thresh_isen_get_event_enable,
    .rearm_events            = thresh_isen_rearm_events,
    .get_event_status        = thresh_isen_get_event_status,
    .get_reading_factors     = thresh_isen_get_reading_factors,
    .get_reading_offset	     = thresh_isen_get_reading_offset,
    .set_reading_offset	     = thresh_isen_set_reading_offset,    
};

/* TODO
   - maybe we should split the implementation of both sensor types, getting crowded here
   - also there are some redundant things which could be merged
*/

static pp_tp_ipmi_sens_t*
disc_ipmi_sens_create(const char* id, pp_tp_sensdev_t* sensor,
		      unsigned char snum, ipmi_sdr_header_t* defsdr, 
		      pp_tp_cond_t* presence);
static void disc_isen_dtor(pp_tp_obj_t* o);
static void disc_isen_setup_for_sdr(pp_tp_disc_ipmi_sens_t* isens,
				    ipmi_sdr_header_t* sdr);
static int disc_isen_set_sdr(pp_tp_ipmi_sens_t* s, ipmi_sdr_header_t* sdr);
static int disc_isen_get_reading(pp_tp_ipmi_sens_t *s,
				 unsigned char* resp_code,
				 ipmi_sensor_get_reading_rs_t* resp_data);
static int disc_isen_set_event_enable(pp_tp_ipmi_sens_t* s,
				      ipmi_set_sensor_event_enable_rq_t* rq);
static int disc_isen_get_event_enable(pp_tp_ipmi_sens_t* s,
				      ipmi_get_sensor_event_enable_rs_t* rs);
static int disc_isen_rearm_events(pp_tp_ipmi_sens_t* s,
				  ipmi_rearm_sensor_events_rq_t* rq);
static int disc_isen_get_event_status(pp_tp_ipmi_sens_t* s,
				      ipmi_sensor_get_event_status_rs_t* rs);
static void disc_isen_check_events(pp_tp_disc_ipmi_sens_t* this, int reading);
static void disc_isen_check_state(pp_tp_disc_ipmi_sens_t* this, 
				  int reading,
				  unsigned char state,
				  unsigned char sensor_type, 
				  unsigned char sensor_no,
				  unsigned char reading_type);
static void disc_isen_fire_assertion_event(pp_tp_disc_ipmi_sens_t* o, 
					   unsigned short event,
					   unsigned char sensor_type, 
					   unsigned char sensor_no,
					   unsigned char reading_type);
static void disc_isen_fire_deassertion_event(pp_tp_disc_ipmi_sens_t* o,
					     unsigned short event,
					     unsigned char sensor_type, 
					     unsigned char sensor_no,
					     unsigned char reading_type);
static void disc_isen_recv_reading(pp_tp_sensdev_subscriber_t * subs,
				   pp_tp_sensdev_t* source, int reading);
static void disc_isen_recv_presence_reading(pp_tp_sensdev_subscriber_t * subs,
					    pp_tp_sensdev_t* source, int reading);


static pp_tp_ipmi_sens_vtable_t disc_isen_vt = {
    .set_sdr                 = disc_isen_set_sdr,
    .get_reading             = disc_isen_get_reading,
    .set_hysteresis          = NULL,
    .get_hysteresis          = NULL,
    .set_thresholds          = NULL,
    .get_thresholds          = NULL,
    .set_event_enable        = disc_isen_set_event_enable,
    .get_event_enable        = disc_isen_get_event_enable,
    .rearm_events            = disc_isen_rearm_events,
    .get_event_status        = disc_isen_get_event_status,
    .get_reading_factors     = NULL,
    .get_reading_offset	     = NULL,
    .set_reading_offset      = NULL
};

/*
 * public implementation
 * ------------------------
 */

/* 
 * see header for ctor argument documentation
 * 
 * Philosophy of (SDR) data held in the IPMI-Sensor object is as follows:
 * 
 * - The type of a sensor and all that is needed to interpret readings 
 *   is defined in the sensor driver (pp_tp_sensdev_t). The sensor
 *   does this by offering a function to create a partially filled
 *   'Default-SDR'. This function is then called by each ipmi sensor
 *   at startup to create it's own instance of the default SDR. The
 *   default SDRs are still partially defined at that time.
 * 
 * - The Default SDR is then filled with data from the both Config SDR
 *   and the ISens parameters. The config SDR mainly contains generic
 *   information like thresholds and event masks that can be shared
 *   among several IPMI sensors. The SDR data from the ISens contains
 *   instance-specific information like sensor number, name, entity id
 *   and entity instance. This information will complete the partially
 *   defined SDR from the sensdev and form a complete 'Default-SDR'.
 *   
 * - During SDRR initialization the Default SDR is requested and
 *   checked. The SDRR may then assign another SDR to the Isensor
 *   with set_sdr(). In this case the default SDR will be completely
 *   replaced with the new SDR.
 * 
 * - If the SDR is requested a pointer to that SDR is passed without
 *   passing responsibility for its memory. The SDR belongs to the
 *   IPMI sensor.
 */
pp_tp_obj_t* pp_tp_ipmi_sens_ctor (const char* id, vector_t* args) {
    const char* cname = "[ISen]";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_tp_ipmi_sens_t* o = NULL;
    pp_tp_sensdev_t* sensor;
    pp_tp_cfg_sdr_t* sdrcfg;
    pp_tp_cond_t* presence = NULL;
    ipmi_sdr_header_t* sdr;
    unsigned char snum, rtype;
    char* desc;
    unsigned char entity_id, entity_inst;

    // scan args
    if (pp_tp_arg_scanf(args, 0, &err, "o<s>d<c>sd<c>d<c>o<r>|o<sc>",
			&sensor, &snum, &desc, &entity_id, &entity_inst,
			&sdrcfg, &presence) != 7) {
        pp_bmc_log_perror("%s (%s) failed: %s ",
			  cname, id, pp_strstream_buf(&err));
        goto finally;
    }
    
    // create default sdr
    if (NULL == (sdr = pp_bmc_tp_sensdev_create_default_sdr(sensor))) {
	pp_bmc_log_error("%s (%s) ctor failed: sensor device %s has no SDR !",
			 cname, id, pp_tp_obj_to_string(&sensor->base));
        goto finally;
    }
    
    // copy config sdr to default SDR
    if ((rtype = sdr_event_type(sdr)) == IPMI_EVENT_READING_TYPE_THRESHOLD) {
	if (!PP_TP_OBJ_IS_TYPE(PP_TP_THRESH_SDR, sdrcfg)) {
	    pp_bmc_log_error("%s (%s) ctor failed: SDR (%s) isn't threshold type!",
			     cname, id, ((pp_tp_obj_t*)sdrcfg)->id);
	    goto finally;
	}
	if (pp_tp_thresh_cfg_sdr_fill_sdr((pp_tp_thresh_sdr_t*)sdrcfg, snum,
					  desc, entity_id, entity_inst, sdr)
	    == PP_ERR) {
	    pp_bmc_log_error("%s (%s) ctor failed: default sdr does not match "
			     "config sdr", cname, id);
	    goto finally;
	}	
	o = thresh_ipmi_sens_create(id, sensor, snum, sdr, presence);
    } else if (ipmi_sdr_event_type_is_discrete(rtype)) {
	if (!PP_TP_OBJ_IS_TYPE(PP_TP_DISC_SDR, sdrcfg)) {
	    pp_bmc_log_error("%s (%s) ctor failed: SDR (%s) isn't discrete type!",
			     cname, id, ((pp_tp_obj_t*)sdrcfg)->id);
	    goto finally;
	}
	if (pp_tp_disc_cfg_sdr_fill_sdr((pp_tp_disc_sdr_t*)sdrcfg, snum,
                                    desc, entity_id, entity_inst, sdr)
	    == PP_ERR) {
	    pp_bmc_log_error("%s (%s) ctor failed: default sdr does not match "
			     "config sdr", cname, id);
	    goto finally;
	}      	
	o = disc_ipmi_sens_create(id, sensor, snum, sdr, presence);
    } else {
	pp_bmc_log_error("%s (%s) failed: reading type %d (%s) not "
			 "supported!", cname, id, rtype,
			 pp_tp_obj_to_string(&sensor->base));
    }
    if (NULL == o) free(sdr);
 finally:
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)o;
}

pp_tp_obj_t* pp_tp_ipmi_sens_full_sdr_ctor (const char* id, vector_t* args) {
    const char* cname = "[ISenFullSDR]";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_tp_ipmi_sens_t* o = NULL;
    ipmi_sdr_header_t* sdr;
    pp_tp_sensdev_t* sensor;
    pp_tp_sdr_init_values_t sv;   
    pp_tp_cond_t*  presence = NULL;
    unsigned char rtype;
    

    // scan args
    if (pp_tp_arg_scanf(args, 0, &err,
			"o<s>d<c>sd<c>d<c>d<c>d<c>d<c>d<s>d<s>d<c>d<c>d<c>d<c>"
			"d<c>d<c>d<c>d<c>d<c>d<s>d<c>d<s>d<s>d<c>d<c>d<c>d<c>"
			"d<c>d<c>d<c>d<c>d<c>d<c>d<c>d<c>d<c>d<c>d<c>d<c>d<c>d<c>"
			"d<c>|o<sc>",
			&sensor, &sv.num, &sv.ids, &sv.eid, &sv.ein, &sv.cap, &sv.sty,
			&sv.erc, &sv.mask.threshold.ltm, &sv.mask.threshold.utm, &sv.mask.threshold.stm,
			&sv.mask.threshold.rtm, &sv.adf, &sv.run, &sv.mnc, &sv.pct, &sv.bun,
			&sv.mun, &sv.type.full.lin, &sv.type.full.m, &sv.type.full.tol, &sv.type.full.b,
			&sv.type.full.acc, &sv.type.full.aex, &sv.type.full.sed, &sv.type.full.rex,
			&sv.type.full.bex, &sv.type.full.acf, &sv.type.full.nrd, &sv.type.full.nma,
			&sv.type.full.nmi, &sv.type.full.sma, &sv.type.full.smi, &sv.type.full.nru,
			&sv.type.full.cru, &sv.type.full.ncu, &sv.type.full.nrl, &sv.type.full.crl,
			&sv.type.full.ncl, &sv.pos, &sv.neg, &sv.oem, &presence) != 43) {
        pp_bmc_log_perror("%s (%s) failed: %s ",
			  cname, id, pp_strstream_buf(&err));
        goto finally;
    }

    // create sdr
    if (NULL == (sdr = (ipmi_sdr_header_t*) ipmi_sdr_full_create(&sv))) {
	pp_bmc_log_error("%s (%s) ctor failed: sensor device %s has no SDR !",
			 cname, id, pp_tp_obj_to_string(&sensor->base));
        goto finally;
    }

    if ((rtype = sdr_event_type(sdr)) == IPMI_EVENT_READING_TYPE_THRESHOLD) {
	o = thresh_ipmi_sens_create(id, sensor, sv.num, sdr, presence);
    } else if (ipmi_sdr_event_type_is_discrete(rtype)) {
	o = disc_ipmi_sens_create(id, sensor, sv.num, sdr, presence);
    } else {
	pp_bmc_log_error("%s (%s) failed: reading type %d (%s) not "
			 "supported!", cname, id, rtype,
			 pp_tp_obj_to_string(&sensor->base));
    }

 finally:
    return (pp_tp_obj_t*)o; 
}

pp_tp_obj_t* pp_tp_ipmi_sens_compact_sdr_ctor (const char* id, vector_t* args) {
    const char* cname = "[ISenCompactSDR]";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_tp_ipmi_sens_t* o = NULL;
    ipmi_sdr_header_t* sdr;
    pp_tp_sensdev_t* sensor;    
    pp_tp_sdr_init_values_t sv;
    pp_tp_cond_t*  presence = NULL;
    unsigned char rtype;

    // scan args
    if (pp_tp_arg_scanf(args, 0, &err,
			"o<s>d<c>sd<c>d<c>d<c>d<c>d<c>d<s>d<s>d<s>d<c>d<c>"
			"d<c>d<c>d<c>d<c>d<c>d<c>d<c>d<c>d<c>d<c>d<c>d<c>|o<sc>",
			&sensor, &sv.num, &sv.ids, &sv.eid, &sv.ein, &sv.cap, &sv.sty,
			&sv.erc, &sv.mask.discrete.eam, &sv.mask.discrete.edm, &sv.mask.discrete.rdm,
			&sv.adf, &sv.run, &sv.mnc, &sv.pct, &sv.bun, &sv.mun, &sv.type.compact.sed,
			&sv.type.compact.ist, &sv.type.compact.shc, &sv.type.compact.eis,
			&sv.type.compact.iso, &sv.pos, &sv.neg, &sv.oem,
			&presence) != 26) {
        pp_bmc_log_perror("%s (%s) failed: %s ",
			  cname, id, pp_strstream_buf(&err));
        goto finally;
    }

    // create default sdr
    if (NULL == (sdr = (ipmi_sdr_header_t*) ipmi_sdr_compact_create(&sv))) {
	pp_bmc_log_error("%s (%s) ctor failed: sensor device %s has no SDR !",
			 cname, id, pp_tp_obj_to_string(&sensor->base));
        goto finally;
    }

    if ((rtype = sdr_event_type(sdr)) == IPMI_EVENT_READING_TYPE_THRESHOLD) {
	o = thresh_ipmi_sens_create(id, sensor, sv.num, sdr, presence);
    } else if (ipmi_sdr_event_type_is_discrete(rtype)) {
	o = disc_ipmi_sens_create(id, sensor, sv.num, sdr, presence);
    } else {
	pp_bmc_log_error("%s (%s) failed: reading type %d (%s) not "
			 "supported!", cname, id, rtype,
			 pp_tp_obj_to_string(&sensor->base));
    }

finally:
    return (pp_tp_obj_t*)o;
}

static pp_tp_ipmi_sens_t*
thresh_ipmi_sens_create(const char* id, pp_tp_sensdev_t* sensor,
			unsigned char snum, ipmi_sdr_header_t* defsdr, 		
			pp_tp_cond_t* presence)
{
    pp_tp_thresh_ipmi_sens_t* o = NULL;

    o = malloc(sizeof(pp_tp_thresh_ipmi_sens_t));
    pp_tp_ipmi_sens_init(&o->base, PP_TP_THRESH_IPMI_SENS, id, 
			 thresh_isen_dtor, &thresh_isen_vt, sensor, 
			 snum, defsdr, presence,
			 thresh_isen_recv_reading,
			 thresh_isen_recv_presence_reading);

    // setup thresh ipmi sensor for sdr
    thresh_isen_setup_for_sdr(o, defsdr);

    // check for a sensor specific offset
    thresh_isen_load_sensor_offset(o);

    return (pp_tp_ipmi_sens_t*)o;
}

static pp_tp_ipmi_sens_t*
disc_ipmi_sens_create(const char* id, pp_tp_sensdev_t* sensor,
		      unsigned char snum, ipmi_sdr_header_t* defsdr, 
		      pp_tp_cond_t* presence)
{
    pp_tp_disc_ipmi_sens_t* o = NULL;

    o = malloc(sizeof(pp_tp_disc_ipmi_sens_t));
    pp_tp_ipmi_sens_init(&o->base, PP_TP_DISC_IPMI_SENS, id, 
                         disc_isen_dtor, &disc_isen_vt, sensor, 
                         snum, defsdr, presence,
			 disc_isen_recv_reading,
			 disc_isen_recv_presence_reading);

    // setup disc ipmi sensor for sdr
    disc_isen_setup_for_sdr(o, defsdr);
    
    return (pp_tp_ipmi_sens_t*)o;
}

void
pp_tp_ipmi_sens_init(pp_tp_ipmi_sens_t* o, pp_tp_obj_type_t type,
		     const char* id, pp_tp_obj_dtor_func_t dtor,
		     pp_tp_ipmi_sens_vtable_t* vt, 
		     pp_tp_sensdev_t* sensor, u_char number,
		     ipmi_sdr_header_t* sdr, pp_tp_cond_t* presence_cond,
		     pp_tp_sensdev_subscriber_recv_reading_func_t recv_reading,
		     pp_tp_sensdev_subscriber_recv_reading_func_t recv_presence_reading)
{
    pp_tp_obj_init(&o->base, type, id, dtor);
    o->vt = vt;
    o->sensor = pp_tp_sensdev_duplicate(sensor);
    o->number = number;
    o->sdr = sdr; // claim responsibility, was created by create_default_sdr
    o->sensor_subscribers = vector_new(NULL, 0, NULL);
    
    /* register for reading updates with sensor-device */
    o->subscriber.recv_reading = recv_reading;
    pp_bmc_tp_sensdev_subscribe(sensor, &o->subscriber);

    /* subscribe for getting changes of the presence condition */
    o->presence = pp_tp_cond_duplicate(presence_cond);
    if (o->presence != NULL) {
	o->presence_subscriber.recv_reading = recv_presence_reading;
	pp_tp_cond_subscribe(o->presence, &o->presence_subscriber);
    }
}

void pp_tp_ipmi_sens_cleanup(pp_tp_ipmi_sens_t* o) {
    pp_bmc_tp_sensdev_unsubscribe(o->sensor, &o->subscriber);
    vector_delete(o->sensor_subscribers);
    if (o->presence != NULL) {
	pp_tp_cond_unsubscribe(o->presence, &o->presence_subscriber);
	pp_tp_cond_release(o->presence);
    }
    free(o->sdr);
    pp_tp_sensdev_release(o->sensor);
    pp_tp_obj_cleanup(&o->base);
}   

void pp_tp_ipmi_sens_subscribe(pp_tp_ipmi_sens_t* this,
			       pp_tp_ipmi_sens_subscriber_t* subscriber) {
    assert(PP_TP_OBJ_IS_TYPE(PP_TP_IPMI_SENS, this));
    vector_add(this->sensor_subscribers, subscriber);
}

int pp_tp_ipmi_sens_unsubscribe(pp_tp_ipmi_sens_t* this,
				pp_tp_ipmi_sens_subscriber_t* subscriber) {
    size_t i, vs = vector_size(this->sensor_subscribers);
    assert(PP_TP_OBJ_IS_TYPE(PP_TP_IPMI_SENS, this));

    for (i = 0; i < vs; ++i) {
        if (subscriber == vector_get(this->sensor_subscribers, i)) {
            vector_remove(this->sensor_subscribers, i);
            return PP_SUC;
        }
    }
    
    // not found
    return PP_ERR;
}

void pp_tp_ipmi_sens_notify_subscribers(pp_tp_ipmi_sens_t* this, int value) {
    size_t i, vs = vector_size(this->sensor_subscribers);
    pp_tp_ipmi_sens_subscriber_t* subscriber;

    assert(PP_TP_OBJ_IS_TYPE(PP_TP_IPMI_SENS, this));
    
    for (i = 0; i < vs; ++i) {
        subscriber = (pp_tp_ipmi_sens_subscriber_t*)
	    vector_get(this->sensor_subscribers, i);
        subscriber->recv_reading(subscriber, this, value);
    }
}


/*
 * private implementation
 * -----------------------
 */
static void thresh_isen_dtor(pp_tp_obj_t* o) {
    pp_tp_thresh_ipmi_sens_t* this = (pp_tp_thresh_ipmi_sens_t*)o;
    assert(this != NULL);
    assert(PP_TP_OBJ_IS_TYPE(PP_TP_THRESH_IPMI_SENS, this));
    pp_tp_ipmi_sens_cleanup(&this->base);
    free(this);
}

static void thresh_isen_setup_for_sdr(pp_tp_thresh_ipmi_sens_t* isens,
				      ipmi_sdr_header_t* sdr) {
    ipmi_sdr_threshold_t* sdr_thresh;
    ipmi_sdr_evt_mask_t* sdr_mask;
    
    if (NULL != (sdr_thresh = ipmi_sdr_get_threshold_ptr(sdr))) {
	isens->threshold = *sdr_thresh;
	isens->all_events_enabled = 1;
    } else {
	isens->all_events_enabled = 0;
	pp_bmc_log_warn("[ThreshISENS] %s(#%d): no Threshold-SDR (type: %#.2x)"
			", events globally disabled!",
			pp_tp_obj_to_string((pp_tp_obj_t*)isens),
			isens->base.number, sdr->type);
    }

    if (NULL != (sdr_mask = ipmi_sdr_get_event_mask_ptr(sdr))) {
	isens->event.assertion.enabled =
	    le16_to_cpu(sdr_mask->threshold.assert_event_le16);
	isens->event.deassertion.enabled =
	    le16_to_cpu(sdr_mask->threshold.deassert_event_le16);
    } else {
	isens->event.assertion.enabled = 0;
	isens->event.deassertion.enabled = 0;
	pp_bmc_log_warn("[ThreshISENS] %s(#%d): no Event-Mask-SDR "
			"(type: %#.2x), all events disabled!",
			pp_tp_obj_to_string((pp_tp_obj_t*)isens),
			isens->base.number, sdr->type);
    }
    
    isens->event.assertion.fired = 0;
    isens->event.deassertion.fired = 0;
    isens->event.state = 0;

    // TODO how to initialize rearm???
    isens->auto_rearm = 1;
    isens->scanning_enabled = 1;
}

static int thresh_isen_set_sdr(pp_tp_ipmi_sens_t* s, ipmi_sdr_header_t* sdr) {
    int ret;
    if (sdr == NULL) {
	ret = PP_ERR;
    } else if (sdr_sens_type(s->sdr) != sdr_sens_type(sdr) ||
	       sdr_event_type(s->sdr) != sdr_event_type(sdr)) {
	pp_bmc_log_error("[TreshISENS] set_sdr: incompatible SDR sens-type/"
			 "evt-type: current=%#.2x/%#.2 new=%#.2x/%#.2",
			 sdr_sens_type(s->sdr), sdr_event_type(s->sdr),
			 sdr_sens_type(sdr), sdr_event_type(sdr));
	ret = PP_ERR;
    } else {
	assert(PP_TP_OBJ_IS_TYPE(PP_TP_THRESH_IPMI_SENS, s));
	free(s->sdr);
	s->sdr = ipmi_sdr_copy(sdr);
	thresh_isen_setup_for_sdr((pp_tp_thresh_ipmi_sens_t*)s, sdr);
	ret = PP_SUC;
    }
    return ret;
}

static int thresh_isen_get_reading(pp_tp_ipmi_sens_t *s,
				   unsigned char* resp_code,
				   ipmi_sensor_get_reading_rs_t* r) {
    pp_tp_thresh_ipmi_sens_t* this = (pp_tp_thresh_ipmi_sens_t*)s;
    int ret = PP_SUC;
    unsigned char value, readable;
    int rv;

    memset(r, 0, sizeof(ipmi_sensor_get_reading_rs_t));

    r->rt.unavailable  = s->presence ?
	!pp_tp_sensdev_get_reading((pp_tp_sensdev_t*)s->presence) : 0;
    r->rt.scan_enabled = this->scanning_enabled;
    r->rt.evts_enabled = this->all_events_enabled;

    if (r->rt.unavailable) {
	*resp_code = IPMI_ERR_SUCCESS;
	pp_bmc_log_debug("[ThreshISens] '%s' GetReading: unavailable, pres-cond false",
            pp_tp_obj_to_string(&s->base));
    } else if (0 > (rv = pp_tp_sensdev_get_reading(s->sensor))) {
	r->rt.unavailable = 1;
	*resp_code = IPMI_ERR_SUCCESS;
	pp_bmc_log_debug("[ThreshISens] '%s' GetReading: unavailable, invalid reading",
            pp_tp_obj_to_string(&s->base));
    } else {
	/* add sensor specific offset
	 * and saturate warning when too large/small */
	if (rv + this->offset > 255) {
	    pp_bmc_log_warn("[ThreshISens] '%s' GetReading: exceeds maximum (reading=%d, offset=%d)",
			    pp_tp_obj_to_string(&s->base), rv, this->offset);
	    rv = 255;
	}
	if (rv + this->offset < 0) {
	    pp_bmc_log_warn("[ThreshISens] '%s' GetReading: below minimum (reading=%d, offset=%d)",
			    pp_tp_obj_to_string(&s->base), rv, this->offset);

	    rv = 0;
	}

	r->rt.value = value = (unsigned char)rv + this->offset;

	// TODO: make sure sdr_mask_ptr can't be zero
	readable = ipmi_sdr_get_event_mask_ptr(s->sdr)->threshold.read;
	if (readable & IPMI_SENSOR_THRESHOLD_LOWER_NON_CRIT)
	    r->rt.below_lower_non_crit =
		(value <= this->threshold.lower.non_critical);
	if (readable & IPMI_SENSOR_THRESHOLD_LOWER_CRIT)
	    r->rt.below_lower_crit =
		(value <= this->threshold.lower.critical);
	if (readable & IPMI_SENSOR_THRESHOLD_LOWER_NON_RECOV)
	    r->rt.below_lower_non_recov =
		(value <= this->threshold.lower.non_recover);
	if (readable & IPMI_SENSOR_THRESHOLD_UPPER_NON_CRIT)
	    r->rt.above_upper_non_crit =
		(value >= this->threshold.upper.non_critical);
	if (readable & IPMI_SENSOR_THRESHOLD_UPPER_CRIT)
	    r->rt.above_upper_crit =
		(value >= this->threshold.upper.critical);
	if (readable & IPMI_SENSOR_THRESHOLD_UPPER_NON_RECOV)
	    r->rt.above_upper_non_recov =
		(value >= this->threshold.upper.non_recover);

	*resp_code = IPMI_ERR_SUCCESS;
	pp_bmc_log_debug("[ThreshISens] '%s' GetReading: 0x%x/0x%hhx/0x%hhx",
			 pp_tp_obj_to_string(&s->base), value,
			 ((unsigned char*)&r->rt)[1],
			 ((unsigned char*)&r->rt)[2]);
    }
    return ret;
}

static int thresh_isen_set_hysteresis(pp_tp_ipmi_sens_t* s,
				      ipmi_set_sensor_hysteresis_rq_t* hyst) {
    pp_tp_thresh_ipmi_sens_t* this = (pp_tp_thresh_ipmi_sens_t*)s;
    this->threshold.hysteresis.positive = hyst->pos_going;
    this->threshold.hysteresis.negative = hyst->neg_going;
    return PP_SUC;
}

static int thresh_isen_get_hysteresis(pp_tp_ipmi_sens_t* s,
				      ipmi_get_sensor_hysteresis_rs_t* hyst) {
    pp_tp_thresh_ipmi_sens_t* this = (pp_tp_thresh_ipmi_sens_t*)s;
    hyst->pos_going = this->threshold.hysteresis.positive;
    hyst->neg_going = this->threshold.hysteresis.negative;
    return PP_SUC;
}

static int thresh_isen_set_thresholds(pp_tp_ipmi_sens_t* s,
				      ipmi_set_sensor_thresholds_rq_t* rq){
    pp_tp_thresh_ipmi_sens_t* this = (pp_tp_thresh_ipmi_sens_t*)s;
    unsigned char writeable;

    writeable = ipmi_sdr_get_event_mask_ptr(s->sdr)->threshold.set;
    
    if (rq->mask & writeable & IPMI_SENSOR_THRESHOLD_LOWER_NON_CRIT)
        this->threshold.lower.non_critical = rq->lower_non_crit;
#ifndef NDEBUG
    else if(rq->mask & IPMI_SENSOR_THRESHOLD_LOWER_NON_CRIT) {
        pp_bmc_log_warn("lower non critical threshold for sensor %s (%#.2hx) "
                        "is not writeable!", s->base.id, (int)rq->sensor_no);
    }
#endif /* !NDEBUG */
    if (rq->mask & writeable & IPMI_SENSOR_THRESHOLD_LOWER_CRIT)
        this->threshold.lower.critical = rq->lower_crit;
#ifndef NDEBUG
    else if(rq->mask & IPMI_SENSOR_THRESHOLD_LOWER_CRIT) {
        pp_bmc_log_warn("lower critical threshold for sensor %s (%#.2hx) "
                        "is not writeable!", s->base.id, (int)rq->sensor_no);
    }
#endif /* !NDEBUG */
    if (rq->mask & writeable & IPMI_SENSOR_THRESHOLD_LOWER_NON_RECOV)
        this->threshold.lower.non_recover = rq->lower_non_recov;
#ifndef NDEBUG
    else if(rq->mask & IPMI_SENSOR_THRESHOLD_LOWER_NON_RECOV) {
        pp_bmc_log_warn("lower non recoverable threshold for sensor %s (%#.2hx) "
                        "is not writeable!", s->base.id, (int)rq->sensor_no);
    }
#endif /* !NDEBUG */
    if (rq->mask & writeable & IPMI_SENSOR_THRESHOLD_UPPER_NON_CRIT)
        this->threshold.upper.non_critical = rq->upper_non_crit;
#ifndef NDEBUG
    else if(rq->mask & IPMI_SENSOR_THRESHOLD_UPPER_NON_CRIT) {
        pp_bmc_log_warn("upper non critical threshold for sensor %s (%#.2hx) "
                        "is not writeable!", s->base.id, (int)rq->sensor_no);
    }
#endif /* !NDEBUG */
    if (rq->mask & writeable & IPMI_SENSOR_THRESHOLD_UPPER_CRIT)
        this->threshold.upper.critical = rq->upper_crit;
#ifndef NDEBUG
    else if(rq->mask & IPMI_SENSOR_THRESHOLD_UPPER_CRIT) {
        pp_bmc_log_warn("upper critical threshold for sensor %s (%#.2hx) "
                        "is not writeable!", s->base.id, (int)rq->sensor_no);
    }
#endif /* !NDEBUG */
    if (rq->mask & writeable & IPMI_SENSOR_THRESHOLD_UPPER_NON_RECOV)
        this->threshold.upper.non_recover = rq->upper_non_recov;
#ifndef NDEBUG
    else if(rq->mask & IPMI_SENSOR_THRESHOLD_UPPER_NON_RECOV) {
        pp_bmc_log_warn("upper non recoverable threshold for sensor %s (%#.2hx) "
                        "is not writeable!", s->base.id, (int)rq->sensor_no);
    }
#endif /* !NDEBUG */
    return PP_SUC;
}

static int thresh_isen_get_thresholds(pp_tp_ipmi_sens_t* s,
				      ipmi_get_sensor_thresholds_rs_t* rs){
    pp_tp_thresh_ipmi_sens_t* this = (pp_tp_thresh_ipmi_sens_t*)s;
    unsigned char readable;

    readable = ipmi_sdr_get_event_mask_ptr(s->sdr)->threshold.read;
    
    rs->mask = readable;
    if (readable & IPMI_SENSOR_THRESHOLD_LOWER_NON_CRIT)
        rs->lower_non_crit =  this->threshold.lower.non_critical;
    if (readable & IPMI_SENSOR_THRESHOLD_LOWER_CRIT)
        rs->lower_crit = this->threshold.lower.critical;
    if (readable & IPMI_SENSOR_THRESHOLD_LOWER_NON_RECOV)
        rs->lower_non_recov  = this->threshold.lower.non_recover;
    if (readable & IPMI_SENSOR_THRESHOLD_UPPER_NON_CRIT)
        rs->upper_non_crit = this->threshold.upper.non_critical;
    if (readable & IPMI_SENSOR_THRESHOLD_UPPER_CRIT)
        rs->upper_crit = this->threshold.upper.critical;
    if (readable & IPMI_SENSOR_THRESHOLD_UPPER_NON_RECOV)
        rs->upper_non_recov = this->threshold.upper.non_recover;
    return PP_SUC;
}

static int thresh_isen_set_event_enable(pp_tp_ipmi_sens_t* s,
					ipmi_set_sensor_event_enable_rq_t* rq){
    pp_tp_thresh_ipmi_sens_t* this = (pp_tp_thresh_ipmi_sens_t*)s;
    ipmi_sdr_sensor_t* sdr_sensor;

    sdr_sensor = ipmi_sdr_get_sensor_ptr(s->sdr);

    if(sdr_sensor->caps.event_msg == IPMI_SENSOR_CAPS_EVENT_NONE) {
	return PP_ERR;
    }
    
    this->scanning_enabled = rq->scan_enabled;
    this->all_events_enabled = rq->evts_enabled;

    if(sdr_sensor->caps.event_msg == IPMI_SENSOR_CAPS_EVENT_PER_TRESH) {
	if (rq->enable_events) {
	    // enable selected events
	    this->event.assertion.enabled |=
		le16_to_cpu(rq->assertion_events_le16);
	    this->event.deassertion.enabled |= 
		le16_to_cpu(rq->deassertion_events_le16);
	} else if (rq->disable_events) {
	    // disable selected events
	    this->event.assertion.enabled &=
		~le16_to_cpu(rq->assertion_events_le16);
	    this->event.deassertion.enabled &= 
		~le16_to_cpu(rq->deassertion_events_le16);
	}
    }
    return PP_SUC;
}
    
static int thresh_isen_get_event_enable(pp_tp_ipmi_sens_t* s,
					ipmi_get_sensor_event_enable_rs_t* rs){
    pp_tp_thresh_ipmi_sens_t* this = (pp_tp_thresh_ipmi_sens_t*)s;
    
    rs->scan_enabled        = this->scanning_enabled;
    rs->evts_enabled        = this->all_events_enabled;
    rs->assertion_events_le16   = cpu_to_le16(this->event.assertion.enabled);
    rs->deassertion_events_le16 = cpu_to_le16(this->event.deassertion.enabled);
    return PP_SUC;
}

static int thresh_isen_rearm_events(pp_tp_ipmi_sens_t* s,
				    ipmi_rearm_sensor_events_rq_t* rq) {
    pp_tp_thresh_ipmi_sens_t* this = (pp_tp_thresh_ipmi_sens_t*)s;

    if (rq->rearm_only_selected) {
        // re-arm selected events
        this->event.assertion.fired   &= ~le16_to_cpu(rq->assertion_events_le16);
        this->event.deassertion.fired &= ~le16_to_cpu(rq->deassertion_events_le16);
    } else {
        // re-arm all events
        this->event.assertion.fired   = 0;
        this->event.deassertion.fired = 0;
    }
    // TODO: immediately retrigger event generation
    return PP_SUC;
}

static int thresh_isen_get_event_status(pp_tp_ipmi_sens_t* s,
					ipmi_sensor_get_event_status_rs_t* rs){
    pp_tp_thresh_ipmi_sens_t* this = (pp_tp_thresh_ipmi_sens_t*)s;
    
    rs->scan_enabled        = this->scanning_enabled;
    rs->evts_enabled        = this->all_events_enabled;
    rs->unavailable         = 0; // TODO: set a usefull value
    rs->assertion_events_le16    = cpu_to_le16(this->event.assertion.fired);
    rs->deassertion_events_le16 = cpu_to_le16(this->event.deassertion.fired);
    return PP_SUC;
}

/*
 * threshold checking and event generation
 */
static void thresh_isen_recv_reading(pp_tp_sensdev_subscriber_t * subs,
				     pp_tp_sensdev_t* source UNUSED, int reading) {
    pp_tp_ipmi_sens_t* o = PP_TP_INTF_2_OBJ_CAST(subs, pp_tp_ipmi_sens_t,
						 subscriber);
    pp_tp_thresh_ipmi_sens_t* this = (pp_tp_thresh_ipmi_sens_t*)o;
    assert(reading < 256);

    /* don't do anything if invalid reading, sensor scanning disabled,
     * no sdr configured or sensor presence condition exists and false */
    if ((reading < 0) || !this->scanning_enabled ||
	((this->base.presence) && !pp_bmc_tp_cond_is_true(this->base.presence))) {
	return;
    }

    // FIXME: calculate event state even if events are disabled
    if (!this->all_events_enabled) return;

    // TODO: remove this assert, no crash if SDR is wrong, can legally happen!!
    assert(sdr_event_type(this->base.sdr) ==IPMI_EVENT_READING_TYPE_THRESHOLD);

    thresh_isen_check_events(this, reading);
}

static void thresh_isen_recv_presence_reading(pp_tp_sensdev_subscriber_t * subs,
					      pp_tp_sensdev_t* source UNUSED, int reading)
{
    pp_tp_ipmi_sens_t* o = PP_TP_INTF_2_OBJ_CAST(subs, pp_tp_ipmi_sens_t,
						 presence_subscriber);
    pp_tp_thresh_ipmi_sens_t* this = (pp_tp_thresh_ipmi_sens_t*)o;
    int sensor_reading;
    
    sensor_reading = pp_tp_sensdev_get_reading(o->sensor);
    
    if (reading > 0 && sensor_reading >= 0) {
	thresh_isen_check_events(this, sensor_reading);
    }
}

static void thresh_isen_check_events(pp_tp_thresh_ipmi_sens_t* this,
				     int reading)
{
    unsigned char sensor_type;
    unsigned char sensor_no = this->base.number;
    unsigned char thresh_valid;
    
    thresh_valid = ipmi_sdr_get_event_mask_ptr(this->base.sdr)->threshold.read;
    sensor_type = sdr_sens_type(this->base.sdr);
    
/*
    pp_bmc_log_notice("##### check thesh for sens '%s': lower=%d/%d/%d "
 		      "value=%d upper=%d/%d/%d",
	 	      ((pp_tp_obj_t*)this)->id,
		      this->threshold.lower.non_recover,
		      this->threshold.lower.critical,
	              this->threshold.lower.non_critical,
		      reading,
	              this->threshold.upper.non_critical,
	              this->threshold.upper.critical,
	              this->threshold.upper.non_recover);
*/

    if (thresh_valid & IPMI_SENSOR_THRESHOLD_LOWER_NON_CRIT) {
	thresh_isen_check_thresh(this, reading, 
				 this->threshold.lower.non_critical,
				 IPMI_SENSOR_EVENT_LOWER_NON_CRIT_GOING_LO,
				 IPMI_SENSOR_EVENT_LOWER_NON_CRIT_GOING_HI,
				 sensor_type, sensor_no);
    }
    if (thresh_valid & IPMI_SENSOR_THRESHOLD_LOWER_CRIT) {
	thresh_isen_check_thresh(this, reading, 
				 this->threshold.lower.critical,
				 IPMI_SENSOR_EVENT_LOWER_CRIT_GOING_LO,
				 IPMI_SENSOR_EVENT_LOWER_CRIT_GOING_HI,
				 sensor_type, sensor_no);
    }
    if (thresh_valid & IPMI_SENSOR_THRESHOLD_LOWER_NON_RECOV) {
	thresh_isen_check_thresh(this, reading, 
				 this->threshold.lower.non_recover,
				 IPMI_SENSOR_EVENT_LOWER_NON_RECOV_GOING_LO,
				 IPMI_SENSOR_EVENT_LOWER_NON_RECOV_GOING_HI,
				 sensor_type, sensor_no);
    }

    if (thresh_valid & IPMI_SENSOR_THRESHOLD_UPPER_NON_CRIT) {
	thresh_isen_check_thresh(this, reading, 
				 this->threshold.upper.non_critical,
				 IPMI_SENSOR_EVENT_UPPER_NON_CRIT_GOING_LO,
				 IPMI_SENSOR_EVENT_UPPER_NON_CRIT_GOING_HI,
				 sensor_type, sensor_no);
    }
    if (thresh_valid & IPMI_SENSOR_THRESHOLD_UPPER_CRIT) {
	thresh_isen_check_thresh(this, reading, 
				 this->threshold.upper.critical,
				 IPMI_SENSOR_EVENT_UPPER_CRIT_GOING_LO,
				 IPMI_SENSOR_EVENT_UPPER_CRIT_GOING_HI,
				 sensor_type, sensor_no);
    }
    if (thresh_valid & IPMI_SENSOR_THRESHOLD_UPPER_NON_RECOV) {
	thresh_isen_check_thresh(this, reading, 
				 this->threshold.upper.non_recover,
				 IPMI_SENSOR_EVENT_UPPER_NON_RECOV_GOING_LO,
				 IPMI_SENSOR_EVENT_UPPER_NON_RECOV_GOING_HI,
				 sensor_type, sensor_no);
    }    
}

// see ipmi 2.0 spec, page 431 for the meaning of low and high
// low means, the reading is going below normal value or returning from below normal.
static void thresh_isen_check_thresh(pp_tp_thresh_ipmi_sens_t* o,
				     unsigned char reading, 
				     unsigned char threshold,
				     unsigned short lo_event,
				     unsigned short hi_event,
				     unsigned char sensor_type, 
				     unsigned char sensor_no) {
    int s_reading;
    int s_threshold;

    unsigned char analog_data_format = ipmi_sdr_get_analog_data_format(o->base.sdr);

    switch ( analog_data_format ) {
      case IPMI_ANALOG_DATA_FORMAT_1_COMPL:
	  s_reading = (signed char) reading;
	  s_threshold = (signed char) threshold;
	  break;
      case IPMI_ANALOG_DATA_FORMAT_2_COMPL:
	  s_reading = ((signed char) reading) + 1;
	  s_threshold = ((signed char) threshold) + 1;
	  break;
      default:
	  s_reading = reading;
	  s_threshold = threshold;	 
	  break;
    }

    if (o->event.state & lo_event) {
        // reading was below this lower threshold at the last poll
        if (s_reading >= s_threshold + o->threshold.hysteresis.negative)
            thresh_isen_fire_deassertion_event(o, lo_event, reading, threshold,
					       sensor_type, sensor_no);
    } else {
        if (s_reading < s_threshold)
	    // reading just went below lower threshold
            thresh_isen_fire_assertion_event(o, lo_event, reading, threshold, 
					     sensor_type, sensor_no);
    }

    if (o->event.state & hi_event) {
        // reading was above this upper threshold before
        if (s_reading < s_threshold - o->threshold.hysteresis.positive)
            thresh_isen_fire_deassertion_event(o, hi_event, reading, threshold,
					       sensor_type, sensor_no);
    } else {
        if (s_reading >= s_threshold)
	    // reading newly climbed above this upper threshold
            thresh_isen_fire_assertion_event(o, hi_event, reading, threshold, 
					     sensor_type, sensor_no);
    }
}

static void thresh_isen_fire_assertion_event(pp_tp_thresh_ipmi_sens_t* o, 
					     unsigned short event,
					     unsigned char reading, 
					     unsigned char threshold,
					     unsigned char sensor_type, 
					     unsigned char sensor_no) {
    unsigned short sdr_assert_event =
	le16_to_cpu(ipmi_sdr_get_event_mask_ptr(o->base.sdr)->
		    threshold.assert_event_le16);
    
    o->event.state |= event; // enter state
    // auto-rearm opposite event
    if (o->auto_rearm) o->event.deassertion.fired &= ~event; 

    if (event                                    // mask the right bit
        & sdr_assert_event                       // evt gen is supported
        & o->event.assertion.enabled             // evt gen is enabled
        & ~o->event.assertion.fired) {           // evt hasn't been sent, alr.

	pp_bmc_receive_event(0, 0x20, 0, 0, sensor_type, sensor_no,
			     0 /* assert */, IPMI_EVENT_READING_TYPE_THRESHOLD,
			     0x50 + ffs(event) - 1, reading, threshold);

	pp_bmc_log_debug("[ThreshIPMISens] assertion event '%s' "
			 "fired for sensor '%s'",
		 ipmi_get_reading_name(IPMI_EVENT_READING_TYPE_THRESHOLD,
				       sensor_type, ffs(event) - 1),
			 pp_tp_ipmi_sens_to_string(&o->base));
    }

    // track overall system health
    if (event & ~o->event.assertion.fired) {
        o->event.assertion.fired |= event;
	pp_tp_ipmi_sens_notify_subscribers(&o->base, event);
    }
}

static void thresh_isen_fire_deassertion_event(pp_tp_thresh_ipmi_sens_t* o,
						unsigned short event,
						unsigned char reading, 
						unsigned char threshold,
						unsigned char sensor_type, 
						unsigned char sensor_no) {
    unsigned short sdr_deassert_event =
	le16_to_cpu(ipmi_sdr_get_event_mask_ptr(o->base.sdr)->
		    threshold.deassert_event_le16);
    o->event.state &= ~event; // exit state
    // auto-rearm opposite event
    if (o->auto_rearm) o->event.assertion.fired &= ~event; 

    if (event // event...
        & sdr_deassert_event             // ...must be supported
        & o->event.deassertion.enabled   // ...must be enabled
        & ~o->event.deassertion.fired) { // ...must not be fired already

        pp_bmc_receive_event(0, 0x20, 0, 0, sensor_type, sensor_no,
                             1 /* deassertion */,
			     IPMI_EVENT_READING_TYPE_THRESHOLD,
                             0x50 + ffs(event) - 1, reading, threshold);

        pp_bmc_log_debug("[ThreshIPMISens] deassertion event '%s' "
			 "fired for sensor '%s'",
                       ipmi_get_reading_name(IPMI_EVENT_READING_TYPE_THRESHOLD,
					     sensor_type, ffs(event) - 1),
                         pp_tp_ipmi_sens_to_string(&o->base));
    }

    // track overall system health
    if (event & ~o->event.deassertion.fired) {
        o->event.deassertion.fired |= event;
	pp_tp_ipmi_sens_notify_subscribers(&o->base,
					   PP_ISENS_NOTIFY_DEASSERTION(event));
    }
}

static int
thresh_isen_get_reading_factors(pp_tp_ipmi_sens_t* s,
				unsigned char reading UNUSED,
			ipmi_get_sensor_reading_factors_rs_t* rs UNUSED)
{
    ipmi_sdr_full_sensor_t* fsdr;
    
    assert(s != NULL);
    // TODO: crep FIXME!!!
    assert(sdr_event_type(s->sdr) == IPMI_EVENT_READING_TYPE_THRESHOLD);

    fsdr = (ipmi_sdr_full_sensor_t*)s->sdr;
    if (fsdr->linearization == IPMI_LINEARIZATION_LINEAR) {
	return PP_ERR;
    }

    /* TODO implement me when using non-linear sensors */
    
    return PP_SUC;
}

static signed char
thresh_isen_get_reading_offset(pp_tp_ipmi_sens_t* s)
{
    pp_tp_thresh_ipmi_sens_t *this = (pp_tp_thresh_ipmi_sens_t*) s;
    
    return this->offset;
}

static int
thresh_isen_set_reading_offset(pp_tp_ipmi_sens_t* s,
			       signed char offset)
{
    pp_tp_thresh_ipmi_sens_t *this = (pp_tp_thresh_ipmi_sens_t*) s;
    
    this->offset = offset;
    thresh_isen_save_sensor_offset(this);
  
    return 0;
}

static void
thresh_isen_load_sensor_offset(pp_tp_thresh_ipmi_sens_t* this)
{
    int offset = 0;
    char* id = NULL;

    this->offset = 0;

    if (PP_FAILED(pp_cfg_get_nodflt(&id, "ipmi.sensor.offset[%u].id",
				    this->base.number))) {
	goto bail;
    }
    
    if (pp_strcmp_safe(id, ((pp_tp_obj_t*)this)->id) != 0) {
	pp_bmc_log_warn("Mismatch while loading sensor offset for nr. "
			"%d (expected '%s', got '%s')",
			this->base.number, id, ((pp_tp_obj_t*)this)->id);

	// clear the (probably old) offset entry in config fs
	pp_cfg_remove("ipmi.sensor.offset[%u]", this->base.number);

	goto bail;
    }

    if (PP_SUCCED(pp_cfg_get_int_nodflt(&offset,"ipmi.sensor.offset[%u].value",
					this->base.number))) {
	this->offset = offset;
    }    
    
    pp_bmc_log_debug("[ThreshIPMISens] load sensor offset %d "
		     "for sensor %d (id '%s')",
		     this->offset, this->base.number,((pp_tp_obj_t*)this)->id);

 bail:
    return;
}

static void
thresh_isen_save_sensor_offset(pp_tp_thresh_ipmi_sens_t* this)
{
    pp_bmc_log_debug("[ThreshIPMISens] save sensor offset "
		     "%d for sensor %d (id '%s')",
		     this->offset, this->base.number,
		     ((pp_tp_obj_t*)this)->id);

    pp_cfg_set(((pp_tp_obj_t*)this)->id, "ipmi.sensor.offset[%u].id",
	       this->base.number);
    pp_cfg_set_int(this->offset, "ipmi.sensor.offset[%u].value",
		   this->base.number);

    pp_cfg_save(DONT_FLUSH);
}

/*
 * Discrete IPMI Sensor
 */

static void disc_isen_dtor(pp_tp_obj_t* o) {
    pp_tp_disc_ipmi_sens_t* this = (pp_tp_disc_ipmi_sens_t*)o;
    assert(this != NULL);
    assert(PP_TP_OBJ_IS_TYPE(PP_TP_DISC_IPMI_SENS, o));
    pp_tp_ipmi_sens_cleanup(&this->base);
    free(this);
}

static void disc_isen_setup_for_sdr(pp_tp_disc_ipmi_sens_t* isens,
				     ipmi_sdr_header_t* sdr) {
    ipmi_sdr_evt_mask_t* sdr_mask;
    
    if (NULL != (sdr_mask = ipmi_sdr_get_event_mask_ptr(sdr))) {
	isens->event.assertion.enabled =
	    le16_to_cpu(sdr_mask->discrete.assert_event_le16);
	isens->event.deassertion.enabled =
	    le16_to_cpu(sdr_mask->discrete.deassert_event_le16);
    } else {
	isens->event.assertion.enabled = 0;
	isens->event.deassertion.enabled = 0;
	pp_bmc_log_warn("[DiscISENS] %s(#%d): no Event-Mask-SDR (type: %#.2x),"
			" all events disabled!",
			pp_tp_obj_to_string((pp_tp_obj_t*)isens),
			isens->base.number, sdr->type);
    }

    isens->event.deassertion.fired = 0;
    isens->event.assertion.fired = 0;
    isens->all_events_enabled = 1;
    isens->event.state = 0;

    isens->auto_rearm = 1;
    isens->scanning_enabled = 1;
}

static int disc_isen_set_sdr(pp_tp_ipmi_sens_t* s, ipmi_sdr_header_t* sdr) {
    int ret;
    if (sdr == NULL) {
	ret = PP_ERR;
    } else if (sdr_sens_type(s->sdr) != sdr_sens_type(sdr) ||
	       sdr_event_type(s->sdr) != sdr_event_type(sdr)) {
	pp_bmc_log_error("[DiscISENS] set_sdr: incompatible SDR sens-type/"
			 "evt-type: current=%#.2x/%#.2 new=%#.2x/%#.2",
			 sdr_sens_type(s->sdr), sdr_event_type(s->sdr),
			 sdr_sens_type(sdr), sdr_event_type(sdr));
	ret = PP_ERR;
    } else {
	assert(PP_TP_OBJ_IS_TYPE(PP_TP_DISC_IPMI_SENS, s));
	free(s->sdr);
	s->sdr = ipmi_sdr_copy(sdr);
	disc_isen_setup_for_sdr((pp_tp_disc_ipmi_sens_t*)s, sdr);
	ret = PP_SUC;
    }
    return ret;
}

static int disc_isen_get_reading(pp_tp_ipmi_sens_t *s,
				 unsigned char* resp_code,
				 ipmi_sensor_get_reading_rs_t* r) {
    pp_tp_disc_ipmi_sens_t* this = (pp_tp_disc_ipmi_sens_t*)s;
    int ret = PP_SUC;
    int value, readable;

    memset(r, 0, sizeof(ipmi_sensor_get_reading_rs_t));
    r->rd.unavailable  =  s->presence ?
	!pp_tp_sensdev_get_reading((pp_tp_sensdev_t*)s->presence) : 0;
    r->rd.scan_enabled = this->scanning_enabled;
    r->rd.evts_enabled = this->all_events_enabled;

    if (r->rd.unavailable) {
	*resp_code = IPMI_ERR_SUCCESS;
	pp_bmc_log_debug("[DiscISens] '%s' GetReading: unavailable, pres-cond"
			 " false", pp_tp_obj_to_string(&s->base));
    } else if (0 > (value = pp_tp_sensdev_get_reading(s->sensor))) {
	r->rd.unavailable = 1;
	*resp_code = IPMI_ERR_SUCCESS;
	pp_bmc_log_debug("[DiscISens] '%s' GetReading: unavailable, invalid"
			 " reading", pp_tp_obj_to_string(&s->base));
    } else {
	// TODO: make sure sdr_mask_ptr can't be zero
	readable = le16_to_cpu(ipmi_sdr_get_event_mask_ptr(s->sdr)->
			       discrete.read_le16);
	if (readable & IPMI_SENSOR_EVENT_STATE_BIT0)
	    r->rd.state0 = (value & IPMI_SENSOR_EVENT_STATE_BIT0) ? 1 : 0;
	if (readable & IPMI_SENSOR_EVENT_STATE_BIT1)
	    r->rd.state1 = (value & IPMI_SENSOR_EVENT_STATE_BIT1) ? 1 : 0;
	if (readable & IPMI_SENSOR_EVENT_STATE_BIT2)
	    r->rd.state2 = (value & IPMI_SENSOR_EVENT_STATE_BIT2) ? 1 : 0;
	if (readable & IPMI_SENSOR_EVENT_STATE_BIT3)
	    r->rd.state3 = (value & IPMI_SENSOR_EVENT_STATE_BIT3) ? 1 : 0;
	if (readable & IPMI_SENSOR_EVENT_STATE_BIT4)
	    r->rd.state4 = (value & IPMI_SENSOR_EVENT_STATE_BIT4) ? 1 : 0;
	if (readable & IPMI_SENSOR_EVENT_STATE_BIT5)
	    r->rd.state5 = (value & IPMI_SENSOR_EVENT_STATE_BIT5) ? 1 : 0;
	if (readable & IPMI_SENSOR_EVENT_STATE_BIT6)
	    r->rd.state6 = (value & IPMI_SENSOR_EVENT_STATE_BIT6) ? 1 : 0;
	if (readable & IPMI_SENSOR_EVENT_STATE_BIT7)
	    r->rd.state7 = (value & IPMI_SENSOR_EVENT_STATE_BIT7) ? 1 : 0;
	if (readable & IPMI_SENSOR_EVENT_STATE_BIT8)
	    r->rd.state8 = (value & IPMI_SENSOR_EVENT_STATE_BIT8) ? 1 : 0;
	if (readable & IPMI_SENSOR_EVENT_STATE_BIT9)
	    r->rd.state9 = (value & IPMI_SENSOR_EVENT_STATE_BIT9) ? 1 : 0;
	if (readable & IPMI_SENSOR_EVENT_STATE_BIT10)
	    r->rd.state10= (value & IPMI_SENSOR_EVENT_STATE_BIT10) ? 1 : 0;
	if (readable & IPMI_SENSOR_EVENT_STATE_BIT11)
	    r->rd.state11= (value & IPMI_SENSOR_EVENT_STATE_BIT11) ? 1 : 0;
	if (readable & IPMI_SENSOR_EVENT_STATE_BIT12)
	    r->rd.state12= (value & IPMI_SENSOR_EVENT_STATE_BIT12) ? 1 : 0;
	if (readable & IPMI_SENSOR_EVENT_STATE_BIT13)
	    r->rd.state13= (value & IPMI_SENSOR_EVENT_STATE_BIT13) ? 1 : 0;
	if (readable & IPMI_SENSOR_EVENT_STATE_BIT14)
	    r->rd.state14= (value & IPMI_SENSOR_EVENT_STATE_BIT14) ? 1 : 0;
	
        *resp_code = IPMI_ERR_SUCCESS;
	pp_bmc_log_debug("[DiscISens] '%s' GetReading: 0x%x", 
			 pp_tp_obj_to_string(&s->base), value);
    }
    return ret;
}

/* TODO >-8 next 4 are currently equal to the threshold version,
 * merge into base class? */
static int disc_isen_set_event_enable(pp_tp_ipmi_sens_t* s,
				      ipmi_set_sensor_event_enable_rq_t* rq)
{
    pp_tp_disc_ipmi_sens_t* this = (pp_tp_disc_ipmi_sens_t*)s;
    ipmi_sdr_sensor_t* sdr_sensor;

    sdr_sensor = ipmi_sdr_get_sensor_ptr(s->sdr);

    if(sdr_sensor->caps.event_msg == IPMI_SENSOR_CAPS_EVENT_NONE) {
	return PP_ERR;
    }
    
    this->scanning_enabled = rq->scan_enabled;
    this->all_events_enabled = rq->evts_enabled;

    if(sdr_sensor->caps.event_msg == IPMI_SENSOR_CAPS_EVENT_PER_TRESH) {
	if (rq->enable_events) {
	    // enable selected events
	    this->event.assertion.enabled |=
		le16_to_cpu(rq->assertion_events_le16);
	    this->event.deassertion.enabled |= 
		le16_to_cpu(rq->deassertion_events_le16);
	} else if (rq->disable_events) {
	    // disable selected events
	    this->event.assertion.enabled &=
		~le16_to_cpu(rq->assertion_events_le16);
	    this->event.deassertion.enabled &= 
		~le16_to_cpu(rq->deassertion_events_le16);
	}
    }

    return PP_SUC;    
}

static int disc_isen_get_event_enable(pp_tp_ipmi_sens_t* s,
				      ipmi_get_sensor_event_enable_rs_t* rs)
{
    pp_tp_disc_ipmi_sens_t* this = (pp_tp_disc_ipmi_sens_t*)s;
    
    rs->scan_enabled        = this->scanning_enabled;
    rs->evts_enabled        = this->all_events_enabled;
    rs->assertion_events_le16   = cpu_to_le16(this->event.assertion.enabled);
    rs->deassertion_events_le16 = cpu_to_le16(this->event.deassertion.enabled);
    return PP_SUC;
}

static int disc_isen_rearm_events(pp_tp_ipmi_sens_t* s,
				  ipmi_rearm_sensor_events_rq_t* rq)
{
    pp_tp_disc_ipmi_sens_t* this = (pp_tp_disc_ipmi_sens_t*)s;

    if (rq->rearm_only_selected) {
        // re-arm selected events
        this->event.assertion.fired   &= ~le16_to_cpu(rq->assertion_events_le16);
        this->event.deassertion.fired &= ~le16_to_cpu(rq->deassertion_events_le16);
    } else {
        // re-arm all events
        this->event.assertion.fired   = 0;
        this->event.deassertion.fired = 0;
    }
    // TODO: immediately retrigger event generation
    return PP_SUC;   
}

static int disc_isen_get_event_status(pp_tp_ipmi_sens_t* s,
				      ipmi_sensor_get_event_status_rs_t* rs)
{
    pp_tp_disc_ipmi_sens_t* this = (pp_tp_disc_ipmi_sens_t*)s;
    
    rs->scan_enabled        = this->scanning_enabled;
    rs->evts_enabled        = this->all_events_enabled;
    rs->unavailable         = 0; // TODO: set a usefull value
    rs->assertion_events_le16    = cpu_to_le16(this->event.assertion.fired);
    rs->deassertion_events_le16  = cpu_to_le16(this->event.deassertion.fired);
    return PP_SUC;
}
/* TODO -------- >-8 ----------- */

static void disc_isen_recv_reading(pp_tp_sensdev_subscriber_t* s, 
				   pp_tp_sensdev_t* source UNUSED, int reading) {
    pp_tp_ipmi_sens_t* o = PP_TP_INTF_2_OBJ_CAST(s, pp_tp_ipmi_sens_t,
						 subscriber);
    pp_tp_disc_ipmi_sens_t* this = (pp_tp_disc_ipmi_sens_t*)o;
    
    // TODO: check assert
    assert(sdr_event_type(o->sdr) != IPMI_EVENT_READING_TYPE_THRESHOLD);

    /* don't do anything if invalid reading, sensor scanning disabled,
     * no sdr configured or sensor presence condition exists and false */
    if ((reading < 0) || !this->scanning_enabled ||
	((this->base.presence) && !pp_bmc_tp_cond_is_true(this->base.presence))) {
        return;
    }

    disc_isen_check_events(this, reading);
}

static void disc_isen_recv_presence_reading(pp_tp_sensdev_subscriber_t* subs,
					    pp_tp_sensdev_t* source UNUSED, int reading)
{
    pp_tp_ipmi_sens_t* o = PP_TP_INTF_2_OBJ_CAST(subs, pp_tp_ipmi_sens_t,
						 presence_subscriber);
    pp_tp_disc_ipmi_sens_t* this = (pp_tp_disc_ipmi_sens_t*)o;
    int sensor_reading;

    sensor_reading = pp_tp_sensdev_get_reading(o->sensor);
    
    if (reading > 0 && sensor_reading >= 0) {
	disc_isen_check_events(this, sensor_reading);
    }   
}

static void disc_isen_check_events(pp_tp_disc_ipmi_sens_t* this, int reading)
{
    unsigned char reading_type =  sdr_event_type(this->base.sdr);
    unsigned char sensor_type = sdr_sens_type(this->base.sdr);
    unsigned char sensor_no = this->base.number;
    int i;
    
    for (i = 0; i < IPMI_SENSOR_EVENT_STATE_COUNT; i++) {
	disc_isen_check_state(this, reading, i,
			      sensor_type, sensor_no, reading_type);
    }
}

static void disc_isen_check_state(pp_tp_disc_ipmi_sens_t* this, 
				  int reading,
				  unsigned char state,
				  unsigned char sensor_type, 
				  unsigned char sensor_no,
				  unsigned char reading_type) {
    unsigned short event = (0x01 << state);

    if (this->event.state & event) {
	if ((reading & event) == 0) {
	    disc_isen_fire_deassertion_event(this, event, sensor_type,
					     sensor_no, reading_type);
	}
    } else {
	if (reading & event) {
	    disc_isen_fire_assertion_event(this, event, sensor_type, sensor_no,
					   reading_type);
	}
    }
}

static void disc_isen_fire_assertion_event(pp_tp_disc_ipmi_sens_t* o, 
					   unsigned short event,
					   unsigned char sensor_type, 
					   unsigned char sensor_no,
					   unsigned char reading_type) {
    unsigned short sdr_assert_event =
	le16_to_cpu(ipmi_sdr_get_event_mask_ptr(o->base.sdr)->
		    threshold.assert_event_le16);
    
    o->event.state |= event; // enter state
    // auto-rearm opposite event
    if (o->auto_rearm) o->event.deassertion.fired &= ~event; 

    if (event                          // event...
        & sdr_assert_event             // ...must be supported
        & o->event.assertion.enabled   // ...must be enabled
        & ~o->event.assertion.fired) { // ...must not be fired already

        pp_bmc_receive_event(0, 0x20, 0, 0, sensor_type, sensor_no,
			     0 /* assertion */, reading_type,
			     ffs(event) - 1, 0x00, 0x00);

        pp_bmc_log_debug("[DiscIPMISens] assertion event '%s' "
			 "fired for sensor '%s'",
			 ipmi_get_reading_name(reading_type, sensor_type,
					       ffs(event) - 1),
                         pp_tp_ipmi_sens_to_string(&o->base));
    }

    // track system health
    if (event & ~o->event.assertion.fired) {
        o->event.assertion.fired |= event;
	pp_tp_ipmi_sens_notify_subscribers(&o->base, event);
    }
}

static void disc_isen_fire_deassertion_event(pp_tp_disc_ipmi_sens_t* o,
					     unsigned short event,
					     unsigned char sensor_type, 
					     unsigned char sensor_no,
					     unsigned char reading_type) {
    unsigned short sdr_deassert_event =
	le16_to_cpu(ipmi_sdr_get_event_mask_ptr(o->base.sdr)->
		    threshold.deassert_event_le16);

    o->event.state &= ~event; // exit state

    // auto-rearm opposite event
    if (o->auto_rearm) o->event.assertion.fired &= ~event; 

    if (event                            // event...
        & sdr_deassert_event             // ...must be supported
        & o->event.deassertion.enabled   // ...must be enabled
        & ~o->event.deassertion.fired) { // ...must not be fired already
	
        pp_bmc_receive_event(0, 0x20, 0, 0, sensor_type, sensor_no,
			     1 /* deassertion */, reading_type,
			     ffs(event) - 1, 0x00, 0x00);

        pp_bmc_log_debug("[DiscIPMISens] deassertion event '%s' "
			 "fired for sensor '%s'",
                         ipmi_get_reading_name(reading_type, sensor_type,
					       ffs(event) - 1),
                         pp_tp_ipmi_sens_to_string(&o->base));
    }

    // track system health
    if (event & ~o->event.deassertion.fired) {
        o->event.deassertion.fired |= event;
	pp_tp_ipmi_sens_notify_subscribers(&o->base,
					   PP_ISENS_NOTIFY_DEASSERTION(event));
    }
}

static unsigned char sdr_sens_type(ipmi_sdr_header_t* sdr) {
    unsigned char ret;
    unsigned char* type_ptr;

    if (NULL != (type_ptr = ipmi_sdr_get_sensor_type_ptr(sdr))) {
	ret = *type_ptr;
    } else {
	ret = IPMI_SENSOR_TYPE_RESERVED;
    }
    return ret;
}
    

static unsigned char sdr_event_type(ipmi_sdr_header_t* sdr) {
    unsigned char ret;
    unsigned char* rtype_ptr;

    if (NULL != (rtype_ptr = ipmi_sdr_get_event_type_ptr(sdr))) {
	ret = *rtype_ptr;
    } else {
	ret = IPMI_EVENT_READING_TYPE_UNSPECIFIED;
    }
    return ret;
}

