/**
 * tp_muxed_scanner.c
 *
 * Scanner object periodically scans sensors
 * 
 * (c) 2006 Peppercon AG, <mkl@peppercon.de>
 */

#include <unistd.h>
#include <fcntl.h>
#include <pthread.h>

#include <pp/base.h>
#include <pp/selector.h>
#include <pp_gpio.h>

#include <pp/bmc/debug.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_gpio_multi_act.h>
#include <pp/bmc/tp_scan_sensdev.h>

#include <pp/bmc/tp_muxed_scanner.h>
#include <pp/bmc/tp_gpio_multi_act.h>

#include <drivers/pp_tach_sens.h>

typedef struct {
    int bank_addr;
    vector_t *sensors;
} sensor_bank_t;

static void muxed_scanner_dtor(pp_tp_obj_t* this);
static int provoke_sensor_update(const int item_id UNUSED, void* ctx);
static void delete_bank_vector(void * void_entry);

pp_tp_obj_t* pp_tp_muxed_scanner_ctor(const char* id, vector_t* args)
{
    const char* msg = "[MuxedScanner] '%d' failed: %s";
    size_t i, vs = vector_size(args);
    pp_tp_gpio_multi_act_t* gma;
    pp_tp_muxed_scanner_t* msc = malloc(sizeof(pp_tp_muxed_scanner_t));
    sensor_bank_t *s_bank;
    pp_tp_arg_t* arg;
    
    pp_tp_obj_init((pp_tp_obj_t*)msc, PP_TP_MUXED_SCANNER, id, muxed_scanner_dtor);

    msc->sensor_banks = vector_new(NULL, 0, delete_bank_vector);
    
    for (i = 0; i < vs; ++i) {
	arg = (pp_tp_arg_t*)vector_get2(args, i);

	switch(arg->type) {
	  case PP_TP_ARG_OBJ:
	      if ( i != 0 ) {
		  pp_bmc_log_error(msg, i, "gpio-dev must come first");
		  goto bail;
	      }
	      if (!PP_TP_OBJ_IS_TYPE(PP_TP_GPIO_MULTI_ACT, arg->value.obj)) {
		  pp_bmc_log_error(msg, i, "invalid object type");
		  goto bail;
	      }
	      gma = (pp_tp_gpio_multi_act_t*)arg->value.obj;
	      msc->gpio_act = pp_tp_gpio_multi_act_duplicate(gma);
	      break;
	  case PP_TP_ARG_INT:
	      if (i == 0) {
		  pp_bmc_log_error(msg, i, "gpio-dev must come first");
		  goto bail;
	      } else if ( i == 1 ) {
		  if ( arg->value.intval < 4000 ) {
		      pp_bmc_log_error(msg, i, "minimum delay time is 4000 ms");
		      goto bail;
		  } else {
		      msc->delay = arg->value.intval;
		  }
	      } else {
		  s_bank = malloc(sizeof(sensor_bank_t));
		  s_bank->bank_addr = arg->value.intval;
		  s_bank->sensors = vector_new(NULL, 0, NULL);
		  vector_add(msc->sensor_banks, s_bank);
	      }
	      break;
	  default:
	      pp_bmc_log_error(msg, i, "invalid argument type");
	      goto bail;
	}
    }

    if ( vector_size(msc->sensor_banks) > 0 ) {
	s_bank = (sensor_bank_t*)vector_get(msc->sensor_banks, 0);
	/* set gpio_act to first sensor bank */
	pp_tp_gpio_multi_act_set(msc->gpio_act, s_bank->bank_addr, 0);
	msc->current_bank = 0;
	/* schedule scan function */
	msc->scanner_hndl = pp_select_add_to(msc->delay, 1, provoke_sensor_update, (void*)msc);
    }

    return (pp_tp_obj_t*)msc;
 bail:
    muxed_scanner_dtor((pp_tp_obj_t*)msc);
    return NULL;

}

int pp_tp_muxed_scanner_register(pp_tp_scan_sensdev_t* s, pp_tp_muxed_scanner_t *this, u_int bank_num)
{
    size_t i;
    const char* errmsg = "[MuxedScanner] %s: %s ";

    for ( i = 0; i < vector_size(this->sensor_banks); i++ ) {
	sensor_bank_t *bank = (sensor_bank_t*)vector_get(this->sensor_banks, i);
	
	if ( i == bank_num ) {	    	    
	    pp_bmc_log_debug("[MuxedScanner] Add Element to Scanner Bank %d (bank number %x)",
			     i, bank->bank_addr);
	    /* add sensor to bank vector */
	    vector_add(bank->sensors, s);	    
	    break;
	}
    }

    if ( i == vector_size(this->sensor_banks) ) {
	pp_bmc_log_error(errmsg, ___F, " Bank Number not valid");
	return PP_ERR;
    }

    return PP_SUC;
}

static int provoke_sensor_update(const int item_id UNUSED, void* ctx)
{
    size_t i;
    pp_tp_muxed_scanner_t* this = (pp_tp_muxed_scanner_t*)ctx;
    u_int last_bank = this->current_bank;
    sensor_bank_t *s_bank =
	(sensor_bank_t*)vector_get(this->sensor_banks, this->current_bank);

    /* update all sensors of this bank */
    for ( i = 0; i < vector_size(s_bank->sensors); i++ ) {
	pp_tp_scan_sensdev_t* s =
	    (pp_tp_scan_sensdev_t*) vector_get(s_bank->sensors, i);
	s->scannable.update(&s->scannable);
    }

    /* switch to next bank, skip empty ones */
    do {
	this->current_bank = this->current_bank >= vector_size(this->sensor_banks) - 1 ?
	    0 : this->current_bank + 1;	
	s_bank = (sensor_bank_t*)vector_get(this->sensor_banks, this->current_bank);
    } while ( vector_size(s_bank->sensors) <= 0 && this->current_bank != last_bank );    

    /* switch if bank has changed */
    if ( this->current_bank != last_bank ) {
	pp_tp_gpio_multi_act_set(this->gpio_act, s_bank->bank_addr, 0);
    }

    return 0;
}

static void delete_bank_vector(void * void_entry)
{
    vector_delete(((sensor_bank_t*)void_entry)->sensors);
}

void muxed_scanner_dtor(pp_tp_obj_t* o)
{
    pp_tp_muxed_scanner_t* this = (pp_tp_muxed_scanner_t*)o;

    pp_select_remove_to(this->scanner_hndl);
    vector_delete(this->sensor_banks);
    if (this->gpio_act) {	
	pp_tp_gpio_multi_act_release(this->gpio_act);
    }
    pp_tp_obj_cleanup(o);
}
