/**
 * tp_sens_has_value_cond.c
 *
 * Condition object taht is true if a sensor has one from
 * a list of specified values
 * 
 * (c) 2005 Peppercon AG, Ronald Wahl <rwa@peppecon.de>
 */

#include <pp/bmc/debug.h>
#include <pp/bmc/topo_factory.h>
#include <pp/bmc/tp_sensdev.h>
#include <pp/bmc/tp_sens_has_value_cond.h>

static void
sens_has_value_cond_recv_reading(pp_tp_sensdev_subscriber_t* subscriber,
				 pp_tp_sensdev_t* source, int reading);
static int get_cond_value(pp_tp_sens_has_value_cond_t* this, int reading);
static void sens_has_value_cond_dtor(pp_tp_obj_t* this);

pp_tp_obj_t*
pp_tp_sens_has_value_cond_ctor(const char* id, vector_t* args)
{
    const char* errmsg = "[SensHasValueCond] %s: '%s' failed (%s)";
    pp_strstream_t err = PP_STRSTREAM_INITIALIZER;
    pp_tp_sens_has_value_cond_t* cond = NULL;
    pp_tp_sensdev_t* sensor;
    vector_t * values;
    int i, arg_cnt, value_cnt;

    arg_cnt = vector_size(args);

    if (arg_cnt < 2) {
	pp_bmc_log_error("ERROR: %s '%s' failed: too few arguments", ___F, id);
	goto bail;
    }

    if (pp_tp_arg_scanf(args, 0, &err, "o<s>", &sensor) != 1) {
	pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
	goto bail;
    }

    value_cnt = arg_cnt - 1;

    values = vector_new(NULL, value_cnt, NULL);
    for (i = 0; i < value_cnt; ++i) {
	int value;
        if (pp_tp_arg_scanf(args, i + 1, &err, "d", &value) != 1) {
	    pp_bmc_log_error(errmsg, ___F, id, pp_strstream_buf(&err));
	    vector_delete(values);
            goto bail;
        }
	vector_add(values, (void*)(long)value);
    }

    cond = calloc(1, sizeof(pp_tp_sens_has_value_cond_t));
    if (PP_FAILED(pp_tp_sens_has_value_cond_init(cond,
						 PP_TP_SENS_HAS_VALUE_COND, id,
						 sens_has_value_cond_dtor,
						 sensor, values))) {
	pp_bmc_log_error(errmsg, ___F, id, "init");
	free(cond);
	cond = NULL;
    }
 bail:
    pp_strstream_free(&err);
    return (pp_tp_obj_t*)cond;
}

static void sens_has_value_cond_dtor(pp_tp_obj_t* this) {
    pp_tp_sens_has_value_cond_cleanup((pp_tp_sens_has_value_cond_t*)this);
    free(this);
}

int
pp_tp_sens_has_value_cond_init(pp_tp_sens_has_value_cond_t* this,
			       pp_tp_obj_type_t type,
			       const char * id, pp_tp_obj_dtor_func_t dtor,
			       pp_tp_sensdev_t* sensor, vector_t * values)
{
    if (PP_SUCCED(pp_tp_cond_init(&this->base, type, id, dtor))) {
	this->base.base.reading = 0;
	this->sensor = pp_tp_sensdev_duplicate(sensor);
	this->values = values;
	this->sens_subscriber.recv_reading = sens_has_value_cond_recv_reading;
	pp_bmc_tp_sensdev_subscribe(this->sensor, &this->sens_subscriber);
	return PP_SUC;
    }
    return PP_ERR;
}

void
pp_tp_sens_has_value_cond_cleanup(pp_tp_sens_has_value_cond_t* this)
{
    pp_bmc_tp_sensdev_unsubscribe(this->sensor, &this->sens_subscriber);
    pp_tp_sensdev_release(this->sensor);
    vector_delete(this->values);
    pp_tp_cond_cleanup(&this->base);
}

static void
sens_has_value_cond_recv_reading(pp_tp_sensdev_subscriber_t* subscriber,
				 pp_tp_sensdev_t* source UNUSED, int reading)
{
    pp_tp_sens_has_value_cond_t* this;
    this = PP_TP_INTF_2_OBJ_CAST(subscriber, pp_tp_sens_has_value_cond_t,
				 sens_subscriber);
    
    int cond_value = get_cond_value(this, reading);

    pp_tp_cond_set_reading(&this->base, cond_value);
}

static int
get_cond_value(pp_tp_sens_has_value_cond_t* this, int reading)
{
    size_t i, size = vector_size(this->values);

    for (i = 0; i < size; ++i) {
	if (vector_get_int(this->values, i) == reading) return 1;
    }
    return 0;
}
