#!/usr/bin/perl -w
###############################################################################
# This is a script for building all modules of the firmware. The build process
# won't stop if a single module fails. However the overall script will return
# 1 if an error occured.
#

use strict;

#
###############################################################################
# imports

use FindBin;
use lib "$FindBin::Bin";

use Getopt::Std;
use File::Path;
use pp_hw_data;
use pp_build_lib;

#
###############################################################################
# global variables

my $erla_flags_arm =
{
 "cross"		=> "pp-wrapper-",
 "arch"			=> "arm",
 "realcross"		=> "armv4-pp#TOOLCHAIN_VERSION#-linux-uclibc-",
 "host_system"		=> "armv4-pp#TOOLCHAIN_VERSION#-linux",
 "defines"		=> "-D_FILE_OFFSET_BITS=64",
 "werror"		=> "-Werror",
 "coptflags"		=> "-Os -fomit-frame-pointer",
 "coptspeedflags"	=> "-O3 -fomit-frame-pointer",
 "cxxoptflags"		=> "-Os",
 "cxxoptspeedflags"	=> "-O3",
 "dbgflags"		=> "-g",
 "dbgoptflags"		=> "-g -O",
 "profileflags"		=> "-pg -O -fprofile-arcs -ftest-coverage",
 "ldflags"		=> "",
};

my $erla_flags_powerpc =
{
 "cross"		=> "pp-wrapper-",
 "arch"			=> "powerpc",
 "realcross"		=> 'powerpc-pp#TOOLCHAIN_VERSION#-linux-uclibc-',
 "host_system"		=> 'powerpc-pp#TOOLCHAIN_VERSION#-linux',
 "defines"		=> "-D_FILE_OFFSET_BITS=64",
 "werror"		=> "-Werror",
 "coptflags"		=> "-Os -fomit-frame-pointer -mcpu=405",
 "coptspeedflags"	=> "-O3 -fomit-frame-pointer -mcpu=405",
 "cxxoptflags"		=> "-Os -mcpu=405",
 "cxxoptspeedflags"	=> "-O3 -mcpu=405",
 "dbgflags"		=> "-g",
 "dbgoptflags"		=> "-g -O",
 "profileflags"		=> "-pg -O -fprofile-arcs -ftest-coverage",
 "ldflags"		=> "",
};

my $psetup_flags =
{
 "cross"		=> "",
 "arch"			=> "x86",
 "realcross"		=> "",
 "host_system"		=> "i586-unknown-linux-gnu",
 "defines"		=> "",
 "werror"		=> "",
 "coptflags"		=> "-O2 -fomit-frame-pointer -march=i586",
 "coptspeedflags"	=> "-O3 -fomit-frame-pointer -march=i586",
 "cxxoptflags"		=> "-O2 -march=i586",
 "cxxoptspeedflags"	=> "-O3 -march=i586",
 "dbgflags"		=> "-g",
 "dbgoptflags"		=> "-g -O",
 "profileflags"		=> "-pg -O -fprofile-arcs -ftest-coverage",
 "ldflags"		=> "",
};

my $kxng_flags =
{
 "cross"		=> "pp-wrapper-",
 "arch"			=> "x86",
 "realcross"		=> 'i586-pp#TOOLCHAIN_VERSION#-linux-uclibc-',
 "host_system"		=> 'i586-pp#TOOLCHAIN_VERSION#-linux',
 "defines"		=> "",
 "werror"		=> "-Werror",
 "coptflags"		=> "-Os -fomit-frame-pointer -march=i586",
 "coptspeedflags"	=> "-O3 -fomit-frame-pointer -march=i586",
 "cxxoptflags"		=> "-Os -march=i586",
 "cxxoptspeedflags"	=> "-O3 -march=i586",
 "dbgflags"		=> "-g",
 "dbgoptflags"		=> "-g -O",
 "profileflags"		=> "-pg -O -fprofile-arcs -ftest-coverage",
 "ldflags"		=> "",
};

my $ccf_flags =
{
 "cross"		=> "",
 "arch"			=> "x86",
 "realcross"		=> "",
 "host_system"		=> "i686-unknown-linux-gnu",
 "defines"		=> "-DLDAP_THREAD_SAFE",
 "werror"		=> "",
 "coptflags"		=> "-O2 -fomit-frame-pointer -march=i686",
 "coptspeedflags"	=> "-O3 -fomit-frame-pointer -march=i686",
 "cxxoptflags"		=> "-O2 -march=i686",
 "cxxoptspeedflags"	=> "-O3 -march=i686",
 "dbgflags"		=> "-g -DDEBUG -D_DEBUG",
 "dbgoptflags"		=> "-g -O",
 "profileflags"		=> "-pg -O -fprofile-arcs -ftest-coverage",
 "ldflags"		=> "-L\${DESTDIR}/lib -L\${DESTDIR}/usr/lib",
};

my $kvmviewerlinux_flags =
{
 "cross"		=> "",
 "arch"			=> "x86",
 "realcross"		=> "",
 "host_system"		=> "i686-unknown-linux-gnu",
 "defines"		=> "-DLDAP_THREAD_SAFE",
 "werror"		=> "",
 "coptflags"		=> "-O2 -fomit-frame-pointer -march=i686",
 "coptspeedflags"	=> "-O3 -fomit-frame-pointer -march=i686",
 "cxxoptflags"		=> "-O2 -march=i686",
 "cxxoptspeedflags"	=> "-O3 -march=i686",
 "dbgflags"		=> "-g",
 "dbgoptflags"		=> "-g -O",
 "profileflags"		=> "-pg -O -fprofile-arcs -ftest-coverage",
 "ldflags"		=> "-L\${DESTDIR}/lib -L\${DESTDIR}/usr/lib",
};

my $kiratool_flags =
{
 "cross"		=> "",
 "arch"			=> "x86",
 "realcross"		=> "",
 "host_system"		=> "i686-unknown-linux-gnu",
 "defines"		=> "-DLDAP_THREAD_SAFE",
 "werror"		=> "",
 "coptflags"		=> "-O2 -fomit-frame-pointer -march=i686",
 "coptspeedflags"	=> "-O3 -fomit-frame-pointer -march=i686",
 "cxxoptflags"		=> "-O2 -march=i686",
 "cxxoptspeedflags"	=> "-O3 -march=i686",
 "dbgflags"		=> "-g",
 "dbgoptflags"		=> "-g -O",
 "profileflags"		=> "-pg -O -fprofile-arcs -ftest-coverage",
 "ldflags"		=> "-L\${DESTDIR}/lib -L\${DESTDIR}/usr/lib",
};

my $fwtype_board_flags	=
{
 "erla"		=> {
		    "kira" => $erla_flags_arm,
		    "lara" => $erla_flags_powerpc,
		    "kxng" => $kxng_flags,
		   },
 "psetup"	=> {
		    "none" => $psetup_flags,
		   },
 "ccf"		=> {
		    "none" => $ccf_flags,
		   },
 "kvmviewerlinux"		=> {
		    "none" => $kvmviewerlinux_flags,
		   },
 "kiratool"		=> {
		    "none" => $kiratool_flags,
		   },
 "production"	=> {
		    # for now, share flags with "erla"
		    "kira" => $erla_flags_arm,
		    "lara" => $erla_flags_powerpc,
		   },
};

my @flashdisk_parts = ( "oem" );

my $install_program = `which install 2> /dev/null || echo install_not_found`;
chomp $install_program;
# preserve timestamps on install to prevent unnecessary rebuilds
my $install_options = "-p";
# preserving security context on SELinux systems increases install-performance
if (!system("[ -x /usr/sbin/selinuxenabled ] && /usr/sbin/selinuxenabled") &&
    !system("$install_program -P /bin/sh .sh 2> /dev/null && rm -f .sh")) {
    $install_options .= " -P";
}

my $sh_vars =
{
 'toolchain_version'		=> "#TOOLCHAIN_VERSION#",
 'path'				=> "\${FW_TOPDIR}/bin:\${PATH}",
 'destdir'			=> "\${FW_TOPDIR}/install_root",
 'flashdir'			=> "\${FW_TOPDIR}/flash_root",
 'pp_build_sys_dir'		=> "\${FW_TOPDIR}/build_sys",
 'pp_doc_sys_dir'		=> "\${FW_TOPDIR}/build_sys/bin/doc",
 'pp_java_home'			=> "/opt/pp#TOOLCHAIN_VERSION#-j2sdk",
 'fw_defines_common'		=> "",
 'fw_cflags_common'		=> "-pipe -fno-common",
 'fw_cflags_speed_common'	=> "-pipe -fno-common",
 'fw_cxxflags_common'		=> "-pipe",
 'fw_cxxflags_speed_common'	=> "-pipe",
 'fw_ldflags_common'		=> "",
 'fw_includes'			=> "",
 'fw_defines'			=> "-D_REENTRANT -D_GNU_SOURCE -DOS_POSIX",
 'fw_cflags'			=> "-Wall -W -Wfloat-equal -Wundef -Wshadow -Wbad-function-cast -Wcast-qual -Wwrite-strings -Wstrict-prototypes -Wmissing-prototypes -Wmissing-declarations -Wredundant-decls -Wnested-externs",
 'fw_cxxflags'			=> "-Wall -W -Wfloat-equal -Wundef -Wshadow -Wcast-qual -Wwrite-strings -Wredundant-decls",
 'fw_ldflags'			=> "",
 'fw_cflags_3rdparty'		=> "",
 'fw_cxxflags_3rdparty'		=> "",
 'fw_ldflags_3rdparty'		=> "",
 'hwid'				=> "",
 'hwid_int'			=> "",
 'platform'			=> "", 
 'lightoem_dir'			=> "",
 'wxconfig'			=> "wx-config --unicode",
 'install'			=> "$install_program $install_options"
};
my $mk_vars = {}; # automatically generated from $sh_vars

my $toolchain_cfg_file	= "toolchain.cfg";
my $build_fw_target_cfg_file = "build_fw_target.cfg";
my $build_fw_cfg_file	= "build_fw.cfg";
my $config_mk		= "Config.mk";
my $config_sh		= "Config.sh";
my $config_site		= "config.site";
my $config_site_in	= "build_sys/config.site.in";
my $finalize_script	= "build_sys/finalize.sh";
my $fw_topdir		= "";
my $flags		= "";
my $webs_filepp_flags	= "";
my $fw_type		= "";
my $board		= "";
my $subboard		= "";
my $product		= "";
my $oem			= "";
my $oem_dir		= "";
my $lightoem		= "";
my $lightoem_dir	= "";
my $lightoem_dir_rel	= "";
my $oem_templatedir	= "mkfirmware/oem_template";
my $build_type		= "";
my $platform		= "";
my $toolchain_version	= "";
my $in_sdk		= 0;
my $errors		= 0;
my $remove_install_root	= 0;
my $preserve_install_root = 0;
my $operate_on_all_comps= 0;
my $opts		= {};
my $toolchain_cfg	= {};
my $target_cfg		= {};
my $build_cfg		= {};
my @operations		= ();
my @all_components	= ();
my @components		= ();
my @nobuild_components	= ();

my $starttime = time();
my $endtime;

#
###############################################################################
# subroutines

sub usage {
    print <<EOF;

Usage: build_fw <operation> [ subdir1 subdir2 ... subdirN ]

    operation - ( clean | build | cleanbuild | linkincl | copyincl <dest_dir> )

EOF
};

sub is_bool_option_true ($) {
    my $option = shift;
    return ($option =~ /true/i or
	    $option =~ /yes/i or
	    $option =~ /on/i or
	    $option =~ /1/);
};

sub rmtree_but_root ($) {
    my $rmdir = shift;

    opendir DIR, "$rmdir" or return;
    my @entries = grep(!/^root$/ && !/^\.+$/, readdir(DIR));
    closedir DIR;

    foreach(@entries) {
        rmtree(["$rmdir/$_"], 0, 1);
    }
}

#
###############################################################################
# main program

# determine the firmware root directory and change into it
$fw_topdir = $ARGV[0]; shift;
if (!defined($fw_topdir)) {
    print "\nNo firmware root directory specified!\n";
    usage;
    exit 1;
} elsif (!chdir($fw_topdir)) {
    die "\nCannot change into '$fw_topdir': $!\n\n";
}

# get options - this must be called after the firmware root and the operation
getopts('t:C', $opts);

# determine and check the requested operation
my $op = $ARGV[0]; shift;
my $opparam = "";
if (!defined($op)) {
    print "\nNo operation specified!\n";
    usage;
    exit 1;
} elsif (!grep(/^$op$/, ("clean", "build", "cleanbuild", "linkincl", "copyincl"))) {
    print "\nInvalid operation $op specified!\n";
    usage;
    exit 1;
} elsif ($op eq "copyincl") {
    if ($#ARGV < 0) {
	print "\noperation $op requires a parameter!\n";
	usage;
	exit 1;
    }
    $opparam = $ARGV[0]; shift;
    $opparam =~ s/\s*(\S*)\s*/$1/;
    if ( $opparam !~ /\/.*/) {
	my $dir = `pwd`;
	chomp($dir);
	$opparam = "$dir/$opparam";
    }
}

# set @operations
if ($op eq "cleanbuild") {
    @operations = ( "clean", "linkincl", "build" );
} elsif ($op ne "clean" and $op ne "linkincl") {
    @operations = ( "linkincl", $op );
} else {
    @operations = ( $op );
}

# read toolchain configuration
if (!open(CFG, $toolchain_cfg_file)) {
    print "\nCannot open $toolchain_cfg_file: $!\n\n";
    exit 1;
}
while (<CFG>) {
    $_ =~ s/#.*//;
    $_ =~ /^(\s*[^=]+?)\s*=\s*(.*?)\s*$/;
    if (defined($1) and defined($2) and $1 ne "" and $2 ne "") {
	$toolchain_cfg->{$1} = $2;
    }
}
close CFG;

if (defined($toolchain_cfg->{"version"})) {
    $toolchain_version = $toolchain_cfg->{"version"};
}

# read target configuration
if (!open(CFG, $build_fw_target_cfg_file)) {
    print "\nCannot open $build_fw_target_cfg_file: $!\n\n";
    print "A sample file can be found in 'build_sys'.\n\n";
    exit 1;
}
while (<CFG>) {
    $_ =~ s/#.*//;
    $_ =~ /^(\s*[^=]+?)\s*=\s*(.*?)\s*$/;
    if (defined($1) and defined($2) and $1 ne "" and $2 ne "") {
	$target_cfg->{$1} = $2;
    }
}
close CFG;

# check build type
if (!defined($target_cfg->{"build_type"})
    or !grep(/^$target_cfg->{"build_type"}$/, @build_types)) {
    print "\nNo or invalid build type specified in $build_fw_target_cfg_file.\n\n";
    exit 1;
}
$build_type = $target_cfg->{"build_type"};

# map fw_type, board, subboard and product to the hardware id
if (!defined($target_cfg->{"fw_type"})) {
    $target_cfg->{"fw_type"} = "erla"; # compatibility
}

if (!defined($target_cfg->{"fw_type"})
    or !defined($hw_id_map->{$target_cfg->{"fw_type"}})) {
    print "\nNo or invalid fw_type specified in $build_fw_target_cfg_file.\n\n";
    exit 1;
}
$fw_type = $target_cfg->{"fw_type"};
if (!defined($target_cfg->{"board"})
    or !defined($hw_id_map->{$fw_type}->{$target_cfg->{"board"}})) {
    print "\nNo or invalid board specified in $build_fw_target_cfg_file.\n\n";
    exit 1;
}
$board = $target_cfg->{"board"};
if (!defined($target_cfg->{"subboard"})
    or !defined($hw_id_map->{$fw_type}->{$board}->{$target_cfg->{"subboard"}})) {
    print "\nNo or invalid subboard specified in $build_fw_target_cfg_file.\n\n";
    exit 1;
}
$subboard = $target_cfg->{"subboard"};
if (!defined($target_cfg->{"product"})
    or !defined($hw_id_map->{$fw_type}->{$board}->{$subboard}->{$target_cfg->{"product"}})) {
    print "\nNo or invalid product specified in $build_fw_target_cfg_file.\n\n";
    exit 1;
}
$product = $target_cfg->{"product"};
if (!defined($target_cfg->{"oem"})) {
    print "\nNo oem specified in $build_fw_target_cfg_file.\n\n";
    exit 1;
}
$oem = $target_cfg->{"oem"};

# check for the OEM directory
if ($board eq $product) {
    $oem_dir = "OEM/${board}_$oem";
} else {
    $oem_dir = "OEM/${board}_${product}_$oem";
}
opendir DIR, 'OEM' or die "\nCannot open OEM directory.\n\n";
grep ( -d "$oem_dir", readdir(DIR) ) or die "\n$oem_dir doesn't exist.\n\n";
closedir DIR;

if (defined($target_cfg->{"in_sdk"}) and
    $target_cfg->{"in_sdk"} eq "1") {
    $in_sdk = 1;
}

if (defined($target_cfg->{"preserve_install_root"}) and
    is_bool_option_true($target_cfg->{"preserve_install_root"}) eq "1") {
    $preserve_install_root = 1;
}

# additional light OEM overlay stuff
if (defined($target_cfg->{"lightoem"})) {
    $lightoem = $target_cfg->{"lightoem"};
    $lightoem_dir_rel = $oem_dir . "_$lightoem";
    opendir DIR, 'OEM' or die "\nCannot open OEM directory.\n\n";
    if (!grep ( -d "$lightoem_dir_rel", readdir(DIR) )) {
	print "\n$lightoem_dir_rel doesn't exist, ignoring lightoem $lightoem.\n";
	$lightoem = "";
	$lightoem_dir_rel = "";
    }
    closedir DIR;
}

if ($lightoem_dir_rel ne "") {
	$sh_vars->{'lightoem_dir'} = "\${FW_TOPDIR}/$lightoem_dir_rel";
}

# set flags
$flags = $fwtype_board_flags->{$fw_type}->{$board};

# determine the hardware id
$sh_vars->{'hwid'} = $hw_id_map->{$fw_type}->{$board}->{$subboard}->{$product};
$sh_vars->{'hwid_int'} = "0x" . $sh_vars->{'hwid'};

# generate various DEFINES
$sh_vars->{'fw_defines_common'} .= " -DPP_FW_TYPE_" . uc($fw_type) . " -DPP_BOARD_" . uc($board);
$webs_filepp_flags .= " -DPP_FW_TYPE_" . uc($fw_type) . " -DPP_BOARD_" . uc($board);
if ($subboard ne "none") {
    $sh_vars->{'fw_defines_common'} .= " -D" . uc($board) . "_" . uc($subboard);
    $webs_filepp_flags .= " -D" . uc($board) . "_" . uc($subboard);
}
$platform = `uname | cut -b 1-6`;
chomp($platform);
$sh_vars->{'fw_defines_common'} .= " -DPP_HWID=" . $sh_vars->{'hwid'};
$webs_filepp_flags .= " -DPP_HWID=" . $sh_vars->{'hwid'};
$sh_vars->{'fw_defines_common'} .= " -DPP_HWID_INT=" . $sh_vars->{'hwid_int'};
$webs_filepp_flags .= " -DPP_HWID_INT=" . $sh_vars->{'hwid_int'};
$sh_vars->{'fw_defines_common'} .= " -DPP_PRODUCT=$product -DPRODUCT_" . uc($product);
$webs_filepp_flags .= " -DPP_PRODUCT=$product -DPRODUCT_" . uc($product);
if ($product =~ /01ip/) {
    $sh_vars->{'fw_defines_common'} .= " -DPRODUCT_XX01IP_ANY";
    $webs_filepp_flags .= " -DPRODUCT_XX01IP_ANY";
}
if ($build_type eq "final") {
    $sh_vars->{'fw_defines'} .= " -DNDEBUG";
}

$sh_vars->{'fw_defines'} .= " " . $flags->{"defines"};

# add dmalloc define for memory debugging according to build type
if ($build_type eq "dmalloc") {
    $sh_vars->{'fw_defines'} .= " -DDMALLOC";
}

$sh_vars->{'fw_defines'} .= " -DPP_BUILD_TYPE=$build_type";
$sh_vars->{'fw_defines'} .= " -DPP_BUILD_TYPE_".uc($build_type);

$sh_vars->{'fw_defines_common'} .= " -DOEM_" . uc($oem) . " -DPP_OEM=$oem";
$webs_filepp_flags .= " -DOEM_" . uc($oem);

# set opt/profile/debug flags
if ($build_type eq "debug") {
    $sh_vars->{'fw_cflags_common'}	   .= " " . $flags->{"dbgflags"};
    $sh_vars->{'fw_cflags_speed_common'}    = $sh_vars->{'fw_cflags_common'};
    $sh_vars->{'fw_cxxflags_common'}	   .= " " . $flags->{"dbgflags"};
    $sh_vars->{'fw_cxxflags_speed_common'}  = $sh_vars->{'fw_cxxflags_common'};
    $sh_vars->{'wxconfig'}  .= " --debug";
} elsif ($build_type eq "debugopt" ||
	 $build_type eq "dmalloc") {
    $sh_vars->{'fw_cflags_common'}	   .= " " . $flags->{"dbgoptflags"};
    $sh_vars->{'fw_cflags_speed_common'}    = $sh_vars->{'fw_cflags_common'};
    $sh_vars->{'fw_cxxflags_common'}	   .= " " . $flags->{"dbgoptflags"};
    $sh_vars->{'fw_cxxflags_speed_common'}  = $sh_vars->{'fw_cxxflags_common'};
} elsif ($build_type eq "profile") {
    $sh_vars->{'fw_cflags_common'}	   .= " " . $flags->{"profileflags"};
    $sh_vars->{'fw_cflags_speed_common'}    = $sh_vars->{'fw_cflags_common'};
    $sh_vars->{'fw_cxxflags_common'}	   .= " " . $flags->{"profileflags"};
    $sh_vars->{'fw_cxxflags_speed_common'}  = $sh_vars->{'fw_cxxflags_common'};
} else {
    $sh_vars->{'fw_cflags_common'}	   .= " " . $flags->{"coptflags"};
    $sh_vars->{'fw_cflags_speed_common'}   .= " " . $flags->{"coptspeedflags"};
    $sh_vars->{'fw_cxxflags_common'}	   .= " " . $flags->{"cxxoptflags"};
    $sh_vars->{'fw_cxxflags_speed_common'} .= " " . $flags->{"cxxoptspeedflags"};
}

# HACK: Determine linux kernel and uClibc directory
# 	 - assume that only one linux kernel is configured
# (skip missing directories)

if (!get_components(\@all_components, \@components, 1, $fw_type, $board, $subboard, $product, "none")) {
    die "\nError during determining the required components\n";
}
$flags->{"linux_kernel_dir"} = "";
$flags->{"uclibc_dir"} = "";
foreach my $component (@components) {
    if ( -d "$component/arch" and -d "$component/kernel") {
	$flags->{"linux_kernel_dir"} = "$component";
    }
    if ( -d "$component/libc" and -d "$component/ldso") {
	$flags->{"uclibc_dir"} = "$component";
    }
}

if ($flags->{"linux_kernel_dir"} eq "") {
    # some firmware types have no kernel
    #die "\nError determining linux kernel directory\n";
    $flags->{"linux_kernel_dir"} = "__unknown__";
}

if ($flags->{"uclibc_dir"} eq "") {
    # some firmware types have no uclibc
    #die "\nError determining uclibc directory\n";
    $flags->{"uclibc_dir"} = "__unknown__";
}

@all_components = ();
@components = ();

# fill-out templates
$flags->{"realcross"} =~ s/#TOOLCHAIN_VERSION#/$toolchain_version/;
$flags->{"host_system"} =~ s/#TOOLCHAIN_VERSION#/$toolchain_version/;
$sh_vars->{"toolchain_version"} =~ s/#TOOLCHAIN_VERSION#/$toolchain_version/;
$sh_vars->{"pp_java_home"} =~ s/#TOOLCHAIN_VERSION#/$toolchain_version/;

# generate $mk_vars
foreach my $var (keys %{$sh_vars}) {
    my $value = $sh_vars->{$var};
    $value =~ s/\${(.*?)}/\$($1)/g;
    $mk_vars->{$var} = $value;
}

# generate Config.mk
open(CONFIG_MK, ">$config_mk") or die "\nCannot open '$config_mk': $!\n\n";
print CONFIG_MK <<EOF;
-include $fw_topdir/pp_features.mk
TOOLCHAIN_VERSION	= $mk_vars->{'toolchain_version'}
CROSS			= \$(if \$(USE_REAL_CROSS),$flags->{'realcross'},$flags->{'cross'})
DISTCC			= \$(if \$(USE_DISTCC),\$(if \$(USE_REAL_CROSS),distcc,),)
AS			= \$(CROSS)as
AR			= \$(CROSS)ar
CXX			= \$(CROSS)g++
CC			= \$(strip \$(DISTCC) \$(CROSS)gcc)
CPP			= \$(CROSS)cpp
LD			= \$(CROSS)ld
NM			= \$(CROSS)nm
OBJCOPY			= \$(CROSS)objcopy
OBJDUMP			= \$(CROSS)objdump
RANLIB			= \$(CROSS)ranlib
SIZE			= \$(CROSS)size
STRINGS			= \$(CROSS)strings
STRIP			= \$(CROSS)strip
XGT			= xgettext

BINDIR			= /bin
SBINDIR			= /sbin
LIBDIR			= /lib
USRBINDIR		= /usr/bin
USRSBINDIR		= /usr/sbin
USRLIBDIR		= /usr/lib
UTESTDIR		= /utest
TRANSLATIONDIR		= /usr/po
TRANSLATIONTEMPDIR	= $fw_topdir/translation

FW_TOPDIR		= $fw_topdir
DESTDIR			= $mk_vars->{'destdir'}
FLASHDIR		= $mk_vars->{'flashdir'}
PP_BUILD_SYS_DIR	= $mk_vars->{'pp_build_sys_dir'}
PP_DOC_SYS_DIR		= $mk_vars->{'pp_doc_sys_dir'}
PP_JAVA_HOME		= $mk_vars->{'pp_java_home'}
FW_KERNEL_INCLUDES	= -include \$(FW_TOPDIR)/include/pp/features.h -I\$(FW_TOPDIR)/include/kernel
FW_INCLUDES		= \$(FW_KERNEL_INCLUDES) -I\$(FW_TOPDIR)/include -I\$(DESTDIR)/include 
FW_DEFINES_COMMON	= $mk_vars->{'fw_defines_common'}
FW_DEFINES		= \$(FW_DEFINES_COMMON) $mk_vars->{'fw_defines'}
FW_CPPFLAGS		= \$(FW_DEFINES) \$(FW_INCLUDES)
FW_CPPFLAGS_KERNEL	= \$(FW_DEFINES_COMMON) \$(FW_KERNEL_INCLUDES)
FW_CFLAGS		= \$(FW_CPPFLAGS) $mk_vars->{'fw_cflags_common'} $mk_vars->{'fw_cflags'} $flags->{'werror'}
FW_CFLAGS_SPEED		= \$(FW_CPPFLAGS) $mk_vars->{'fw_cflags_speed_common'} $mk_vars->{'fw_cflags'} $flags->{'werror'}
FW_CXXFLAGS		= \$(FW_CPPFLAGS) $mk_vars->{'fw_cxxflags_common'} $mk_vars->{'fw_cxxflags'} $flags->{'werror'}
FW_CXXFLAGS_SPEED	= \$(FW_CPPFLAGS) $mk_vars->{'fw_cxxflags_speed_common'} $mk_vars->{'fw_cxxflags'} $flags->{'werror'}
FW_LDFLAGS		= $mk_vars->{'fw_ldflags_common'} $mk_vars->{'fw_ldflags'} $flags->{'ldflags'}
FW_CPPFLAGS_3RDPARTY	= \$(FW_DEFINES_COMMON) \$(FW_INCLUDES)
FW_CFLAGS_3RDPARTY	= \$(FW_CPPFLAGS_3RDPARTY) $mk_vars->{'fw_cflags_common'}
FW_CXXFLAGS_3RDPARTY	= \$(FW_CPPFLAGS_3RDPARTY) $mk_vars->{'fw_cxxflags_common'}
FW_LDFLAGS_3RDPARTY	= $mk_vars->{'fw_ldflags_common'} $flags->{'ldflags'}

override INSTALL	= $mk_vars->{'install'}
override INSTALL_PROGRAM= \$(INSTALL) -m 755
override INSTALL_DATA	= \$(INSTALL) -m 644

HOST_SYSTEM		= $flags->{'host_system'}
PP_FW_TYPE		= $fw_type
PP_BOARD		= $board
PP_SUBBOARD		= $subboard
PP_PRODUCT		= $product
PP_OEM			= $oem
PP_LIGHTOEM		= $lightoem
PP_HWID			= $mk_vars->{'hwid'}
PP_HWID_INT		= $mk_vars->{'hwid_int'}
PP_PLATFORM		= $mk_vars->{'platform'}
OEMDIR			= \$(FW_TOPDIR)/$oem_dir
REL_OEMDIR		= $oem_dir
LIGHTOEMDIR		= $mk_vars->{'lightoem_dir'}
REL_LIGHTOEMDIR		= $lightoem_dir_rel
OEM_TEMPLATEDIR		= \$(FW_TOPDIR)/$oem_templatedir
PP_BUILD_TYPE		= $build_type
LINUX_KERNEL_DIR	= $flags->{'linux_kernel_dir'}
UCLIBC_DIR		= $flags->{'uclibc_dir'}
WEBS_FILEPP_FLAGS	= $webs_filepp_flags
PP_BUILD_ARCH           = $flags->{'arch'}
WXCONFIG		= $mk_vars->{'wxconfig'}
PATH		       := $mk_vars->{'path'}
export PATH
EOF
close CONFIG_MK;

# generate Config.sh
open(CONFIG_SH, ">$config_sh") or die "\nCannot open '$config_sh': $!\n\n";
print CONFIG_SH <<EOF;
set -a
TOOLCHAIN_VERSION="$sh_vars->{'toolchain_version'}"
[ -f $fw_topdir/pp_features.sh ] && source $fw_topdir/pp_features.sh
FW_TOPDIR="$fw_topdir"
if [ -n "\${USE_REAL_CROSS}" ]; then
  CROSS="$flags->{'realcross'}"
else
  CROSS="$flags->{'cross'}"
fi
CXX="\${CROSS}g++"
CC="\${CROSS}gcc"
CPPROG="cp -p"
INSTALL="$sh_vars->{'install'}"

DESTDIR="$sh_vars->{'destdir'}"
FLASHDIR="$sh_vars->{'flashdir'}"
PP_BUILD_SYS_DIR="$sh_vars->{'pp_build_sys_dir'}"
PP_DOC_SYS_DIR="$sh_vars->{'pp_doc_sys_dir'}"
PP_JAVA_HOME="$sh_vars->{'pp_java_home'}"
FW_KERNEL_INCLUDES="-include \${FW_TOPDIR}/include/pp/features.h -I\${FW_TOPDIR}/include/kernel"
FW_INCLUDES="\${FW_KERNEL_INCLUDES} -I\${FW_TOPDIR}/include -I\${DESTDIR}/include"
FW_DEFINES_COMMON="$sh_vars->{'fw_defines_common'}"
FW_DEFINES="\${FW_DEFINES_COMMON} $sh_vars->{'fw_defines'}"
FW_CPPFLAGS="\${FW_DEFINES} \${FW_INCLUDES}"
FW_CPPFLAGS_KERNEL="\${FW_DEFINES_COMMON} \${FW_KERNEL_INCLUDES}"
FW_CFLAGS="\${FW_CPPFLAGS} $sh_vars->{'fw_cflags_common'} $sh_vars->{'fw_cflags'} $flags->{'werror'}"
FW_CFLAGS_SPEED="\${FW_CPPFLAGS} $sh_vars->{'fw_cflags_speed_common'} $sh_vars->{'fw_cflags'} $flags->{'werror'}"
FW_CXXFLAGS="\${FW_CPPFLAGS} $sh_vars->{'fw_cxxflags_common'} $sh_vars->{'fw_cxxflags'} $flags->{'werror'}"
FW_CXXFLAGS_SPEED="\${FW_CPPFLAGS} $sh_vars->{'fw_cxxflags_speed_common'} $sh_vars->{'fw_cxxflags'} $flags->{'werror'}"
FW_LDFLAGS="$sh_vars->{'fw_ldflags_common'} $sh_vars->{'fw_ldflags'} $flags->{'ldflags'}"
FW_CPPFLAGS_3RDPARTY="\${FW_DEFINES_COMMON} \${FW_INCLUDES}"
FW_CFLAGS_3RDPARTY="\${FW_CPPFLAGS_3RDPARTY} $sh_vars->{'fw_cflags_common'}"
FW_CXXFLAGS_3RDPARTY="\${FW_CPPFLAGS_3RDPARTY} $sh_vars->{'fw_cxxflags_common'}"
FW_LDFLAGS_3RDPARTY="$sh_vars->{'fw_ldflags_common'} $flags->{'ldflags'}"

HOST_SYSTEM="$flags->{'host_system'}"
PP_FW_TYPE="$fw_type"
PP_BOARD="$board"
PP_SUBBOARD="$subboard"
PP_PRODUCT="$product"
PP_OEM="$oem"
PP_LIGHTOEM="$lightoem"
PP_HWID="$sh_vars->{'hwid'}"
PP_HWID_INT="$sh_vars->{'hwid_int'}"
PP_PLATFORM="$sh_vars->{'platform'}"
OEMDIR="\${FW_TOPDIR}/$oem_dir"
REL_OEMDIR="$oem_dir"
LIGHTOEMDIR="$sh_vars->{'lightoem_dir'}"
REL_LIGHTOEMDIR="$lightoem_dir_rel"
OEM_TEMPLATEDIR="\${FW_TOPDIR}/$oem_templatedir"
PP_BUILD_TYPE="$build_type"
LINUX_KERNEL_DIR="$flags->{'linux_kernel_dir'}"
UCLIBC_DIR="$flags->{'uclibc_dir'}"
PP_BUILD_ARCH="$flags->{'arch'}"
PATH="$sh_vars->{'path'}"
CONFIG_SITE="\${FW_TOPDIR}/$config_site"
set +a
source \${FW_TOPDIR}/build_sys/Build.inc.sh
EOF

if ($fw_type ne "ccf") {
    print CONFIG_SH <<EOF;
EOF
}
    
close CONFIG_SH;

# link Config.mk and Config.sh into various directories to
# reach compatibility to old non-flat eric_firmware repo
my @dirs = ( "pplinux", "pp_shared", "debug_tools", "pplinux/debug_tools", "pp_shared/debug_tools" );
foreach my $dir (@dirs) {
    if ( -d $dir ) {
	my $prefix = $dir;
	$prefix =~ s/[^\/\.]+/../g;
	unlink "$dir/$config_mk";
	symlink "$prefix/$config_mk", "$dir/$config_mk"
	    or die "\nSymlinking '$dir/$config_mk -> $prefix/$config_mk' failed: $!\n\n";
	unlink "$dir/$config_sh";
	symlink "$prefix/$config_sh", "$dir/$config_sh"
	    or die "\nSymlinking '$dir/$config_sh -> $prefix/$config_sh' failed: $!\n\n";
    }
}

# create config.site
open(CONFIG_SITE_IN, "<${config_site_in}") or die "\nCannot open '${config_site_in}': $!\n\n";
open(CONFIG_SITE, ">${config_site}") or die "\nCannot open '${config_site}' for writing: $!\n\n";
while (<CONFIG_SITE_IN>) {
	$_ =~ s/\@FW_TOPDIR\@/$fw_topdir/g;
	$_ =~ s/\@CPPFLAGS\@/\$FW_CPPFLAGS_3RDPARTY/g;
	$_ =~ s/\@CFLAGS\@/\$FW_CFLAGS_3RDPARTY/g;
	$_ =~ s/\@CXXFLAGS\@/\$FW_CXXFLAGS_3RDPARTY/g;
	$_ =~ s/\@LDFLAGS\@/\$FW_LDFLAGS_3RDPARTY/g;
	print CONFIG_SITE;
}
close CONFIG_SITE;
close CONFIG_SITE_IN;

# determine components that shall not be built (skip missing directories)
if (!get_components(\@all_components, \@nobuild_components, 1, $fw_type, $board, $subboard, $product, "nobuild")) {
    die "\nError during determining the required components\n";
}

@all_components = ();

# determine required components (skip missing directories)
if (!get_components(\@all_components, \@components, 1, $fw_type, $board, $subboard, $product, "none")) {
    die "\nError during determining the required components\n";
}

# filter requested directories if necessary
#   - @ARGV contains the user requested directories
#   - order in build_fw.cfg is preserved
#   - if the user specified nothing a complete build is done
if ($#ARGV >= 0) {
    $operate_on_all_comps = 0;
    my @tmp_components = ();
    foreach my $component (@components) {
        my $component_escaped = $component;
        $component_escaped =~ s/\+/\\\+/gs;
	if (grep(/^$component_escaped\/*$/, @ARGV)) {
	    push @tmp_components, $component;
	}
    }
    @components = @tmp_components;
} else {
    $operate_on_all_comps = 1;
    # determine if install_root/flash_root should be removed later
    $remove_install_root = (($op =~ "clean") || ($op =~ "build")) &&
                           ($build_type eq "final" ||
                            $preserve_install_root != 1);
}

# skip all components that have no executable Build script or a "nobuild" tag
my @tmp_components = ();    
COMPONENT: foreach my $component (@components) {
    if ( -x "$component/Build") {
	foreach my $nobuild_component (@nobuild_components) {
	    next COMPONENT if ($component eq $nobuild_component);
	}
	push @tmp_components, $component;
    }
}
@components = @tmp_components;
@tmp_components = ();    
foreach my $component (@all_components) {
    if ( -x "$component/Build" ) {
	push @tmp_components, $component;
    }
}
@all_components = @tmp_components;

# determine the install root suffix
my $suffix = ($build_type eq "final" ? "final" : "devel");

# remove install root if full build
if ($remove_install_root) {
    if($build_type eq "final") {
        # on final build, remove directory completely
        rmtree([ "install_root_$suffix" ], 0, 1);
    } else {
        # keep install_root for NFS! delete all but root
        rmtree_but_root("install_root_$suffix");
    }
    ( -d "install_root_$suffix/bin" ) and die "\nRemoving 'install_root_$suffix' failed.\n\n";
}

# create install root directory - create links
eval { mkpath([ "install_root_$suffix" ]) }; # mkpath doesnt whine about already existing dir
if ($@) { die "\nCreating 'install_root_$suffix' failed: $!\n\n"; }
unlink "install_root";
symlink "install_root_$suffix", "install_root"
    or die "\nSymlinking 'install_root -> install_root_$suffix' failed: $!\n\n";

# create flash directory for eric and lara boards
if ($board eq "eric" or $board eq "lara" or $board eq "kira") {
    unlink "flash_root";

    if ($remove_install_root) {
        rmtree([ "flash_root_$suffix" ], 0, 1);
        ( -d "flash_root_$suffix" ) and die "\nRemoving 'flash_root_$suffix' failed.\n\n";
    }

    if ($build_type eq "final") {
	eval { mkpath([ "flash_root_$suffix" ]) }; # mkpath doesnt whine about already existing dir
	if ($@) { die "\nCreating 'flash_root_$suffix' failed: $!\n\n"; }

	symlink "flash_root_$suffix", "flash_root"
	    or die "\nSymlinking 'flash_root -> flash_root_$suffix' failed: $!\n\n";	
    } else {
	# on devel we use part of our install_root like flash
	eval { mkpath([ "install_root_$suffix/flashdisk" ]) };
	symlink "install_root_$suffix/flashdisk", "flash_root"
	    or die "\nSymlinking 'flash_root -> install_root_$suffix/flashdisk' failed: $!\n\n";
    }

    foreach my $flashdisk_part (@flashdisk_parts) {
	eval { mkpath([ "flash_root/$flashdisk_part" ]) };
    }
}

# create OEM template dir for collecting OEM data for outside users
if ($board eq "eric" or $board eq "lara" or $board eq "kira" or $board eq "kxng") {
    eval { mkpath([ "$oem_templatedir" ]) };
    if ($@) { die "\nCreating '$oem_templatedir' failed: $!\n\n"; }
}

# do the build
foreach my $operation (@operations) {
    my $rc;
    my $success;
    my $signal = 0;
    my $in_sdk_param = $in_sdk ? "IN_SDK=1" : "";
    my @tmp_components = $operation eq "clean" ? reverse @components : @components;
    # do some extra stuff if we operate on all components
    # NOTE: this must be done before cleaning the components
    if ($operate_on_all_comps) {
	# recreate pp_features.* before cleaning components
	if ($operation eq "clean") {
	    system "make -f build_sys/gen_pp_features.mk $in_sdk_param clean all";
	}
	# extra stuff which is done in real firmware builds and not for the SDK
	if (!$in_sdk) {
	    # remove local include directory on "linkincl" operation
	    if ($operation eq "linkincl") {
		rmtree([ "include" ], 0, 1);
		( -d "include" ) and die "\nRemoving 'include' failed.\n\n";
	    }
	}
    }

    if ($operation eq "linkincl") {
	print "\033]0;linking includes\007";
	print "===============================================================================\n";
	print "linking includes\n";
	print "-------------------------------------------------------------------------------\n\n";
	eval { mkpath([ "include/pp" ]) };
	if ($@) { die "\nCreating 'include/pp' failed: $!\n\n"; }
	$rc = system "make -f build_sys/gen_pp_features.mk $in_sdk_param";
	if ($rc) { $errors++; }
	foreach my $component (@components, @nobuild_components) {
	    $success = 1;
	    $rc = system "cd $component && ( [ ! -r ./Build ] || ./Build linkincl )";
	    if ($rc) { $success = 0; }
	    if ($rc & 127) { $signal = $rc & 127; }
	    if (not $success) { $errors++; }
	    if ($signal) { last; }
	}
	# call symlinks twice to fully simplify links - this is a symlinks bug
	$rc = system "symlinks -csr ./include/ > /dev/null";
	if ($rc) { $errors++; }
	$rc = system "symlinks -csr ./include/ > /dev/null";
	if ($rc) { $errors++; }
	print "\n-------------------------------------------------------------------------------\n";
	print !$errors ? "SUCCESS" : "ERROR", $signal ? "(signal $signal)" : "",
	    ": linking includes\n";
	print "===============================================================================\n\n";
    } else {
	my $counter=0;
	foreach my $component (@tmp_components) {
	    $success = 1;
	    $counter++;
	    print "\033]0;$operation on '$component' for $board/$subboard/$product ($counter of ".scalar(@tmp_components).", errors: " . $errors . ")\007";
	    print "===============================================================================\n";
	    print "$operation on '$component' for $board/$subboard/$product\n";
	    print "-------------------------------------------------------------------------------\n\n";
	    if ($operation eq "copyincl") {
		$rc = system "cd $component && ./Build copyincl $opparam";
		if ($rc) { $success = 0; }
		if ($rc & 127) { $signal = $rc & 127; }
	    } elsif ($operation eq "clean") {
		$rc = system "cd $component && ./Build clean";
		if ($rc) { $success = 0; }
		if ($rc & 127) { $signal = $rc & 127; }
	    } elsif ($success and !$signal and $operation eq "build") {
		$rc = system "cd $component && ./Build";
		if ($rc) { $success = 0; }
		if ($rc & 127) { $signal = $rc & 127; }
	    }
	    print "\n-------------------------------------------------------------------------------\n";
	    print $success ? "SUCCESS" : "ERROR", $signal ? "(signal $signal)" : "",
		": $operation on '$component' for $board/$subboard/$product\n";
	    print "===============================================================================\n\n";
	    
	    if (not $success) { $errors++; }
	    if ($signal) { last; }
	}

	# do some extra cleanup stuff if we operate on all components
	# NOTE: this must be done after cleaning the components
	if ($operate_on_all_comps and $operation eq "clean") {
	    # remove pp_features.*
	    system "make -f build_sys/gen_pp_features.mk clean";

	    # extra stuff which is done in real firmware builds and not for the SDK
	    if (!$in_sdk) {
		# remove local include directory on "clean" and "linkincl" operation
		rmtree([ "include" ], 0, 1);
		( -d "include" ) and die "\nRemoving 'include' failed.\n\n";
	    }
	    # remove local bin directory on "clean" operation
	    rmtree([ "bin" ], 0, 1);
	    ( -d "bin" ) and die "\nRemoving 'bin' failed.\n\n";
	}
    }

    # run finalize script if final build and no errors occured
    if ($errors == 0 and $build_type eq "final" and $operation eq "build" and scalar(@tmp_components) > 0) {
	$success = 1;
	print "\033]0;Finalizing firmware for $board/$subboard/$product\007";
	print "===============================================================================\n";
	print "Finalizing firmware for $board/$subboard/$product\n";
	print "-------------------------------------------------------------------------------\n\n";
	if (system "$finalize_script") { $success = 0; }
	print "\n-------------------------------------------------------------------------------\n";
	print $success ? "SUCCESS" : "ERROR",
	    ": Finalizing firmware for $board/$subboard/$product\n";
	print "===============================================================================\n\n";
	if (not $success) { $errors++; }
    }
}

# if necessary print a warning about errors
if ($errors) {
    print "*******************************************************************************\n";
    print "WARNING: There where $errors error(s) encountered!\n";
    print "*******************************************************************************\n\n";
}

if ($errors) {
	print "\033]0;build_fw finished with $errors errors\007";	
} else {
	print "\033]0;build_fw finished successfully\007";
}

$endtime = time();

my $timediff = $endtime - $starttime;
my $seconds = $timediff % 60;
my $minutes = ($timediff / 60) % 60;
my $hours = int(($timediff / 60) / 60);

print "elapsed time: $hours h, $minutes m, $seconds s\n\n";

my $print_size_stats = ($build_type eq "final") && $operate_on_all_comps;

if ($print_size_stats) {

    my $size_fw = `du -B M -s $fw_topdir`;
    $size_fw =~ s/^([^\s]*)\s.*/$1/;
    chop $size_fw;

    my $size_install_root = `du -B M -s \"$fw_topdir/install_root/\"`;
    $size_install_root =~ s/^([^\s]*)\s.*/$1/;
    chop $size_install_root;

    print "size of complete firmware dir: $size_fw\n";
    print "size of install root dir: $size_install_root\n";
    my $size_initrd = "";
    if ( -r "$fw_topdir/mkfirmware/initrd.bin" ) {
	$size_initrd = `du -s $fw_topdir/mkfirmware/initrd.bin`;
    } elsif ( -r "$fw_topdir/mkfirmware/rootfs.tgz" ) {
	$size_initrd = `du -s $fw_topdir/mkfirmware/rootfs.tgz`;
    }
    if ( $size_initrd) {
	$size_initrd =~ s/^([^\s]*)\s.*/$1/;
	chop $size_initrd;
	print "size of initrd: $size_initrd kB\n";
    }
}

exit $errors ? 1 : 0;

#
###############################################################################
# Emacs
#
# Local Variables:
# cperl-indent-level:4
# cperl-continued-statement-offset:4
# cperl-continued-brace-offset:-4
# cperl-brace-offset:0
# cperl-brace-imaginary-offset:0
# cperl-label-offset:-2
# cperl-hairy:t
# perl-tab-always-indent:nil
# End:
###############################################################################
