#!/usr/bin/perl -w
###############################################################################
# This is a script to checkout al required firmware components for a specific
# product.
#

package ppsetview;

use strict;

#
###############################################################################
# imports

use FindBin;
use lib "$FindBin::Bin";

use Getopt::Std;
use File::Path;
use File::Find;
use pp_hw_data;
use pp_build_lib;

#
###############################################################################
# global variables

my @all_components = ();
my @components = ();
my @existing_components = ();
my %uncommitted_by_component = ();
my %updated_by_component = ();
my %conflicts_by_component = ();
my %cvserror_by_component = ();
my $opts = {};
my $operation;
my $cvs_module;
my $cvs_root;
my $cvs_tag;

#
###############################################################################
# subroutines

sub usage {
    print <<EOF;

Usage: ppsetview <operation> <options>

    <operations> are selected using capitalized command-line switches.
    <options> are specified using lower case command-line switches.

    -S           Setting a view by checking out the required components from
                 CVS. The view is determined by various configuration items
                 that are specified interactively once.

        -a       Checkout all possible components for a particular firmware
                 type. This must be used for branching and merging.

        -b <cfg> Batch mode. <cfg> is a colon/comma/semicolon seperated list

                 fwtype:board:sub-board:product:oem:light-oem:build-type

        -n       Only checkout new components and don't touch existing. This
                 may lead to inconsistencies on the local CVS tree but reduces
                 CVS traffic quite a lot. Use this option with care!

        -r       Reconfigure the view parameters.
        
        -l       List unnecessary components (e.g. after branching).

        -c       Clean up, i.e. remove unnecessary components.

    -A <name>    Add a new component called <name>.

        -f       Create the component framework. The framework depends on the
                 name of the component. If the name starts with "libpp_" it
                 will create a framework for libraries. Otherwise a framework
                 for executables is created.
                 NOTE: The framework is created only if the component directory
                       does not yet exist on the local side or if it is empty!

    -M           Merge a branch.

        -t       This specifies the firmware type (e.g. erla). It is used for
                 determining the components to merge. This option has a second
                 meaning: If ommitted the build system only is merged.
                 Otherwise the components only.

        -s       The tag in the branch from where to start the merge. On
                 consecutive merges you use the end tag of the last merge as
                 start tag for the current one. This tag is ommitted if a
                 branch is mergend for the first time. In this case the branch
                 point is used by default.

        -e       The tag in the branch up to which changes should be merged.

EOF
};

sub chooser (\@$$$$) {
    my $list = shift;
    my $default = shift;
    my $prompt = shift;
    my $reconf_view = shift;
    my $batch_mode = shift;
    my $menu = "";
    my $cnt = 0;
    my $val;
    my $defval;

    foreach my $list_elem (@{$list}) {
	my $sel = " ";
	if (defined $default and $list_elem eq $default) {
	    $defval = $cnt;
	    $sel = ">";
	}
	$menu .= sprintf "  %s %2u - %s\n", $sel, $cnt++, ($list_elem ne "") ? $list_elem : "<none>";
    }

    if (not defined $defval and $batch_mode) {
	if (scalar(@{$list}) > 0) {
	    die "\nInvalid value '" . (defined $default ? $default : "") . "' entered on prompt '$prompt'.\n\n";
	} else {
	    return "";
	}
    } elsif (not defined $defval or $reconf_view) {
	if (scalar(@{$list}) > 0) {
	    print "\n$menu\n";
	    do {
		print "$prompt ";
		$val = <STDIN>;
		$val =~ s/\s*(\S*)\s*/$1/;
		$val =~ /^$/ and defined($defval) and $val = $defval;
	    } while ($val !~ /^[0-9]+$/ or $val >= $cnt);
	    return $list->[$val];
	} else {
	    return "";
	}
    } else {
	return $list->[$defval];
    }
}

sub cvs_operation ($$$$) {
    my $component = shift;
    my $message = shift;
    my $cvs_operation = shift;
    my $cvslog_handle = shift;
    local *CVS;

    print "$message: ";
    print $cvslog_handle "---- CVS: $message ...'\n";

    # start CVS operation
    if (not open CVS, "$cvs_operation 2>&1 |") {
	print "failed\n";
	print $cvslog_handle "!!!! CVS: Executing cvs command ($cvs_operation) failed ($!).\n\n";
	return 0;
    }

    # copy CVS output to log file
    while (<CVS>) {
	if ($_ =~ /Merging differences /) {
	    $updated_by_component{$component} = 1;
	}
	if ($_ =~ /^C / or $_ =~ /conflicts during merge/) {
	    $conflicts_by_component{$component} = 1;
	} elsif ($_ =~ /^M / or $_ =~ /^A / or $_ =~ /^R /) {
	    $uncommitted_by_component{$component} = 1;
	} elsif ($_ =~ /^U / or $_ =~ /^P /) {
	    $updated_by_component{$component} = 1;
	} 

	print CVSLOG "$_";
    }
    close CVS;

    # evaluate exit code
    if ($? == 0) {
	print "done\n";
	print $cvslog_handle "---- CVS: $message finished.\n\n";
	return 1;
    } else {
	print "failed\n";
	print $cvslog_handle "!!!! CVS: $message failed.\n\n";
	return 0;
    }
}

sub checkout_component ($$$$$) {
    my $cvs_root = shift;
    my $cvs_module = shift;
    my $cvs_tag = shift;
    my $component = shift;
    my $checkout_new_only = shift;

    if ( -d $component ) {
	# determine the CVS repository of the component
	open CVSREPO, "$component/CVS/Repository" or die "\nCannot open $component/CVS/Repository ($!).\n\n";
	my $cvs_module_comp = <CVSREPO>;
	$cvs_module_comp =~ s/\s*(\S*)\s*/$1/;
	$cvs_module_comp =~ s/\/.*//;
	close CVSREPO;
	
	# components have a different cvs module - so if we have a component
	# with the same module as our base directory then someone has
	# added a component to the base directory. This is not allowed but
	# cannot entirely prevented administratively with CVS. Since file
	# checkins are prevented we can assume that only empty directories
	# are in the way that can be pruned using "cvs update -dP".
	if ($cvs_module eq $cvs_module_comp) {
	    if (not cvs_operation($component,
				  "Pruning '$component'",
				  "cvs -z3 update -dP \"$component\"",
				  \*CVSLOG)) {
		$cvserror_by_component{$component} = 1;
		return 0;
	    }
	}
    }

    if ($checkout_new_only) {
	if (-d $component ) {
	    push @existing_components, $component;
	    return 1;
	}
    }

    my $branch_spec = $cvs_tag ne "" ? "-r $cvs_tag" : "-A";
    if (not cvs_operation($component,
			  "Checking out '$component'",
			  "cvs -z3 -d \"$cvs_root\" co -P -d \"$component\" $branch_spec \"fwcomponents/$component\"",
			  \*CVSLOG)) {
	$cvserror_by_component{$component} = 1;
	return 0;
    }

    return 1;
}

sub merge_component ($$$$$) {
    my $cvs_root = shift;
    my $cvs_module = shift;
    my $cvs_start_tag = shift;
    my $cvs_end_tag = shift;
    my $component = shift;

    if ( -d $component ) {
	# determine the CVS repository of the component
	open CVSREPO, "$component/CVS/Repository" or die "\nCannot open $component/CVS/Repository ($!).\n\n";
	my $cvs_module_comp = <CVSREPO>;
	$cvs_module_comp =~ s/\s*(\S*)\s*/$1/;
	$cvs_module_comp =~ s/\/.*//;
	close CVSREPO;
	
	# components have a different cvs module - so if we have a component
	# with the same module as our base directory then someone has
	# added a component to the base directory. This is not allowed but
	# cannot entirely prevented administratively with CVS. Since file
	# checkins are prevented we can assume that only empty directories
	# are in the way that can be pruned using "cvs update -dP".
	if ($cvs_module eq $cvs_module_comp) {
	    if (not cvs_operation($component,
				  "Pruning '$component'",
				  "cvs -z3 update -dP \"$component\"",
				  \*CVSLOG)) {
		$cvserror_by_component{$component} = 1;
		return 0;
	    }
	}
    }

    my $start_tag_spec = $cvs_start_tag ne "" ? "-j $cvs_start_tag" : "";
    my $end_tag_spec = $cvs_end_tag ne "" ? "-j $cvs_end_tag" : "";
    if (not cvs_operation($component,
			  "Merging '$component'",
			  "cvs -z3 -d \"$cvs_root\" co -P -d \"$component\" $start_tag_spec $end_tag_spec \"fwcomponents/$component\"",
			  \*CVSLOG)) {
	$cvserror_by_component{$component} = 1;
	return 0;
    }

    return 1;
}

sub remove_hidden_fwcomponents_root () {
    # removing hidden fwcomponents root (uff we call "rm -rf" - be cautious!)
    print "Removing hidden fwcomponents root: ";
    system("rm -rf .fwcomponents");
    if ($? == 0) {
	print "done\n";
    } else {
	print "failed (ignored)\n";
    }
    return 0;
}

#
###############################################################################
# main program

# save command-line parameters
my @save_ARGV = @ARGV;

# determine the firmware root directory and change into it
my $fw_topdir = $ARGV[0]; shift;
if (!defined($fw_topdir)) {
    print STDERR "\nNo firmware root directory specified!\n";
    #usage;
    exit 1;
} elsif (!chdir($fw_topdir)) {
    die "\nCannot change into '$fw_topdir': $!\n\n";
}

# get options - this must be called after the firmware root and the operation
getopts('Sab:nrlcA:fMt:s:e:', $opts);

# check command-line options
if (defined($opts->{"S"})) {
    $operation = "setview";
}
if (defined($opts->{"A"})) {
    if (defined($operation)) {
	die "Error: Conflicting operations specified!\n";
    }
    $operation = "add";
}
if (defined($opts->{"M"})) {
    if (defined($operation)) {
	die "Error: Conflicting operations specified!\n";
    }
    $operation = "merge";
}
if (!defined($operation)) {
    print STDERR "You must specify at least one operation\n";
    usage();
    exit 1;
}

# determine the CVS root
open CVSROOT, "CVS/Root" or die "\nCannot open CVS/Root ($!).\n\n";
$cvs_root = <CVSROOT>;
$cvs_root =~ s/\s*(\S*)\s*/$1/;
close CVSROOT;

# determine the CVS module
open CVSREPO, "CVS/Repository" or die "\nCannot open CVS/Repository ($!).\n\n";
$cvs_module = <CVSREPO>;
$cvs_module =~ s/\s*(\S*)\s*/$1/;
$cvs_module =~ s/\/.*//;
close CVSREPO;

# determine branch
if (open CVSTAG, "CVS/Tag") {
    $cvs_tag = <CVSTAG>;
    $cvs_tag =~ s/\s*\S(\S*)\s*/$1/;
    close CVSTAG;
} elsif ( -f "CVS/Tag" ) {
    # possible permission problem
    die "\nCannot read CVS/Tag ($!).\n\n";
} else {
    $cvs_tag = ""
}

#------------------------------------------------------------------------------
# SETVIEW OPERATION
#------------------------------------------------------------------------------
if ($operation eq "setview") {
    # check setview command-line options
    my $checkout_all = defined($opts->{"a"});
    my $batch_mode = defined($opts->{"b"});
    my $checkout_new_only = defined($opts->{"n"});
    my $reconf_view = defined($opts->{"r"});
    my $list = defined($opts->{"l"});
    my $cleanup = defined($opts->{"c"});

    my $fw_type;
    my $board;
    my $subboard;
    my $product;
    my $oem = "";
    my $lightoem = "";
    my $build_type;
    my $preserve_install_root = "no";

    if ($batch_mode) {
	($fw_type, $board, $subboard, $product, $oem, $lightoem, $build_type) = split /[,;:]/, $opts->{"b"};
    }

    open CVSLOG, ">cvs.log" or die "\nCannot open cvs.log for writing ($!).\n\n";

    # generate md5sums of ourself to check modification later
    system ("md5sum build_sys/ppsetview.pl build_sys/pp_build_lib.pm build_sys/pp_hw_data.pm > ppsetview.md5");

    # update the base directory (NOTE: subdirectories must be specified here manually!)
    my @base_subdirs = ( "build_sys" );
    if (not cvs_operation("<base>", "Updating base directory", "cvs -z3 update -l -P", \*CVSLOG)) {
	exit 1;
    }
    foreach my $base_subdir (@base_subdirs) {
	if (not cvs_operation("<base>", "Updating directory '$base_subdir'", "cvs -z3 update -dP $base_subdir", \*CVSLOG)) {
	    exit 1;
	}
    }

    # check if some bad guy modified us and restart in this case
    system ("md5sum -c ppsetview.md5 > /dev/null 2>&1");
    if ($? != 0) {
	close CVSLOG;
	if (defined($conflicts_by_component{"<base>"})) {
	    die "\nWe were modified and a conflict occured. Check cvs.log! Aborting ...\n\n";
	} elsif (defined($cvserror_by_component{"<base>"})) {
	    die "\nWe were modified and a CVS error occured. Check cvs.log! Aborting ...\n\n";
	} else {
	    print "\nWe were modified. Restarting ourself ...\n\n";
	    exec $0 $0, @save_ARGV;
	    die("\nRestarting ourself failed\n\n");
	}
    }

    # read the target configuration if not in batch mode
    if (not $batch_mode and open TARGET_CONFIG, $build_fw_target_cfg_file) {
	my $cnt = 0;
	my $error = 0;
	my $errmsg = "";
	print "\nReading build target configuration file: ";
	while (<TARGET_CONFIG>) {
	    $cnt++;
	    $_ =~ /#.*/ and next;
	    $_ =~ /^$/ and next;
	    $_ =~ /^(\s*[^=]+?)\s*=\s*(.*?)\s*$/;
	    if (defined($1) and defined($2)) {
		if ($1 eq "fw_type") {
		    $fw_type = $2;
		} elsif ($1 eq "board") {
		    $board = $2;
		} elsif ($1 eq "subboard") {
		    $subboard = $2;
		} elsif ($1 eq "product") {
		    $product = $2;
		} elsif ($1 eq "oem") {
		    $oem = $2;
		} elsif ($1 eq "lightoem") {
		    $lightoem = $2;
		} elsif ($1 eq "build_type") {
		    $build_type = $2;
		} elsif ($1 eq "preserve_install_root") {
		    $preserve_install_root = $2;
		} else {
		    $error = 1;
		    $errmsg .= "$build_fw_target_cfg_file: Invalid line $cnt\n";
		}
	    }
	}
	close TARGET_CONFIG;
	if ($error) {
	    print "failed\n";
	    die "\n$errmsg\n\n";
	} else {
	    print "done\n";
	}
    }

    # ask for firmware type
    my @fw_types = sort keys %{$hw_id_map};
    $fw_type = chooser(@fw_types, $fw_type, "Choose firmware type>", $reconf_view, $batch_mode);

    # ask for board
    my @boards = sort keys %{$hw_id_map->{$fw_type}};
    $board = chooser(@boards, $board, "Choose board>", $reconf_view, $batch_mode);

    # ask for subboard
    my @subboards = sort keys %{$hw_id_map->{$fw_type}->{$board}};
    $subboard = chooser(@subboards, $subboard, "Choose sub-board>", $reconf_view, $batch_mode);

    # ask for product
    my @products = sort keys %{$hw_id_map->{$fw_type}->{$board}->{$subboard}};
    $product = chooser(@products, $product, "Choose product>", $reconf_view, $batch_mode);

    # determine required components (do *not* skip missing directories)
    if (!get_components(\@all_components, \@components, 0, $fw_type, $board, $subboard, $product, "none")) {
	die "\nError during determining the required components\n";
    }

    # do we want all components?
    if ($checkout_all) {
	@components = @all_components;
    }

    # cut off any subtrees of components
    for (0 .. $#components) {
	$components[$_] =~ s/\/.*//;
    }

    # list or clean up?
    if (($list || $cleanup) && ! $checkout_all) {
        for (0 .. $#all_components) {
            $all_components[$_] =~ s/\/.*//;
        }
        
        # from perl cookbook 4.8.2.3
        my %seen;
        @seen {@all_components} = ( );
        delete @seen {@components};
        
        my @clean_components = ();
        foreach my $component (keys %seen) {
            opendir(DIR, $component) && push(@clean_components, $component);
            closedir(DIR);
        }
        my $clean_components = join(" ", @clean_components);
        
        if (! $clean_components) {
            print "\nNothing to clean up.\n";
        } else {
            if ($cleanup) {
                my @not_cleaned_components = ();
                print "\nRemoving unnecessary components:\n$clean_components\n\nPress return to continue...";
                my $foo = <STDIN>;
                
                COMPONENT: foreach my $clean_component (@clean_components) {
                    local *CVS;
                    print "Verifying status of $clean_component: ";

                    # similar to cvs_operation but do not log!
                    if (not open CVS, "cvs -n -z3 -Q up -dP demontic 2>&1 |") {
                        print "failed\n";
                        push(@not_cleaned_components, $clean_component);
                        next COMPONENT;
                    }

                    # parse CVS output
                    while (<CVS>) {
                        if ($_ =~ /^[ARMC\?] /) {
                            # component was somehow modified, request user interaction
	                    close CVS;
                            print "modified\n";
                            push(@not_cleaned_components, $clean_component);
                            next COMPONENT;
                        } 
                    }
                    close CVS;
                    
                    system "rm -rf $clean_component";
                    print "ok... removing...\n";
                }
                
                my $not_cleaned_components = join(" ", @not_cleaned_components);
                if ($not_cleaned_components) {
                    print "\nPlease check an manually remove not yet cleaned components:\n$not_cleaned_components\n";
                }
            } else {    
                print "\nSuggest to clean unnecessary components:\n$clean_components\n\nCall ppsetview -S -c to clean up.\n";
            }
        }
        
        # stop here?
        exit 0;
    }

    # skip duplicate and the build_sys component
    my @new_components = ();
    foreach my $component (@components) {
	$component =~ /^build_sys$/ and next; # build_sys is not a real component
	$component =~ /^Source(\/.*|)$/ and next; # "Source" is a special Raritan component.
	my $dup = 0;
	foreach my $new_component (@new_components) {
	    $new_component eq $component and $dup = 1 and last;
	}
	$dup or push @new_components, $component;
    }
    @components = @new_components;

    my %oems;
    sub _find_oems () {
	if ( -d $_ ) {
	    if ($_ =~ /^([^_]+)_([^_]+)$/ and $board eq $product and $1 eq $board) {
		$oems{$2} = 1;
	    } elsif ($_ =~ /^([^_]+)_([^_]+)_([^_]+)/ and $1 eq $board and $2 eq $product) {
		$oems{$3} = 1;
	    }
	}
    }

    my %lightoems;
    sub _find_lightoems () {
	if ( -d $_ ) {
	    if ($_ =~ /^([^_]+)_([^_]+)_([^_]+)_([^_]+)$/ and $1 eq $board and $2 eq $product and $3 eq $oem) {
		$lightoems{$4} = 1;
	    }
	}
    }

    foreach my $component (@all_components) {
	if ($component eq "OEM") {
	    print "\n";
	    if (checkout_component($cvs_root, $cvs_module, $cvs_tag, $component, $checkout_new_only)) {
		# scan OEM directory
		find({ wanted => \&_find_oems, no_chdir => 0 }, $component);
		my @oems = sort keys %oems;
		# ask for OEMs
		$oem = chooser(@oems, $oem, "Choose OEM>", $reconf_view, $batch_mode);
		
		# scan OEM directory
		find({ wanted => \&_find_lightoems, no_chdir => 0 }, $component);
		my @lightoems = sort keys %lightoems;
		if (scalar(@lightoems) > 0) {
		    unshift @lightoems, "";
		}
		# ask for light-OEMs
		$lightoem = chooser(@lightoems, $lightoem, "Choose light-OEM>", $reconf_view, $batch_mode);
	    }
	    last;
	}
    }

    # ask for the build type
    $build_type = chooser(@build_types, $build_type, "Choose build type>", $reconf_view, $batch_mode);

    # write the target configuration
    print "\nWriting build target configuration file: ";
    if (!open TARGET_CONFIG, ">$build_fw_target_cfg_file") {
	print "failed\n";
	die "\nCannot open '$build_fw_target_cfg_file' for writing ($!)\n\n";
    }

    print TARGET_CONFIG <<EOF;
###############################################################################
# This file determines for which target the firmware will be build
#
# ATTENTION: DO NOT EDIT THIS FILE! IT IS CREATED AUTOMATICALLY.
#

fw_type = $fw_type
board = $board
subboard = $subboard
product = $product
oem = $oem
lightoem = $lightoem
build_type = $build_type
EOF
    close TARGET_CONFIG;
    print "done\n\n";

    print "---- Configuration --------------------------------------\n";
    print "CVS tag       : " . ($cvs_tag ne "" ? $cvs_tag : "MAIN") . "\n";
    print "Firmware type : $fw_type\n";
    print "Board         : $board\n";
    print "Sub-board     : $subboard\n";
    print "Product       : $product\n";
    print "OEM           : $oem\n";
    print "Light-OEM     : $lightoem\n";
    print "Build type    : $build_type\n";
    print "---------------------------------------------------------\n\n";

    # set default checkout CVS server
    my $checkout_cvs_root = $cvs_root;
    my $mirror_present = 0;

    # check for r/o mirror server
    open MIRROR, "$ENV{HOME}/.ppsetview/mirror" and $mirror_present = 1;
    if ($mirror_present) {
	my $cvs_mirror = <MIRROR>;
	$cvs_mirror =~ s/\s*(\S*)\s*/$1/;
	$checkout_cvs_root = $cvs_mirror;
    }

    # check if the user needs to enter authentication data for the cvs mirror
    system "cvs -d '$checkout_cvs_root' status fwbase/build_fw 2>&1 | grep -q 'cvs login'";
    if ($? == 0) {
    	# user must enter valid authentication data
	do {
	    system "cvs -d '$checkout_cvs_root' login";
	} while ($? != 0);
    }
    
    foreach my $component (@components) {
	$component =~ /^OEM$/ and next; # skip OEM component - we already processed it
	checkout_component($checkout_cvs_root, $cvs_module, $cvs_tag, $component, $checkout_new_only);
	if ($mirror_present) {
		`find $component -name Root -exec sed -i s,$checkout_cvs_root,$cvs_root, {} \\;`; 
		# do a CVS update after checking out from the mirror
	    	cvs_operation($component, "Updating directory '$component'", "cvs -z3 update -dP $component", \*CVSLOG)
	}	
    }
    close CVSLOG;
    print "\n";

    if (scalar(@existing_components) > 0) {
	print "Already checked out components that are left untouched:\n\n";
	foreach my $component (sort @existing_components) {
            # keep the two leading spaces - required for automatic build system!
	    print "  $component\n";
	}
	print "\n";
    }

    if (scalar(keys %uncommitted_by_component) > 0) {
	print "Components with uncommitted files:\n\n";
	foreach my $component (sort keys %uncommitted_by_component) {
            # keep the two leading spaces - required for automatic build system!
	    print "  $component\n";
	}
	print "\n";
    }

    if (scalar(keys %updated_by_component) > 0) {
	print "Components with updated files:\n\n";
	foreach my $component (sort keys %updated_by_component) {
            # keep the two leading spaces - required for automatic build system!
	    print "  $component\n";
	}
	print "\n";
    }

    if (scalar(keys %conflicts_by_component) > 0) {
	print "Components with conflicts:\n\n";
	foreach my $component (sort keys %conflicts_by_component) {
            # keep the two leading spaces - required for automatic build system!
	    print "  $component\n";
	}
	print "\n";
    }

    if (scalar(keys %cvserror_by_component) > 0) {
	print "CVS errors:\n\n";
	foreach my $component (sort keys %cvserror_by_component) {
            # keep the two leading spaces - required for automatic build system!
	    print "  $component\n";
	}
	print "\n";
    }
    if (scalar(keys %conflicts_by_component) > 0 or scalar(keys %cvserror_by_component) > 0) {
	die "\nYou should check cvs.log for errors and/or merge conflicts!\n\n";
    }
#------------------------------------------------------------------------------
# ADD OPERATION
#------------------------------------------------------------------------------
} elsif ($operation eq "add") {
    my $new_component = $opts->{"A"};
    if ($new_component =~ /^[^A-Za-z0-9]+/) {
	die "Error: Component name must start with an alphanumeric character!\n";
    }
    my $create_framework = defined($opts->{"f"});

    open CVSLOG, ">cvs.log" or die "\nCannot open cvs.log for writing ($!).\n\n";

    # checkout fwcomponents root and check if the requested component already exists
    if (not cvs_operation("",
			  "Checking out fwcomponents root",
			  "cvs -z3 -f -d \"$cvs_root\" co -l -d .fwcomponents fwcomponents",
			  \*CVSLOG)) {
	close CVSLOG;
	exit 1;
    }

    # try checking out the new component to check if it already exists (must fail!)
    if (cvs_operation($new_component, 
		      "Checking out new component '$new_component' which must fail",
		      "cd .fwcomponents && cvs -z3 -f -d \"$cvs_root\" co -l -d \"$new_component\" \"fwcomponents/$new_component\"",
		      \*CVSLOG)) {
	remove_hidden_fwcomponents_root();
	die "\nThe component '$new_component' already exists.\n\n";
    }

    # handle the case when the new component directory exists already on the local side
    if ( -d "$new_component" ) {
	if ( -d "$new_component/CVS" ) {
	    remove_hidden_fwcomponents_root();
	    die "\nThe component $new_component seems to be in CVS already.\n\n"
	} elsif ($create_framework) {
	    if (opendir DIR, "$new_component") {
		my @dir_contents = grep !/^\.\.?$/, readdir DIR;
		if (scalar(@dir_contents) > 0) {
		    remove_hidden_fwcomponents_root();
		    die "\nThe component directory must be empty to add a framework!\n\n"
		}
		closedir DIR;
	    }
	}
    }

    # create the new component directory in case it doesn't exist yet
    if (not -d ".fwcomponents/$new_component" and not mkdir ".fwcomponents/$new_component") {
	remove_hidden_fwcomponents_root();
	die "\nCannot create .fwcomponents/$new_component ($!).\n\n";
    }
	
    # add root directory of the new component to cvs
    if (not cvs_operation($new_component,
			  "Adding new component '$new_component'",
			  "cvs -z3 -d \"$cvs_root\" add \".fwcomponents/$new_component\"",
			  \*CVSLOG)) {
	close CVSLOG;
	remove_hidden_fwcomponents_root();
	exit 1;
    }

    # remove the hidden fwcomponents root directory
    remove_hidden_fwcomponents_root();

    # optionally create framework
    if ($create_framework) {
	# determine framework to use
	my $framework = $new_component =~ /^libpp_(.*)/ ? "libpp_example" : "example";
	my $unprefixed_name = $new_component =~ /^libpp_(.*)/ ? $1 : $new_component;
	my $unprefixed_name_upcase = $unprefixed_name;
	$unprefixed_name_upcase =~ tr/a-z/A-Z/;
	# export framework
	my $branch_spec = $cvs_tag ne "" ? "-r $cvs_tag" : "-r HEAD";
	if (not cvs_operation($new_component,
			      "Exporting framework for '$new_component'",
			      "cvs -z3 -d \"$cvs_root\" export -d \"$new_component\" $branch_spec \"fwcomponents/$framework\"",
			      \*CVSLOG)) {
	    close CVSLOG;
	    remove_hidden_fwcomponents_root();
	    exit 1;
	}
	# patch framework
	print "Patching framework: ";	
	system("find ./$new_component -type f -exec sed -i -e 's,example,$unprefixed_name,' -e 's,EXAMPLE,$unprefixed_name_upcase,' {} \\;");
	if ($? == 0) {
	    print "done\n";
	} else {
	    print "failed (ignored)\n";
	}
    }

    # checkout root directory of the new component
    my $branch_spec = $cvs_tag ne "" ? "-r $cvs_tag" : "-A";
    if (not cvs_operation($new_component,
			  "Checking out new component '$new_component'",
			  "cvs -z3 -f -d \"$cvs_root\" co -d \"$new_component\" $branch_spec \"fwcomponents/$new_component\"",
			  \*CVSLOG)) {
	close CVSLOG;
	remove_hidden_fwcomponents_root();
	exit 1;
    }

    close CVSLOG;
#------------------------------------------------------------------------------
# MERGE OPERATION
#------------------------------------------------------------------------------
} elsif ($operation eq "merge") {
    # check merge command-line options
    my $fw_type = $opts->{"t"};
    my $cvs_start_tag = defined($opts->{"s"}) ? $opts->{"s"} : "";
    my $cvs_end_tag = $opts->{"e"};

    if (!defined($cvs_end_tag)) {
	die "\nOption -e <tag/branch> must be specified!\n\n";
    }

    open CVSLOG, ">cvs.log" or die "\nCannot open cvs.log for writing ($!).\n\n";

    if (!defined($fw_type)) {
	# merge the base directory (NOTE: subdirectories must be specified here manually!)
	my $start_tag_spec = $cvs_start_tag ne "" ? "-j $cvs_start_tag" : "";
	my $end_tag_spec = $cvs_end_tag ne "" ? "-j $cvs_end_tag" : "";
	my @base_subdirs = ( "build_sys" );
	cvs_operation("<base>", "Merging base directory", "cvs -z3 update -l -P $start_tag_spec $end_tag_spec", \*CVSLOG);
	foreach my $base_subdir (@base_subdirs) {
	    cvs_operation("<base>", "Merging directory '$base_subdir'", "cvs -z3 update -dP $start_tag_spec $end_tag_spec $base_subdir", \*CVSLOG);
	}
    } else {
	# get component list
	if (!get_components(\@all_components, \@components, 0, $fw_type, "", "", "", "none")) {
	    die "\nError during determining the components to merge\n";
	}

	# cut off any subtrees of components
	for (0 .. $#all_components) {
	    $all_components[$_] =~ s/\/.*//;
	}

	# skip duplicate and the build_sys component
	my @new_all_components = ();
	foreach my $component (@all_components) {
	    $component =~ /^build_sys$/ and next; # build_sys is not a real component
	    $component =~ /^Source(\/.*|)$/ and next; # "Source" is a special Raritan component.
	    my $dup = 0;
	    foreach my $new_component (@new_all_components) {
		$new_component eq $component and $dup = 1 and last;
	    }
	    $dup or push @new_all_components, $component;
	}
	@all_components = @new_all_components;

	foreach my $component (@all_components) {
	    merge_component($cvs_root, $cvs_module, $cvs_start_tag, $cvs_end_tag, $component);
	}
    }
    close CVSLOG;
    print "\n";

    if (scalar(keys %uncommitted_by_component) > 0) {
	print "Components with uncommitted files:\n\n";
	foreach my $component (sort keys %uncommitted_by_component) {
            # keep the two leading spaces - required for automatic build system!
	    print "  $component\n";
	}
	print "\n";
    }

    if (scalar(keys %updated_by_component) > 0) {
	print "Components with updated files:\n\n";
	foreach my $component (sort keys %updated_by_component) {
            # keep the two leading spaces - required for automatic build system!
	    print "  $component\n";
	}
	print "\n";
    }

    if (scalar(keys %conflicts_by_component) > 0) {
	print "Components with conflicts:\n\n";
	foreach my $component (sort keys %conflicts_by_component) {
            # keep the two leading spaces - required for automatic build system!
	    print "  $component\n";
	}
	print "\n";
    }

    if (scalar(keys %cvserror_by_component) > 0) {
	print "CVS errors:\n\n";
	foreach my $component (sort keys %cvserror_by_component) {
            # keep the two leading spaces - required for automatic build system!
	    print "  $component\n";
	}
	print "\n";
    }
    if (scalar(keys %conflicts_by_component) > 0 or scalar(keys %cvserror_by_component) > 0) {
	die "\nYou should check cvs.log for errors and/or merge conflicts!\n\n";
    }
}

# HACK: This commits Entries.Log to Entries
system "cvs status dummy > /dev/null 2>&1";

#
###############################################################################
# Emacs
#
# Local Variables:
# cperl-indent-level:4
# cperl-continued-statement-offset:4
# cperl-continued-brace-offset:-4
# cperl-brace-offset:0
# cperl-brace-imaginary-offset:0
# cperl-label-offset:-2
# cperl-hairy:t
# perl-tab-always-indent:nil
# End:
###############################################################################
