#include <stdio.h>
#include <stdlib.h>
#include <pp/base.h>
#include <pp/cfg.h>
#include <liberic_config.h>
#include <sys/statfs.h>
#include <sys/param.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>
#include <errno.h>
#include <regex.h>
#if defined(PP_FEAT_CASE_INSENSITIVE_LOGINS)
#include <ctype.h>
#include <pp/um.h>
#endif /* PP_FEAT_CASE_INSENSITIVE_LOGINS */

#include "cfg_conv.h"

static int init(void);
static void cleanup(void);

typedef struct {
    char*               old_key;
    char*               new_key;
    regex_t             old_regex;
} key_conv_t;

typedef struct {
    key_conv_t*         key_conv;
    char*               old_value;
    char*               new_value;
} value_conv_t;

typedef struct {
    char                filepath[MAXPATHLEN + 1];
    FILE*               fp;
    struct statfs       statfs_buf;
} file_ctx_t;

typedef int (*parse_cb_t)(char *line, int line_nr, file_ctx_t *parsed_file, va_list);

#define CONFIG_DIR "/config_flash2/config"
static const char* key_conv_fn = "/etc/newcfgkeys.txt";
static const char* value_conv_fn = "/etc/newcfgvalues.txt";
static const char* cfg_fn = "/config/!config";
static const char* cd_version_str = "cd_version";

/* prototype to prevent compiler warning */
int cfg_conv_main(int mode);

static key_conv_t*
create_key_conv(const char* old_key, const char* old_key_wc, const char* new_key)
{
    key_conv_t *key_conv = malloc(sizeof(key_conv_t));
    char key[MAX_OPT_KEY_LEN + 3];
    
    sprintf(key, "^%s$", old_key_wc);
    if (0 != regcomp(&key_conv->old_regex, key, REG_EXTENDED)) {
        pp_log("could not create regex for '%s'!\n", old_key_wc);
        free(key_conv);
        return NULL;
    }
    key_conv->old_key = strdup(old_key);
    key_conv->new_key = strdup(new_key);
    return key_conv;
}

static value_conv_t*
create_value_conv(pp_hash_t *old_keys_hash, 
                  const char* old_key, const char* new_key,
                  const char* old_value, const char* new_value)
{
    value_conv_t *value_conv = malloc(sizeof(value_conv_t));
    const char *ptr1, *ptr2;

    if (NULL == (value_conv->key_conv = pp_hash_get_entry(old_keys_hash, old_key))) {
        pp_log("old key '%s' unknown!\n", old_key);
        free(value_conv);
        return NULL;
    }
    for (ptr1 = new_key, ptr2 = value_conv->key_conv->new_key;
	 *ptr1 && *ptr2;
	 ++ptr1, ++ptr2) {
        if (*(ptr1 - 1) != '%' && *ptr1 != *ptr2) {
            pp_log("conversion key mismatch: '%s' =! '%s'\n", 
                   new_key, value_conv->key_conv->new_key);
            free(value_conv);
            return NULL;
        }
    }
    value_conv->old_value = strdup(old_value);
    value_conv->new_value = strdup(new_value);
    return value_conv;
}

static void
destroy_key_conv(void *k)
{
    key_conv_t *key_conv = (key_conv_t*)k;
    free(key_conv->old_key);
    free(key_conv->new_key);
    free(key_conv);
}

static void
destroy_value_conv(void *v)
{
    value_conv_t *value_conv = (value_conv_t*)v;
    free(value_conv->old_value);
    free(value_conv->new_value);
    free(value_conv);
}

static int
load_key_cb(char *line, int line_nr, file_ctx_t *parsed_file UNUSED, va_list args)
{
    char *ptr, *old_key = line;
    char old_key_wc[MAX_OPT_KEY_LEN + 1], new_key[MAX_OPT_KEY_LEN + 1];
    key_conv_t *key_conv;
    pp_hash_t *old_keys_hash;
    int idx;
    static const char wc_digit[] = "([0-9]+)";
    static const char wc_string[] = "([^[:cntrl:] ]+)";
    
    for (ptr = line, idx = 0; 
        *ptr != ' ' && *ptr != '\t' && *ptr != '\n' && *ptr;
	 ++ptr, ++idx) {
        if (idx > MAX_OPT_KEY_LEN) return PP_ERR;
        if (*ptr == '%') {
            /* a wildcard */
            ++ptr;
            switch (*ptr) {
	      case 'u':
		  if (idx + sizeof(wc_digit) > MAX_OPT_KEY_LEN) return PP_ERR;
		  memcpy(old_key_wc + idx, wc_digit, sizeof(wc_digit));
		  idx += sizeof(wc_digit) - 1;
		  break;
	      case 's':
		  if (idx + sizeof(wc_string) > MAX_OPT_KEY_LEN) return PP_ERR;
		  memcpy(old_key_wc + idx, wc_string, sizeof(wc_string));
		  idx += sizeof(wc_string) - 1;
		  break;
	      default:
		  pp_log("parse error: undefined wildcard '%%%c' in line %d\n",
			 *ptr, line_nr + 1);
		  return PP_ERR;
            }
        } else {
            old_key_wc[idx] = *ptr;
	}
    }
    if (!ptr || *ptr == '\n') return PP_ERR;
    old_key_wc[idx] = '\0';
    *ptr = '\0'; /* end of old key */

    for (ptr++; *ptr == ' ' || *ptr == '\t'; ++ptr); // seek new_key

    for (idx = 0;
	 *ptr != ' ' && *ptr != '\t' && *ptr != '\n' && *ptr; 
	 ++ptr, ++idx) {
        if (idx > MAX_OPT_KEY_LEN) return PP_ERR;
        new_key[idx] = *ptr;            
        if (*ptr == '%') {
            /* FIXME: consistency check */
            /* for new keys, convert wildcard to %s */
            ++ptr;
            ++idx;
            switch(*ptr) {
	      case 'u':
	      case 's':
		  if(idx > MAX_OPT_KEY_LEN) return PP_ERR;
		  new_key[idx] = 's';            
		  break;
	      default:
		  pp_log("parse error: undefined wildcard '%%%c' in line %d\n",
			 *ptr, line_nr + 1);
		  return PP_ERR;
            }
        }
    }
    if (*ptr != '\n') return PP_ERR;
    new_key[idx] = '\0';

/*
    for(new_key = ptr; *ptr != ' ' && *ptr != '\t' && *ptr != '\n' && *ptr; ++ptr);
    if(*ptr != '\n' || ptr - new_key > MAX_OPT_KEY_LEN)
        return PP_ERR;
    *ptr = '\0';
*/

    if (NULL == (key_conv = create_key_conv(old_key, old_key_wc, new_key))) {
        return PP_ERR;
    }

    old_keys_hash = va_arg(args, pp_hash_t*);
    if (NULL != pp_hash_get_entry(old_keys_hash, old_key)) {
        pp_log("parse error: '%s' redefined in line %d\n", 
               old_key, line_nr + 1);
        return PP_ERR;
    }
    pp_hash_set_entry(old_keys_hash, old_key, key_conv, destroy_key_conv);
    
    return PP_SUC;
}

static int
load_value_cb(char *line, int line_nr, file_ctx_t *parsed_file UNUSED, 
              va_list args)
{
//    char prev_key[MAX_OPT_KEY_LEN + 1];
//    char prev_value[MAX_OPT_VALUE_LEN + 1];
//    char key[MAX_OPT_KEY_LEN + 1];
//    char value[MAX_OPT_VALUE_LEN + 1];
    char *prev_key, *prev_value, *key, *value;
    char hash_key[MAX_OPT_KEY_LEN + MAX_OPT_VALUE_LEN + 2];
    value_conv_t *value_conv;

    pp_hash_t *old_values_hash = va_arg(args, pp_hash_t*);
    pp_hash_t *old_keys_hash = va_arg(args, pp_hash_t*);
    char **prev_line_ptr = va_arg(args, char**);
    char *prev_line = *prev_line_ptr;

    if (prev_line == NULL) {
        prev_line = strdup(line);
        /* do not free prev_line, since it's referenced by prev_line_ptr */
        *prev_line_ptr = prev_line;
        return PP_SUC;
    }
    
    /* there were previous key/value... */
    if (PP_FAILED(eric_config_parse_line(prev_line, &prev_key, &prev_value, 
					 parsed_file->fp, line_nr,
					 parsed_file->filepath))
	|| PP_FAILED(eric_config_parse_line(line, &key, &value, 
					    parsed_file->fp, line_nr, 
					    parsed_file->filepath))) {
        return PP_ERR;
    }
    
    if (NULL == (value_conv = create_value_conv(old_keys_hash, prev_key, key,
						prev_value, value))) {
        /* if we got an error vor prev_line/line, try line/succ_line... */
        free(prev_line);
        prev_line = strdup(line);
        /* do not free prev_line, since it's referenced by prev_line_ptr */
        *prev_line_ptr = prev_line;
        return PP_SUC;
    }

    //printf("create_value_conv %s %s %s %s\n", value_conv->key_conv->old_key, value_conv->key_conv->new_key, value_conv->old_value, value_conv->new_value);
    snprintf(hash_key, sizeof(hash_key), "%s=%s", prev_key, prev_value);
    if (NULL != pp_hash_get_entry(old_values_hash, hash_key)) {
        pp_log("parse error: '%s' redefined in line %d\n", 
               hash_key, line_nr + 1);
        return PP_ERR;
    }
    pp_hash_set_entry(old_values_hash, hash_key, value_conv,
                      destroy_value_conv);
    
    free(prev_line);
    *prev_line_ptr = NULL;
    return PP_SUC;
}

static void
close_file(file_ctx_t *file)
{
    struct flock fl = {
	l_type:   F_UNLCK,
	l_whence: SEEK_SET,
	l_start:  0,
	l_len:    0,
	l_pid:    getpid()
    };
    
    if (file && file->fp) {
        /* do not unlock on nfs volume - used for development */
        if(file->statfs_buf.f_type != 0x6969 && 
           fcntl(fileno(file->fp), F_SETLKW, fl) == -1){
            pp_log_err("%s(): fcntl(): %s\n", ___F, strerror(errno));
        }
    
        fclose(file->fp);
    }
    
    free(file);
}

static file_ctx_t*
open_file(const char *name, const char *mode)
{
    struct statfs statfs_buf;
    FILE * fp;
    struct flock fl = {
	l_type:   F_RDLCK,
	l_whence: SEEK_SET,
	l_start:  0,
	l_len:    0,
	l_pid:    getpid()
    };
    file_ctx_t *file = malloc(sizeof(file_ctx_t));

    if (*name == '/') {
        snprintf(file->filepath, sizeof(file->filepath), "%s", name);
    } else {
        snprintf(file->filepath, sizeof(file->filepath), "%s/%s", 
                 CONFIG_DIR, name);
    }

    if ((fp = fopen(file->filepath, mode)) == NULL) {
	pp_log_err("%s(): fopen() of '%s': %s\n", ___F, file->filepath, strerror(errno));
        close_file(file);
	return NULL;
    }

    if (fstatfs(fileno(fp), &statfs_buf) == -1) {
	pp_log_err("%s(): fstatfs(): %s", ___F, strerror(errno));
        close_file(file);
	return NULL;
    }

    /* do not lock on nfs volume - used for development */
    if(statfs_buf.f_type != 0x6969 && fcntl(fileno(fp), F_SETLKW, fl) == -1) {
	pp_log_err("%s(): fcntl(): %s", ___F, strerror(errno));
        close_file(file);
	return NULL;
    }
    
    file->statfs_buf = statfs_buf;
    file->fp = fp;
    return file;
}

static int
parse_file(const char * name, parse_cb_t cb, ...) {
    va_list args;
    FILE * fp;
    u_int line;
    int max_len = MAX_OPT_KEY_LEN + 1 + MAX_OPT_VALUE_LEN + 2; // key=value\n\0
    char ln[max_len];
    file_ctx_t *file;
    int ret = PP_SUC;

    if((file = open_file(name, "r")) == NULL) {
	return PP_ERR;
    }

    fp = file->fp;

    for (line = 0; !feof(fp) && !ferror(fp) && fgets(ln, max_len, fp); ++line) {
        /* skip comments and empty lines */
        if (*ln != '#' && *ln != '\n') {
	    //printf("parsing %s line %d: %s", name, line + 1, ln);
            va_start(args, cb);
            if (PP_FAILED(cb(ln, line, file, args))) {
                pp_log("%s: parse error at line '%d'\n", name, line + 1);
                ret = PP_ERR;
	    }
            va_end(args);
        }
    }

    close_file(file);

    return ret;
}

static int
reset_file(const char* name)
{
    char filepath[MAXPATHLEN + 1];
    FILE * fp;

    if (*name == '/') {
        snprintf(filepath, sizeof(filepath), "%s", name);
    } else {
        snprintf(filepath, sizeof(filepath), "%s/%s", CONFIG_DIR, name);
    }

    if ((fp = fopen(filepath, "w")) == NULL) {
	pp_log_err("%s(): fopen() of '%s': %s", ___F, filepath, strerror(errno));
        return PP_ERR;
    }
    
    return PP_SUC;
}

static const char*
convert_to_new_value(const char *old_value, pp_hash_t *old_values_hash,
                     const key_conv_t *key_conv)
{
    const char *new_value = old_value;
    value_conv_t *value_conv;
    char hash_key[MAX_OPT_KEY_LEN + MAX_OPT_VALUE_LEN + 2];
    
    snprintf(hash_key, sizeof(hash_key), "%s=%s", key_conv->old_key, old_value);
    
    if (NULL != (value_conv = pp_hash_get_entry(old_values_hash, hash_key))) {
        new_value = value_conv->new_value;
	//printf("converted '%s'='%s' to '%s'='%s'\n", key_conv->old_key, old_value, key_conv->new_key, new_value);
    }
    //else printf("conversion '%s'='%s' to '%s'='%s' failed\n", key_conv->old_key, old_value, key_conv->new_key, new_value);
    
    return new_value;
}

static int
convert_to_new_dev_cb(char *line, int line_nr, file_ctx_t *parsed_file, va_list args)
{
    pp_hash_t *old_keys_hash = va_arg(args, pp_hash_t*);
    pp_hash_t *old_values_hash = va_arg(args, pp_hash_t*);
    //char old_key[MAX_OPT_KEY_LEN + 1];
    //char old_value[MAX_OPT_VALUE_LEN + 1];
    char *old_key, *old_value;
    key_conv_t *key_conv;
    regmatch_t match_exp[5];
    char match[4][MAX_OPT_KEY_LEN + 1];
    int i, match_sz, wc_sz;
    char *ptr;
    
    if (PP_FAILED(eric_config_parse_line(line, &old_key, &old_value, 
					 parsed_file->fp, line_nr, parsed_file->filepath))) {
        return PP_ERR;
    }
    //printf("convert_to_new_dev_cb key/value: '%s'='%s'\n", old_key, old_value);
    for (key_conv = pp_hash_get_first_entry(old_keys_hash);
	 key_conv != NULL;
	 key_conv = pp_hash_get_next_entry(old_keys_hash)) {
        if (!regexec(&key_conv->old_regex, old_key, 5, match_exp, 0)) {
            for (i = 1; i < 5; ++i) {
                if (match_exp[i].rm_eo < 0) break;
                match_sz = match_exp[i].rm_eo -  match_exp[i].rm_so;
                strncpy(match[i-1], old_key + match_exp[i].rm_so, match_sz);
                match[i-1][match_sz] = '\0';
            }
            --i;
            for (ptr = key_conv->new_key, wc_sz = 0; ptr && *ptr; ++ptr) {
                if (*ptr == '%') ++wc_sz;
	    }
            if (wc_sz > i) {
                /* more wildcards in new key than extracted from old key  */
                pp_log("new key '%s' contains %d wildcards more than old key '%s'\n",
		       key_conv->new_key, i - wc_sz, old_key);
                return PP_ERR;
            }
            goto found;
        }
    }

    /* old key not found in conversion table  */
    pp_log("old key '%s' not found in conversion table, dropped\n", old_key);
    return PP_ERR;
    
 found:

    /* Drop conversion for new key = '#'.
     * Indicates that a key is not any more present in new cfg.
     * Not really necessary, since inexistant keys are dropped anyway
     */
    if (*key_conv->new_key == '#') return PP_SUC;
    
    //printf("pp_cfg_set(%s, %s)\n", convert_to_new_value(old_value, old_values_hash, key_conv), key_conv->new_key);
    return pp_cfg_set(convert_to_new_value(old_value, old_values_hash, key_conv), 
                      key_conv->new_key, match[0], match[1], match[2], match[3]);
}

static int
convert_to_new_acl(const char *key, char *value, const char *uid)
{
    u_int u, perm_sz;
    const char *acl_name;
    char *perm[3];
    char *ptr;
    static const char *new_acl_key = "user[%s].acl[%s][%u]";
    
    /* assert a acl-name following after "acl_" prefix */
    if (!(acl_name = key + 4)) return PP_ERR;
    
    /* check for old acl_excl perm */
    if (!strcmp(acl_name, "excl") && !strcmp(value + 1, "allow")) {
        if (*value == '+') {
            if (PP_FAILED(pp_cfg_set("+view", new_acl_key, uid, acl_name, 0))
		|| PP_FAILED(pp_cfg_set("+change", new_acl_key, uid, acl_name, 1))) {
                return PP_ERR;
	    }
        } else if (PP_FAILED(pp_cfg_set("-view", new_acl_key, uid, acl_name, 0))
		   || PP_FAILED(pp_cfg_set("-change", new_acl_key, uid, acl_name, 1))) {
	    return PP_ERR;
        }
        return PP_SUC;
    }
    
    /* count number of permissions stored in acl-value */
    for (ptr = value, perm[0] = value, perm_sz = 1; *ptr; ++ptr) {
        if (*ptr == ':') {
            *ptr = '\0';
            /* assert a permission following after "prev_perm:" */
            if (!(ptr + 1)) return PP_ERR;
            perm[perm_sz] = ptr + 1;
            ++perm_sz;
        }
    }
    
    for (u = 0; u < perm_sz; ++u) {
        if (PP_FAILED(pp_cfg_set(perm[u], new_acl_key, uid, acl_name, u))) {
            return PP_ERR;
	}
    }
    
    return PP_SUC;
}

static int
convert_to_new_user_cb(char *line, int line_nr, file_ctx_t *parsed_file, 
                       va_list args)
{
    pp_hash_t *old_keys_hash = va_arg(args, pp_hash_t*);
    pp_hash_t *old_values_hash = va_arg(args, pp_hash_t*);
    char *uid = va_arg(args, char*);
    //char old_key[MAX_OPT_KEY_LEN + 1];
    //char old_value[MAX_OPT_VALUE_LEN + 1];
    char *old_key, *old_value;
    char user_key[MAX_OPT_KEY_LEN + 6];
    key_conv_t *key_conv;
    regmatch_t match_exp[5];
    char match[4][MAX_OPT_KEY_LEN + 1];
    int i, match_sz, wc_sz;
    char *ptr;
    
    if (PP_FAILED(eric_config_parse_line(line, &old_key, &old_value, 
					 parsed_file->fp, line_nr, parsed_file->filepath))) {
        return PP_ERR;
    }
    //printf("convert_to_new_user_cb key/value: '%s'='%s'\n", old_key, old_value);

    /* acls are handled specially */
    if (!strncmp(old_key, "acl_", 4)) {
        return convert_to_new_acl(old_key, old_value, uid);
    }
        
    snprintf(user_key, sizeof(user_key), "user:%s", old_key);
    for (key_conv = pp_hash_get_first_entry(old_keys_hash);
	 key_conv != NULL;
	 key_conv = pp_hash_get_next_entry(old_keys_hash)) {
        if (!regexec(&key_conv->old_regex, user_key, 5, match_exp, 0)) {
            for (i = 1; i < 5; ++i) {
                if (match_exp[i].rm_eo < 0) break;
                match_sz = match_exp[i].rm_eo -  match_exp[i].rm_so;
                strncpy(match[i-1], user_key + match_exp[i].rm_so, match_sz);
                match[i-1][match_sz] = '\0';
            }
            --i;
            for (ptr = user_key, wc_sz = -1; ptr && *ptr; ++ptr) {
                if (*ptr == '%') ++wc_sz;
	    }
            if (wc_sz > i) {
                /* more wildcards in new key than values extracted from old key  */
                pp_log("new key '%s' contains %d wildcards more than old "
                       "key '%s'\n", user_key, wc_sz - i, old_key);
                return PP_ERR;
            }
            goto found;
        }
    }

    /* old key not found in conversion table  */
    pp_log("old key '%s' not found in conversion table, dropped\n", user_key);
    return PP_ERR;
    
 found:
    /* Drop conversion for new key = '#'.
       Indicates that a key is not any more present in new cfg.
       Not really necessary, since inexistant keys are dropped anyway */
    if (*key_conv->new_key == '#') return PP_SUC;
    
//printf("pp_cfg_set(%s, %s)\n", convert_to_new_value(old_value, old_values_hash, key_conv), key_conv->new_key);
    return pp_cfg_set(convert_to_new_value(old_value, old_values_hash, key_conv), 
                      key_conv->new_key, uid, match[0], match[1], match[2], match[3]);
}

static void
convert_to_new(const char *old_fn, pp_hash_t *old_keys_hash, 
               pp_hash_t *old_values_hash)
{
    char uid[21];
    char *user;
    
    //printf("convert_to_new(%s, %s)\n", old_fn, new_fn);
    
    uid[0] = 0;
    
    /* parse old cfg file */
    if (sscanf(old_fn, "%20[0123456789]", uid) == 1
	|| !strcmp(old_fn, "administrator") /* administrator group */
	|| !strcmp(old_fn, "user") /* user group */ ) {
        /* user profile */
        
        if(!uid[0]) {
            snprintf(uid, 21, old_fn);
        }
        
        parse_file(old_fn, (parse_cb_t)convert_to_new_user_cb, 
                   old_keys_hash, old_values_hash, uid);
        if (PP_SUCCED(pp_cfg_get(&user, "user[%s].tmp_login", uid))) {
            // remove unused key
            pp_cfg_remove("user[%s].tmp_login", uid);
            // rename roles to lower case
            if (!strcmp(user, "Administrator")) {
                strcpy(user, "administrator");
            } else if (!strcmp(user, "User")) {
                strcpy(user, "user");
            }
            // rename user[123] to user[tweb]
            pp_cfg_rename(user, "user[%s]", uid); 
            free(user);
        }
    } else {
        /* device profile */
        parse_file(old_fn, (parse_cb_t)convert_to_new_dev_cb, 
                   old_keys_hash, old_values_hash);
    }
}

static int
conv_old_cfg_to_new_cfg(void)
{
    pp_hash_t *old_keys_hash = NULL, *old_values_hash = NULL;
    DIR *dir;
    struct dirent *entry;
    char **prev_line_ptr = malloc(sizeof(char**));
    *prev_line_ptr = NULL;
    int ret = PP_ERR;
    
    old_keys_hash = pp_hash_create(400);
    old_values_hash = pp_hash_create(20);
    
    /* parse key conversion file */    
    if (PP_FAILED(parse_file(key_conv_fn, (parse_cb_t)load_key_cb, 
			     old_keys_hash))) {
        pp_log("failed to parse '%s'\n", key_conv_fn);
        goto bail;
    }
    
    /* parse value conversion file */    
    if (PP_FAILED(parse_file(value_conv_fn, (parse_cb_t)load_value_cb, 
			     old_values_hash, old_keys_hash, prev_line_ptr))) {
        pp_log("failed to parse '%s'\n", value_conv_fn);
        goto bail;
    }

    /* create/reset new config file */
    if (PP_FAILED(reset_file(cfg_fn))) goto bail;
    
    /* convert old config files */
    dir = opendir(CONFIG_DIR);
    if (dir) {
        while (NULL != (entry = readdir(dir))) {
            /* only parse real config files */
            if (strcmp(entry->d_name, ".")
		&& strcmp(entry->d_name, "..")
		&& strcmp(entry->d_name, cfg_fn)
		&& strcmp(entry->d_name, "!global_defaults")
		&& strncmp(entry->d_name, "newcfg", 6)
		&& entry->d_name[strlen(entry->d_name) - 1] != '~') {
		convert_to_new(entry->d_name, old_keys_hash, old_values_hash);
	    }
        }
        closedir(dir);
    }
    ret = PP_SUC;

 bail:
    pp_hash_delete(old_keys_hash);
    pp_hash_delete(old_values_hash);
    free(*prev_line_ptr);
    free(prev_line_ptr);
    
    return ret;
}

static void check_pw_change(void) {
    char *default_pw_hash = NULL, *pw_hash = NULL;
    int need_pw_change;
    static const char admin_pw_hash[] = "user[0].pw_hash";
    static const char admin_need_pw_change[] = "user[0].need_pw_change";
    
    if(PP_ERR == pp_cfg_is_enabled_at_layer_nodflt(PP_PROFILE_LOCAL,
                                                   &need_pw_change,
                                                   admin_need_pw_change) &&
       PP_ERR != pp_cfg_get_at_layer(PP_PROFILE_LOCAL_CODE, &default_pw_hash,
                                     admin_pw_hash) &&
       PP_ERR != pp_cfg_get_at_layer_nodflt(PP_PROFILE_LOCAL, &pw_hash,
                                            admin_pw_hash) &&
       pp_strcmp_safe(default_pw_hash, pw_hash)) {
        /**
         * need password change flag is not set locally (e.g. due to strong
         * password enforcement etc.) but by default and password set for admin
         * differs from default -> no need to change
         */
        pp_cfg_set_enabled(0, admin_need_pw_change);
    }
    
    free(pw_hash);
    free(default_pw_hash);
}

#if defined(PP_FEAT_CASE_INSENSITIVE_LOGINS)
static int check_login_upper(void) {
    int cnt = 0;
    
    if(PP_ERR != pp_um_init()) {
        vector_t *users = pp_um_get_all_users();
        u_int u, users_sz = vector_size(users);
        
        for(u = 0; u < users_sz; ++u) {
            char *user = vector_get(users, u);
            
            if(user && *user) {
                char *c, *user_dc = strdup(user);
                int local_cnt = 0;
                
                for(c = user_dc; *c; ++c) {
                    if(isupper(*c)) {
                        *c = tolower(*c);
                        ++local_cnt;
                    }
                }
                
                if(local_cnt) {
                    char buf[65];
                    char *new_user = user_dc;
                    int ret;
                    
                    while((ret = pp_um_user_rename(user, new_user)) != PP_SUC) {
                        if (errno != PP_UM_ERR_USER_ALREADY_EXISTS || 
                            sizeof(buf) < strlen(new_user) + 1) {
                            /* maximum size of groupname reached, break */
                            pp_log("WARNING: could not convert user '%s'!\n",
                                   user);
                            break;
                        } else {
                            /* user of that name exists, append '_' to username
                               and try again */
                            snprintf(buf, sizeof(buf), "%s_", new_user);
                            new_user = buf;
                        }
                    }

                    /* did we change anything? */
                    if(ret == PP_SUC) {
                        pp_log("converted user '%s' to '%s'!\n", 
                               user, new_user);
                        ++cnt;
                    }
                }
                
                free(user_dc);
            }
        }
        
        vector_delete(users);
        pp_um_cleanup();
    }
    
    return cnt;
}
#endif /* PP_FEAT_CASE_INSENSITIVE_LOGINS */

int
cfg_conv_main(int mode)
{
    int ret, local_version = -1, code_version = -1, dirty = 0;
    int local_settings;
    
    if (PP_FAILED(ret = init())) goto bail;

    local_settings = PP_ERR == pp_cfg_local_layer_is_empty();
    pp_cfg_get_int_at_layer(PP_PROFILE_LOCAL_CODE, &code_version,
                            cd_version_str); 
    pp_cfg_get_int_at_layer_nodflt(PP_PROFILE_LOCAL, &local_version,
                                   cd_version_str); 
    
    /* determine, which conversion to process */
    switch(mode) {
        case CONVERT_CONFIG_FS:
            if(local_settings && PP_ERR != (ret = conv_old_cfg_to_new_cfg())) {
                dirty = 1;
            }
            break;
        case CONVERT_RAASIP_UM:
            if(code_version >= CODE_VERSION_RAASIP_UM && 
               local_version < CODE_VERSION_RAASIP_UM) {
                /* v0 -> v1: change user management to RAASIP */
                if(local_settings) {
                    ret = conf_old_um_to_raasip();
                }
                if(PP_ERR != ret) {
                    pp_cfg_set_int_at_layer(PP_PROFILE_LOCAL,
                                            CODE_VERSION_RAASIP_UM,
                                            cd_version_str);
                    dirty = 1;
                }
            }
            break;
        case CONVERT_COMMON_UM:
            if(code_version >= CODE_VERSION_COMMON_UM) {
                if(local_version < CODE_VERSION_RAASIP_UM) {
                    pp_log("local configuration does not match minimum "
                           "requirements for conversion to v2\n");
                    ret = PP_ERR;
                } else if(local_version < CODE_VERSION_COMMON_UM) {
                    /* v1 -> v2: common um, rename IPMI permissions */
                    if(local_settings) {
                        ret = conf_common_um();
                    }
                    if(PP_ERR != ret) {
                        pp_cfg_set_int_at_layer(PP_PROFILE_LOCAL,
                                                CODE_VERSION_COMMON_UM,
                                                cd_version_str);
                        dirty = 1;
                    }
                }
            }
            break;
        case CHECK_PW_CHANGE:
            if(code_version >= CODE_VERSION_FORCE_PW_CHANGE) {
                if(local_version < CODE_VERSION_RAASIP_UM) {
                    pp_log("local configuration does not match minimum "
                           "requirements for conversion to v2\n");
                    ret = PP_ERR;
                } else {
                    check_pw_change();
                    if(local_version < CODE_VERSION_FORCE_PW_CHANGE) {
                        pp_cfg_set_int_at_layer(PP_PROFILE_LOCAL,
                                                CODE_VERSION_FORCE_PW_CHANGE,
                                                cd_version_str);
                    }
                    dirty = 1;
                }
            }
            break;
        case CHECK_LOGIN_UPPER:
#if defined(PP_FEAT_CASE_INSENSITIVE_LOGINS)
            if(check_login_upper() > 0) {
                dirty = 1;
            }
#endif /* PP_FEAT_CASE_INSENSITIVE_LOGINS */
            break;
        case SPLIT_SEC_S_ACL:
            if(code_version >= CODE_VERSION_SPLIT_SEC_S_ACL) {
                if(local_version < CODE_VERSION_RAASIP_UM) {
                    pp_log("local configuration does not match minimum "
                           "requirements for conversion to v2\n");
                    ret = PP_ERR;
                } else if(local_version < CODE_VERSION_SPLIT_SEC_S_ACL) {
                    /* v3 -> v4: split sec_s to auth_s & log_s & log_v & security_s */
                    if(local_settings) {
                        ret = conf_split_sec_s_acl();
                    }
                    if(PP_ERR != ret) {
                        pp_cfg_set_int_at_layer(PP_PROFILE_LOCAL,
                                                CODE_VERSION_SPLIT_SEC_S_ACL,
                                                cd_version_str);
                        dirty = 1;
                    }
                }
            }
            break;

        case CONVERT_SYS_PROFILE:
            {
                int do_flush = 0;
                char *value;
                int profile;
                if (PP_SUCCED( pp_cfg_get_with_prof_nodflt(&value, &profile, "serial") )) {
                    if (profile > PP_PROFILE_SYSTEM) {
                        pp_cfg_remove("serial");
                        pp_cfg_set_at_layer(PP_PROFILE_SYSTEM, value, "serial");
                        do_flush = 1;
                    }
                }
                if (PP_SUCCED( pp_cfg_get_with_prof_nodflt(&value, &profile, "network.mac") )) {
                    if (profile > PP_PROFILE_SYSTEM) {
                        pp_cfg_remove("network.mac");
                        pp_cfg_set_at_layer(PP_PROFILE_SYSTEM, value, "network.mac");
                        do_flush = 1;
                    }
                }
                if (do_flush) {
                    pp_cfg_save_layer(PP_PROFILE_SYSTEM, DO_FLUSH);
                    dirty = 1;
                }
            }
            break;

        default:
            pp_log("unknown mode, nothing to convert?\n");
            ret = PP_ERR;
            break;
    }
    
    if(dirty) {
        pp_cfg_save(DO_FLUSH);
    }

 bail:    
    cleanup();
    return ret;
}

static int
init(void)
{
    if (PP_FAILED(pp_base_init("cfg_conv", LOG_NOT_SILENT))) {
	pp_log("Initializing base-library failed.\n");
	return PP_ERR;
    }

    if (PP_FAILED(eric_config_init(FLUSH_IN_BACKGROUND))) {
	pp_log("Initializing config-library failed.\n");
	return PP_ERR;
    }

    if (PP_FAILED(pp_cfg_init(PP_CD_FNAME_DEFAULT, PP_CD_OEM_SKEL_FNAME_DEFAULT))) {
	pp_log("Initializing cfg-library failed.\n");
	return PP_ERR;
    }

    return PP_SUC;
}

static void
cleanup(void)
{
     pp_cfg_cleanup();
     eric_config_cleanup();
     pp_base_cleanup();
}
