#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pp/base.h>
#include <pp/cfg.h>
#include <liberic_config.h>

#undef DEBUG
#ifdef DEBUG
# define D(fmt, args...)        { printf(fmt, ##args); }
#else
# define D(fmt, args...)        { }
#endif

#define RTC_EEP_DEV	"/dev/rtc_eep"
#define RTC_EEP_MAGIC	"VALID\n"
#define RTC_EEP_SIZE	512

#define GATHER_RET_OK	 0
#define GATHER_RET_FAIL	-1
#define GATHER_RET_INIT	-2

typedef struct cfg_option_s {
    const char * name;
    char * rtc;
    char * cfg;
    size_t min_size;
} cfg_option_t;

cfg_option_t options[] = {
    { "network.mac",	   NULL, NULL, 17 },
    { "serial",		   NULL, NULL, 10 },
    { NULL,		   NULL, NULL,  0 }
};
  
int config_rtceep_main(int argc, char ** argv); /* prototype to avoid warning */

static int  init(void);
static void cleanup(void);
static int  gather_opts(void);
static int  sync_with_eep(void);
static int  restore_from_eep(void);
static int  save_to_eep(void);

int
config_rtceep_main(int argc, char ** argv)
{
    int command;
    int dosync=0, restore=0, ret=0, gather;

    if (init()) {
	pp_log("Initialization failed\n");
	exit(1);
    }
    
    while ((command = getopt(argc, argv, "sr")) != -1) {
	switch(command) {
	  case 's':
	      dosync++;
	      break;
	  case 'r':
	      restore++;
	      break;
	}
    }

    if (dosync && restore) {
	pp_log("Ambiguous options, only use (-s)ync or (-r)estore\n");
	goto fail;
    }

    gather = gather_opts();
    switch (gather) {
      case GATHER_RET_FAIL:
	  pp_log("Gathering options failed, exiting\n");
	  goto fail;
	  break;
      case GATHER_RET_INIT:
	  pp_log("Gathering from RTC failed, initializing\n");
	  save_to_eep();
	  break;	  
      default:
	  if (dosync)
	      ret = sync_with_eep();
	  else if (restore)
	      ret = restore_from_eep();
	  break;
    }
    
    
    cleanup();
    exit(ret);

 fail:
    cleanup();
    exit(1);
}

static int
init(void)
{
    if (pp_base_init("config_rtceep", LOG_NOT_SILENT) != 0) {
	pp_log("Initializing base-library failed.\n");
	return -1;
    }

    if (eric_config_init(FLUSH_IN_BACKGROUND) != 0) {
	pp_log("Initializing config-library failed.\n");
	return -1;
    }

    return 0;
}

static void
cleanup()
{
    eric_config_cleanup();
    pp_base_cleanup();
}


/**
 * gather_opts:
 * -read config values from RTC and config fs
 *  for later handling
*/
static int
gather_opts(void)
{
    FILE* fp = NULL;
    cfg_option_t* opt;
    char magic[16]; 
    char *tmp;
    int rtc_ok = 0;
  
    if ((fp = fopen(RTC_EEP_DEV, "r")) == NULL) {
	pp_log("Could not open RTC EEPROM\n");
	goto fail;
    }    

    /* check magic sequence */
    if (fgets(magic, 16, fp)) {
	if (!strcmp(magic, RTC_EEP_MAGIC)) {
	    rtc_ok = 1;
	}
    }
    
    for (opt = options; opt->name != NULL; opt++) {
	opt->rtc = NULL;
        pp_cfg_get(&opt->cfg, opt->name);
	if (!opt->cfg) {
	    pp_log("Cannot read %s from config fs\n", opt->name);
	}

	if (rtc_ok) {
	    if ((opt->rtc = malloc(MAX_OPT_VALUE_LEN)) == NULL) {
		pp_log_err("%s(): malloc()", ___F);
		exit(1);
	    } else if (feof(fp) || !fgets(opt->rtc, MAX_OPT_VALUE_LEN, fp)) {
		pp_log("Cannot read %s from rtc\n", opt->name);
		free(opt->rtc);
		opt->rtc = NULL;
	    } else {
		opt->rtc[MAX_OPT_VALUE_LEN-1] = '\0';
		tmp = strchr(opt->rtc, '\n');
		if (tmp) { *tmp = '\0'; }
		D("Name: %s, RTC: %s, Config: %s\n", opt->name, opt->rtc, opt->cfg);
	    }
	} else {
	    if (opt->cfg && (opt->rtc = strdup(opt->cfg)) == NULL) {
		pp_log_err("%s(): strdup()", ___F);
		exit(1);
	    }
	    D("Name: %s, no RTC, Config: %s\n", opt->name, opt->cfg);
	}
    }
   
    fclose(fp);

    return (rtc_ok ? GATHER_RET_OK : GATHER_RET_INIT);
    
 fail:
    if (fp) fclose(fp);
    return GATHER_RET_FAIL;    
}

/**
 * sync_with_eep:
 * -used to sync config options with RTC ones
 *  when config fs seems to be ok
 * -updates RTC content or single config options
 *  if they are empty or wrong (less than usual) size
*/
static int
sync_with_eep()
{
    cfg_option_t* opt;
    int cfg_changed = 0, rtc_changed = 0;
    int ret = -1;

    for (opt = options; opt->name != NULL; opt++) {
	if (opt->cfg == NULL || strlen(opt->cfg) < opt->min_size) {
	    /* config option NULL or to short -> probably invalid */
	    pp_log("%s < min size, restoring with %s\n", opt->name, opt->rtc);
	    if (pp_cfg_set(opt->rtc, opt->name) != PP_SUC) {
		pp_log("Error during config write of %s\n", opt->name);
	    } else {
		cfg_changed = 1;
	    }
	} else if (opt->rtc == NULL || strcmp(opt->rtc, opt->cfg)) {
	    /* cfg seems ok but rtc & cfg differ -> update rtc */
	    pp_log("%s differs, saving to rtc\n", opt->name);
	    free(opt->rtc);
	    if ((opt->rtc = strdup(opt->cfg)) == NULL) {
		pp_log_err("%s(): strdup()", ___F);
		exit(1);
	    }
	    rtc_changed = 1;
	} 
    }
    
    if (cfg_changed) {
	D("Config changed\n");
	/* flushing must be done after this program exits! */
	if (pp_cfg_save(DONT_FLUSH) != PP_SUC) {
	    pp_log("Error during config save\n");	    
	}
    }

    if (rtc_changed) {
	D("RTC changed\n");
	if (save_to_eep()) {
	    pp_log("Error during RTC save\n");
	    goto bail;
	}
    }

    ret = 0;
    
 bail:
    /* free allocated resources */
    for (opt = options; opt->name != NULL; opt++) {
	free(opt->cfg);
	free(opt->rtc);
    }
    return ret;
}

/**
 * restore_from_eep:
 * -used to restore options from RTC 
 *  when config fs is damaged
*/
static int
restore_from_eep()
{
    cfg_option_t* opt;

    for (opt = options; opt->name != NULL; opt++) {
	if (pp_cfg_set(opt->rtc, opt->name) != PP_SUC) {
	    pp_log("Error during config write of %s\n", opt->name);
	}
    }

    /* flushing must be done after this program exits! */
    if (pp_cfg_save(DONT_FLUSH) != PP_SUC) {
	pp_log("Error during config save\n");
    }
    
    return 0;
}

/**
 * save_to_eep:
 * -saves current option to rtc
*/
static int
save_to_eep(void)
{
    FILE* fp = NULL;
    cfg_option_t* opt;
    int ret = -1;

    if ((fp = fopen(RTC_EEP_DEV, "w"))==NULL) {
	pp_log("Could not open RTC EEPROM\n");
	goto bail;
    }

    if (fprintf(fp, RTC_EEP_MAGIC) < 0) {
	pp_log("Error during rtc write of magic\n");
	goto bail;
    }
    
    for (opt = options; opt->name != NULL; opt++) {
	D("Saving %s as %s to RTC\n", opt->name, opt->rtc);
	if (fprintf(fp, "%s\n", opt->rtc ? opt->rtc : "") < 0) {
	    pp_log("Error during rtc write of %s\n", opt->name);
	    goto bail;
	}
    }

    ret = 0;

 bail:
    if (fp) fclose(fp);
    return ret;
}
