#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <ctype.h>
#include "client.h"

#define PROG_NAME	"dhcpcd"

#include <liberic_config.h>
#include <pp/base.h>
#include <pp/um.h>
#include <pp/cfg.h>
#include <pp_kernel_common.h>

extern dhcpOptions	DhcpOptions;
extern dhcpInterface	DhcpIface;

PP_SYM_HIDDEN int
saveConfigInit()
{
    static u_char pp_base_initialized = 0;
    static u_char eric_config_initialized = 0;
    static u_char pp_cfg_initialized = 0;
    static u_char pp_um_initialized = 0;

    if (!pp_base_initialized && pp_base_init(PROG_NAME, LOG_NOT_SILENT) != 0) {
	return -1;
    }
    pp_base_initialized = 1;

    /*
     * Since we play with fork() in dhcpd we should not use a thread for
     * background flushing. There seems to be a bug in the pthread_create() call
     * that may deadlock.
     */
    if (!eric_config_initialized && eric_config_init(FLUSH_IN_FOREGROUND) != 0) {
	return -1;
    }
    eric_config_initialized = 1;

    if (!pp_cfg_initialized && PP_FAILED(pp_cfg_init(PP_CD_FNAME_DEFAULT, PP_CD_OEM_SKEL_FNAME_DEFAULT))) {
	return -1;
    }
    pp_cfg_initialized = 1;

    if (!pp_um_initialized && pp_um_init() != 0) {
	    return -1;
    }
    pp_um_initialized = 1;

    return 0;
}

/*
 * Copy those bits of a string which are alphanumeric or in a
 * "safe" list of characters.
 */
static void
safecopy(unsigned char *out, unsigned char *string, int len)
{
	char safe[] = "./:-_=+[]~()%&*^#@! ";
	int i, c;

	for (i = 0 ; i < len; i++) {
		c = string[i] ;
		out[i] = (isalnum(c) || strchr(safe, c) != NULL) ? c : '?';
	}
	out[i] = 0;
} 

/* Prints the string passed - stolen from bootpc */
static void
safeprint(FILE * fp, char *name, unsigned char *cookie, int len)
{   
	char * buf;
	if (len == -1)
		len = strlen((char *)cookie);
	if (len > 0) {
		buf = (char *)malloc(len);
		safecopy((unsigned char *)buf, cookie, len);
		fprintf(fp, "%s='%s'\n", name, buf) ;
		free(buf);
	}
}

PP_SYM_HIDDEN void
printVendorSpecificInfo(FILE * fp, const unsigned char * info, int len)
{
	char name[9];
	int i = 0; 
	while (i < len && (unsigned char)info[i] != endOption) {
		int subtag = info[i];
		int sublen;
		/* If we arn't at the end of the cookie and we will need it extract len */
		if ((i < len - 1) && (subtag != padOption)) {
			sublen = info[i+1];
		} else {
			sublen = 0;
		}
		/* Warn if the "length" takes us out of the cookie and truncate */
		if (sublen + i > len) {
			/* And truncate in any case even with no warning */
			sublen = len - i;
		}
		snprintf(name, sizeof(name), "T%3.3d_%3.3d", vendorSpecificInfo, subtag) ;
		safeprint(fp, name, (unsigned char *)info+i+2, sublen);
		i += 2 + sublen;
	}
}

static char *
getVendorSpecificInfoByTag(const unsigned char * info, size_t len, int tag)
{
    size_t i = 0;
    while (i < len && info[i] != endOption) {
	int subtag = info[i];
	int sublen;
	/* If we arn't at the end of the cookie and we will need it extract len */
	if ((i < len - 1) && (subtag != padOption)) {
	    sublen = info[i+1];
	} else {
	    sublen = 0;
	}
	/* Warn if the "length" takes us out of the cookie and truncate */
	if (sublen + i > len) {
	    /* And truncate in any case even with no warning */
	    sublen = len - i;
	}
	if (subtag == tag && sublen > 0) {
	    char * subinfo = (char *)malloc(sublen+1);
	    safecopy(subinfo, (unsigned char *)info+i+2, sublen);
	    subinfo[sublen] = '\0';
	    return subinfo;
	}
	i += 2 + sublen;
    }
    return NULL;
}

static void
ericSaveOptions(void)
{
    int fd;

    if ((fd = creat("/var/run/dont_restart_rc.net", 0644)) >= 0) {
	close(fd);
    }
    
    pp_cfg_save(DO_FLUSH);
}

PP_SYM_HIDDEN void
saveConfig(void)
{
    const char * ipaddr_name;
    const char * netmask_name;
    const char * gateway_name;
    char * old_ipaddr = NULL;
    char * old_netmask = NULL;
    char * old_gateway = NULL;
    char * old_dns_ip_1 = NULL;
    char * old_dns_ip_2 = NULL;
    char * old_ntp_server_1 = NULL;
    char * old_ntp_server_2 = NULL;
    char * old_local_profile_tag = NULL;
    char * old_eth_speed = NULL;
    char * old_eth_duplex = NULL;
    char new_ipaddr[16];
    char new_netmask[16];
    char new_gateway[16];
    char new_dns_ip_1[16];
    char new_dns_ip_2[16];
    char new_ntp_server_1[16];
    char new_ntp_server_2[16];
    char * new_local_profile_tag = NULL;
    char * new_eth_speed = NULL;
    char * new_eth_duplex = NULL;
    int need_saving = 0;
   
    if (saveConfigInit() == -1) return;

    ipaddr_name = "network.ipaddr";
    netmask_name = "network.netmask";
    gateway_name = "network.gateway";

    pp_cfg_get_nodflt(&old_ipaddr, ipaddr_name);
    pp_cfg_get_nodflt(&old_netmask, netmask_name);
    pp_cfg_get_nodflt(&old_gateway, gateway_name);
    pp_cfg_get_nodflt(&old_dns_ip_1, "network.dns_ip_1");
    pp_cfg_get_nodflt(&old_dns_ip_2, "network.dns_ip_2");
    pp_cfg_get_nodflt(&old_ntp_server_1, "time.proto.ntp.server_1");
    pp_cfg_get_nodflt(&old_ntp_server_2, "time.proto.ntp.server_2");
    pp_cfg_get_nodflt(&old_local_profile_tag, "profile_tag.local");
    pp_cfg_get_nodflt(&old_eth_speed, "network.eth_speed");
    pp_cfg_get_nodflt(&old_eth_duplex, "network.eth_duplex");
    new_ipaddr[0] = '\0';
    new_netmask[0] = '\0';
    new_gateway[0] = '\0';
    new_dns_ip_1[0] = '\0';
    new_dns_ip_2[0] = '\0';
    new_ntp_server_1[0] = '\0';
    new_ntp_server_2[0] = '\0';

    if (inet_ntop(AF_INET, &DhcpIface.ciaddr, new_ipaddr,
		  sizeof(new_ipaddr)) == NULL) {
	/* FIXME: report error */
	goto fail;
    }

    sethostname(new_ipaddr, sizeof(new_ipaddr));

    if (inet_ntop(AF_INET, DhcpOptions.val[subnetMask], new_netmask,
		  sizeof(new_netmask)) == NULL) {
	/* FIXME: report error */
	goto fail;
    }

    if (DhcpOptions.len[routersOnSubnet] > 3) {
	if (inet_ntop(AF_INET, DhcpOptions.val[routersOnSubnet],
		      new_gateway, sizeof(new_gateway)) == NULL) {
	    /* FIXME: report error */
	    goto fail;
	}
    }

    if (DhcpOptions.len[dns] > 7) {
	if (inet_ntop(AF_INET, &((char *)DhcpOptions.val[dns])[4],
		      new_dns_ip_2, sizeof(new_dns_ip_2)) == NULL) {
	    /* FIXME: report error */
	    goto fail;
	}
    }

    if (DhcpOptions.len[dns] > 3) {
	if (inet_ntop(AF_INET, DhcpOptions.val[dns],
		      new_dns_ip_1, sizeof(new_dns_ip_1)) == NULL) {
	    /* FIXME: report error */
	    goto fail;
	}
    }

    if (DhcpOptions.len[ntpServers] > 7) {
	if (inet_ntop(AF_INET, &((char *)DhcpOptions.val[ntpServers])[4],
		      new_ntp_server_2, sizeof(new_ntp_server_2)) == NULL) {
	    /* FIXME: report error */
	    goto fail;
	}
    }

    if (DhcpOptions.len[ntpServers] > 3) {
	if (inet_ntop(AF_INET, DhcpOptions.val[ntpServers],
		      new_ntp_server_1, sizeof(new_ntp_server_1)) == NULL) {
	    /* FIXME: report error */
	    goto fail;
	}
    }

    new_local_profile_tag =
	getVendorSpecificInfoByTag(DhcpOptions.val[vendorSpecificInfo],
				   DhcpOptions.len[vendorSpecificInfo],
				   1 /* the profile tag */);
    new_eth_speed =
	getVendorSpecificInfoByTag(DhcpOptions.val[vendorSpecificInfo],
				   DhcpOptions.len[vendorSpecificInfo],
				   2 /* ethernet speed */);
    new_eth_duplex =
	getVendorSpecificInfoByTag(DhcpOptions.val[vendorSpecificInfo],
				   DhcpOptions.len[vendorSpecificInfo],
				   3 /* ethernet duplex mode */);

    if ((old_ipaddr == NULL && new_ipaddr[0] != '\0') ||
	(old_ipaddr != NULL && strcmp(old_ipaddr, new_ipaddr))) {

        pp_cfg_set(new_ipaddr, ipaddr_name);
	need_saving = 1;
    }

    if ((old_netmask == NULL && new_netmask[0] != '\0') ||
	(old_netmask != NULL && strcmp(old_netmask, new_netmask))) {

        pp_cfg_set(new_netmask, netmask_name);
	need_saving = 1;
    }

    if ((old_gateway == NULL && new_gateway[0] != '\0') ||
	(old_gateway != NULL && strcmp(old_gateway, new_gateway))) {

        pp_cfg_set(new_gateway, gateway_name);
	need_saving = 1;
    }

    if ((old_dns_ip_1 == NULL && new_dns_ip_1[0] != '\0') ||
	(old_dns_ip_1 != NULL && strcmp(old_dns_ip_1, new_dns_ip_1))) {

        pp_cfg_set(new_dns_ip_1, "network.dns_ip_1");
	need_saving = 1;
    }

    if ((old_dns_ip_2 == NULL && new_dns_ip_2[0] != '\0') ||
	(old_dns_ip_2 != NULL && strcmp(old_dns_ip_2, new_dns_ip_2))) {

        pp_cfg_set(new_dns_ip_2, "network.dns_ip_2");
	need_saving = 1;
    }

    if ((old_ntp_server_1 == NULL && new_ntp_server_1[0] != '\0') ||
	(old_ntp_server_1 != NULL &&
	 strcmp(old_ntp_server_1, new_ntp_server_1))) {

        pp_cfg_set(new_ntp_server_1, "time.proto.ntp.server_1");
	need_saving = 1;
    }

    if ((old_ntp_server_2 == NULL && new_ntp_server_2[0] != '\0') ||
	(old_ntp_server_2 != NULL &&
	 strcmp(old_ntp_server_2, new_ntp_server_2))) {

        pp_cfg_set(new_ntp_server_2, "time.proto.ntp.server_2");
	need_saving = 1;
    }

    if ((old_local_profile_tag == NULL && new_local_profile_tag != NULL) ||
	(old_local_profile_tag != NULL && new_local_profile_tag != NULL &&
	 strcmp(old_local_profile_tag, new_local_profile_tag))) {

	if (!new_local_profile_tag
	    || strcspn(new_local_profile_tag, "\r\n") == strlen(new_local_profile_tag)) {
            pp_cfg_set(new_local_profile_tag, "profile_tag.local");
	    need_saving = 1;
	}
    }

    if ((old_eth_speed == NULL && new_eth_speed != NULL) ||
	(old_eth_speed != NULL && new_eth_speed != NULL &&
	 strcmp(old_eth_speed, new_eth_speed))) {
	if (!new_eth_speed
	    || !strcmp(new_eth_speed, "10")
	    || !strcmp(new_eth_speed, "100")
	    || !strcmp(new_eth_speed, "auto")) {
            pp_cfg_set(new_eth_speed, "network.eth_speed");
	    need_saving = 1;
	}
    }

    if ((old_eth_duplex == NULL && new_eth_duplex != NULL) ||
	(old_eth_duplex != NULL && new_eth_duplex != NULL &&
	 strcmp(old_eth_duplex, new_eth_duplex))) {
	if (!new_eth_duplex
	    || !strcmp(new_eth_duplex, "half")
	    || !strcmp(new_eth_duplex, "full")
	    || !strcmp(new_eth_duplex, "auto")) {
            pp_cfg_set(new_eth_duplex, "network.eth_duplex");
	    need_saving = 1;
	}
    }

    if (need_saving) ericSaveOptions();

 fail:
    free(old_ipaddr);
    free(old_netmask);
    free(old_gateway);
    free(old_dns_ip_1);
    free(old_dns_ip_2);
    free(old_ntp_server_1);
    free(old_ntp_server_2);
    free(old_local_profile_tag);
    free(new_local_profile_tag);
    free(old_eth_speed);
    free(new_eth_speed);
    free(old_eth_duplex);
    free(new_eth_duplex);
}
