/*
 * Dropbear - a SSH2 server
 * 
 * Copyright (c) 2002,2003 Matt Johnston
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE. */

#include "includes.h"
#include "dbutil.h"
#include "session.h"
#include "buffer.h"
#include "signkey.h"
#include "runopts.h"

typedef struct {
    pp_net_conn_data_t * conn_data;
    struct sockaddr_storage remoteaddr;
} dropbear_client_t;

static int initialized = 0;

void pp_dropbear_handle_connection(pp_net_conn_data_t * conn_data);
static u_short get_port_ssh(void);
static void client_cleanup_handler(void * _cl);

/* start dropbear thread from eric main process */
int
pp_dropbear_init(void)
{
    if (!initialized) {

	if (PP_FAILED(eric_net_register_proto(PP_NET_PROTOCOL_TYPE_SSH, "SSH", 1, get_port_ssh,
					      eric_net_init_bio_standard, pp_dropbear_handle_connection))) {
	    return PP_ERR;
	}

	_dropbear_exit = svr_dropbear_exit;
	_dropbear_log = svr_dropbear_log;
	session_globals_init();

	svr_opts.rsakeyfile = NULL;
	svr_opts.dsskeyfile = NULL;
	svr_opts.rsakeyfile = RSA_PRIV_FILENAME;
	svr_opts.dsskeyfile = DSS_PRIV_FILENAME;
	svr_opts.bannerfile = NULL;
	svr_opts.banner = NULL;
	svr_opts.forkbg = 0; /* important: don't fork! */
	svr_opts.norootlogin = 0;
	svr_opts.noauthpass = 0;
	svr_opts.norootpass = 0;
	svr_opts.inetdmode = 0;
	svr_opts.portcount = 1;
	svr_opts.ports[0] = strdup(DROPBEAR_DEFPORT);
	svr_opts.reconf_ports = 1;
	svr_opts.hostkey = NULL;
	opts.nolocaltcp = 0;
	opts.noremotetcp = 0;
#ifdef DO_MOTD
	svr_opts.domotd = 0;
#endif
#ifndef DISABLE_SYSLOG
	svr_opts.usingsyslog = 0;
#endif
	initialized = 1;
    }

#ifndef DISABLE_SYSLOG
    if (svr_opts.usingsyslog) startsyslog();
#endif

    /*
     * Now we can setup the hostkeys - needs to be after logging is on,
     * otherwise we might end up blatting error messages to the socket.
     */
    loadhostkeys();

    return PP_SUC;
}

void
pp_dropbear_cleanup(void)
{
    eric_net_unregister_proto(PP_NET_PROTOCOL_TYPE_SSH);
    initialized = 0;
}

void
pp_dropbear_handle_connection(pp_net_conn_data_t * conn_data)
{
    dropbear_client_t * cl = calloc(1, sizeof(dropbear_client_t));
    char client_ip[INET_ADDRSTRLEN];
    int fd;

    pthread_setspecific(session_globals_key, new_session_globals());

    pthread_cleanup_push(client_cleanup_handler, cl);

    cl->conn_data = conn_data;

    pp_bio_get_peer_name(conn_data->bio, client_ip, sizeof(client_ip));

    dropbear_log(LOG_INFO, "Child connection from %s", client_ip);

    fd = BIO_get_fd(conn_data->bio, NULL);
    if (fd >= 0) {
	/* start the session */
	svr_session(fd, client_ip, client_ip);
    } else {
	pp_log("%s(): BIO not initialized.\n", ___F);
    }

    pthread_cleanup_pop(1);
}

static u_short
get_port_ssh(void)
{
    int enabled;
    u_short port = 0;
    pp_cfg_is_enabled(&enabled, "network.ssh_enabled");
    if (enabled) pp_cfg_get_ushort(&port, "network.ssh_port");
    return port;
}

static void
client_cleanup_handler(void * _cl)
{
    dropbear_client_t * cl = (dropbear_client_t *)_cl;
    GLOBALS(g);
    g->terminated = 1;
    close(g->ses.sock);
    delete_session_globals(g->thread_id);
    BIO_free_all(cl->conn_data->bio);
    free(cl->conn_data);
    free(cl);
    dropbear_log(LOG_INFO, "Child thread exited.");
}

