package nn.pp.drvredir;

import java.io.*;
import java.util.*;

class DrvRedirDriveLinux extends DrvRedirIso {
    boolean mediumChanging = false;

    // find a list of all available drives under linux
    public Drive[] getAvailableDrives() {
        Vector devs = new Vector();
        File devDir = new File("/dev");
        String files[] = devDir.list();

        for (int i = 0; files != null && i < files.length; i++) {
            String s = files[i];
            if (s.startsWith("hd") ||
                s.startsWith("sd") ||
                s.startsWith("scd") ||
                (s.startsWith("fd") && s.length() < 4) ||
                s.startsWith("cdrom")) {
                devs.add("/dev/" + files[i]);
            }
        }
        
        if (devs.size() == 0) {
            return null;
        }

        Drive[] ret = new Drive[devs.size()];
        for (int i = 0; i < devs.size(); i++) {
            ret[i] = new Drive((String)devs.elementAt(i), null);
        }
        
        return ret;
    }

    void openFile(String directory, String filename, boolean readOnly) throws IOException {
        super.openFile(directory, filename, readOnly);
        mediumChanging = false;
    }

    boolean isRemovable(RandomAccessFile file) throws IOException {
        // check whether this is a removable or fixed drive at all by calling super's method
        boolean s = super.isRemovable(file) || super.isFixed(file);
        if (!s) return s;
        
        // if the filename ends with a number, we assume it a removable
        return filenameEndsWithNumber();
    }

    boolean isFixed(RandomAccessFile file) throws IOException {
        // check whether this is a removable or fixed drive at all by calling super's method
        boolean s = super.isRemovable(file) || super.isFixed(file);
        if (!s) return s;
        
        // if the filename does not end with a number, we assume it a fixed disk
        return !filenameEndsWithNumber();
    }
    
    private boolean filenameEndsWithNumber() {
        char last = filename.charAt(filename.length() - 1);
        return last == '0' || last == '1' || last == '2' || last == '3' || last == '4' ||
            last == '5' || last == '6' || last == '7' || last == '8' || last == '9';
    }

    public int getMediumChangeState() {
        try {
            RandomAccessFile f = new RandomAccessFile(new File(directory, filename), "r");
            
            // we have access to the medium, now check the size
            if (mediumChanging) {
                // there was no medium before, so we must have a new one
                return CD_EXT_MEDIUM_CHANGED;
            }
            
            // check whether we have a new medium size now
            int lsec = (int)(f.length() / (long)sectorSize) - 1;
            f.close();
            if (lsec == lastSectorNo) {
                return CD_EXT_MEDIUM_NO_CHANGE;
            } else {
                return CD_EXT_MEDIUM_CHANGED;
            }
        } catch (IOException e) {
            // we cannot access the drive, so no medium is present
            if (mediumChanging) {
                return CD_EXT_MEDIUM_NO_CHANGE;
            } else {
                return CD_EXT_MEDIUM_REMOVED;
            }
        }
    }

    void mediumRemovedForChange() throws Exception {
        mediumChanging = true;
    }
    
    void mediumInsertedAfterChange() throws Exception {
        mediumChanging = false;
    }
}
