package nn.pp.drvredir;

import java.io.*;
import nn.pp.rc.*;

class DrvRedirIso implements DriveAccess {
    RandomAccessFile file;
    String filename, directory;
    boolean readOnly;
    
    private final static int ISO_SECTOR_SIZE = 2048;
    private final static int FLOPPY_SECTOR_SIZE = 512;
    private final static int REMOVABLE_SECTOR_SIZE = 512;
    private final static int FIXED_SECTOR_SIZE = 512;
    
    private int mspType = MSPProto.MSP_DISC_TYPE_CDROM;
    protected int sectorSize = ISO_SECTOR_SIZE;
    protected int lastSectorNo = 0;
    
    private boolean debug = false;
    
    // NOTE: overwritten in DrvRedirDriveLinux
    void openFile(String directory, String filename, boolean readOnly) throws IOException {
        this.readOnly = readOnly;
        this.directory = directory;
        this.filename = filename;
        file = new RandomAccessFile(new File(directory, filename),
            readOnly ? "r" : "rw");
        if (file == null) {
            throw new IOException(T._("Could not open file"));
        }
        
        adjustParameters();

        // Last sector number, _not_ total sector count!
        lastSectorNo = (int)(file.length() / (long)sectorSize) - 1;
    }
    
    void closeFile() {
        if (file != null) {
            try {
                file.close();
            } catch (IOException ignore) {
            }
        }
        file = null;
    }
    
    private void adjustParameters() throws IOException {
        if (isFloppy(file)) {
            mspType = MSPProto.MSP_DISC_TYPE_FLOPPY;
            sectorSize = FLOPPY_SECTOR_SIZE;
        } else if (isIso(file)) {
            mspType = MSPProto.MSP_DISC_TYPE_CDROM;
            sectorSize = ISO_SECTOR_SIZE;
        } else if (isRemovable(file)) {
            mspType = MSPProto.MSP_DISC_TYPE_REMOVABLE;
            sectorSize = REMOVABLE_SECTOR_SIZE;
        } else if (isFixed(file)) {
            mspType = MSPProto.MSP_DISC_TYPE_SOLID;
            sectorSize = FIXED_SECTOR_SIZE;
        } else {
            throw new IOException(T._("No valid image file found"));
        }
    }
    
    boolean isFloppy(RandomAccessFile file) throws IOException {
        long len = file.length();
        if (len > 2880 * 1024 || (len % FLOPPY_SECTOR_SIZE) != 0) {
            if (debug) System.out.println("Image file is not a floppy image!");
            return false;
        }
        if (debug) System.out.println("Image file is floppy image");
        return true;
    }
    
    private final static int TAGLEN = 5;
    private final static int ISO9660_MAGIC_POS = 0x8001;
    private final static String ISOTAG = "CD001";

    boolean isIso(RandomAccessFile file) throws IOException {
        long len = file.length();

        if (len < ISO9660_MAGIC_POS + TAGLEN || (len % FLOPPY_SECTOR_SIZE) != 0) {
            if (debug) System.out.println("Image file is not a ISO image, size doesn't match!");
        }
        
        byte imageTagBytes[] = new byte[TAGLEN];
        
        try {
            long fp = file.getFilePointer();
            file.seek(ISO9660_MAGIC_POS);
            file.readFully(imageTagBytes, 0, TAGLEN);
            file.seek(fp);
        } catch (IOException e) {
            return false;
        }
        
        String imageTag = new String(imageTagBytes);
        if (!ISOTAG.equals(imageTag)) {
            if (debug) System.out.println("Image file is not a ISO image, tag doesn't match (" + imageTag + " - " + ISOTAG + ")");
            return false;
        }
        if (debug) System.out.println("Image file is ISO image");
        return true;
    }
    
    // NOTE: overwritten in DrvRedirDriveLinux
    boolean isRemovable(RandomAccessFile file) throws IOException {
        long len = file.length();
        if ((len % REMOVABLE_SECTOR_SIZE) != 0) {
            if (debug) System.out.println("Image file is not a removable image!");
            return false;
        }
        if (debug) System.out.println("Image file is removable image");
        return true;
    }
    
    // NOTE: overwritten in DrvRedirDriveLinux
    boolean isFixed(RandomAccessFile file) throws IOException {
        return false;
    }
    
    public boolean mediumChangeSupported() {
        return false;
    }
    
    // NOTE: overwritten in DrvRedirDriveLinux
    public Drive[] getAvailableDrives() {
        return null;
    }
    
    public int getMSPDriveType() {
        return mspType;
    }
    
    public int getSectorSize() {
        return sectorSize;
    }
    
    public int getSectorNo() throws IOException {
        return lastSectorNo;
    }
    
    synchronized public void readCDSectors(long startSector, long noSectors, byte dataBuf[]) throws IOException {
        file.seek(startSector * sectorSize);
        file.readFully(dataBuf, 0, (int)(noSectors * sectorSize));
    }
    
    synchronized public void writeCDSectors(long startSector, long noSectors, byte dataBuf[]) throws IOException {
        if (readOnly) {
            throw new IOException(T._("Writing not supported"));
        } else {
            file.seek(startSector * sectorSize);
            file.write(dataBuf, 0, (int)(noSectors * sectorSize));
        }
    }
    
    // NOTE: overwritten in DrvRedirDriveLinux
    public int getMediumChangeState() {
        return CD_EXT_MEDIUM_NO_CHANGE;
    }
}
