package nn.pp.drvredir;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.text.*;

import nn.pp.rc.*;

class StatusLabel extends Label {
    StatusLabel() {
        setDisconnected();
    }
    
    void setDisconnected() {
        setText(T._("Not connected"));
    }
    
    void setDrive(String drive) {
        setText(T._("Using Drive ") + drive);
    }

    void setIso(String iso) {
        setText(T._("Using ") + iso);
    }
}

class DrvRedirSettingsPanel extends Panel implements ActionListener {
    // member variables
    DrvRedirNative drvAccess;
    RFBProfile profile;
    ConnectionHandler handler;
    PrintStream logger;
    boolean readOnly;
    
    Label drvLabel;
    StatusLabel statusLabel;
    Button connectDriveButton;
    Button connectIsoButton;
    Button disconnectButton;
    
    int msNo;
    int msIndex;
    
    // debugging
    void debug(String s) {
    	if (false) System.out.println(s);
    }
    
    // Constructor
    public DrvRedirSettingsPanel(RFBProfile profile, PrintStream logger,
				 int msNo, int msIndex, DrvRedirNative drvAccess,
				 boolean readOnly) {
	this.profile = profile;
    	this.logger = logger;
    	this.msNo = msNo;
    	this.msIndex = msIndex;
    	this.drvAccess = drvAccess;
    	this.readOnly = readOnly;
    	
    	addLayout();
    	
    	disconnect(true);
    }
    
    // panel handling
    public void close() {
    	disconnect(true);
    	hide();
    }
    
    public void setEnabled(boolean enabled) {
    	if (!enabled) {
    	    disconnect(true);
    	    enableControls(true, false);
    	}
    	super.setEnabled(enabled);
    }
    
    private void addLayout() {
        setLayout(new BorderLayout());
        
        Panel p1 = new Panel(new FlowLayout(FlowLayout.LEFT, 0, 1));
    	if (msNo == 1) {
    	    p1.add(drvLabel = new Label(T._("Drive:")));
    	} else {
    	    p1.add(drvLabel = new Label(T._("Drive") + " " + (msIndex + 1) + ":"));
    	}
    	
        p1.add(new Label(""));
    	p1.add(connectDriveButton = new Button(T._("Connect Drive")));
    	p1.add(connectIsoButton = new Button(T._("Connect ISO")));
    	p1.add(disconnectButton = new Button(T._("Disconnect")));
        p1.add(new Label(""));
        
        add(p1, BorderLayout.WEST);
    	add(statusLabel = new StatusLabel(), BorderLayout.CENTER);
    	
    	if (connectDriveButton != null) connectDriveButton.addActionListener(this);
    	if (connectIsoButton != null) connectIsoButton.addActionListener(this);
    	if (disconnectButton != null) disconnectButton.addActionListener(this);
    	
    	enableControls(false, false);
    }
    
    private void enableControls(boolean disableAll, boolean connected) {
    	drvLabel.setEnabled(!disableAll);
    	statusLabel.setEnabled(!disableAll);
    	
    	if (connectDriveButton != null) connectDriveButton.setEnabled(disableAll ? false : !connected);
    	if (connectIsoButton != null) connectIsoButton.setEnabled(disableAll ? false : !connected);
    	if (disconnectButton != null) disconnectButton.setEnabled(disableAll ? false : connected);
    }
    
    // action callbacks
    public void actionPerformed(ActionEvent e) {
    	Object src = e.getSource();
    	
    	if (src == connectDriveButton) {
    	    if (drvAccess != null) {
    	        connectDriveWindows();
    	    } else {
    	        connectDriveLinux();
    	    }
    	} else if (src == connectIsoButton) {
    	    connectIso();
    	} else if (src == disconnectButton) {
    	    disconnect(true);
    	}
    }

    private Frame getParentFrame() {
        Frame frame = null;
        Container parent = this;
        while (parent != null) {
            if (parent instanceof Frame) {
                frame = (Frame)parent;
                break;
            }
            parent = parent.getParent();
        }
        
        return frame;
    }

    private void connectDriveWindows() {
        ConnectionHandlerDrive handlerDrive;
    	handler = handlerDrive = new ConnectionHandlerDrive(profile, this, drvAccess, logger, msIndex, readOnly);
    	
    	try {
    	    handlerDrive.chooseDrive(getParentFrame());
    	    handler.connect();
    	    enableControls(false, true);
    	    logger.println(MessageFormat.format(T._("Drive Redirection established (Drive {0})."),
    	        new Object[] { handlerDrive.drive }));
    	    statusLabel.setDrive(handlerDrive.drive);
    	} catch (Exception e) {
    	    logger.println(e.getMessage() == null ? T._("Error opening device") : e.getMessage());
    	    disconnect(false);
    	}
    }
    
    private void connectDriveLinux() {
        ConnectionHandlerDriveLinux handlerLinux;
        handler = handlerLinux = new ConnectionHandlerDriveLinux(profile, this, logger, msIndex, readOnly);
        
        try {
    	    handlerLinux.chooseDrive(getParentFrame());
    	    handler.connect();
    	    enableControls(false, true);
    	    logger.println(MessageFormat.format(T._("Drive Redirection established (Drive {0})."),
    	        new Object[] { handlerLinux.drive }));
    	    statusLabel.setDrive(handlerLinux.drive);
        } catch (Exception e) {
    	    logger.println(e.getMessage() == null ? T._("Error opening device") : e.getMessage());
    	    disconnect(false);
    	}
    }
    
    private void connectIso() {
        ConnectionHandlerIso handlerIso;
        handler = handlerIso = new ConnectionHandlerIso(profile, this, logger, msIndex);
        
        try {
            handlerIso.chooseIsoFile(getParentFrame());
            //if (true) throw new IOException("blabla");
            handler.connect();
    	    enableControls(false, true);
    	    logger.println(MessageFormat.format(T._("ISO Drive Redirection established ({0})."),
    	        new Object[] { handlerIso.filename }));
    	    statusLabel.setIso(handlerIso.filename);
    	} catch (Exception e) {
    	    logger.println(e.getMessage() == null ? T._("Error opening file") : e.getMessage());
    	    disconnect(false);
    	}
    }
    
    private void disconnect(boolean showInfo) {
    	debug("Disconnecting.");
    	
    	if (handler != null) {
    	    handler.disconnect();
    	    handler = null;
    	}
    	
    	enableControls(false, false);
    	if (showInfo) {
    	    logger.println(T._("Drive Redirection disconnected."));
    	    statusLabel.setDisconnected();
    	}
    }
    
    public void disconnected(boolean server) {
    	debug("Have been disconnected by " + (server ? "Server" : "User"));
    	if (server) {
    	    disconnect(false);
    	    logger.println(T._("Disconnected by the device!"));
    	} else {
    	    logger.println(T._("Successfully disconnected!"));
    	}
    	statusLabel.setDisconnected();
    }
}
