package nn.pp.drvredir;

import java.io.*;
import java.net.*;
import java.text.*;
import javax.net.ssl.*;
import java.security.*;

import nn.pp.rc.*;

class MSPProto extends ErlaConnector {
    /*****************************************************************************
     * Initial handshaking message
     *****************************************************************************/
    final static String protoInitMsg = "e-RIC MSP P";
    final static String mspProtocolVersion = "e-RIC MSP 01.02\n";
    
    /*****************************************************************************
     * Message types
     *****************************************************************************/

    /* client -> server*/
    final static int MSP_TYPE_LOGIN		= 0x00;
    final static int MSP_TYPE_RQ_CONNECTION	= 0x01;
    final static int MSP_TYPE_SEND_DATA		= 0x02;	/* used in both directions */
    final static int MSP_TYPE_QUIT_CONNECTION	= 0x03;	/* used in both directions */
    final static int MSP_TYPE_PING		= 0x04;	/* used in both directions */
    final static int MSP_TYPE_PONG		= 0x05;	/* used in both directions */
    final static int MSP_TYPE_SESSION_ID	= 0x06;
    
    /* server -> client */
    final static int MSP_TYPE_RSP_CONNECTION	= 0x80;
    final static int MSP_TYPE_RQ_DATA		= 0x81;
    final static int MSP_TYPE_DATA_ACK		= 0x82;
    
    /* sub messages */
    final static int MSP_QUIT_USER_CANCELLED	= 0x00;
    final static int MSP_QUIT_DEVICE_CANCELLED	= 0x01;

    final static int MSP_DATA_OKAY		= 0x00;	// request completed successfully
    final static int MSP_DATA_ERROR		= 0x01;	// error in client (e.g. no memory)
    final static int MSP_DATA_NO_CD		= 0x02;	// currently no cd inserted
    final static int MSP_DATA_NOT_COMPLETE	= 0x03;	// not all requested sectors read,

    final static int MSP_CONNECTION_RESP_OKAY			= 0x00;	// only if connection is established
    final static int MSP_CONNECTION_RESP_NOT_AVAILABLE		= 0x01;
    final static int MSP_CONNECTION_RESP_ALREADY_CONNECTED	= 0x02;
    final static int MSP_CONNECTION_RESP_ALREADY_IMAGE		= 0x03;
    final static int MSP_CONNECTION_RESP_PROTOCOL_ERROR		= 0x04;
    final static int MSP_CONNECTION_RESP_AUTH_FAILED		= 0x05;
    final static int MSP_CONNECTION_RESP_NO_PERMISSION		= 0x06;
    final static int MSP_CONNECTION_RESP_INTERNAL_ERROR		= 0x07;
    final static int MSP_CONNECTION_RESP_NO_SUCH_MS_INDEX	= 0x08;

    final static int MSP_DISC_TYPE_CDROM	= 0x00;
    final static int MSP_DISC_TYPE_FLOPPY	= 0x01;
    final static int MSP_DISC_TYPE_REMOVABLE	= 0x02;
    final static int MSP_DISC_TYPE_SOLID	= 0x03;
    final static int MSP_DISC_NONE		= 0xff;

    /*****************************************************************************
     * Members
     *****************************************************************************/

    RFBProfile profile;
    PrintStream logger;
    DriveAccess drvAccess;
    
    boolean writeSupport;
    int msIndex;

    private Socket                    sock;
    private DataInputStream           is;
    private DataOutputStream          os;
    private boolean                   connected = false;
    private boolean                   supportProto_1_2 = false;
    
    /*****************************************************************************
     * Methods
     *****************************************************************************/

    MSPProto(RFBProfile prof, PrintStream logger, DriveAccess drvAccess, int msIndex) {
    	super(prof, logger, "MSP");
    	this.profile = prof;
    	this.logger = logger;
    	this.drvAccess = drvAccess;
    	this.msIndex = msIndex;
    }
    
    static void debug(String s) {
    	if (false) System.out.println(s);
    }
    
    void initializeMSPConnection(boolean writeSupport) throws Exception {
    	this.writeSupport = writeSupport;
    	
    	connectNetwork();
    	sendClientInit();
    	negotiateProtocolVersion();
    	if (profile.username != null && profile.password != null) {
    	    sendLogin(profile.username, profile.password);
    	} else {
    	    sendAuth();
    	}
    	processResponse();
    	sendConnectionRequest(false);
    	processResponse();
    }
    
    void closeConnection(boolean sendQuitMessage) {
    	if (!connected) {
    	    return;
    	}
    	
    	if (sendQuitMessage) {
    	    try {
    	    	sendQuitMessage(MSP_QUIT_USER_CANCELLED);
    	    } catch (Exception ignore) { }
    	}
    	
    	disconnectNetwork();
    	
    	connected = false; 
    }
    
    void processProtocol() throws IOException {
    	if (!connected) {
    	    throw new IOException(T._("Cannot process protocol, not connected."));
    	}
    	
    	while(connected) {
    	    int type = is.readByte() & 0xff;
    	    
    	    switch (type) {
		case MSP_TYPE_PING:
		    processPingMessage();
		    break;

		case MSP_TYPE_PONG:
		    processPongMessage();
		    break;

		case MSP_TYPE_QUIT_CONNECTION:
		    processQuitMessage();
		    break;

		case MSP_TYPE_RQ_DATA:
		    processDataRequestMessage();
		    break;

		case MSP_TYPE_SEND_DATA:
		    processSendDataMessage();
		    break;

		default:
		    debug("Unknown MSP message received: " + Integer.toHexString(type));
		    throw new IOException(T._("Unknown protocol message received."));
    	    }
    	}
    }
    
    void sendMediumRemoval() throws IOException {
        sendConnectionRequest(true);
    }
    
    void sendMediumChange() throws IOException {
        sendConnectionRequest(false);
    }
    
    /* helper methods */
    private void connectNetwork() throws IOException {
    	logger.println(MessageFormat.format(T._("Connecting Drive Redirection to {0}"),
    	    new Object[] { profile.remoteHost }));
 	if(profile.sslRequired || profile.sslRequested)
 	    sock = connectSSL(profile.remoteHost, profile.sslPort);
 	if(sock == null && !profile.sslRequired)
 	    sock = connect(profile.remoteHost, profile.primaryPort);
	if(sock == null && !profile.sslRequired && profile.useProxy)
	    sock = connectProxy(profile.proxyHost, profile.proxyPort,
				profile.remoteHost, profile.primaryPort);
	if(sock == null) {
	    throw new IOException(sockerr + " (" + T._("no connect options left") + ")");
	}

	connected = true;
	
	os = new DataOutputStream(sock.getOutputStream());
	is = new DataInputStream(new BufferedInputStream(sock.getInputStream(), 32768));;
    }
    
    private void disconnectNetwork() {
    	try {
    	    sock.close();
    	} catch (IOException e) {
    	    e.printStackTrace(logger);
    	}
    	
    }

    private synchronized void sendClientInit() throws IOException {
    	debug("Sending Client initialization message.");
    	os.write(protoInitMsg.getBytes());
    	
    	/* this is a bug in the MSP proto, but at this position of the protocol
    	   handling (the first message), it can't be fixed. */
    	int l = protoInitMsg.length();
    	for (int i = l; i < 19; i++) {
    	    os.write(0);
    	}
    }
    
    private synchronized void negotiateProtocolVersion() throws IOException {
    	debug("Reading server version message.");
    	byte b[] = new byte[16];
	is.readFully(b);
	if ((b[0] != 'e') || (b[1] != '-') || (b[2] != 'R') ||
	    (b[3] != 'I') || (b[4] != 'C') || (b[5] != ' ') ||
	    (b[6] != 'M') || (b[7] != 'S') || (b[8] != 'P') ||
	    (b[9] != ' ') || (b[10]  < '0') || (b[10] > '9')  ||
	    (b[11] < '0') || (b[11] > '9') || (b[12] != '.')||
	    (b[13] < '0') || (b[13] > '9') || (b[14] < '0') ||
	    (b[14] > '9') || (b[15] != '\n')) {
	    throw new IOException(MessageFormat.format(T._("Host {0} hasn't a valid server version"),
	                          new Object[] { profile.remoteHost }));
	}

	int serverMajor = (b[10] - '0') * 10 + (b[11] - '0');
	int serverMinor = (b[13] - '0') * 10 + (b[14] - '0');
	
	debug("MSP server supports protocol version " + serverMajor + "." + serverMinor + ".");

	// parse the version information
	int version = serverMajor * 1000 + serverMinor;
	if (version >= 1002) {
	    supportProto_1_2 = true;
	}
	
	debug("Sending Client version message.");
	
	os.write(mspProtocolVersion.getBytes());
    }
    
    private synchronized void sendLogin(String user, String pass) throws IOException {
    	debug("Logging in. Username: " + user + " Password: " + pass);
    	
    	os.writeByte(MSP_TYPE_LOGIN & 0xff);
    	os.writeByte(0);	// pad
    	os.writeShort(user.length() & 0xffff);
    	os.writeShort(pass.length() & 0xffff);
    	os.write(user.getBytes(), 0, user.length());
    	os.write(pass.getBytes(), 0, pass.length());
    }
    
    private synchronized void sendAuth() throws Exception {
    	debug("Sending session id.");
    	
    	os.writeByte(MSP_TYPE_SESSION_ID & 0xff);
    	
	byte[] challenge_msg = new byte[64 + 9];
	byte[] challenge = new byte[64];
	debug("1");
	is.readFully(challenge_msg);
	System.arraycopy(challenge_msg, 9, challenge, 0, challenge.length);
	
	byte[] response = new byte[32 + 9];
	System.arraycopy(("MSP RESP=").getBytes("ISO-8859-1"), 0, response, 0, 9);
	getChallengeResponse(challenge, response, 9);
	os.write(response);
	os.flush();
    }
    
    private synchronized void processResponse() throws IOException {
    	debug("Processing connection response message.");
    	
    	int type = is.read();
    	int ack = is.read();
    	int reason = is.read();
    	
    	debug("Got connection response, ack = " + ack + ", reason = " + reason);
    	
    	if (ack == 0 || reason != MSP_CONNECTION_RESP_OKAY) {
    	    switch (reason) {
		case MSP_CONNECTION_RESP_AUTH_FAILED:
		    throw new IOException(T._("Authentication failed."));
		case MSP_CONNECTION_RESP_NO_PERMISSION:
		    throw new IOException(T._("You are not allowed to establish Drive Redirection."));
		case MSP_CONNECTION_RESP_NOT_AVAILABLE:
		    throw new IOException(T._("Drive Redirection not available."));
		case MSP_CONNECTION_RESP_ALREADY_CONNECTED:
		    throw new IOException(T._("There is already a Drive Reconnection active on this device."));
		case MSP_CONNECTION_RESP_ALREADY_IMAGE:
		    throw new IOException(T._("Another virtual image is already set on this device."));
		case MSP_CONNECTION_RESP_NO_SUCH_MS_INDEX:
		    throw new IOException(T._("Mass storage index not available."));
		default:
		    throw new IOException(T._("Other response error") + ": " + reason);
    	    }
    	}
    	
    	debug("Successfully logged in.");
    }
    
    private synchronized void sendConnectionRequest(boolean noMedium) throws IOException {
    	int driveType;
    	
    	if (noMedium) {
    	    driveType = MSP_DISC_NONE;
    	} else {
    	    driveType = drvAccess.getMSPDriveType();
    	}
    	
    	if (supportProto_1_2) {
    	    os.write(MSP_TYPE_RQ_CONNECTION & 0xff);
    	    os.write(msIndex);
    	    os.write(0);			// ID
    	    os.write(writeSupport ? 0 : 1);	// write support
    	    os.write(driveType);
    	    os.write(0);			// pad
    	    os.writeShort(drvAccess.getSectorSize() & 0xffff);
    	    os.writeInt(drvAccess.getSectorNo() & 0xffffffff);
    	} else {
    	    os.write(MSP_TYPE_RQ_CONNECTION & 0xff);
    	    os.write(0);			// ID
    	    os.write(writeSupport ? 0 : 1);	// write support
    	    os.write(driveType);
    	    os.writeShort(drvAccess.getSectorSize() & 0xffff);
    	    os.writeShort(0);			// pad
    	    os.writeInt(drvAccess.getSectorNo() & 0xffffffff);
    	}
    }
    
    private String getQuitReason(int reason) {
    	switch(reason) {
    	    case MSP_QUIT_USER_CANCELLED:
    	    	return T._("User cancelled connection");
    	    case MSP_QUIT_DEVICE_CANCELLED:
    	    	return T._("Device cancelled connection");
    	}
    	
    	return T._("Unknown quit reason");
    }
    
    private synchronized void sendQuitMessage(int reason) throws IOException {
    	debug("Sending quit message, reason = " + getQuitReason(reason) + " (" + reason + ")");
    	
    	os.write(MSP_TYPE_QUIT_CONNECTION & 0xff);
    	os.write(reason & 0xff);
    }
    
    private synchronized void processPingMessage() throws IOException {
    	debug("Got ping message, sending pong.");
    	os.write(MSP_TYPE_PONG & 0xff);
    }
    
    private synchronized void processPongMessage() throws IOException {
    	debug("Got pong message.");
    	// nothing to do here
    }
    
    private synchronized void processQuitMessage() throws IOException {
    	int reason = is.readByte() & 0xff;
    	
    	debug("Got quit message, reason = " + getQuitReason(reason) + " (" + reason + ")");
    	
    	throw new IOException(getQuitReason(reason));
    }

    private byte readbuf[] = null;
    private byte writebuf[] = null;

    private void checkReadBufSize(int newSize) {
    	if(readbuf == null || readbuf.length < newSize) {
    	    readbuf = new byte[newSize];
    	}
    }

    private void checkWriteBufSize(int newSize) {
    	if(writebuf == null || writebuf.length < newSize) {
    	    writebuf = new byte[newSize];
    	}
    }
    
    private synchronized void processDataRequestMessage() throws IOException {
    	debug("Reading data request message.");
    	
    	int sectorSize = drvAccess.getSectorSize();
    	
    	is.read();		// pad
    	int count = is.readShort() & 0xffff;
    	long startsec = (long)(is.readInt() & 0xffffffff);	// make it unsigned
    	
    	debug("Requested " + count + " sectors beginning from " + startsec);
    	
    	int code = MSP_DATA_OKAY;
    	
    	// fetch the data from the device
    	try {
    	    checkReadBufSize(count * sectorSize);
    	} catch (Exception e) {
    	    code = MSP_DATA_ERROR;
    	    logger.println(T._("Could not incerase read buffer."));
    	    e.printStackTrace();
    	}
    	
    	if (code == MSP_DATA_OKAY) {
    	    try {
    	    	drvAccess.readCDSectors(startsec, count, readbuf);
    	    } catch (Exception e) {
    	    	code = MSP_DATA_NO_CD;
    	    	System.out.println("Could not read drive sectors.");
    	    	e.printStackTrace();
    	    }
    	}
    	
    	if (code != MSP_DATA_OKAY) {
    	    count = 0;
    	}
    	
    	// send the reply information
    	os.writeByte(MSP_TYPE_SEND_DATA & 0xff);
    	os.writeByte(code & 0xff);
    	os.writeShort(code == MSP_DATA_OKAY ? (count & 0xffff) : 0);
    	os.writeShort(code == MSP_DATA_OKAY ? (sectorSize & 0xffff) : 0);
    	os.writeShort(0);	// pad
    	os.writeInt(0);		// only server->client
    	
    	// send the data
    	if (code == MSP_DATA_OKAY) {
    	    os.write(readbuf, 0, count * sectorSize);
    	}
    }

    private synchronized void processSendDataMessage() throws IOException {
    	is.readByte();		// code, only client->server
    	int count = is.readShort() & 0xffff;
    	int sectorSize = is.readShort() & 0xffff;
    	is.readShort();		// pad
    	long startsec = (long)(is.readInt() & 0xffffffff);	// make it unsigned
    	
    	checkWriteBufSize(count * sectorSize);
    	is.readFully(writebuf, 0, count * sectorSize);
    	
    	int ack = 0;
    	
    	try {
    	    if (writeSupport) {
    	    	drvAccess.writeCDSectors(startsec, count, writebuf);
    	    	ack = 1;
    	    } else {
    	    	debug("Cannot write data because device is read only.");
    	    }
    	} catch (Exception e) {
    	    logger.println(T._("Could not write drive sectors."));
    	    e.printStackTrace();
    	}
    	
    	// write the ack
    	os.writeByte(MSP_TYPE_DATA_ACK & 0xff);
    	os.writeByte(ack & 0xff);
    }

}
