/*
 * ImageButton Class
 * realizes a n-state button which can display image
 * and features mouse highlighting
 */

package nn.pp.rc;

import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import java.lang.*;
import java.util.*;

public class ImageButton extends Canvas
    implements MouseListener, ImageObserver {

    private Dimension	dimSize, dimRealSize, imageSize;
    private Image  	image = null;
    private Image       grayImage = null;
    private String 	label = null;
    private int 	imageIndex;
    private byte	state, oldstate;
    private Color	colHlNorm, colHlHl, colShd, colHlShd, colFg, colBg;
    private Font 	fontLabel;
    private FontMetrics fontMetrics;
    private Image	offscreenImage = null;
    private boolean	active=true;
    private transient Vector alisteners = null, mlisteners = null;
    private boolean     alwaysInactive = false;
    final static byte StateNormal	= 0x00;
    final static byte StateHighlight	= 0x01;
    final static byte StatePressed	= 0x02;

    public ImageButton(String label)
    {
	this.label = label;
	state = StateNormal;
	setDefaultValues();
	addMouseListener(this);
	dimRealSize = new Dimension(0,0);
    }

    public ImageButton(Image image, Dimension imageSize, int imageIndex)
    {
	this.image	= image;
	this.imageSize	= imageSize;
	this.imageIndex = imageIndex;
	setDefaultValues();
	addMouseListener(this);
	dimRealSize = new Dimension(0,0);
    }

    public ImageButton(Image image, Dimension imageSize, int imageIndex, boolean alwaysInactive)
    {
	this.image	= image;
	this.imageSize	= imageSize;
	this.imageIndex = imageIndex;
	this.alwaysInactive = alwaysInactive;
	setDefaultValues();
	addMouseListener(this);
	dimRealSize = new Dimension(0,0);
    }
    
    public ImageButton(String label, Image image, Dimension imageSize, int imageIndex)
    {
	this.label	= label;
	this.image	= image;
	this.imageSize	= imageSize;
	this.imageIndex = imageIndex;
	setDefaultValues();
	addMouseListener(this);
	dimRealSize = new Dimension(0,0);
    }


    
    private int darkness = 0xffafafaf;
    
    private Image getGrayImage(Image image) {
	if (grayImage == null) {
	    ImageFilter filter = new GrayFilter(darkness);
	    ImageProducer producer = new FilteredImageSource(image.getSource(), filter);
	    grayImage = createImage(producer);
	}
	return grayImage;
    }

    class GrayFilter extends RGBImageFilter {
	private int darkness = 0xffafafaf;
	public GrayFilter() {
	    canFilterIndexColorModel = true;
	}
	public GrayFilter(int darkness) {
	    this();
	    this.darkness = darkness;
	}
	public int filterRGB(int x, int y, int rgb) {
	    return(rgb & darkness);
	}
    }
    
    public void addActionListener(ActionListener l) {
	if (alisteners == null) alisteners = new Vector();
	alisteners.addElement(l);
    }

    public void removeActionListener(ActionListener l) {
	if (alisteners == null) return;
	alisteners.removeElement(l);
    }

    public void setImageIndex(int imageIndex) {
	this.imageIndex = imageIndex;
	repaint();
    }
    
    public void setDefaultValues()
    {
	int itemWidth, itemHeight;

	colBg	    = SystemColor.control;
	colFg	    = SystemColor.controlText;
	colHlHl	    = SystemColor.controlLtHighlight;
	colHlNorm   = SystemColor.controlHighlight;
	colShd	    = SystemColor.controlShadow;
	colHlShd    = SystemColor.controlDkShadow;
	fontLabel   = new Font("SansSerif", Font.PLAIN, 12);
	fontMetrics = getFontMetrics(fontLabel);
	if (image != null && label != null) {
	    itemWidth  = fontMetrics.stringWidth(label)+fontMetrics.getAscent() + imageSize.width;
	    itemHeight = fontMetrics.getHeight()+fontMetrics.getLeading();
	    if (itemHeight < imageSize.height) itemHeight = imageSize.height;
	    dimSize 	= new Dimension(itemWidth, itemHeight);

	} else if (image != null) {
	    dimSize = new Dimension(imageSize.width,imageSize.height);
	} else if (label != null) {
	    dimSize 	= new Dimension(fontMetrics.stringWidth(label)+fontMetrics.getAscent(),
					fontMetrics.getHeight()+fontMetrics.getLeading());
	}
    }

    public void setEnabled(boolean state) {
	this.active = state;
	repaint();
    }
	
    public void addNotify()
    {
	super.addNotify();

    }

    public Dimension getPreferredSize()
    {
	return dimSize;
    }

    public Dimension getMinimumSize()
    {
	return dimSize;
    }

    public void update(Graphics g) {
	Graphics offscreenGraphics;

	if (dimRealSize.width != getSize().width ||
	    dimRealSize.height!= getSize().height) {
	    offscreenImage = createImage(getSize().width, getSize().height);
	    dimRealSize.setSize(getSize().width, getSize().height);
	}

	offscreenGraphics = offscreenImage.getGraphics();

	paint(offscreenGraphics);

	g.drawImage(offscreenImage, 0, 0, null);
    }

    public void paint(Graphics g)
    {
	int textX=0, textY=0;
	int itemWidth, itemHeight;
	int picX=0, picY=0;

	g.setFont(fontLabel);
	g.setColor(colBg);
	g.fillRect(0, 0 , getSize().width-1, getSize().height-1);

	// draw border of button, depending on current state
	switch (state) {
	  case StateNormal:
	      g.setColor(colHlNorm);
	      break;
	  case StateHighlight:
	      g.setColor(colHlHl);
	      break;
	  case StatePressed:
	      g.setColor(colHlShd);
	      break;
	}

	g.drawLine(0, 0, getSize().width-1, 0);
	g.drawLine(0, 0, 0, getSize().height-1);

	switch (state) {
	  case StateNormal:
	      g.setColor(colShd);
	      break;
	  case StateHighlight:
	      g.setColor(colHlShd);
	      break;
	  case StatePressed:
	      g.setColor(colHlHl);
	      break;
	}

	g.drawLine(getSize().width-1, 0, getSize().width-1, getSize().height-1);
	g.drawLine(0, getSize().height-1, getSize().width-1, getSize().height-1);

	g.setColor((active || alwaysInactive)?colFg:colShd);
	if (label !=null && image != null) {
	    // draw both image and label
	    itemWidth  = fontMetrics.stringWidth(label) + imageSize.width;
	    itemHeight = fontMetrics.getHeight() + fontMetrics.getAscent() + fontMetrics.getLeading();
	    if (fontMetrics.getHeight() < imageSize.height) itemHeight = imageSize.height;

	    picX = (getSize().width - itemWidth)/2;
	    picY = (getSize().height - itemHeight)/2 + 1;

	    textX = (getSize().width - itemWidth)/2 + imageSize.width;
	    textY = (getSize().height - fontMetrics.getHeight())/2
		+ fontMetrics.getAscent() + fontMetrics.getLeading() - 1;
	    
	    g.drawImage((active || alwaysInactive)?image:getGrayImage(image),
			picX, picY, picX+imageSize.width-1, picY+imageSize.height-1,
			imageIndex*imageSize.width, 0, (imageIndex+1)*imageSize.width-1,
			imageSize.height-1, this);
	    g.drawString(label, textX, textY);
	} else if (label != null) {
	    // draw only the label
	    textX = (getSize().width - fontMetrics.stringWidth(label))/2;
	    textY = (getSize().height - fontMetrics.getHeight())/2
		+ fontMetrics.getAscent() + fontMetrics.getLeading() - 1;

	    g.drawString(label, textX, textY);
	} else if (image != null) {
	    // draw only the image
	    picX = (getSize().width - imageSize.width)/2;
	    picY = (getSize().height - imageSize.height)/2;

	    g.drawImage((active || alwaysInactive)?image:getGrayImage(image),
			picX, picY, picX+imageSize.width-1, picY+imageSize.height-1,
			imageIndex*imageSize.width, 0, (imageIndex+1)*imageSize.width-1,
			imageSize.height-1, this);
	}
    }
    
    public boolean imageUpdate(Image img, int infoflags, int x, int y,
			       int width, int height) {
	if ((infoflags & ImageObserver.ALLBITS) != 0) {
	    repaint();
	    return false;
	} else {
	    return true;
	}
    }

    /* Listener Methods */

    public void mouseReleased(MouseEvent evt)
    {
	if (!active) return;
	
	state = StateHighlight;
	repaint();

	/* notify action listeners */
	if (alisteners != null && !alisteners.isEmpty()) {
	    ActionEvent event = new ActionEvent(this, 0, null);

	    synchronized(alisteners) {
		Enumeration e = alisteners.elements();
		while (e.hasMoreElements()) {
		    ActionListener l = (ActionListener) e.nextElement();
		    l.actionPerformed(event);
		}
	    }
	}
    }

    public void mousePressed(MouseEvent evt)
    {
	if (!active) return;
	
	state = StatePressed;
	repaint();
    }

    public void mouseEntered(MouseEvent evt)
    {
	if (!active) return;
	
	state = StateHighlight;
	repaint();
    }

    public void mouseExited(MouseEvent evt)
    {
	if (!active) return;
	
	state = StateNormal;
	repaint();
    }
    
    public void mouseMoved(MouseEvent evt)   {}    
    public void mouseDragged(MouseEvent evt) {}    

    public void mouseClicked(MouseEvent evt) {
	if (mlisteners != null && !mlisteners.isEmpty()) {
	    synchronized(mlisteners) {
		Enumeration e = mlisteners.elements();
		while (e.hasMoreElements()) {
		    MouseListener l = (MouseListener) e.nextElement(); 
		    l.mouseClicked(evt);
		}
	    }
	}	
    }
}
