package nn.pp.rc;

import java.awt.*;
import java.awt.event.*;
import java.io.*;

/**
 * The Logger class is a graphical Component
 * used to log text output of an application.
 * The logger shows always the last line in a
 * TextField and allows to display the History in 
 * an scrollable Textarea using the History Button
 * An application writes into the logger using a
 * classical PrintWriter, the same stream that is 
 * used by the java.lang.System.out field.
 * The stream can also be set as the default output
 * Stream for an java virtual maching using the 
 * System.setOut method
 *
 * @author Thomas Breitfeld, Peppercon
 * @version $Id: Logger.java,v 1.5 2003-07-08 09:31:21 miba Exp $
 */
public class Logger extends Panel {

    private boolean stdout;
    PrintStream mystream;
    Label stateline;
    HistoryFrame details;
    //Button detailButton;
    private boolean tiny = false;
    FocusHandler fh;
    
    /**
     * initializes the Logger with the given Text
     */
    public Logger(String text, FocusHandler focusHandler, boolean stdoutOutput) {
	super(new BorderLayout());
	fh = focusHandler;
	stdout = stdoutOutput;
	mystream = new PrintStream(new LoggerStream(), false);
	stateline = new Label(text);
	stateline.setFont(new Font("Helvetica", Font.PLAIN, 12));
	details = new HistoryFrame(text);
	add(stateline, BorderLayout.CENTER);
	stateline.addMouseListener(new MouseAdapter() {
		public void mouseClicked(MouseEvent e) {
		    if(e.getClickCount() > 1) {
			fh.handleFocus((Component)e.getSource());
			details.show();
			//details.setState(Frame.NORMAL);
			details.toFront();
		    }
		}
	    });			
    }

    public PrintStream getPrintStream() {
	return mystream;
    }

    public Dimension getPreferredSize() {
	return new Dimension(0,0);
    }
	
    /**
     * closes a open History Window and reclaims all
     * the associated resources to the window system
     */
    public void close() {
	details.close();
    }

    public void setTiny(boolean tiny) {
	if(this.tiny == tiny) return;
	this.tiny = tiny;
	if(tiny) {
	    remove(stateline);
	} else {
	    add(stateline, BorderLayout.CENTER);
	}
	validate();
    }

	
    /**
     * This inner class is an implementation 
     * of an output stream that converts the received
     * bytes back to char and Strings and writes them 
     * out to Loggers graphical output areas, namely
     * the TextField and the TextArea
     */
    class LoggerStream extends OutputStream {

	private boolean nltrunc = false;

	public void write(int b) throws IOException {
	    byte[] bb = new byte[1];
	    bb[0] = (byte)b;
	    write(bb);
	}

	/**
	 * adds the last line of the given text to the 
	 * text field and the complete text to the 
	 * TextArea
	 * In case the last character of the text is a 
	 * newline, we won't write this to the TextField
	 * but remember it, so the next time the textfield
	 * is going to be overwriten instead of appended
	 */
	public void write(byte b[]) throws IOException {
	    if(stdout) System.out.write(b);
	    String s = new String(b);
	    update(s);
	}

	public void write(byte b[], int off, int len) throws IOException {
	    if(stdout) System.out.write(b, off, len);
	    String s = new String(b, off, len);
	    update(s);
	}

	private void update(String s) {
	    details.append(s);
	    String ns = s.replace('\t', ' ');
	    if(nltrunc) stateline.setText(getLastLine(ns));
	    else stateline.setText(stateline.getText() + getLastLine(ns));
	}

	private String getLastLine(String s) {
	    int endi = s.length();
	    if(endi == 0) return s;
	    if(s.charAt(endi-1) == '\n') {
		nltrunc = true;
		if(--endi == 0) return "";
	    } else nltrunc = false;
	    int starti = s.lastIndexOf('\n', endi);
	    return s.substring(starti < 0 ? 0 : starti, endi);
	}

	public void flush() throws IOException {}
	public void close() throws IOException {}
    }
}

class HistoryFrame extends Frame {

    private TextArea myta;

    public HistoryFrame(String text) {
	super("Remote Console Log");

	addWindowListener(new WindowAdapter() {
		public void windowClosing(WindowEvent e) {
		    HistoryFrame.this.hide();
		}
	    });
	myta = new TextArea(text, 25, 80,
			    TextArea.SCROLLBARS_VERTICAL_ONLY);
	myta.setEditable(false);
	myta.setBackground(Color.white);
	setLayout(new BorderLayout());
	add(myta, BorderLayout.CENTER);
	pack();
    }

    public void append(String s) {
	myta.append(s);
	myta.setCaretPosition(myta.getText().length() + 8192);
    }

    public void close() {
	myta.setText("");
	dispose();
    }
}
