package nn.pp.rc;

import java.io.*;
import java.awt.*;
import java.util.*;
import java.awt.event.*;
import nn.pp.rckbd.*;

/**
 * The RCHandler is the super class for the RFB and RDP Handler classes
 *
 * RCHandler is a thread and should be started as such
 *
 * subclasses: RFBHandler, RDPHandler
 */
public abstract class RCHandler
    extends Thread
    implements KeyboardListener {

    public static final boolean debug = false;

    protected RCHandler child;
    protected RCHandler parent;
    
    protected ServerConsolePanelBase  scp;
    protected RCCanvasPanel           canvas;
    protected ChatFrame               chat;
    protected MouseHandler            mhandler1;
    protected MouseHandler            mhandler2;
    protected TrafficMonitor          tmoni;
    protected VideoSettingsFrame      vsframe;
   
    protected PrintStream             logger;
    
    protected RCProto                 proto;
    protected RFBProfile              profile;
    
    public boolean fps   = false;

    private Semaphore startsema;
    
    /* abstract function definitions */
    protected abstract int processProtocol() throws IOException;

    protected abstract void writeKeyboardEvent(byte keycode) throws IOException;

    protected abstract void writeKvmSwitchEvent(short newport) throws IOException;
    
    protected abstract void establishConnection() throws IOException;
    
    public void sendPlay() throws IOException { }

    public void sendPause() throws IOException { }

    public void sendStop() throws IOException { }

    public void sendSpeed(int speed) throws IOException { }
    
    public void sendSeek(Date pos) throws IOException { }
    
    public void setBlank(boolean blank) { }

    public abstract void distributeProto();
    
    protected abstract RCRenderer getRenderer();
    
    /* constructors */
    public RCHandler() {
    }
    
    public RCHandler (RCHandler parent, RFBProfile profile) {
	setClasses(parent.scp, parent.canvas, parent.chat, parent.tmoni,
		   parent.vsframe, parent.mhandler1, parent.mhandler2,
		   parent.logger, profile);
	this.parent = parent;
	parent.child = this;
    }
    
    public void setChat(ChatFrame chat) {
        this.chat = chat;
    }
    
    public RCHandler (ServerConsolePanelBase scp, RCCanvasPanel canvas,
		       ChatFrame chat, TrafficMonitor tmoni,
		       VideoSettingsFrame vsframe,
		       MouseHandler mhandler1, MouseHandler mhandler2,
		       PrintStream logger, RFBProfile profile) {
    	setClasses(scp, canvas, chat, tmoni, vsframe, mhandler1, mhandler2, logger, profile);
    }
    
    protected void setClasses(ServerConsolePanelBase scp, RCCanvasPanel canvas,
			      ChatFrame chat, TrafficMonitor tmoni,
			      VideoSettingsFrame vsframe,
			      MouseHandler mhandler1, MouseHandler mhandler2,
			      PrintStream logger, RFBProfile profile) {
	this.scp = scp;
	this.canvas = canvas;
	this.chat = chat;
	this.tmoni = tmoni;
	this.vsframe = vsframe;
	this.mhandler1 = mhandler1;
	this.mhandler2 = mhandler2;
	this.logger = logger;
	this.profile = profile;
	startsema = new Semaphore(1);
    }
    
    public void setMouseHandlers(MouseHandler mhandler1, MouseHandler mhandler2) {
        this.mhandler1 = mhandler1;
        this.mhandler2 = mhandler2;
    }
    
    /* common functions */
    
    /**
     * enable/disable fps
     */
    public void switchFPS() {
    	fps = !fps;
    }

    /**
     * the getMouseHandler function returns the primary mouse handler
     * of the RC handler
     */
    public MouseHandler getMouseHandler() {
    	return mhandler1;
    }
    
    public void setScreenSize(int w, int h) {
    }
    
    /**
     * the connect method will start a thread that tries to
     * connect to the host specified in RFBProfile using
     * the given Parameters.
     * If the connection succeeds the thread will connect
     * the ServerConsoleCanvas to the RFBprotocoll and
     * and drives it.
     * If the connection failes the thread will terminate
     * after writing an error messages
     */
    public void connect() {
	startsema.get();
	if(proto.connected()) disconnectFunc();
	start();
    }

    /**
     * The disconnect method will close an open connection
     * in case there is one. This will cause the active
     * thread of the Frame to terminate.
     * The Frame can be set active again by calling
     * startwork with the same or another profile
     */
    public void disconnect() {
	startsema.get();
	// before this proto will disconnect, our child has to disconnect
	if (child != null) {
	    child.disconnect();
	}
	disconnectFunc();
	startsema.post();
    }

    /**
     * disconnect functionality without semaphore
     * we need that in case connect would like to do
     * some disconnect as well
     */
    protected void disconnectFunc() {
	if(proto != null && proto.connected()) {
	    proto.close();
	    proto = null;
	}
	try {
	    join(500);
	    // netscape 4.7 seems to have a bug with respect to
	    // sockets. they don't throw an execption if another
	    // thread is closing them. So in case our thread is still
	    // alive after we return from the join, we bruttally
	    // destroy it. Not nice but OK
	    if(isAlive()) {
		System.out.println("Frame: cleanup: thread " + this +
				   "still alive, killing forcefully!");
		stop();
		if (parent != null) {
		    parent.distributeProto();
		}
	    }
	} catch(Exception e) {
	    e.printStackTrace();
	}
    }

    /**
     * That is the active method of our frame trying
     * to connect to a server and running the protocoll
     */
    public void run() {
	// connecting the proto
	try {
	    // we sleep a very little bit to let things cool down
	    Thread.sleep(500);
	    establishConnection();
	    scp.connected(this);
	} catch(Exception e) {
	    logException(T._("unable to connect!"), e);
	    e.printStackTrace();
	    // hack alert, but if the Exception does only contain the
	    // information as string...
	    if (e.toString().indexOf("exclusive access active") != -1) {
		scp.setExclusiveMode(true);
	    }
	    scp.disconnected(this);
	    disposeConnect();
	    return;
	} finally {
	    startsema.post();
	}

	try {
	    // distribute the proto and other stuff
	    distributeProto();

	    // and run it
	    int q = processProtocol();
	    
	    showQuitMsg(q);
	} catch (Exception e) {
	    e.printStackTrace();
	    logException(T._("terminated!"), e);
	} finally {
	    // cleanup resources
	    proto.close();
	    scp.disconnected(this);
	    dispose();
	}
    }
    
    protected abstract void showQuitMsg(int q);
    
    protected abstract void dispose();

    protected abstract void disposeConnect();

    protected String mode() {
    	return "RFB";
    }

    private void logException(String text, Exception e) {
	String msg = e.getMessage();
	if(debug) e.printStackTrace();
	if (msg != null) {
	    logger.println(mode() + ": " + T._("error") + ": " + e);
	}
	logger.println(mode() + ": " + text + " "
		       + (msg != null ? msg : e.toString()) + "!");
    }

    /**
     * handle events from the KeyEventhandler
     */
    public void fired(KeyboardEvent e) {
	try {
	    writeKeyboardEvent(e.keycode);
	} catch (IOException ignore) {
	    logger.println(T._("Can't write keycode:") + " " + ignore);
	}
    }

}
