package nn.pp.rc;

import java.util.*;

public class RFBEncoding
{
    protected static final int modeAuto		= 0;
    protected static final int modeFix		= 1;
    protected static final int modeAdvanced	= 2;
    protected static final int modeDefault	= modeAuto;
    
    protected static final int fixVideoHi	= 0;
    protected static final int fixVideo		= 1;
    protected static final int fixLanHi		= 2;
    protected static final int fixLan		= 3;
    protected static final int fixDsl		= 4;
    protected static final int fixUmts		= 5;
    protected static final int fixIsdn		= 6;
    protected static final int fixModem		= 7;
    protected static final int fixGprs		= 8;
    protected static final int fixGsm		= 9;
    protected static final int fixDefault	= fixLanHi;

    protected static final int comprVideo	= 0;
    protected static final int comprNone	= 1;
    protected static final int comprLevel1	= 2;
    protected static final int comprLevel2	= 3;
    protected static final int comprLevel3	= 4;
    protected static final int comprLevel4	= 5;
    protected static final int comprLevel5	= 6;
    protected static final int comprLevel6	= 7;
    protected static final int comprLevel7	= 8;
    protected static final int comprLevel8	= 9;
    protected static final int comprLevel9	= 10;
    protected static final int comprDefault	= comprNone;
    
    protected static final int color16Bit	= 0;
    protected static final int color8Bit	= 1;
    protected static final int color4Bit	= 2;
    protected static final int color4Grey	= 3;
    protected static final int color3Grey	= 4;
    protected static final int color2Grey	= 5;
    protected static final int color1Grey	= 6;
    protected static final int colorDefault	= color16Bit;

    private boolean supportHw;
    private boolean lossy;

    protected Vector modes, fixEncs, comps, colors;

    protected encMode currentMode;
    protected encFix  currentFix;
    protected encCompression currentComp;
    protected encColorDepth currentColor;

    protected class encodingParam {
	public int id;
	public String shortname;
	public String name;
	public boolean supportHw;

	public encodingParam(int id, String name, String shortname, boolean supportHw) {
	    this.id = id; this.name = name; this.shortname = shortname; this.supportHw = supportHw;
	}

	public String toString() {
	    return new String("name="+name+"(short="+shortname+",id="+id+")");
	}	
    }
    
    protected class encMode extends encodingParam {
	public encMode(int id, String name, String shortname, boolean supportHw) {
	    super(id, name, shortname, supportHw);
	}
    }

    protected class encFix extends encodingParam {
	public int compression;
	public int color;
	public boolean useCache;
	public boolean lossy;

	public encFix(int id, String name, String shortname, boolean supportHw,
	              int compression, int color, boolean lossy) {
	    
	    super(id, name, shortname, supportHw);
	    
	    this.compression = compression;
	    this.color = color;
  
	    this.lossy = lossy;
	}
    }
       
    protected class encCompression extends encodingParam {
	public int   rfbEncoding;
	public int   rfbCompressLevel;
	public int[] supportedColors;

	public boolean useCache;
	public boolean useLRLE;
	public boolean needTrueColor;
	
	public encCompression(int id, String name, String shortname, boolean supportHw,
	                      int encoding, int level, int[] colors, boolean cache, boolean lrle) {

	    super(id, name, shortname, supportHw);

	    this.rfbEncoding = encoding;
	    this.rfbCompressLevel = level;
	    
	    this.supportedColors = colors;

	    this.useCache = cache;
	    this.useLRLE = lrle;
	}
    }

    protected class encColorDepth extends encodingParam {
	public boolean needTrueColor;
	
	// options for non-lrle encodings
	int subformat;
	
	// options for lrle encoding
	int rfbLRLESubencLossy;
	int rfbLRLESubencLossless;

	public encColorDepth(int id, String name, String shortname, boolean supportHw, int subformat,
			     int rfbLRLESubencLossy, int rfbLRLESubencLossless, boolean truecolor) {
	    
	    super(id, name, shortname, supportHw);

	    this.subformat = subformat;
	    this.rfbLRLESubencLossy = rfbLRLESubencLossy;
	    this.rfbLRLESubencLossless = rfbLRLESubencLossless;

	    this.needTrueColor = truecolor;
	}
    }
       
    protected encMode[] init_modes = {
	new encMode(modeAuto,     T._("Automatic"),  "auto", true),
	new encMode(modeFix,      T._("Predefined"), "preconf", true),
	new encMode(modeAdvanced, T._("Advanced"),   "manual", true)
    };
   
    protected encFix[] init_fixEncs = {
	new encFix(fixVideoHi, T._("Video Optimized Hicolor"),  "videohi", false, comprVideo,  color16Bit,   false),
	new encFix(fixVideo,   T._("Video Optimized"),	        "video",   false, comprVideo,  color8Bit,    false),
	new encFix(fixLanHi,   T._("LAN Hicolor"),		"lanhi",   true,  comprNone,   color16Bit,   false),
	new encFix(fixLan,     T._("LAN"),			"lan",	   true,  comprNone,   color8Bit,    false),
	new encFix(fixDsl,     T._("DSL"),			"dsl",     true,  comprLevel2, color8Bit,    false),
	new encFix(fixUmts,    T._("UMTS"),			"umts",    true,  comprLevel4, color8Bit,    true),
        new encFix(fixIsdn,    T._("ISDN"),			"isdn",    true,  comprLevel6, color4Bit,    true),
	new encFix(fixModem,   T._("Modem"),			"modem",   true,  comprLevel7, color2Grey,   true),
	new encFix(fixGprs,    T._("GPRS"),			"gprs",    true,  comprLevel8, color2Grey,   true),
	new encFix(fixGsm,     T._("GSM"),			"gsm",     true,  comprLevel9, color1Grey,   true)	
    };
  
    protected int[] colorDepths16_8 = { color16Bit, color8Bit };
    protected int[] colorDepths8421 = { color8Bit,
					color4Bit,
					color4Grey,
					color2Grey,
					color1Grey };
    
    protected encCompression[] init_comps = {
	new encCompression(comprVideo,  T._("Video Optimized"),	"video",        false, RFBproto.EncodingRawVSC,  0, colorDepths16_8, false, true),
	new encCompression(comprNone,   "0"+T._(" - none"),	"uncompressed", true,  RFBproto.EncodingHextile, 0, colorDepths16_8, false, true),
	new encCompression(comprLevel1, "1"+T._(" - fastest"),	"comp1",        true,  RFBproto.EncodingTight,   1, colorDepths8421, false, true),
	new encCompression(comprLevel2, "2",			"comp2",        true,  RFBproto.EncodingTight,   2, colorDepths8421, false, true),
	new encCompression(comprLevel3, "3",			"comp3",        true,  RFBproto.EncodingTight,   3, colorDepths8421, false, true),
	new encCompression(comprLevel4, "4",			"comp4",	true,  RFBproto.EncodingTight,   4, colorDepths8421, false, true),
	new encCompression(comprLevel5, "5",			"comp5",	true,  RFBproto.EncodingTight,   5, colorDepths8421, false, true),
	new encCompression(comprLevel6, "6",			"comp6",	true,  RFBproto.EncodingTight,   6, colorDepths8421, true,  true),
	new encCompression(comprLevel7, "7",			"comp7",	true,  RFBproto.EncodingTight,   7, colorDepths8421, true,  true),
	new encCompression(comprLevel8, "8",			"comp8",	true,  RFBproto.EncodingTight,   8, colorDepths8421, true,  true),
	new encCompression(comprLevel9, "9"+T._(" - best"),	"comp9",	true,  RFBproto.EncodingTight,   9, colorDepths8421, true,  true)
    };
	
    protected encColorDepth[] init_colors = {
	new encColorDepth(color16Bit,	T._("16 Bit Hicolor"),		"color_16bpp",     true, 0,
			  RFBproto.LRLESubenc15bitDirectLossy,	RFBproto.LRLESubenc15bitDirectLossless,  true),
	new encColorDepth(color8Bit,	T._("8 Bit - 256 Colors"),	"color_8bpp",      true, 8,
			  RFBproto.LRLESubenc7bitDirectLossy,	RFBproto.LRLESubenc7bitDirectLossless,  false),
	new encColorDepth(color4Bit,	T._("4 Bit - 16 Colors"),	"color_4bpp",      true, 4,
			  RFBproto.LRLESubenc4bitPaletteLossy,	RFBproto.LRLESubenc4bitPaletteLossless, false),
	new encColorDepth(color4Grey,	T._("4 Bit - Grayscale"),	"grey_4bpp",       true, 3,
			  RFBproto.LRLESubenc4bitGreyLossy,	RFBproto.LRLESubenc4bitGreyLossless,    false),
	new encColorDepth(color2Grey,	T._("2 Bit - Grayscale"),	"grey_2bpp",       true, 2,
			  RFBproto.LRLESubenc2bitGreyLossy,	RFBproto.LRLESubenc2bitGreyLossless,    false),
	new encColorDepth(color1Grey,	T._("1 Bit - Black/White"),	"grey_1bpp",       true, 1,
			  RFBproto.LRLESubenc1bitGreyLossy,	RFBproto.LRLESubenc1bitGreyLossless,    false),
    };
      
    public RFBEncoding()
    {
	this(true);
    }

    public RFBEncoding(boolean supportHw)
    {
	this.supportHw = supportHw;
	this.lossy = false;

	initCompressionTypes();
	
	currentMode = lookupModeFromID(modeDefault);
	currentFix = lookupFixFromID(fixDefault);
	currentColor = lookupColorFromID(colorDefault);
	currentComp = lookupCompFromID(comprDefault);
	
	if (currentMode.id == modeFix) {
	    setFixEncoding(currentFix.name);
	}
    }

    public RFBEncoding(boolean supportHw, String selEnc, String fixEnc,
		       String encAdvCR, String encAdvCD)
    {
	this(supportHw);
		
	/* convert external IDs to internal ones
	   TODO: add sanity check and fallback against possible-encodings lists? */

	///*
	System.out.println("RFBEncoding: selEnc=" + selEnc);
	System.out.println("RFBEncoding: fixEnc=" + fixEnc);
	System.out.println("RFBEncoding: encAdvCR=" + encAdvCR);
	System.out.println("RFBEncoding: encAdvCD=" + encAdvCD);
	//*/

	currentMode = lookupModeFromShortname(selEnc);
	if (currentMode == null) currentMode = lookupModeFromID(modeDefault);
	
	currentFix = lookupFixFromShortname(fixEnc);
	if (currentFix == null) currentFix = lookupFixFromID(fixDefault);

	currentColor = lookupColorFromShortname(encAdvCD);
	if (currentColor == null) currentColor = lookupColorFromID(colorDefault);

	currentComp = lookupCompFromShortname(encAdvCR);
	if (currentComp == null) currentComp = lookupCompFromID(comprDefault);

	/* map fix values to compression/color */
	if (currentMode.id == modeFix) {
	    setFixEncoding(currentFix.name);
	}

	System.out.println(this);
    }

    protected void initCompressionTypes()
    {
	int i;

	modes	= new Vector();
	fixEncs = new Vector();
	comps   = new Vector();
	colors  = new Vector();
	
	for (i = 0; i < init_modes.length; i++) {
	    if (!supportHw || init_modes[i].supportHw) {
	        modes.addElement(init_modes[i]);
	    }
	}
	for (i = 0; i < init_fixEncs.length; i++) {
	    if (!supportHw || init_fixEncs[i].supportHw) {
	        fixEncs.addElement(init_fixEncs[i]);
	    }
	}
	for (i = 0; i < init_comps.length; i++) {
	    if (!supportHw || init_comps[i].supportHw) {
	        comps.addElement(init_comps[i]);
	    }	    
	}
	for (i = 0; i < init_colors.length; i++) {
	    if (!supportHw || init_colors[i].supportHw) {
	        colors.addElement(init_colors[i]);
	    }
	}
    }

    protected encodingParam lookupParamFromID(int id, Vector v)
    {
	for (Enumeration el = v.elements(); el.hasMoreElements(); ) {
	    encodingParam entry = (encodingParam)el.nextElement();
	    if (entry.id == id) return entry;
	}
	return null;
    }

    protected encodingParam lookupParamFromName(String name, Vector v)
    {
	for (Enumeration el = v.elements(); el.hasMoreElements(); ) {
	    encodingParam entry = (encodingParam)el.nextElement();
	    if (entry.name.equals(name)) return entry;
	}
	return null;
    }

    protected encodingParam lookupParamFromShortname(String shortname, Vector v)
    {
	for (Enumeration el = v.elements(); el.hasMoreElements(); ) {
	    encodingParam entry = (encodingParam)el.nextElement();
	    if (entry.shortname.equals(shortname)) return entry;
	}
	return null;
    }
    
    protected encMode lookupModeFromID(int id) {
	return (encMode) lookupParamFromID(id, modes);
    }
    protected encFix lookupFixFromID(int id)  {
	return (encFix) lookupParamFromID(id, fixEncs);
    }   
    protected encCompression lookupCompFromID(int id) {
	return (encCompression) lookupParamFromID(id, comps);
    }
    protected encColorDepth lookupColorFromID(int id) {
	return (encColorDepth) lookupParamFromID(id, colors);
    }

    protected encMode lookupModeFromName(String name) {
	return (encMode) lookupParamFromName(name, modes);
    }
    protected encFix lookupFixFromName(String name)  {
	return (encFix) lookupParamFromName(name, fixEncs);
    }   
    protected encCompression lookupCompFromName(String name) {
	return (encCompression) lookupParamFromName(name, comps);
    }
    protected encColorDepth lookupColorFromName(String name) {
	return (encColorDepth) lookupParamFromName(name, colors);
    }

    protected encMode lookupModeFromShortname(String name) {
	return (encMode) lookupParamFromShortname(name, modes);
    }
    protected encFix lookupFixFromShortname(String name)  {
	return (encFix) lookupParamFromShortname(name, fixEncs);
    }   
    protected encCompression lookupCompFromShortname(String name) {
	return (encCompression) lookupParamFromShortname(name, comps);
    }
    protected encColorDepth lookupColorFromShortname(String name) {
	return (encColorDepth) lookupParamFromShortname(name, colors);
    }    

    public String toString()
    {
	return	"RFBEncoding mode=" + currentMode +
		", fix=" + currentFix +
		", compression=" + currentComp +
		", color=" + currentColor +
	        ", lossy=" + lossy;
    }

    public int[] getRFBEncodings(boolean honorAuto)
    {
	int[] encodings = new int[4];
	int index = 0;

	if (honorAuto == true && currentMode.id == modeAuto) {
	    encCompression autoComp = lookupCompFromID(comprLevel5);
	    encColorDepth autoColor = lookupColorFromID(color8Bit);
	    int encoding;
	    
	    if (supportHw) {		
		encoding = RFBproto.EncodingLRLEHard;
		encoding |= autoColor.rfbLRLESubencLossless << RFBproto.EncodingParamSubencShift;
		encoding |= autoComp.rfbCompressLevel << RFBproto.EncodingParamZLIBShift;

		encodings[index++] = RFBproto.EncodingAutomaticHW;
	    } else {
		encoding  = autoComp.rfbEncoding;
		encoding |= autoColor.subformat << RFBproto.EncodingParamSubencShift;
		encoding |= autoComp.rfbCompressLevel << RFBproto.EncodingParamZLIBShift;

		encodings[index++] = RFBproto.EncodingAutomaticSW;
	    }
	    
	    encodings[index++] = encoding;
	} else {
	    if (supportHw && currentComp.useLRLE) {
		int encoding = RFBproto.EncodingLRLEHard;
		if (lossy) {
		    encoding |= currentColor.rfbLRLESubencLossy << RFBproto.EncodingParamSubencShift;
		} else {
		    encoding |= currentColor.rfbLRLESubencLossless << RFBproto.EncodingParamSubencShift;
		}
		encoding |= currentComp.rfbCompressLevel << RFBproto.EncodingParamZLIBShift;

		if (currentComp.id == comprVideo) {
		    encoding |= RFBproto.EncodingParamVideoOptMask;
		}

		encodings[index++] = encoding;
	    } else {
		int encoding  = currentComp.rfbEncoding;
		encoding |= currentColor.subformat << RFBproto.EncodingParamSubencShift;
		encoding |= currentComp.rfbCompressLevel << RFBproto.EncodingParamZLIBShift;

		if (currentComp.id == comprVideo) {
		    encoding |= RFBproto.EncodingParamVideoOptMask;
		}
		encodings[index++] = encoding;
	
		if (currentComp.useCache) {
		    encodings[index++] = RFBproto.EncodingUseTightCache;
		}
	    }
	}

	return encodings;
    }

    public boolean isTightCacheUsed()
    {
	return currentComp.useCache;
    }
    
    public boolean isModeAuto()
    {
	return (currentMode.id == modeAuto) ? true : false;
    }

    public boolean needsTrueColor()
    {
	if (supportHw) {
	    return currentComp.useLRLE || (currentMode.id == modeAuto);
	} else {
	    return (currentMode.id != modeAuto) && currentColor.needTrueColor;
	}
    }
    
    public String[] getFixNames()
    {
	String[] names = new String[fixEncs.size()];
	int i = 0;
	
	for (Enumeration el = fixEncs.elements(); el.hasMoreElements(); ) {
	    encFix entry = (encFix)el.nextElement();
	    names[i++] = entry.name;
	}
	return names;
    }
    public String[] getCompressionNames()
    {
	String[] names = new String[comps.size()];
	int i = 0;
	
	for (Enumeration el = comps.elements(); el.hasMoreElements(); ) {
	    encCompression  entry = (encCompression)el.nextElement();
	    names[i++] = entry.name;
	}
	return names;
    }
    public String[] getColorNames()
    {
	String[] names = new String[colors.size()];
	int i = 0;
	
	for (Enumeration el = colors.elements(); el.hasMoreElements(); ) {
	    encColorDepth entry = (encColorDepth)el.nextElement();
	    names[i++] = entry.name;
	}
	return names;
    }

    public String getCurrentFixName()
    {
	return currentFix.name;
    }
    public String getCurrentCompName()
    {
	return currentComp.name;
    }
    public String getCurrentColorName()
    {
	return currentColor.name;
    }
    public boolean isColorPossible(String name)
    {
	// hw encoding always supports all colors
	if (supportHw) return true;
	
	encColorDepth color = lookupColorFromName(name);
	if (color == null) return false;

	for (int i = 0; i < currentComp.supportedColors.length; i++) {
	    if (currentComp.supportedColors[i] == color.id) return true;
	}

	return false;
    }
    public boolean isHwSupported()
    {
	return supportHw;
    }
    public boolean isLossy()
    {
	return lossy;
    }

    public boolean setFixEncoding(String name)
    {
	encFix fix = lookupFixFromName(name);

	if (fix == null) return false;

	encCompression comp = lookupCompFromID(fix.compression);
	encColorDepth color = lookupColorFromID(fix.color);
	
	if (comp == null || color == null ||
	    (currentComp == comp && currentColor == color && fix.lossy != lossy)) return false;

	currentComp = comp;
	currentColor= color;
	lossy = fix.lossy;
	currentMode = lookupModeFromID(modeFix);
	
	return true;
    }
    public boolean setCompression(String name)
    {
	encCompression comp = lookupCompFromName(name);

	if (comp == null || currentComp == comp) return false;

	currentComp = comp;
	
	if (supportHw == false) {
	    // check if color still possible, otherwise change it
	    boolean found = false;
	    for (int i = 0; i < currentComp.supportedColors.length; i++) {
		if (currentComp.supportedColors[i] == currentColor.id) found = true;
	    }
	    if (found == false) {
		// we assume the first color entry for a compression is the default
		currentColor = lookupColorFromID(currentComp.supportedColors[0]);
	    }    
	}
	currentMode = lookupModeFromID(modeAdvanced);

	return true;
    }
    public boolean setColorDepth(String name)
    {
	encColorDepth color = lookupColorFromName(name);

	if (color == null || currentColor == color) return false;

	currentColor = color;
	currentMode = lookupModeFromID(modeAdvanced);

	return true;
    }
    public boolean setLossyState(boolean lossy)
    {
	if (this.lossy == lossy) return false;
	
	this.lossy = lossy;
	currentMode = lookupModeFromID(modeAdvanced);

	return true;
    }
}
