package nn.pp.rc;

import java.io.*;
import java.util.*;
import java.awt.*;
import java.awt.event.*;
import nn.pp.rckbd.*;

/**
 * The RFBHandler is the actual protocoll handler running
 * the RFB protocoll.
 *
 * It makes use of two Co-classes:
 * RFBproto:    is the low level, message oriented network stuff
 * RFBRenderer: is a polymorphic class depending on the actual
 *              client capabilities and higher level protocoll
 *              parameters responsible to bring the bits on the screen
 *
 * RFBHandler is a thread and should be started as such
 */
public class RFBHandler_01_22
    extends RFBHandler {

    Dimension mysize;

    RFBproto_01_22 rfbProto;
    final static int     REPAINT_TIMEOUT = 300; /* ms */

    public RFBHandler_01_22 (RCHandler parent, RFBProfile profile) {
	super(parent, profile);
	loadClasses();
    }
    
    public RFBHandler_01_22 (ServerConsolePanelBase scp, RCCanvasPanel canvas,
		       ChatFrame chat, TrafficMonitor tmoni,
		       VideoSettingsFrame vsframe,
		       MouseHandler mhandler1, MouseHandler mhandler2,
		       PrintStream logger, RFBProfile profile) {
	super(scp, canvas, chat, tmoni, vsframe, mhandler1, mhandler2, logger, profile);
	// turn on/off fps in the status bar
	tmoni.enableFPS(true);
	loadClasses();
    }
    
    /* an RFBHandler might be instantiated without parameters in the constructor;
       then, init() must be called afterwards */
    public RFBHandler_01_22() {
    }
    
    public void init(ServerConsolePanelBase scp, RCCanvasPanel canvas,
		     ChatFrame chat, TrafficMonitor tmoni,
		     VideoSettingsFrame vsframe,
		     MouseHandler mhandler1, MouseHandler mhandler2,
		     PrintStream logger, RFBProfile profile) {
    	setClasses(scp, canvas, chat, tmoni, vsframe, mhandler1, mhandler2, logger, profile);
	tmoni.enableFPS(true);
	loadClasses();
	if (sock != null) {
	    rfbProto.setSocket(sock);
	}
	if (secDesc != null) rfbProto.setSecDesc(secDesc);
	if (versionMajor != -1 && versionMinor != -1) {
	    rfbProto.setProtocolVersion(versionMajor, versionMinor);
	}
    }
    
    private void loadClasses() {
	proto = super.rfbProto = rfbProto = new RFBproto_01_22(profile, logger, this);
	loadRFBRenderer(params.encoding.needsTrueColor() ? 16 : 8, true);
	repaint_timer = new GenericTimer(REPAINT_TIMEOUT, this);
    }
    
    protected void showQuitMsg(int q) {
    	logger.println(mode() + ": " + T._("terminated") +
    	               ": " + RFBproto.getQuitReasonMsg(q));
    }
    
    private void changeSpTimes() {
	if (forensicPanel != null && rfbProto.start != null && rfbProto.end != null) {
    	    SpEvent sp = new SpEvent();
    	    sp.setSessionTimes(rfbProto.start, rfbProto.end);
    	    forensicPanel.handleSpEvent(sp);
    	    sp.played();
    	    forensicPanel.handleSpEvent(sp);
	}
    }
    
    private void handleSpSessionInfo() throws IOException {
        RecordedSessionInfo info = rfbProto.readSessionInfo();
        if (forensicPanel != null) {
            forensicPanel.setSessionInfo(info);
        }
        if (scp != null && scp.scframe != null) {
            scp.scframe.setTitle("Replaying " + info.hostName  + " (" + info.hostAddress + ")");
        }
    }
    
    protected void establishConnection() throws IOException {
    	rfbProto.establishConnection();
	scp.setSoftKbdLayout(rfbProto.kbdlayout);
	if ((rfbProto.connectionFlags & (RFBproto.ConnectionFlagSasSw | RFBproto.ConnectionFlagSp)) != 0) {
	    boolean localOnly = ((rfbProto.connectionFlags & RFBproto.ConnectionFlagSasLocalOnly) != 0);
	    boolean noKbd = ((rfbProto.connectionFlags >> RFBproto.ConnectionFlagSasKbdShift) & RFBproto.ConnectionFlagSasKbdMask)
	    	== RFBproto.ConnectionFlagSasKbdNoKbd;
	    forensicPanel = scp.addForensicButton(
	    	(rfbProto.connectionFlags & RFBproto.ConnectionFlagSasSw) != 0,
	    	localOnly, noKbd,
	    	(rfbProto.connectionFlags & RFBproto.ConnectionFlagSp) != 0);
	    if (forensicPanel != null) {
	    	forensicPanel.setRCHandler(this);
	    }
	    changeSpTimes();
	}
    }
    
    protected void writeKeyboardEvent(byte keycode) throws IOException {
	rfbProto.writeKeyboardEvent(keycode);
    }

    protected void writeKvmSwitchEvent(short newport) throws IOException {
	rfbProto.writeKvmSwitchEvent(newport);
    }
    
    private void processFbFormatUpdate(boolean force) {
	mysize = new Dimension(rfbProto.framebufferWidth,
			       rfbProto.framebufferHeight);
	canvas.adjustSize(mysize, force);
    }
   
    protected void parseServerCommand() {
    	if(rfbProto.serverCommand == null || rfbProto.serverCommandParams == null) {
    	    return;
    	}

        String command = rfbProto.serverCommand.toLowerCase();
        String params = rfbProto.serverCommandParams;

	if(command.equals("exclusive_mode")) {
	    if (params.equals("active")) {
		scp.setExclusiveMode(true);
	    } else {
		scp.setExclusiveMode(false);
	    }
	} else if(command.equals("rc_users")) {
	    System.out.println("rc_users: " + params);
	    scp.setShareMode(Integer.parseInt(params));
	}
	else if(command.equals("wlan_quality")) {
	    scp.setWLANQuality(Integer.parseInt(params));
	}
	else if(command.equals("charset")) {
	    T.setCharset(params);
	}
        // unknown command
        else {
            System.out.println("got unknown RFB command: " + command);
    	}
    }
       
    void sendFramebufferUpdateRequest() throws IOException {
	rfbProto.writeFramebufferUpdateRequest(0, 0, rfbProto.framebufferWidthPadded, 
					       rfbProto.framebufferHeightPadded, true);
    }
    
    void sendFullFramebufferUpdateRequest() throws IOException {
	rfbProto.writeFullFramebufferUpdateRequest();
    }
    
    protected void readScreenUpdate() throws IOException {
	StopWatch watch = new StopWatch();
	int count = 0;

	repaint_timer.start();
	rfbProto.readFramebufferUpdate();
	tmoni.increaseFPSCount();	
	for (int i = 0; i < rfbProto.updateNRects; i++) {
	    rfbProto.readFramebufferUpdateRectHdr();
            rfbProto.prepareUpdateRect();
	      
	    switch (rfbProto.updateRectEncoding & RFBproto.EncodingMask) {
		case RFBproto.EncodingRaw:
		    rdr.drawRawRect(rfbProto.updateRectX, rfbProto.updateRectY,
				    rfbProto.updateRectW, rfbProto.updateRectH,
				    rfbProto.bigEndian);
		    break;
		    
		case RFBproto.EncodingRawVSC:
		    rdr.drawRawVSCRect(rfbProto.updateRectX, rfbProto.updateRectY,
				       rfbProto.updateRectW, rfbProto.updateRectH,
				       rfbProto.bigEndian);
		    break;
		    
		case RFBproto.EncodingHextile:
		    rdr.drawHextileRect(rfbProto.updateRectX,
					rfbProto.updateRectY,
					rfbProto.updateRectW,
					rfbProto.updateRectH,
					rfbProto.bigEndian);
		    break;

		case RFBproto.EncodingTight:
		    rdr.drawTightRect(rfbProto.updateRectX, rfbProto.updateRectY,
				      rfbProto.updateRectW, rfbProto.updateRectH);
		    break;
		    
		case RFBproto.EncodingUseTightCache:
		    rdr.drawTightCachedRect(rfbProto.updateRectX, rfbProto.updateRectY,
				            rfbProto.updateRectW, rfbProto.updateRectH);
		    break;
		    
		case RFBproto.EncodingCopyRect:
		    rfbProto.readCopyRect();
		    rdr.drawCopyRect(rfbProto.copyRectSrcX, rfbProto.copyRectSrcY,
		                     rfbProto.updateRectX, rfbProto.updateRectY,
				     rfbProto.updateRectW, rfbProto.updateRectH);
		    break;
		case RFBproto.EncodingLRLESoft:
		case RFBproto.EncodingLRLEHard:
		    int size = rfbProto.readHWencSize();			    
		    rdr.drawLRLERect(rfbProto.updateRectEncoding,
				     rfbProto.updateRectX,
				     rfbProto.updateRectY,
				     rfbProto.updateRectW,
				     rfbProto.updateRectH);
		    rfbProto.readHWencPadding(size); 
		    break;
		
		case RFBproto.EncodingJpeg:
		    rdr.drawJpegRect(rfbProto.updateRectX, rfbProto.updateRectY,
				     rfbProto.updateRectW, rfbProto.updateRectH);
		    break;
		    
		default:
		    throw new IOException
			("Unknown RFB rectangle encoding "
			 + rfbProto.updateRectEncoding);
	    } // end switch
	    // slow network connections repaint,
	    // triggered by repaint timer
	    if(repaint_scheduled) {
		rdr.repaint();
		repaint_scheduled = false;
	    }		      
	}// end for updateNRects
	repaint_timer.stop();
	rdr.repaint();
	if(fps) {
	    if(++count == 10) {
	        long t = watch.stop();
	        if(t == 0) t = 1;	// prevent division by 0!
		logger.println(" fps=" + 1000.0
				 * (float)count / (float)t
				 + " t=" + t/count + "ms"
				 + " c=" + count);
	        watch.start();
	        count = 0;
	    }
	}
    }

    protected void handleAckPixelFormat() throws IOException {
	rfbProto.readAckPixelFormat();

    	loadRFBRenderer(rfbProto.ack_bitsPerPixel, true);

	rdr.enableTightCache(params.encoding.isTightCacheUsed());
	
	rfbProto.writeFullFramebufferUpdateRequest();
    }
    
    protected void handleScreenUpdate() throws IOException {
	if (pendingFormatChange) {
	    sendFormatAndEncodings(true);
	    pendingFormatChange = false;
	} else {
	    sendFramebufferUpdateRequest();
	}
	readScreenUpdate();
    }
    
    protected void readSasEvent() throws IOException {
    	SasEvent evt = rfbProto.readSasEvent();
    	
    	if (forensicPanel != null) {
    	    forensicPanel.handleSasEvent(evt);
    	}
    }
    
    protected void handleServerFbFormat() throws IOException {
        int framebufferWidthOld, framebufferHeightOld;
        framebufferWidthOld = rfbProto.framebufferWidth;
        framebufferHeightOld = rfbProto.framebufferHeight;
        rfbProto.readServerInit();
        processFbFormatUpdate(false);
        if (params.encoding.isTightCacheUsed() && 
            (framebufferWidthOld != rfbProto.framebufferWidth || framebufferHeightOld != rfbProto.framebufferHeight)) {
            
            rdr.setNewTightCacheSize();
        }
        if (forensicPanel != null) {
            forensicPanel.setFramebufferFormat(rfbProto.framebufferWidth, rfbProto.framebufferHeight, rfbProto.bitsPerPixel);
        }
    }
    
    private int no = 0;
    // FIXME: set this to false
    static boolean sasDebug = false;
    
    protected void readSpTimestamp() throws IOException {
    	Date d = rfbProto.readSpTimestamp();
    	if (forensicPanel != null) {
    	    if (sasDebug) {
    	    	System.out.println("Timestamp " + no++);
    	    }
    	    SpEvent sp = new SpEvent();
    	    sp.setReplayTime(d);
    	    forensicPanel.handleSpEvent(sp);
    	}
    }
    
    public void sendFormatAndEncodings(boolean framebufferUpdateRequest) throws IOException {
	int[] encodings = params.encoding.getRFBEncodings(true);	
	rfbProto.writeSetEncodings(encodings, encodings.length);
	
	loadRFBRenderer(params.encoding.needsTrueColor() ? 16 : 8, false);

	rdr.sendPixelMsg();
	
	if (framebufferUpdateRequest) {
	    rfbProto.writeFullFramebufferUpdateRequest();
	}
    }
    
    void readSpCommand() throws IOException {
    	rfbProto.readSpCommand();
    	
    	switch (rfbProto.spCommand) {
    	    case RFBproto.SpCommandPlay:
    	    case RFBproto.SpCommandPause:
    	    case RFBproto.SpCommandStop:
    	    	if (forensicPanel != null) {
    	    	    forensicPanel.handleSpEvent(new SpEvent(rfbProto.spCommand));
    	    	}
    	    	break;
    	}
    }

    protected int processProtocol() throws IOException {
	// do the post init steps
	sendFormatAndEncodings(false);
	logger.println(T._("Console: rendering") + " " + rdr);
	processFbFormatUpdate(true);
	rdr.sendUpdateMsg();
	
	rdr.setOSDState(rfbProto.osdStateBlanking, rfbProto.osdStateMessage,
			rfbProto.osdStateTimeout);

	// main dispatch loop
	while (true) {

	    // wait for frame buffer updates
	    int msgType = rfbProto.readServerMessageType();
	    
	    // process the protocoll
	    switch (msgType) {
	      case RFBproto.Quit:
	          return rfbProto.readQuitReason();
	      case RFBproto.UTFString:
		  if (chat != null) chat.newMsg(rfbProto.readString());
		  break;
	      case RFBproto.ServerFbFormat:
		  handleServerFbFormat();
		  break;
	      case RFBproto.FramebufferUpdate:
		  handleScreenUpdate();
		  break; /* RFBproto.FramebufferUpdate */
		  
	      case RFBproto.AckPixelFormat:
		  handleAckPixelFormat();
		  break;
		  
	      case RFBproto.Bell:
		  Toolkit.getDefaultToolkit().beep();
		  break;

	      case RFBproto.ConsoleMessage:
		  rfbProto.readConsoleMessage();
		  logger.println(rfbProto.message);
		  break;

	      case RFBproto.ServerCommand:
		  rfbProto.readServerCommand();
		  parseServerCommand();
		  break;

	      case RFBproto.OsdState:
		  rfbProto.readOsdStateMessage();
		  rdr.setOSDState(rfbProto.osdStateBlanking, rfbProto.osdStateMessage,
				  rfbProto.osdStateTimeout);
		  break;

	      case RFBproto.KbdLayout:
		  rfbProto.readKbdLayout();
		  scp.setSoftKbdLayout(rfbProto.kbdlayout);
		  break;

	      case RFBproto.PingRequest:
		  rfbProto.readPing();
		  try {
		      rfbProto.writePingReply(0);
		  } catch (Exception ignore) {
		      logger.println(T._("Can't write pingresponse:") + " " + ignore);
		  }
		  break;

	      case RFBproto.PingReply:
		  rfbProto.readPing();
		  break;
	      
	      case RFBproto.BandwidthRequest:
		  rfbProto.writeBandwidthTick(rfbProto.BandwidthTickStage1);
		  rfbProto.readBandwidthRequest();
		  rfbProto.writeBandwidthTick(rfbProto.BandwidthTickStage2);
		  System.out.println("Bandwidth measurement done.");
		  break;
		  
	      case RFBproto.VideoSettingsEventS2C:
		  rfbProto.readVideoSettingsUpdate();
		  vsframe.updateVideoSettings(rfbProto.vs);
		  break;
		  
	      case RFBproto.VideoQualityEventS2C:
		  rfbProto.readVideoQualityUpdate();
		  break;
		  
	      case RFBproto.RCModeReply:
		  System.out.println("RCModeReply not supported.");
		  break;
		  
	      case RFBproto.SasErrorMsg:
	          rfbProto.readSasError();
	          break;
	          
	      case RFBproto.SasEventMsg:
	          readSasEvent();
	          break;
	      
	      case RFBproto.SpTimestamp:
	          readSpTimestamp();
	          break;
	      
	      case RFBproto.SpCommand:
	          readSpCommand();
	          break;
	          
	      case RFBproto.SpSetSessionTimes:
	          rfbProto.readSpSessionTimes();
	          changeSpTimes();
	          break;
	      
	      case RFBproto.SpSesionInfo:
	          handleSpSessionInfo();
	          break;
	          
	      default:
		  throw new IOException(T._("Unknown RFB message type") + " " + msgType);
	    }
	}
    }

    public void sendPlay() throws IOException {
    	rfbProto.sendSpCommand(RFBproto.SpCommandPlay);
    }

    public void sendPause() throws IOException {
    	rfbProto.sendSpCommand(RFBproto.SpCommandPause);
    }

    public void sendStop() throws IOException {
    	rfbProto.sendSpCommand(RFBproto.SpCommandStop);
    }

    public void sendSpeed(int speed) throws IOException {
    	rfbProto.sendSpCommand(RFBproto.SpCommandSetSpeed, speed, 0);
    }

    public void sendSeek(Date pos) throws IOException {
    	rfbProto.sendSpTimestamp(pos);
    }

}
