package nn.pp.rc;

import java.io.*;
import java.net.*;
import java.text.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.util.zip.*;
import java.lang.reflect.*;
import java.security.*;
import javax.net.ssl.*;

/**
 * The protocoll implementation for the RFB.
 * The protocoll connects to a server using the 
 * given profile in the constructor.
 * 
 * @author Thomas Breitfeld, added RFBProfile and connection magic
 */
public abstract class RFBproto extends ErlaConnector implements RCProto {

    private static final boolean debug = false;
    final String versionMsgPrefix               = "e-RIC RFB ";

    /* TODO sort this stuff */
    final static int FramebufferUpdate          = 0x00;
    final static int SetColourMapEntries        = 0x01; /* unused */
    final static int Bell                       = 0x02;
    final static int Quit                       = 0x03;
    final static int UTFString                  = 0x07;
    final static int KbdLayout                  = 0x09;
    final static int OsdState			= 0x10;
    final static int CapabilityTable		= 0x12;

    final static int AuthCaps			= 0x20;
    final static int SessionChallenge		= 0x21;
    final static int AuthAck			= 0x22;
    
    final static int ServerCutText              = 0x03;
    final static int ServerFbFormat             = 0x80;
    final static int SystemInfo                 = 0x81;
    final static int PasswordChangeReply        = 0x82;
    final static int ConsoleMessage             = 0x83;
    final static int ServerCommand              = 0x84;
    final static int SasEventMsg                = 0x85;
    final static int SasErrorMsg                = 0x86;

    final static int SpTimestamp		= 0x88;


    final static int SetPixelFormat             = 0x00;
    final static int FixColourMapEntries        = 0x01; /* unused */
    final static int SetEncodings               = 0x02;
    final static int FramebufferUpdateRequest   = 0x03;
    final static int KeyboardEvent              = 0x04;
    final static int PointerEvent               = 0x05;
    final static int ScancodeEvent              = 0x06;
    final static int AckPixelFormat             = 0x13;

    final static int Login			= 0x20;
    final static int ChallengeResponse		= 0x21;

    final static int SystemInfoRequest          = 0x80;
    final static int TriggerHostReset           = 0x81;
    final static int TriggerHostPower           = 0x82;
    final static int SetKeyboardMapping         = 0x83;
    final static int PasswordChangeRequest      = 0x84;
    final static int MouseStateEvent		= 0x85; /* dummy */
    final static int MouseSyncEvent		= 0x86;
    final static int UserPropChangeEvent	= 0x87;
    final static int CursorChangeEvent		= 0x88;
    final static int PointerEventRelative       = 0x93;
    final static int PingRequest                = 0x94;
    final static int PingReply                  = 0x95;
    final static int BandwidthRequest		= 0x96;
    final static int BandwidthTick		= 0x97;
    final static int GlobalPropChangeEvent      = 0x9A;

    /* LARA only */    
    final static int VideoSettingsEventS2C	= 0x08; 
    final static int VideoQualityEventS2C	= 0x11;	// ignored for the moment
    final static int KvmSwitchEvent		= 0x89;
    final static int VideoSettingsEventC2S	= 0x90;
    final static int VideoSettingsRequest	= 0x91;
    final static int VideoRefresh		= 0x92;
    final static int VideoQualityEventC2S	= 0x98;	// not used
    final static int VideoQualityRequest	= 0x99;	// not used
    final static byte VSRequestNow		= 0x01;
    final static byte VSRequestRefuse		= 0x02;
    
    // RC mode stuff
    final static int RCModeRequest              = 0xA0;
    final static int RCModeReply                = 0xA1;
    
    final static int SpSetCimId			= 0xA1;
    final static int SpSetSession		= 0xA2; // only client -> server
    final static int SpSesionInfo               = 0xA2; // only server -> client
    final static int SpCommand			= 0xA3;
    final static int SpSetSessionTimes		= 0xA4;
    
    // RC mode messages, currently unused but preserved here
    final static byte RDPEnter                  = 0x00;
    public final static byte RDPLeave           = 0x01;
    
    final static byte RDPEntered                = 0x00;
    final static byte RDPLeft                   = 0x01;
    final static byte RDPNotAvailable           = 0x02;
    
    // quit reason send in quit msg
    final static int  QuitNoPerm                = 0x01;
    final static int  QuitExclAccess            = 0x02;
    final static int  QuitRejected              = 0x03;
    final static int  QuitNoSrvPasswd           = 0x04;
    final static int  QuitLoopback              = 0x05;
    final static int  QuitAuthFailed            = 0x06;
    final static int  QuitKVMPortDenied         = 0x07;
    final static int  QuitTooManyClients        = 0x08;
    final static int  QuitUnexpected            = 0x10;
    final static int  QuitBadProtoVer           = 0x11;
    final static int  QuitProtoError            = 0x12;
    final static int  QuitInternalError         = 0x13;
    final static int  QuitWrongReplayParameter	= 0x14;
    final static int  QuitNoForcedSSLConn	= 0x15;
    
    final static int AuthHttpSessId		= (1 << 0);
    final static int AuthPlain			= (1 << 1);
    final static int AuthMD5			= (1 << 2);
    final static int AuthRDMsession             = (1 << 4);

    final static int ConnectionFlagSasHw	= (1 << 0);
    final static int ConnectionFlagSasSw	= (1 << 1);
    final static int ConnectionFlagSp		= (1 << 2);
    final static int ConnectionFlagSasLocalOnly	= (1 << 3);
    final static int ConnectionFlagSasKbdShift		= 4;
    final static int ConnectionFlagSasKbdMask		= 3;
    final static int ConnectionFlagSasKbdNormal		= 0;
    final static int ConnectionFlagSasKbdNoKbd		= 1;	/* don't record keyboard events */
    final static int ConnectionFlagSasKbdFilter		= 2;	/* filter alphanumeric keys */
    final static int ConnectionFlagSasKbdAllFilterPb	= 3;	/* record all with filtered playback */
    
    final static int FbUpdateFlagTimestamp	= (1 << 0);

    final static int EncodingAuto               = 0;
    final static int EncodingFix                = 1;
    final static int EncodingAdvanced           = 2;
    final static String DefaultSelectedEncoding = "auto";
    final static String DefaultFixEncoding      = "videohi";
    final static String DefaultEncodingAdvCR    = "video";
    final static String DefaultEncodingAdvCD    = "color_16bpp";
    
    final static int EncodingMask		= 0x000000FF;
    final static int EncodingRaw                = 0x00;
    final static int EncodingCopyRect           = 0x01;
    final static int EncodingRRE                = 0x02;
    final static int EncodingCoRRE              = 0x04;
    final static int EncodingHextile            = 0x05;
    final static int EncodingZlib               = 0x06;
    final static int EncodingTight              = 0x07;
    final static int EncodingUseTightCache      = 0x09;
    final static int EncodingRawVSC             = 0x0A;
    final static int EncodingLRLESoft           = 0x0B;
    final static int EncodingJpeg		= 0x0C;
    final static int EncodingLRLEHard           = 0x80;
    final static int EncodingDCT                = 0x81;
    final static int EncodingIsHW		= 0x80;
    final static int EncodingAutomaticSW        = 0x7F;
    final static int EncodingAutomaticHW        = 0xFF;
    
    final static int EncodingParamZLIBShift	= 8;
    final static int EncodingParamZLIBMask	= (0x0F << EncodingParamZLIBShift);
    final static int EncodingParamSubencShift	= 12;
    final static int EncodingParamSubencMask	= (0x0F << EncodingParamSubencShift);
    final static int EncodingParamVideoOptShift	= 16;
    final static int EncodingParamVideoOptMask	= (0x01 << EncodingParamVideoOptShift);
    final static int EncodingParamCompressShift	= 17;
    final static int EncodingParamCompressMask	= (0x01 << EncodingParamCompressShift);
        
    final static int RawVscIsVsc                   = (1 << 0);
    final static int RawVscUseServersPixelFormat   = (1 << 1);
    final static int RawVscBigEndian               = (1 << 2);
    
    final static int HextileRaw                 = (1 << 0);
    final static int HextileBackgroundSpecified = (1 << 1);
    final static int HextileForegroundSpecified = (1 << 2);
    final static int HextileAnySubrects         = (1 << 3);
    final static int HextileSubrectsColoured    = (1 << 4);

    final static int TightStreamId                 = 0x03;
    final static int TightExplicitFilter           = 0x07;
    final static int TightFill                     = 0x08;
    final static int TightCheckForXBitFullColor    = 0x08;
    final static int Tight1BitFullColorBlackWhite  = 0x0A;
    final static int Tight2BitFullColorGrayscale   = 0x0B;
    final static int Tight4BitFullColorGrayscale   = 0x0C;
    final static int Tight4BitFullColor16Colors    = 0x0D;
    final static int TightFillXBit                 = 0x0F;
    final static int TightMaxSubencoding           = 0x0F;
    final static int TightFilterCopy               = 0x00;
    final static int TightFilterPalette            = 0x01;
    final static int TightFilterGradient           = 0x02;

    final static int TightMinToCompress            = 12;
    final static int TightMaxRectSize              = 65536;

    final static int Tight1BitBlackWhite           = 1;
    final static int Tight2BitGrayscale            = 2;
    final static int Tight4BitGrayscale            = 3;
    final static int Tight4Bit16Colors             = 4;
    final static int Tight8Bit256Colors            = 8;

    final static int TightCacheBufferDepth         = 8;
    final static int TightTileDimension            = 16;
    final static int TightCacheTag                 = 0x0C;
    final static int TightCacheClientIndex         = 0x7F;
    final static int TightTileCached               = 0x80;
    final static int TightTileNot_Cached           = 0x00;
    final static int TightCacheAll                 = 0x00;
    final static int TightCacheNone                = 0x04;
    final static int TightCacheMix                 = 0x08;
    
    final static int LRLESubenc15bitDirectLossy	   =  0;
    final static int LRLESubenc15bitDirectLossless =  1;
    final static int LRLESubenc7bitDirectLossy	   =  2;
    final static int LRLESubenc7bitDirectLossless  =  3;
    final static int LRLESubenc4bitPaletteLossy	   =  4;
    final static int LRLESubenc4bitPaletteLossless =  5;
    final static int LRLESubenc4bitGreyLossy	   =  6;
    final static int LRLESubenc4bitGreyLossless    =  7;   
    final static int LRLESubenc3bitGreyLossy	   =  8;
    final static int LRLESubenc3bitGreyLossless	   =  9;   
    final static int LRLESubenc2bitGreyLossy	   = 10;
    final static int LRLESubenc2bitGreyLossless	   = 11;   
    final static int LRLESubenc1bitGreyLossy	   = 12;
    final static int LRLESubenc1bitGreyLossless	   = 13;   

    final static int PointerMoveAbsolute = 0;
    final static int PointerMoveRelative = 1;

    final static byte BandwidthTickStage1 = 1;
    final static byte BandwidthTickStage2 = 2;

    final static byte UnbufferedInStream = 1;
    final static byte BufferedInStream   = 2;

    final static byte BlankStateDontBlank = 0;
    final static byte BlankStateBlank = 1;
    
    /* Explanation (no one could remember this)
       Norm - for button, usually fast unless
	      no universe or mode was switched
       Hard - for menu, always intelligent
       Fast - for hotkey and menu, always fast unless
	      we got no universe
     */
    final static byte MouseSyncNorm	 = 0x00;
    final static byte MouseSyncHard	 = 0x01;
    final static byte MouseSyncFast	 = 0x02;    
    
    final static byte tileWidth		 = 16;
    final static byte tileHeight	 = 16;
    
    final static int SasEventExistingSession	= 0x01;
    final static int SasEventExistingKvmSession	= 0x02;
    final static int SasEventUserLoginFailure	= 0x03;
    final static int SasEventUserSessionOpened	= 0x04;
    final static int SasEventUserSessionClosed	= 0x05;
    final static int SasEventKvmSessionOpened	= 0x06;
    final static int SasEventKvmSessionClosed	= 0x07;
    final static int SasEventKvmExclusiveOn	= 0x08;
    final static int SasEventKvmExclusiveOff	= 0x09;
    final static int SasEventInput		= 0x0A;
    final static int SasEventKvmSwitch		= 0x0B;
    final static int SasEventResetSessions	= 0x0C;

    final static int SasErrorNoInput			= 0x01;
    final static int SasErrorMissedInputEvent		= 0x02;
    final static int SasErrorMissedSessionEvent		= 0x03;
    final static int SasErrorMissedOtherEvent		= 0x04;
    
    final static int SpCommandPlay		= 0x00;
    final static int SpCommandPause		= 0x01;
    final static int SpCommandStop		= 0x02;
    final static int SpCommandSetSpeed		= 0x03;
    final static int SpCommandSeek		= 0x04;	// this command has a different PDU!

    final static int SpSessionInfoColorDepthUnknown    = 0x00;
    final static int SpSessionInfoColorDepth1BitBW     = 0x01;
    final static int SpSessionInfoColorDepth2BitGrey   = 0x02;
    final static int SpSessionInfoColorDepth3BitGrey   = 0x03;
    final static int SpSessionInfoColorDepth4BitGrey   = 0x04;
    final static int SpSessionInfoColorDepth4BitColor  = 0x05;
    final static int SpSessionInfoColorDepth7BitColor  = 0x06;
    final static int SpSessionInfoColorDepth15BitColor = 0x07;

    protected PrintStream       logger;
    protected boolean           connected;
    protected boolean           monitorMode;

    protected Socket            sock;
    public  InputStream	        sock_instream;
    public MonitoringDataInputStream min;
    public MonitoringOutputStream    mout;

    MonitoringDataInputStream        is, is_rect;
    MonitoringOutputStream           os;

    String name;

    String kbdlayout;

    int serverMajor = -1, serverMinor = -1;

    VideoSettings vs = new VideoSettings();

    int framebufferWidth, framebufferWidthPadded, framebufferHeight, framebufferHeightPadded;

    RFBproto(RFBProfile profile, PrintStream logger) {
	super(profile, logger, "RFB");
	this.profile = profile;
	this.logger = logger;
	connected = false;
    }
    
    public boolean connected() {
    	return connected;
    }

    public void setMonitorMode(boolean val) {
	this.monitorMode = val;
    }
    
    void setSocket(Socket sock) {
        this.sock = sock;
    }
    
    void setProtocolVersion(int major, int minor) {
        serverMajor = major;
        serverMinor = minor;
    }
    
    Socket createSocket() throws IOException {
        Socket sock = null;
        ErlaConnector connector = new ErlaConnector(profile, logger, "RFB");
       
        if(profile.sslRequired || profile.sslRequested)
            sock = connector.connectSSL(profile.remoteHost, profile.sslPort);
        if(sock == null && !profile.sslRequired)
            sock = connector.connect(profile.remoteHost, profile.primaryPort);
        if(sock == null && !profile.sslRequired && profile.useProxy)
            sock = connector.connectProxy(profile.proxyHost, profile.proxyPort,
    			    profile.remoteHost, profile.primaryPort);
        if(sock == null)
            throw new IOException(connector.sockerr + " (" + T._("no connect options left") + ")");

	secDesc = connector.secDesc;
        return sock;
    }
    
    void rfbConnect() throws IOException {
 	if (sock == null) {
 	    sock = createSocket();
	}

	mout= new MonitoringOutputStream(sock.getOutputStream());
	os = mout;

	sock_instream = sock.getInputStream();
	//min = new MonitoringInputStream(sock_instream);
	min = new MonitoringDataInputStream(new BufferedInputStream(sock_instream, 32768));
	is = min;
    }
    
    void writeCCSGproxyModePrefix() throws IOException {
	byte[] msg_buf;
	String msg_str;
	int length;
	msg_str = "<CSC_Connect ConnectionID=\"" + profile.rdmConnectionID + "\"/>";
	msg_buf = msg_str.getBytes();
	length = 4 + msg_buf.length + 1; 
	
	os.writeInt(length);
	os.write(msg_buf, 0, msg_buf.length);
	os.writeByte(0);
    }
    
    void writeHelloMsg() throws IOException {
	byte[] auth_bytes = new byte[11];
	System.arraycopy(("e-RIC AUTH=").getBytes("ISO-8859-1"), 0, auth_bytes, 0, auth_bytes.length);
	os.write(auth_bytes);
	os.flush();
    }

    void readVersionMsg() throws IOException {

	byte[] b = new byte[16];
	is.readFully(b);
	if ((b[0] != 'e') || (b[1] != '-') || (b[2] != 'R') ||
	    (b[3] != 'I') || (b[4] != 'C') || (b[5] != ' ') ||
	    (b[6] != 'R') || (b[7] != 'F') || (b[8] != 'B') ||
	    (b[9] != ' ') || (b[10]  < '0') || (b[10] > '9')  ||
	    (b[11] < '0') || (b[11] > '9') || (b[12] != '.')||
	    (b[13] < '0') || (b[13] > '9') || (b[14] < '0') ||
	    (b[14] > '9') || (b[15] != '\n')) {
	    throw new IOException(MessageFormat.format(T._("Host {0} hasn't a valid server version (protocol error)"),
	                          new Object[] { profile.remoteHost }));
	}
	serverMajor = (b[10] - '0') * 10 + (b[11] - '0');
	serverMinor = (b[13] - '0') * 10 + (b[14] - '0');
    }

    void writeVersionMsg() throws IOException {
        String major = "" + serverMajor;
        if (serverMajor < 10) major = "0" + major;
        String minor = "" + serverMinor;
        if (serverMinor < 10) minor = "0" + minor;
	String versionMsg = versionMsgPrefix + major + "." + minor + "\n";

	byte b[] = versionMsg.getBytes();
	os.write(b);
	os.flush();
    }
    
    void negotiateProtocolVersion() throws IOException {
        if (serverMinor == -1 || serverMajor == -1) {
            writeHelloMsg();
            readVersionMsg();
            writeVersionMsg();
        }
        // otherwise, the version number has already been negotiated from outside
    }

    protected abstract void runInitialHandshake() throws IOException;
    
    /**
     * tries to connect to the server using the information 
     * in RFBProfile. After the connection has been successfully
     * made the initial handshake will be run by the constructor
     * as in order to check for the right protocoll version etc.
     * The following algorithm is implemented:
     * 
     * if ssl class then connect sslPort
     * else connect primaryPort
     *    if timeout connecting primaryPort then throw IOException
     * 
     * Additionally, in case the sslRequired flag is set, only the
     * ssl connection will be made. If sslRequired is not set but
     * the sslRequested flag the primaryPort number will be used 
     * only for connecting to the server.
     */
    public void establishConnection() throws IOException {
	logger.println(T._("RFB: construct RFB with the following profile")
		       + ": " + profile);
	rfbConnect();
	
	// Handle CC-SG proxy mode connections
	if (profile.rdmConnectionID != null) {
	    writeCCSGproxyModePrefix();
	}
        
        // the init handshake might hang for obscure reasons
        // the timeout brings us out of it again...
        sock.setSoTimeout(20000);
	
	runInitialHandshake();

        // everything went well, we can switch the timeout off
        sock.setSoTimeout(0);
    }

    public static String getQuitReasonMsg(int qr) {
	switch(qr) {
	  case RFBproto.QuitNoPerm: return T._("no permission");
	  case RFBproto.QuitExclAccess: return T._("exclusive access active");
	  case RFBproto.QuitRejected: return T._("manually rejected");
	  case RFBproto.QuitNoSrvPasswd: return T._("server password disabled");
	  case RFBproto.QuitLoopback: return T._("loopback connection is senseless");
	  case RFBproto.QuitAuthFailed: return T._("authentication failed");
	  case RFBproto.QuitKVMPortDenied: return T._("access to this kvm port denied");
	  case RFBproto.QuitTooManyClients: return T._("too many clients active simultaneously");
	  case RFBproto.QuitUnexpected: return T._("unexpected server error");
	  case RFBproto.QuitBadProtoVer: return T._("bad protocol version");
	  case RFBproto.QuitProtoError: return T._("protocol error");
	  case RFBproto.QuitInternalError: return T._("internal server error");
	  case RFBproto.QuitWrongReplayParameter: return T._("wrong replay parameters");
	  case RFBproto.QuitNoForcedSSLConn: return T._("unable to establish SSL connection");
	}
	return "";
    }
    
    public static String getSasErrorMsg(int qr) {
	switch(qr) {
	  case RFBproto.SasErrorNoInput: return T._("no input events possible");
	  case RFBproto.SasErrorMissedInputEvent: return T._("missed input event");
	  case RFBproto.SasErrorMissedSessionEvent: return T._("missed session event");
	  case RFBproto.SasErrorMissedOtherEvent: return T._("missed other event");
	}
	return "";
    }
    
    public boolean isSecure() {
	return isSecure;
    }

    public String getSecDesc() {
	return secDesc;
    }

    public void setSecDesc(String secDesc) {
	this.secDesc = secDesc;
    }

    public void close() {
        try {
	    sock.close();
	} catch (IOException e) {
	    e.printStackTrace(logger);
	}
	connected = false;
    }
    
    protected void printBuffer(byte[] buf, int size, int brk) {
	for (int i = 0; i < size; i++) {
	    if ((i % brk) == 0) System.out.println("");
	    System.out.print(Integer.toHexString(buf[i]).toUpperCase()+ " ");
	}
	System.out.println("");
    }
    
    //
    // Read integer in compact representation
    //
    
    int readCompactLen() throws IOException {
	int portion = is.readUnsignedByte();
	int len = portion & 0x7F;
	if ((portion & 0x80) != 0) {
	    portion = is.readUnsignedByte();
	    len |= (portion & 0x7F) << 7;
	    if ((portion & 0x80) != 0) {
		portion = is.readUnsignedByte();
		len |= (portion & 0xFF) << 14;
	    }
	}
	return len;
    }

    abstract void writeUserPropChangeEvent(String key, String value) throws IOException;
    abstract void writeGlobalPropChangeEvent(String key, String value) throws IOException;
    abstract void writeVideoSettingsEvent(byte event, short value) throws IOException;
    abstract void writeVideoSettingsRequest(byte request) throws IOException;
    abstract void writeVideoRefresh() throws IOException;
    abstract void writeString(String msg) throws IOException;
    abstract void writeSetPixelFormat(int bitsPerPixel, int depth,
    	boolean bigEndian, boolean trueColour, int redMax, int greenMax,
    	int blueMax, int redShift, int greenShift, int blueShift) throws IOException;
    abstract void writeFullFramebufferUpdateRequest() throws IOException;
    abstract RecordedSessionInfo readSessionInfo() throws IOException;
}
