package nn.pp.rc;

import java.awt.*;
import java.awt.image.*;
import java.io.*;

/*
 * The TrueColor Renderer is a 16 bpp renderer
 * (so its actually a HiColor renderer now, but 16bpp is as true as it gets for us)
 * 
 * @author Thomas Breitfeld, Peppercon AG
 */
public abstract class TrueColorRFBRenderer extends RFBRenderer {

    final static int           sh = 16;      // size of sprite
    final static int           sw = 16;
    protected Dimension        mysize, padsize;
    protected Image            simg;
    private MemoryImageSource  simgsrc;      
    private int[]              simgmem;
    private byte[]             simgmemb;
    private int                transx = 0;
    private int                transy = 0;
    private int[]              intcolors;
    private boolean            lrle_debug = false;

    private LRLEColorDecoder   lrle_dec;  
    private int		       lrle_subenc = -1;
    
    private byte[]             rimgmemb;
    private byte[]             rimgtempb;
    
    abstract public void setInterpol(boolean ip);
    abstract public void paint(Graphics g, boolean scale, Dimension scalesize);
    abstract public void repaint();
    abstract public void sendUpdateMsg() throws IOException;

    public TrueColorRFBRenderer(Component comp) {
        super(comp);

        DirectColorModel colormodel;
        int i;

        // preallocate colors
        colormodel = new DirectColorModel(16, 31 << 11, 63 << 5, 31);
        intcolors = new int[64 * 1024];
        for (i = 0; i < (64 * 1024); i++) {
            intcolors[i] = colormodel.getRGB(i);
        }

        // preallocate colors and greyscale values for LRLE
        colormodel = new DirectColorModel(15, 31 << 10, 31 << 5, 31);
	        
        colormodel = new DirectColorModel(6, 63, 63, 63);
        
    }   
   
    public void sendPixelMsg() throws IOException {
        synchronized(this) {
            rfb.writeSetPixelFormat(16, 16, true, true, 31, 63, 31, 11, 5, 0);
        }
    }

    public void setRenderSize(Dimension d) {
        super.setRenderSize(d);
        mysize = d;
        createSImg(d);
    }

    private void createSImg(Dimension d) {
        if(simg != null) simg.flush();

	padsize = new Dimension(d);
	/* we reserve more memory than visible for padding */
	if (padsize.width % sw != 0) padsize.width = (padsize.width / sw + 1) *  sw;
	if (padsize.height % sh != 0) padsize.height = (padsize.height / sh + 1) * sh;
	
        simgmemb = new byte[2 * padsize.width * padsize.height];
        simgmem = new int[padsize.width * padsize.height];
        simgsrc = new MemoryImageSource(d.width, d.height, simgmem, 0, d.width);
        jimgsrc = simgsrc;
        simgsrc.setAnimated(true);
        //simgsrc.setFullBufferUpdates(false);
        simg = comp.createImage(simgsrc);
    }

    void checkRawBufSize(int newSize) {
    	if(rimgmemb == null || rimgmemb.length < newSize) {
    	    rimgmemb = new byte[newSize];
    	}
    }

    void checkRawTmpBufSize(int newSize) {
    	if(rimgtempb == null || rimgtempb.length < newSize) {
    	    rimgtempb = new byte[newSize];
    	}
    }

    private void byteToIntBuffer(byte[] bb, int off_in, int[] ib, int off, int c, boolean is_be) {
        int bi;
        for(int i = 0; i < c; ++i) {
            bi = 2 * i + off_in;
	    int index;
	    if (is_be) {
		index = (bb[bi + 0] & 0xff) << 8 | bb[bi + 1] & 0xff;
	    } else {
		index = (bb[bi + 1] & 0xff) << 8 | bb[bi + 0] & 0xff;
	    }
            ib[off + i] = intcolors[index];
            //System.out.println(Integer.toHexString(ib[i]));
        }
    }

    public void drawRawRect(int x, int y, int w, int h, boolean bigEndian)
            throws IOException {
        //System.out.println("raw " + x + " " + y + " " + w + " " + h);
	
        // write back the buffered data before reading unbuffered from rfb.is
        finishBufferedReading(rfb.is_rect);
        
        for (int i = y; i < (y + h); i++) {
            rfb.is_rect.readFully(simgmemb, 0, 2 * w);
            byteToIntBuffer(simgmemb, 0, simgmem, i * mysize.width + x, w, bigEndian);
        }
	
        simgsrc.newPixels(x, y, w, h);
        gvimg.setClip(x, y, w, h);
        gvimg.drawImage(simg, 0, 0, null);
        gvimg.setClip(0, 0, mysize.width, mysize.height);
    }

    public void drawRawVSCRect(int x, int y, int w, int h, boolean bigEndian)
	throws IOException {
	int TILE_WIDTH = 16;
	int TILE_HEIGHT = 16;
	
	int flags = rfb.is_rect.readByte();
	
	if((flags & RFBproto.RawVscIsVsc) == 0) {
	    drawRawRect(x, y, w, h, bigEndian);
	    return;
	}

	// read the pixel data from the network
	int c = 2 * w * h;
	checkRawTmpBufSize(c);
	rfb.is_rect.readFully(rimgtempb, 0, c);
	
	// convert from VSC to linear framebuffer
	w = Math.min(w, mysize.width - x);
	h = Math.min(h, mysize.height - y);
	c = 2 * w * h;
	checkRawBufSize(c);
	
	int indexsrc = 0;
	int indexdst = 0;
	int indexsrcstart = 0;
	int lines = 0;
	int rows = 0;
	
	for(int i = 0; i < h; i++) {
	    int cols = 0;
	    for(int j = 0; j < w; j++) {
	    	rimgmemb[indexdst++] = rimgtempb[indexsrc++];
	    	rimgmemb[indexdst++] = rimgtempb[indexsrc++];
	    	cols++;
	    	if(cols == TILE_WIDTH) {
	    	    indexsrc += (TILE_HEIGHT - 1) * TILE_WIDTH * 2;
	    	    cols = 0;
	    	}
	    }
	    lines++;
	    indexsrc = indexsrcstart + lines * TILE_WIDTH * 2;
	    if(lines == TILE_HEIGHT) {
	    	rows++;
	    	indexsrcstart = rows * TILE_HEIGHT * w * 2;
	    	indexsrc = indexsrcstart;
	    	lines = 0;
	    }
	}
	
    	// actually draw the data
        for (int i = y; i < (y + h); i++) {
            byteToIntBuffer(rimgmemb, (i-y)*w*2, simgmem, i * mysize.width + x, w, (flags & RFBproto.RawVscBigEndian) == RFBproto.RawVscBigEndian);
        }
		
        simgsrc.newPixels(x, y, w, h);
        gvimg.setClip(x, y, w, h);
        gvimg.drawImage(simg, 0, 0, null);
        gvimg.setClip(0, 0, mysize.width, mysize.height);
    }

    public void drawCopyRect(int srcX, int srcY, int x, int y, int w, int h)
	throws IOException {
	// FIXME
	throw new IOException(T._("True color CopyRect encoding not implemented"));
    }


    public void drawHextileRect(int x, int y, int w, int h, boolean bigEndian)
	throws IOException {
	int bg = 0, fg = 0, ux, uy, uw, uh;
	int counter = 0;
	for (int ty = y; ty < y + h; ty += sh) {
	    for (int tx = x; tx < x + w; tx += sw) {
		int tw = sw, th = sh;
		if (x + w - tx < 16) tw = x + w - tx;
		if (y + h - ty < 16) th = y + h - ty;
		int subencoding = readBufferedByte(rfb.is_rect);
		counter++;

		if ((subencoding & RFBproto.HextileRaw) != 0) {
		    drawRawRect(tx, ty, tw, th, bigEndian);
		    continue;
		}

		if ((subencoding & RFBproto.HextileBackgroundSpecified) != 0){
		    if (bigEndian) {
		    	bg = intcolors[((readBufferedByte(rfb.is_rect) & 0xff) << 8) | (readBufferedByte(rfb.is_rect) & 0xff)];
		    } else {
		    	bg = intcolors[(readBufferedByte(rfb.is_rect) & 0xff) | ((readBufferedByte(rfb.is_rect) & 0xff) << 8)];
		    }
		}
		fillRect(bg, tx, ty, tw, th);

		if ((subencoding & RFBproto.HextileForegroundSpecified) != 0) {
		    if (bigEndian) {
		    	fg = intcolors[((readBufferedByte(rfb.is_rect) & 0xff) << 8) | (readBufferedByte(rfb.is_rect) & 0xff)];
		    } else {
		    	fg = intcolors[(readBufferedByte(rfb.is_rect) & 0xff) | ((readBufferedByte(rfb.is_rect) & 0xff) << 8)];
		    }
		}
		
		if ((subencoding & RFBproto.HextileAnySubrects) == 0)
		    continue;

		int nSubrects = readBufferedByte(rfb.is_rect);
		counter++;
		translate(tx, ty);
		
		if ((subencoding & RFBproto.HextileSubrectsColoured) != 0) {
		    for (int j = 0; j < nSubrects; j++) {
		    	if (bigEndian) {
		    	    fg = intcolors[((readBufferedByte(rfb.is_rect) & 0xff) << 8) | (readBufferedByte(rfb.is_rect) & 0xff)];
		    	} else {
		    	    fg = intcolors[(readBufferedByte(rfb.is_rect) & 0xff) | ((readBufferedByte(rfb.is_rect) & 0xff) << 8)];
		    	}
			int b1 = readBufferedByte(rfb.is_rect);
			int b2 = readBufferedByte(rfb.is_rect);
			ux = b1 >> 4;
			uy = b1 & 0xf;
			uw = (b2 >> 4) + 1;
			uh = (b2 & 0xf) + 1;
			fillRect(fg, ux, uy, uw, uh);
		    }
		    counter+=(nSubrects * 2);
		} else {
		    for (int j = 0; j < nSubrects; j++) {
			int b1 = readBufferedByte(rfb.is_rect);
			int b2 = readBufferedByte(rfb.is_rect);
			ux = b1 >> 4;
			uy = b1 & 0xf;
			uw = (b2 >> 4) + 1;
			uh = (b2 & 0xf) + 1;
			fillRect(fg, ux, uy, uw, uh);
		    }
		    counter+=(nSubrects * 2);
		}
		translate(-tx, -ty);
	    }
	}
	
	finishBufferedReading(rfb.is_rect);
	
	// increase the monitoring counters manually
	// (see also MonitoringDataInputStream)
	rfb.min.increaseCounters(counter);
	simgsrc.newPixels(x, y, w, h);
	gvimg.setClip(x, y, w, h);
	gvimg.drawImage(simg, 0, 0, null);
	gvimg.setClip(0, 0, mysize.width, mysize.height);
    }
    
    public void drawTightRect(int x, int y, int w, int h)
	throws IOException {
	throw new IOException(T._("True color tight encoding not implemented"));
    }

    public void drawTightCachedRect(int x, int y, int w, int h)
	throws IOException {
	throw new IOException(T._("True color tight encoding not implemented"));
    }

    public void dispose() {
	simg.flush();
    }
    
    public void fillRect(int color, int x, int y, int w, int h) {
	//System.out.println("fill " + x + " " + y + " " + w + " " + h);
	for(int iy = y; iy < y + h; ++iy) {
	    int off = (transy + iy) * mysize.width + transx;
	    for(int ix = x; ix < x + w; ++ix) {
		simgmem[off + ix] = color;
	    }
	}
    }

    public void translate(int tx, int ty) {
	transx += tx;
	transy += ty;
	//System.out.println("trans " + transx + " " + transy);
    }

    public void setNewTightCacheSize() {
    }

    public void enableTightCache(boolean enable) {
    }
	
    private class LRLEColorDecoderConf {
	public boolean is_compact;
	public boolean is_grey;
	public boolean is_map;
	public int  depth;
	public int  grey_depth;

	public LRLEColorDecoderConf(boolean is_map, boolean is_compact, boolean is_grey,
				    int depth, int grey_depth) {
	    this.is_map		= is_map;
	    this.is_compact	= is_compact;
	    this.is_grey	= is_grey;
	    this.depth		= depth;
	    this.grey_depth	= grey_depth;	    
	}
	public String toString() {
	    return new String("Compact="+is_compact+",Grey="+is_grey+",Depth="+depth+",GreyDepth="+grey_depth);
	}
    }
    
    /* ------- LRLE handling --------- */
    private class LRLEColorDecoder {
	public int  subenc;
	public int[] colors;
	public int[] greys;
	public LRLEColorDecoderConf conf;

	private LRLEColorDecoderConf[] configs = {
	    new LRLEColorDecoderConf(false, false, false, 15, 6), // RFBproto.LRLESubenc15bitDirectClean
	    new LRLEColorDecoderConf(false, false, false, 15, 6), // RFBproto.LRLESubenc15bitDirectNoisy
	    new LRLEColorDecoderConf(false, false, false,  7, 4), // RFBproto.LRLESubenc7bitDirectClean
	    new LRLEColorDecoderConf(false, false, false,  7, 4), // RFBproto.LRLESubenc7bitDirectNoisy
	    new LRLEColorDecoderConf(false, true,  false,  4, 4), // RFBproto.LRLESubenc4bitPaletteClean
	    new LRLEColorDecoderConf(false, true,  false,  4, 4), // RFBproto.LRLESubenc4bitPaletteNoisy
	    new LRLEColorDecoderConf(false, true,  true,   4, 4), // RFBproto.LRLESubenc4bitGreyClean
	    new LRLEColorDecoderConf(false, true,  true,   4, 4), // RFBproto.LRLESubenc4bitGreyNoisy
	    new LRLEColorDecoderConf(false, true,  true,   3, 3), // RFBproto.LRLESubenc3bitGreyClean
	    new LRLEColorDecoderConf(false, true,  true,   3, 3), // RFBproto.LRLESubenc3bitGreyNoisy
	    new LRLEColorDecoderConf(true,  false, true,   2, 2), // RFBproto.LRLESubenc2bitGreyClean
	    new LRLEColorDecoderConf(true,  false, true,   2, 2), // RFBproto.LRLESubenc2bitGreyNoisy
	    new LRLEColorDecoderConf(true,  false, true,   1, 1), // RFBproto.LRLESubenc1bitGreyClean
	    new LRLEColorDecoderConf(true,  false, true,   1, 1)  // RFBproto.LRLESubenc1bitGreyNoisy
	};
	
	public LRLEColorDecoder(int new_subenc) {
	    subenc = new_subenc;
	    System.out.println("new LRLE color decoder type " + subenc);
	    
	    createLRLEData();
	    createLRLEColorTables();

	}
	
	private void createLRLEData() {	    	    
	    if (subenc < configs.length) {
		conf = configs[subenc];
		System.out.println(conf.toString());
	    } else {
		System.out.println("ERROR: unsupported color decoder type");		
	    }
	}
	
	private void createLRLEColorTables() {
	    int i;

	    /* create grey table */
	    greys = new int[1 << conf.grey_depth];
	    if (conf.is_grey) colors = new int[1 << conf.grey_depth];
	    
	    for (i = 0; i < (1 << conf.grey_depth); i++) {
		greys[i] = 0xFF000000;
		switch (conf.grey_depth) {
		  case 1:	greys[i] |= ( (i*255)    << 16) | ( (i*255)    << 8) |  (i*255);	break;
		  case 2:	greys[i] |= ( (i* 85)    << 16) | ( (i* 85)    << 8) |  (i* 85);	break;
		  case 3:	greys[i] |= (((i* 73)/2) << 16) | (((i* 73)/2) << 8) | ((i* 73)/2);	break;
		  case 4:	greys[i] |= ( (i* 17)    << 16) | ( (i* 17)    << 8) |  (i* 17);	break;
		  case 5:	greys[i] |= (((i* 33)/4) << 16) | (((i* 33)/4) << 8) | ((i* 33)/4);	break;
		  case 6:	greys[i] |= (((i* 65)/16)<< 16) | (((i* 65)/16)<< 8) | ((i* 65)/16);	break;
		  default:	greys[i] |= 0x00FF00FF;							break;
		}

		if (conf.is_grey) colors[i] = greys[i];
	    }
	    if (conf.is_grey) return;
	    
	    /* calculations adapted from original rle_decode.c,
	       target format is 32 bit ARGB */
	    colors = new int[1 << conf.depth];
	    switch (conf.depth) {
	      case 4:
		  colors[0x00] = 0xFF000000; // black
		  colors[0x01] = 0xFF7F0000; // dark red
		  colors[0x02] = 0xFF007F00; // dark green
		  colors[0x03] = 0xFF7F7F00; // dark yellow
		  colors[0x04] = 0xFF00007F; // dark blue
		  colors[0x05] = 0xFF7F007F; // dark magenta
		  colors[0x06] = 0xFF007F7F; // dark cyan
		  colors[0x07] = 0xFF7F7F7F; // dark grey
		  colors[0x08] = 0xFFC0C0C0; // light grey
		  colors[0x09] = 0xFFFF0000; // light red
		  colors[0x0A] = 0xFF00FF00; // light green
		  colors[0x0B] = 0xFFFFFF00; // light yellow
		  colors[0x0C] = 0xFF0000FF; // light blue
		  colors[0x0D] = 0xFFFF00FF; // light magenta
		  colors[0x0E] = 0xFF00FFFF; // light cyan
		  colors[0x0F] = 0xFFFFFFFF; // white
		  break;
	      case 7:
		  int c7Table[] = { 0, 64, 128, 192, 255 };
		    
		  for (i = 0; i < 125; i++) {
		      colors[i]  = 0xFF000000 |
			  (c7Table[(i / 25)]	<< 16) |
			  (c7Table[(i / 5) % 5] <<  8) |
			   c7Table[(i % 5)];
		  }
		  /* shouldn't be used, so use some 'warning' color */
		  for (i = 125; i < 128; i++) {
		      colors[i] = 0xFFFF0000;
		  }
		  break;
	      case 15:
		  for (i = 0; i < (1 << conf.depth); i++) {
		      colors[i] = 0xFF000000 |
			  (((((i & 0x7C00) >> 10) * 33) / 4) << 16) |
			  (((((i & 0x03E0) >>  5) * 33) / 4) <<  8) |
			   ((((i & 0x001F)      ) * 33) / 4);
		  }
		  break;
	    }
	}
    }

    private void drawLRLEMap(int offset, int x, int y, int w, int h) throws IOException
    {
	int  steps, dy, segment, k, lineBytes, lineRest, color, code, mask;

	steps	  = 8 / lrle_dec.conf.grey_depth;
	mask	  = (1 << lrle_dec.conf.grey_depth) - 1;
	lineBytes = w / steps;
	lineRest  = w % steps;

	for (dy = 0; dy < h; dy++) {
	    for (segment = 0; segment < lineBytes; segment++) {
		code = readBufferedByte(rfb.is_rect);
		for (k = steps - 1; k >= 0; k--) {
		    simgmem[offset + dy * mysize.width + segment * steps + k] = lrle_dec.greys[code & mask];
		    code >>= lrle_dec.conf.grey_depth;
		}
	    }
	    
	    if (lineRest != 0) {
		code = readBufferedByte(rfb.is_rect);
		for (k = lineRest - 1; k >= 0; k--) {
		    simgmem[offset + dy * mysize.width + lineBytes * steps + k] = lrle_dec.greys[code & mask];
		    code >>= lrle_dec.conf.grey_depth;
		}
	    }
	}
    }
    
    public void drawLRLERect(long encoding, int x, int y, int w, int h) throws IOException
    {
        int tx, ty, tw, th;
        int dx, dy, cx, cy;
        int code, run = 0, copy = 0, j;
        int color = 0xFF000000;
        Graphics gvimg = this.gvimg;
        int[] eot = new int[2];
        int[] prevLine = new int[w];
	int subenc = (int)(encoding & RFBproto.EncodingParamSubencMask) >> RFBproto.EncodingParamSubencShift;
	    
	if (lrle_subenc == -1 || lrle_subenc != subenc) {
	    lrle_subenc = subenc;
	    lrle_dec = new LRLEColorDecoder(lrle_subenc);
	}
	
        for (ty = y; ty < y + h; ty += sh) {
            for (tx = x; tx < x + w; tx += sw) {
                int tile_offset = ty * mysize.width + tx;
                tw = sw;
                th = sh;
                if (x + w - tx < 16) tw = x + w - tx;
                if (y + h - ty < 16) th = y + h - ty;

		if (lrle_dec.conf.is_map == true) {		    
		    drawLRLEMap(tile_offset, tx, ty, tw, th);
		} else {
		    dx = dy = 0;
		    cx = cy = 0;

		    if (lrle_debug) System.out.println("Tile ("+tx+","+ty+")");
		
		    while (true) {
			if (lrle_debug) System.out.print("("+dx+","+dy+")");
			// end of tile
			if (dx == 0 && cy + dy == th) {	
			    // thats it, go to the next tile
			    if (lrle_debug) System.out.println(" >>END OF TILE");
			    break;			
			}

			code = readBufferedByte(rfb.is_rect);
			if (lrle_debug) System.out.print(" Code=0x" + Integer.toHexString(code) + " -> ");
			if ((code & 0xE0) == 0xE0) {
			    // line copy
			    copy = 1;
			    run = (code == 0xFF ? readBufferedByte(rfb.is_rect) : (code & 0x1f));
			    if (lrle_debug) System.out.println("COPY " + (run + 1));
			} else if (lrle_dec.conf.is_compact) {
			    copy = 0;
			    if (lrle_dec.conf.depth <= 3) {
				color = lrle_dec.colors[code & 0x07];
				run = code >> 3;
				if (lrle_debug) System.out.println("PIXEL=0x" + Integer.toHexString(code & 0x07) + " (RUN=" + (run+1) + ") -> 0x" + Integer.toHexString(color));
			    } else {
				color = lrle_dec.colors[code & 0x0F];
				run = code >> 4;
				if (lrle_debug) System.out.println("PIXEL=0x" + Integer.toHexString(code & 0x0F) + " (RUN=" + (run+1) + ") -> 0x" + Integer.toHexString(color));
			    }
			} else {			
			    switch ((code & 0xC0) >> 6) {
			      case 0:
			      case 1: // word pixel value
				  if (lrle_dec.conf.depth > 7) {
				      code = (code << 8) | readBufferedByte(rfb.is_rect);
				  }
				  color = lrle_dec.colors[code];
				  if (lrle_debug) System.out.println("PIXEL=0x" + Integer.toHexString(code) + " -> 0x" + Integer.toHexString(color));
				  run = 0;
				  copy = 0;
				  break;
			      case 2:		// grey compression
				  color = lrle_dec.greys[code & 0x3F];
				  if (lrle_debug) System.out.println("GREY=0x" + Integer.toHexString(code) + " -> 0x" + Integer.toHexString(color));
				  run = 0;
				  copy = 0;
				  break;
			      case 3:		// run or line copy
				  run = (code & 0x1f);
				  if (lrle_debug) System.out.println("RUN " + (run + 1));
				  break;
			      default:
				  break;
			    }
			}

			for (j = 0; j <= run; j++) {
			    if (copy == 0) {
				prevLine[dx] = color;
			    }
			    simgmem[tile_offset + (cy+dy) * mysize.width  + cx+dx] = prevLine[dx];
			    ++dx;
			    if (cx + dx == tw) {
				dx = 0;
				dy++;
			    }
			}
		    }    
		}
	    }
        }

	// draw this rect
	simgsrc.newPixels(x, y, w, h);
	gvimg.setClip(x, y, w, h);
	gvimg.drawImage(simg, 0, 0, null);
	gvimg.setClip(0, 0, mysize.width, mysize.height);
        
        finishBufferedReading(rfb.is_rect);
    }
}
