package nn.pp.rc;

import java.net.*;
import java.text.*;
import java.io.*;

/**
 * This special Socket implementation is a Tunnel Socket
 * that tries to open a http tunnel using the http CONNECT
 * method supplied by http proxies. The Tunnel Socket will
 * open a connection to the proxy first and try to run the 
 * Tunnel protocol. If everything goes well it can be used
 * as a normale Socket tunneling the Data transparently
 * through the Proxy http server
 */
public class TunnelSocket extends Socket {

    /**
     * The Tunnel Socket will
     * open a connection to the proxy first and try to run the 
     * Tunnel protocol. If everything goes well it can be used
     * as a normale Socket tunneling the Data transparently
     * through the Proxy http server
     * 
     * @param tunnelhost hostname of the proxy server
     * @param tunnelport portnumber of the proxy server
     * @param host       target host name to connect
     * @param port       target port number to connect
     */
    public TunnelSocket(String tunnelhost, int tunnelport,
			String host, int port) 
	throws IOException {

	// first we try to open a conenction to the proxy
	super(tunnelhost, tunnelport);

	// if that has worked ok we run the tunnel handshake
	doTunnelHandshake(host, port);
    }

    /*
     * Tell our tunnel where we want to CONNECT, and look for the
     * right reply.  Throw IOException if anything goes wrong.
     */
    private void doTunnelHandshake(String host, int port)
	throws IOException {

	OutputStream out = getOutputStream();
	String msg = "CONNECT " + host + ":" + port + " HTTP/1.0\n"
	    + "User-Agent: e-RIC Remote Console Applet"
	    + "\r\n\r\n";
	byte b[];
	try {
	    // We really do want ASCII7 
	    // the http protocol doesn't change with locale.
	    b = msg.getBytes("ASCII7");
	} catch (UnsupportedEncodingException ignored) {
	    // If ASCII7 isn't there, something serious is wrong, but
	    // Paranoia Is Good (tm)
	    b = msg.getBytes();
	}
	out.write(b);
	out.flush();

	// We need to store the reply so we can create a detailed error
	byte reply[] = new byte[200];
	int replyLen = 0;
	int newlinesSeen = 0;
	boolean	headerDone = false; // Done on first newline 
	InputStream in = getInputStream();
	boolean	error = false;

	while (newlinesSeen < 2) {
	    int i = in.read();
	    if (i < 0) {
		throw new IOException(T._("Unexpected EOF from proxy"));
	    }
	    if (i == '\n') {
		headerDone = true;
		++newlinesSeen;
	    } else if (i != '\r') {
		newlinesSeen = 0;
		if (!headerDone && replyLen < reply.length) {
		    reply[replyLen++] = (byte)i;
		}
	    }
	}

	// Converting the byte array to a string is slightly wasteful
	// in the case where the connection was successful, but it's
	// insignificant compared to the network overhead.
	String replyStr;
	try {
	    replyStr = new String(reply, 0, replyLen, "ASCII7");
	} catch (UnsupportedEncodingException ignored) {
	    replyStr = new String(reply, 0, replyLen);
	}

	// We asked for HTTP/1.0, so we should get that back 
	if (!replyStr.startsWith("HTTP/1.0 200")) {
	    throw new IOException(MessageFormat.format(T._(
	        "Unable to tunnel through {0}:{1}. Proxy returns \"{2}\""),
	        new Object[] { host, new Integer(port), replyStr }));
	}
	// tunneling Handshake was successful!
    }
}
