/*
 * VideoSettingsFrame.java
 *
 * handles the LARA video settings
 * 
 */
package nn.pp.rc;

import java.awt.*;
import java.util.*;
import java.text.*;
import java.awt.event.*;
import java.net.*;
import java.io.*;

public class VideoSettingsFrame extends Frame
{
    Component parent;
    PrintStream logger;
    boolean monitorMode, debug = false;
    Dialog myFrame;
    GridBagLayout gbl;
    RFBproto proto;
    int	perms;
    boolean type_full;
    String boardname;

    Vector panels;
    VSButton saveBtn, cancelBtn, resetCurBtn, resetAllBtn;
    VSMouseAdapter mouseAdapter = new VSMouseAdapter();
    Label resInfoLbl;
    
    final static byte VSBrightness	= 0x00;
    final static byte VSContrastRed	= 0x01;
    final static byte VSContrastGreen	= 0x02;
    final static byte VSContrastBlue	= 0x03;
    final static byte VSClock		= 0x04;
    final static byte VSPhase		= 0x05;
    final static byte VSXOffset		= 0x06;
    final static byte VSYOffset		= 0x07;
    final static byte VSResetAll	= 0x08;
    final static byte VSResetCurrent	= 0x09;
    final static byte VSSaveSettings	= 0x0A;
    final static byte VSCancelSettings	= 0x0B;
    final static byte VSAutoAdjust	= 0x0C;
    
    public VideoSettingsFrame(Component parent, String boardname, PrintStream logger,
			      boolean debug, boolean monitor, int vs_perms,
			      boolean type_full) {
        this.parent = parent;
	this.logger = logger;
	this.debug  = debug;
	this.monitorMode = monitor;
	this.perms = vs_perms;
	this.boardname = boardname;
	this.type_full = type_full;
	myFrame = null;
    }

    public void setVisible(boolean v) {
	Window win = getAsWindow();
	if(v && !win.isShowing()) {
	    Point p = parent.getLocationOnScreen();
	    Dimension pdim = parent.getSize();
	    Dimension cdim = win.getSize();
	    int x = pdim.width - cdim.width;
	    int y = pdim.height - cdim.height;
	    p.translate(x, y);
	    win.setLocation(p);
	    if (proto != null) {
		try {
		    proto.writeVideoSettingsRequest(proto.VSRequestNow);
		} catch (IOException e) {
		    if (debug) logger.println(T._("Could not send VideoSettings request"));
		}
	    }
	} else if (!v) {
	    try {
		proto.writeVideoSettingsRequest(proto.VSRequestRefuse);
	    } catch (IOException e) {
		if (debug) logger.println(T._("Could not send VideoSettings request"));
	    }
	}
	win.setVisible(v);
	if(!v) win.dispose();	
    }

    public void close() {
	if(myFrame != null) {
	    myFrame.setVisible(false);
	    myFrame.dispose();
	}
    }

    public Window getAsWindow() {
	if (myFrame == null) {
	    while(!(parent instanceof Frame))
		if(null == (parent = parent.getParent())) break;
	    
	    myFrame = new Dialog((Frame)parent);

	    gbl = new GridBagLayout();

	    panels = new Vector();
	    if (type_full == true) {
		panels.addElement(new VSPanel(T._("Brightness"),
					      VSBrightness, 0, 127, 1, 1, this, gbl, 0, 1));
		panels.addElement(new VSPanel(T._("Contrast Red"),
					      VSContrastRed, 0, 255, 1, 1, this, gbl, 1, 1));
		panels.addElement(new VSPanel(T._("Contrast Green"),
					      VSContrastGreen, 0, 255, 1, 1, this, gbl, 2, 1));
		panels.addElement(new VSPanel(T._("Contrast Blue"),
					      VSContrastBlue, 0, 255, 1, 1, this, gbl, 3, 1));
		panels.addElement(new VSPanel(T._("Clock"),
					      VSClock, 0, 4320, 2, 1, this, gbl, 5, 2));
		panels.addElement(new VSPanel(T._("Phase"),
					      VSPhase, 0, 31, 1, 1, this, gbl, 6, 2));
	    }
	    panels.addElement(new VSPanel(T._("Horizontal Offset"),
					  VSXOffset, 0, 512, 1, 1, this, gbl, 7, 1));
	    panels.addElement(new VSPanel(T._("Vertical Offset"),
					  VSYOffset, 0, 128, 1, 1, this, gbl, 8, 1));

	    saveBtn	= new VSButton(T._("Save Changes"),
	                               VSSaveSettings, this, 1);
	    cancelBtn	= new VSButton(T._("Undo Changes"),
	                               VSCancelSettings, this, 1);
	    resetAllBtn	= new VSButton(T._("Reset All Modes"),
	                               VSResetAll, this, 2,
				       T._("Do you really want to reset all settings to their factory setting?"));
	    resetCurBtn	= new VSButton(T._("Reset this Mode"),
	                               VSResetCurrent, this, 2,
				       T._("Do you really want to reset this mode to its factory setting?"));
	    saveBtn.addMouseListener(mouseAdapter);
	    cancelBtn.addMouseListener(mouseAdapter);
	    resetAllBtn.addMouseListener(mouseAdapter);
	    resetCurBtn.addMouseListener(mouseAdapter);
	    setConstraints(gbl, resetCurBtn, 1, 9, 1, 1, GridBagConstraints.BOTH, 10, 10, GridBagConstraints.EAST);
	    setConstraints(gbl, resetAllBtn, 3, 9, 1, 1, GridBagConstraints.BOTH, 10, 10, GridBagConstraints.WEST);
	    setConstraints(gbl, saveBtn, 0, 11, 1, 1, GridBagConstraints.BOTH, 10, 10, GridBagConstraints.WEST);
	    setConstraints(gbl, cancelBtn, 4, 11, 1, 1, GridBagConstraints.BOTH, 10, 10, GridBagConstraints.EAST);
	    myFrame.add(resetAllBtn);myFrame.add(resetCurBtn);myFrame.add(saveBtn);myFrame.add(cancelBtn);

	    resInfoLbl = new Label("", Label.CENTER);
	    setConstraints(gbl, resInfoLbl, 1, 11, 3, 1, GridBagConstraints.BOTH, 10, 10, GridBagConstraints.EAST);
	    myFrame.add(resInfoLbl);
	    
	    myFrame.setLayout(gbl);
	    myFrame.setBackground(SystemColor.scrollbar);
	    myFrame.setTitle(T._("Video Settings"));
	    myFrame.pack();
	    myFrame.addWindowListener(new WindowAdapter() {
		    public void windowClosing(WindowEvent e) {
			setVisible(false);
		    }
		});

	    if (proto != null) updateVideoSettings(proto.vs);
	}

	return myFrame;
    }

    public void setRFBproto(RFBproto proto) {
	this.proto = proto;
    }

    public void setMonitorMode(boolean mode) {
	this.monitorMode = mode;
    }

    public void updateVideoSettings(VideoSettings vs) {
	if (myFrame == null) return;
	VSPanel panel;

	panel = getPanel(VSBrightness);
	if (panel != null) panel.update(vs.brightness);
	panel = getPanel(VSContrastRed);
	if (panel != null) panel.update(vs.contrast_red);
	panel = getPanel(VSContrastGreen);
	if (panel != null) panel.update(vs.contrast_green);
	panel = getPanel(VSContrastBlue);
	if (panel != null) panel.update(vs.contrast_blue);
	panel = getPanel(VSClock);
	if (panel != null) {
	    panel.updateMinMax(vs.clock-50, vs.clock+50);
	    panel.update(vs.clock);
	}
	panel = getPanel(VSPhase);
	if (panel != null) panel.update(vs.phase);

	panel = getPanel(VSXOffset);
	if (panel != null) panel.update(vs.x_offset);

	panel = getPanel(VSYOffset);
	if (panel != null) {
	    panel.updateMinMax(0, vs.y_max_offset);
	    panel.update(vs.y_offset);
	}

	resInfoLbl.setText(MessageFormat.format(T._("{0}x{1} {2}Hz"),
	                   new Object[] { new Integer(vs.x_res), new Integer(vs.y_res),
	                   new Integer(vs.refresh)} ));
	resInfoLbl.setAlignment(Label.CENTER);
    }

    public void sendEvent(byte event, short value)
    {
	if (null != proto) {
	    try {
		proto.writeVideoSettingsEvent(event, value);
	    } catch (IOException e) {
		if (debug) logger.println(T._("Could not send VideoSettings event"));
	    }
	}
    }

    private VSPanel getPanel(int event)
    {
	for (Enumeration el=panels.elements(); el.hasMoreElements(); ) {
	    VSPanel panel = (VSPanel) el.nextElement();
	    if (panel.getEventType() == event) return panel;
	}
	return null;
    }
    
    private void setConstraints(GridBagLayout gbl, Component obj, int gridx, int gridy, int gridwidth,
				int gridheight, int fill, double weightx, double weighty, int anchor) {
	GridBagConstraints gbc = new GridBagConstraints();
	
	gbc.gridx = gridx; gbc.gridy = gridy; gbc.gridwidth = gridwidth; gbc.gridheight = gridheight;
	gbc.fill = fill; gbc.weightx = weightx; gbc.weighty = weighty; gbc.anchor = anchor;
	gbl.setConstraints(obj, gbc);
    }

    class VSMouseAdapter extends MouseAdapter {
	public void mouseClicked(MouseEvent event) {
	    VSButton btnSource = (VSButton) event.getSource();

	    if (monitorMode) return;

	    if (btnSource.getConfirm()!=null) {
		ConfirmDialog d = new ConfirmDialog(btnSource.getConfirm(),
		                                    T._("Confirmation"), ConfirmDialog.OK_CANCEL, true);
		d.show();
		boolean ok = d.ok;
		d.dispose();
		d = null;
		if(ok == false) return;
	    }
	    
	    if (null != proto) {
		try {
		    proto.writeVideoSettingsEvent(btnSource.getEvent(),
						  (short)0);
		} catch (IOException e) {
		    if (debug) logger.println(T._("Could not send VideoSettings event"));
		}
	    }
	}
    }

    class VSPanel implements AdjustmentListener {
	private MyScrollbar scroll;
	private Label label;
	private MyLabel value;
	private byte event;
	private VideoSettingsFrame parent;
	private int perm_level;

	public VSPanel(String name, byte event, int min, int max, int step,
		       int active, VideoSettingsFrame parent, GridBagLayout gbl,
		       int ypos, int perm_level) 
	{
            this.event = event;
	    this.parent= parent;
	    this.perm_level = perm_level;
	    
	    // create labels and scrollbar
	    label = new Label(name,  Label.CENTER);	    
	    value = new MyLabel(T._("n/a"), Label.CENTER);   
	    scroll= new MyScrollbar(Scrollbar.HORIZONTAL, 0, 10, min, max+10);
	    scroll.addAdjustmentListener(this);
	    scroll.setEnabled(false);
	    scroll.setUnitIncrement(step);

	    setConstraints(gbl, label, 0, ypos, 1, 1, GridBagConstraints.BOTH, 0, 0, GridBagConstraints.WEST);
	    setConstraints(gbl, scroll, 1, ypos, 3, 1, GridBagConstraints.HORIZONTAL, 10, 10, GridBagConstraints.CENTER);
	    setConstraints(gbl, value, 4, ypos, 1, 1, GridBagConstraints.BOTH, 0, 0, GridBagConstraints.EAST);

	    parent.myFrame.add(label); parent.myFrame.add(scroll); parent.myFrame.add(value);
        }

	public void adjustmentValueChanged(AdjustmentEvent ae)
	{
	    short newval = (short) ae.getValue();
	    sendEvent(event, newval);
	    value.setText(Integer.toString(newval));
	}

	public void update(int newval) {
	    scroll.setValue(newval);
	    value.setText(Integer.toString(newval));
	    if (parent.perms >= perm_level) {
		scroll.setEnabled(true);
	    }
	}

	public void updateMinMax(int min, int max) {
	    scroll.setMinimum(min);
	    scroll.setMaximum(max+10);
	}

	public int getEventType() {
	    return this.event;
	}
	
	class MyScrollbar extends Scrollbar {
	    public MyScrollbar(int orientation, int value, int visible,
			     int minimum, int maximum) {
		super(orientation, value, visible, minimum, maximum);
	    }   
	}	

	class MyLabel extends Label {
	    public MyLabel(String text, int alignment) {
		super(text, alignment);
	    }   
	}
    }

    class VSButton extends Button {
	private byte	event;
	private String  confirm=null;
	private VideoSettingsFrame parent;
	private int perm_level;
	
	public VSButton(String label, byte event, VideoSettingsFrame parent, int perm_level)
	{
	    super(label);
	    this.event = event;
	    this.parent= parent;
	    this.perm_level = perm_level;
	    evalPerm();
	}

	public VSButton(String label, byte event, VideoSettingsFrame parent, int perm_level,
			String confirm)
	{
	    super(label);
	    this.event = event;
	    this.parent= parent;
	    this.confirm = confirm;
	    this.perm_level = perm_level;
	    evalPerm();
	}

	private void evalPerm()
	{
	    if (parent.perms < perm_level) {
		this.setEnabled(false);
	    } else {
		this.setEnabled(true);
	    }
	}
	
	public byte getEvent()
	{
	    return event;
	}

	public String getConfirm()
	{
	    return confirm;
	}

    }
}
