package nn.pp.rcrdp;

class Cache {
    private boolean debug = false;
    
    // the (private) cache data
    private RDPBitmap bmpCache[][];
    private int[] bmpIntCache[][];
    private RDPFontGlyph fontCache[][];
    private RDPDataBlob textCache[];
    private RDPCursor cursorCache[];
    private int deskCache[];
    
    // constructor - initialize the cache data
    public Cache() {
    	bmpCache = new RDPBitmap[3][600];
    	bmpIntCache = new int[3][600][];	// same dimension as bmpCache !
    	fontCache = new RDPFontGlyph[12][256];
    	textCache = new RDPDataBlob[256];
    	cursorCache = new RDPCursor[32];
    	deskCache = new int[0x38400];
    }
    
    // get and reallocate (if necessary) a bitmap int array
    public int[] getBitmapInt(int cacheID, int cacheIndex, int size) {
    	if(bmpIntCache[cacheID][cacheIndex] == null || bmpIntCache[cacheID][cacheIndex].length < size) {
    	    bmpIntCache[cacheID][cacheIndex] = new int[size];
    	}    	
    	return bmpIntCache[cacheID][cacheIndex];
    }
    
    // Retrieve a bitmap from the cache
    public RDPBitmap getBitmap(int cacheID, int cacheIndex) {
    	RDPBitmap ret = null;
    	
    	if((cacheID < bmpCache.length) && (cacheIndex < bmpCache[0].length)) {
    	    ret = bmpCache[cacheID][cacheIndex];
    	    if(ret != null) {
    	    	return ret;
    	    }
    	}
    	
    	System.out.println("RDP Cache: unable to get bitmap");
    	return null;
    }
    
    // Store a bitmap in the cache
    public void putBitmap(int cacheID, int cacheIndex, RDPBitmap bitmap) {
    	if((cacheID < bmpCache.length) && (cacheIndex < bmpCache[0].length)) {
    	    if(bmpCache[cacheID][cacheIndex] != null) {
    	    	bmpCache[cacheID][cacheIndex].dispose();
    	    }
    	    bmpCache[cacheID][cacheIndex] = bitmap;
    	}
    	else {
    	    System.out.println("RDP Cache: unable to put bitmap");
    	}
    }
    
    // Retrieve a glyph from the font cache
    public RDPFontGlyph getFont(int font, int character) {
    	RDPFontGlyph glyph = null;
    	
    	if((font < fontCache.length) && (character < fontCache[0].length)) {
    	    glyph = fontCache[font][character];
    	    if(glyph == null) {
    	    	return null;
    	    }
    	    if(glyph.getPixMap() != null) {
    	    	return glyph;
    	    }
    	}
    	System.out.println("RDP Cache: unable to get font");
    	return null;
    }
    
    // Store a glyph in the font cache
    public void putFont(int font, int character, RDPFontGlyph glyph) {
    	if((font < fontCache.length) && (character < fontCache[0].length)) {
    	    fontCache[font][character] = glyph;
    	}
    	else {
    	    System.out.println("RDP Cache: unable to put font");
    	}
    }
    
    // Retrieve a text item from the cache
    public RDPDataBlob getText(int cacheID) {
    	RDPDataBlob text = null;
    	
    	if(cacheID < textCache.length) {
    	    text = textCache[cacheID];
    	    byte[] d = text.getData();
    	    if(d != null) {
    	    	if(debug) {
    	    	    int l = text.getSize();
    	    	    System.out.println("RDP Cache getText: cacheID = " + cacheID + ", length = " + l);
    	    	    for(int i = 0; i < l; i++) {
    	    	    	System.out.print("0x" + Integer.toHexString(d[i] & 0xff) + " ");
    	    	    }
    	    	    System.out.println();
    	    	}
    	    	return text;
    	    }
    	}
    	System.out.println("RDP Cache: unable to get text");
    	return null;
    }
    
    // Store a text item in the cache
    public void putText(int cacheID, RDPDataBlob text) {
    	if(cacheID < textCache.length) {
    	    textCache[cacheID] = text;
    	    if(debug) {
    	    	byte[] d = text.getData();
    	    	int l = text.getSize();
    	    	System.out.println("RDP Cache putText: cacheID = " + cacheID + ", length = " + l);
    	    	for(int i = 0; i < l; i++) {
    	    	    System.out.print("0x" + Integer.toHexString(d[i] & 0xff) + " ");
    	    	}
    	    	System.out.println();
    	    }
    	}
    	else {
    	    System.out.println("RDP Cache: unable to put text");
    	}
    }
    
    // Retrieve desktop data from the cache
    public void getDesktop(int dest[], int destCx, int offset, int x, int y, int cx, int cy) {
    	int len = cx * cy;
    	
    	if((offset + len) <= deskCache.length) {
    	    int indexSrc = offset;
    	    int indexDst = y * destCx + x;
    	    
    	    for(int i = 0; i < cy; i++) {
    	    	System.arraycopy(deskCache, indexSrc, dest, indexDst, cx);
    	    	indexSrc += cx;
    	    	indexDst += destCx;
    	    }
	}
	else {
	    System.out.println("RDP Cache: unable to get desktop");
	}
    }
    
    // Store desktop data in the cache
    public void putDesktop(int src[], int srcCx, int offset, int x, int y, int cx, int cy) {
    	int len = cx * cy;
    	
    	if((offset + len) <= deskCache.length) {
    	    int indexSrc = y * srcCx + x;
    	    int indexDst = offset;
    	    
    	    for(int i = 0; i < cy; i++) {
    	    	System.arraycopy(src, indexSrc, deskCache, indexDst, cx);
    	    	indexSrc += srcCx;
    	    	indexDst += cx;
    	    }
	}
	else {
	    System.out.println("RDP Cache: unable to put desktop");
	}
    }
    
    // Retrieve cursor from cache
    public RDPCursor getCursor(int cacheIndex) {
    	RDPCursor cursor = null;
    	
    	if(cacheIndex < cursorCache.length) {
    	    cursor = cursorCache[cacheIndex];
    	    if(cursor != null) {
    	    	return cursor;
    	    }
    	}
    	
    	System.out.println("RDP Cache: unable to get cursor");
    	return null;
    }
    
    // Store cursor in cache
    public void putCursor(int cacheIndex, RDPCursor cursor) {
    	if(cacheIndex < cursorCache.length) {
    	    cursorCache[cacheIndex] = cursor;
    	}
    	else {
    	    System.out.println("RDP Cache: unable to put cursor");
    	}
    }
}
