package nn.pp.rcrdp;

import java.io.*;
import java.net.*;
import java.text.*;

import nn.pp.rc.T;

/**
 * class MCSLayer
 *
 * the MCS-Layer of the MS RDP protocol
 * (Multipoint Communications Service)
 */
 
public class MCSLayer {
    private static final int sizeOfMCSHdr = 8;
    
    private int mcsUserID = 0;
    
    private ISOLayer isoLayer;
    private SecureLayer secureLayer;
    
    private boolean debug = false;
    
    // internal functions
    
    // debugging
    private void debug(String s) {
    	if(debug) System.out.println(s);
    }
    
    // Parse an ASN.1 BER header
    private int berParseHeader(RDPStream s, int tagval) throws IOException {
    	int tag, len, retLen = 0;
    	
    	if(tagval > 0xff) {
    	    tag = s.inUInt16BE();
	}
	else {
	    tag = s.inUInt8();
	}
	
	if(tagval != tag) {
	    throw new IOException(MessageFormat.format(
	                          T._("berParseHeader: wrong tag received; expected {0}, got {1}"),
	                          new Object[] { new Integer(tagval), new Integer(tag) }));
	}
	
	len = s.inUInt8();
	
	if((len & 0x80) != 0) {
	    len &= ~0x80;	//substract 128
	    retLen = 0;
	    while(len-- != 0) {
		retLen = s.nextBE(retLen);
	    }
	}
	else {
	    retLen = len;
	}
	
	return retLen;
    }
    
    // output an ANS.1 BER header
    private void berOutHeader(RDPStream s, int tagval, int len) throws IOException {
    	if(tagval > 0xff) {
    	    s.outUInt16BE(tagval);
    	}
    	else {
    	    s.outUInt8(tagval);
    	}
    	
    	if(len > 0x80) {
    	    s.outUInt8(0x80 | 2);	// 0x80 for a combined length, 2 for 2 bytes
    	    s.outUInt16BE(len);
    	}
    	else {
    	    s.outUInt8(len);
    	}
    }
    
    // Output an ASN.1 BER integer
    private void berOutInteger(RDPStream s, int val) throws IOException {
    	berOutHeader(s, RDPConstants.berTagInteger, 2);	// 2 bytes
    	s.outUInt16BE(val);
    }
    
    // Output a DOMAIN_PARAMS structure (ASN.1 BER)
    private void outDomainParams(RDPStream s, int maxChannels, int maxUsers, int maxTokens, int maxPDUSize) throws IOException {
	berOutHeader(s, RDPConstants.mcsTagDomainParams, 32);
	berOutInteger(s, maxChannels);
	berOutInteger(s, maxUsers);
	berOutInteger(s, maxTokens);
	berOutInteger(s, 1);	/* num_priorities */
	berOutInteger(s, 0);	/* min_throughput */
	berOutInteger(s, 1);	/* max_height */
	berOutInteger(s, maxPDUSize);
	berOutInteger(s, 2);	/* ver_protocol */
    }
    
    // Parse a DOMAIN_PARAMS structure (ASN.1 BER)
    private void parseDomainParams(RDPStream s) throws IOException {
    	int length = berParseHeader(s, RDPConstants.mcsTagDomainParams);
    	s.incrementPosition(length);
    }
    
    // Send an MCS_CONNECT_INITIAL message (ASN.1 BER)
    private void sendConnectInitial(RDPStream mcsData) throws IOException {
    	int datalen = mcsData.getEnd();
    	int len = 9 + 3 * 34 + 4 + datalen;
    	
    	RDPStream s = isoLayer.init(len + 5);
    	
    	berOutHeader(s, RDPConstants.mcsConnectInitial, len);
    	berOutHeader(s, RDPConstants.berTagOctetString, 1);	//calling domain
    	s.outUInt8(1);
    	berOutHeader(s, RDPConstants.berTagOctetString, 1);	//called domain
    	s.outUInt8(1);
    	
    	berOutHeader(s, RDPConstants.berTagBoolean, 1);
    	s.outUInt8(0xff);					//upward flag
    	
    	outDomainParams(s, 34, 2, 0, 0xffff);			// target params
    	outDomainParams(s, 1, 1, 1, 0x420);			// min params
    	outDomainParams(s, 0xffff, 0xfc17, 0xffff, 0xffff);	// max params
    	
    	berOutHeader(s, RDPConstants.berTagOctetString, datalen);
    	s.outUInt8Array(mcsData.getPacket(), 0, datalen);
    	
    	s.markEnd();
    	isoLayer.send(s);
    }
    
    // Expect a MCS_CONNECT_RESPONSE message (ASN.1 BER)
    private void recvConnectResponse(RDPStream mcsData) throws IOException {
    	int result, len;
    	RDPStream s;
    	
    	s = isoLayer.receive();
    	
    	len = berParseHeader(s, RDPConstants.mcsConnectResponse);
    	len = berParseHeader(s, RDPConstants.berTagResult);
    	
    	result = s.inUInt8();
    	
    	if(result != 0) {
    	    throw new IOException(T._("MCS recvConnectResponse: wrong result"));
    	}
    	
    	len = berParseHeader(s, RDPConstants.berTagInteger);
    	s.incrementPosition(len);	// connect ID
    	parseDomainParams(s);
    	
    	len = berParseHeader(s, RDPConstants.berTagOctetString);
    	secureLayer.processMCSData(s);
    }
    
    // Send an EDrq (Erect Domain Request) message (ASN.1 PER)
    private void sendEDRQ() throws IOException {
    	RDPStream s = isoLayer.init(5);
    	
    	s.outUInt8(RDPConstants.mcsEDRQ << 2);
    	s.outUInt16BE(1);	//height
    	s.outUInt16BE(1);	//width
    	
    	s.markEnd();
    	isoLayer.send(s);
    }
    
    // Send an AUrq (Attach User Request) message (ASN.1 PER)
    private void sendAURQ() throws IOException {
    	RDPStream s = isoLayer.init(1);
    	s.outUInt8(RDPConstants.mcsAURQ << 2);
    	s.markEnd();
    	isoLayer.send(s);
    }
    
    // Expect a AUcf (Attach User Confirm) message (ASN.1 PER)
    private int recvAUCF() throws IOException {
    	int opcode, result, userID = 0;
    	RDPStream s = isoLayer.receive();
    	
    	opcode = s.inUInt8();
    	if((opcode >> 2) != RDPConstants.mcsAUCF) {
    	    throw new IOException(T._("MCS recvAUCF: wrong opcode"));
    	}
    	
    	result = s.inUInt8();
    	if(result != 0) {
    	    throw new IOException(T._("MCS recvAUCF: wrong result"));
    	}
    	
    	if((opcode & 2) != 0) {
    	    userID = s.inUInt16BE();
    	}
    	
    	return userID;
    }
    
    // Send a CJrq (Channel Join Request) message (ASN.1 PER)
    private void sendCJRQ(int chanID) throws IOException {
    	RDPStream s = isoLayer.init(5);
    	
    	s.outUInt8(RDPConstants.mcsCJRQ << 2);
    	s.outUInt16BE(mcsUserID);
    	s.outUInt16BE(chanID);
    	
    	s.markEnd();
    	isoLayer.send(s);
    }
    
    // Expect a CJcf (Channel Join Confirm) message (ASN.1 PER)
    private void recvCJCF() throws IOException {
    	int opcode, result;
    	RDPStream s = isoLayer.receive();
    	
    	opcode = s.inUInt8();
    	if((opcode >> 2) != RDPConstants.mcsCJCF) {
    	    throw new IOException(T._("MCS recvCJCF: wrong opcode"));
    	}
    	
    	result = s.inUInt8();
    	if(result != 0) {
    	    throw new IOException(T._("MCS recvCJCF: wrong result"));
    	}
    	
    	s.incrementPosition(4);		//mcs_userid, req_chanid
    	if((opcode & 2) != 0) {
    	    s.incrementPosition(2);	//join_chanid
    	}
    }    
    
    // public functions
    
    // constructor
    public MCSLayer(SecureLayer sec) {
    	isoLayer = new ISOLayer();
    	secureLayer = sec;
    }
    
    // get the particular layers
    public ISOLayer getISOLayer() {
    	return isoLayer;
    }
    
    public TCPLayer getTCPLayer() {
    	return isoLayer.getTCPLayer();
    }
    
    // Initialise an MCS transport data packet
    public RDPStream init(int maxlen) {
    	RDPStream s = isoLayer.init(maxlen + sizeOfMCSHdr);
    	s.pushLayer(RDPStream.mcsHdrID, sizeOfMCSHdr);
    	
    	return s;
    }
    
    public RDPStream initNew(int maxlen) {
    	RDPStream s = isoLayer.initNew(maxlen + sizeOfMCSHdr);
    	s.pushLayer(RDPStream.mcsHdrID, sizeOfMCSHdr);
    	
    	return s;
    }
    
    // Establish a connection up to the MCS layer
    public void connect(String server, int port, RDPStream mcsData, String username) throws IOException {
    	try {
    	    isoLayer.connect(server, port, username);
    	    sendConnectInitial(mcsData);
    	    recvConnectResponse(mcsData);
    	    sendEDRQ();
    	    sendAURQ();
    	    int userID = recvAUCF();
    	    mcsUserID = userID;
    	    sendCJRQ(mcsUserID + 1001);
    	    recvCJCF();
    	    sendCJRQ(RDPConstants.mcsGlobalChannel);
    	    recvCJCF();
	}
	catch (IOException e) {
	    isoLayer.disconnect();
	    throw new IOException(T._("MCS connect: unable to connect") + " " + e);
	}
    }
    
    // Disconnect from the MCS layer
    public void disconnect() {
    	isoLayer.disconnect();
    }
    
    // Send an MCS transport data packet
    public void send(RDPStream s) throws IOException {
    	int len;
    	
    	s.popLayer(RDPStream.mcsHdrID);
    	len = s.getEnd() - s.getPosition() - sizeOfMCSHdr;
    	len |= 0x8000;
    	
    	s.outUInt8(RDPConstants.mcsSDRQ << 2);
    	s.outUInt16BE(mcsUserID);
    	s.outUInt16BE(RDPConstants.mcsGlobalChannel);  
    	s.outUInt8(0x70); //flags
    	s.outUInt16BE(len);
    	
    	isoLayer.send(s);
    }
    
    // Receive an MCS transport data packet
    public RDPStream receive() throws IOException {
    	RDPStream s = null;
    	
    	int opcode, appid, len;
    	
    	s = isoLayer.receive();
    	if(s == null) return null;
    	
    	opcode = s.inUInt8();		//opcode
    	appid = opcode >> 2;
    	if(appid != RDPConstants.mcsSDIN) {
    	    if(appid != RDPConstants.mcsDPUM) {
    	    	throw new IOException(T._("MCS receive: expected data"));
    	    }
    	    throw new IOException(T._("RDP: Connection closed"));
	}
	
	s.incrementPosition(5);		//userid, channel, flags
	len = s.inUInt8();		//length
	if((len & 0x80) != 0) {
	    s.incrementPosition(1);	//second byte of length
	}
    	
    	return s;
    }
    
    // get the MCS Layer internal userID
    public int getUserID() {
    	return mcsUserID;
    }
    
}
