package nn.pp.rcrdp;

import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import java.io.*;
import java.net.*;
import java.beans.*;
import java.text.*;
import java.util.*;
import nn.pp.rc.*;

public class RDPProto implements RCProto, ActionListener {
    private boolean debug = false;
    
    public final static int keycodePressed         = 0x80;
    public final static int scancodeReleased       = 0x80;

    private RDPHandler handler;
    private RDPProfile rdpProfile;
    private PrintStream logger;
    private Component panel;
    boolean inNormalProtocol = false;
    private int framebufferWidth = 800;
    private int framebufferHeight = 600;
    private RDPRenderer rdr;
    
    private RDPStream stream = null;
    private int nextPacket = 0;
    private int receivedType = 0;
    private int shareID = 0;
    private String username;
    private String hostname;
    
    private Orders orders;
    private Cache cache;
    private BitmapDecompressor bitmapDecompressor;
    
    private SecureLayer secureLayer = null;
    
    private boolean mouseRobotMove = false;
    
    private boolean killMe = false;
    private GenericTimer observeTimer;
    
    public boolean isActive = false;
    
    private byte[] buffer = null;
    private int[] intBuffer = null;
    
    boolean fps = false;
    StopWatch updateWatch;
    int updateCount = 0;
    
    // Constructor
    public RDPProto(RDPHandler handler, RDPProfile rdpProfile, RDPRenderer rdr, Cache cache, PrintStream logger, Component panel, int width, int height, int kbdLayout, String username) {
    	
        if(username == null) {
            this.username = "";
        }
        else {
            this.username = username;
        }

    	// get the hostname of the client host
    	// FIXME: should this be the erla's name or the client's?
    	// I think it should be erla's name, but I leave it here in case
    	// someone has a different opinion
        //try {
            //InetAddress localhost = InetAddress.getLocalHost();
            //hostname = localhost.getHostName();
        //}
        //catch (Exception e) {
        //    hostname = "unknown";
        //}
        
        this.handler = handler;
    	this.logger = logger;
    	this.rdpProfile = rdpProfile;
    	this.rdr = rdr;
    	this.cache = cache;
    	this.panel = panel;
    	framebufferWidth = width;
    	framebufferHeight = height;
    	secureLayer = new SecureLayer(rdpProfile, this, kbdLayout);
    	bitmapDecompressor = new BitmapDecompressor();
    	bitmapDecompressor.setColorTranslator(rdr.getColorTranslator());
    	orders = new Orders(cache, bitmapDecompressor, rdr, panel);
    	observeTimer = new GenericTimer(5000, this);
    	
	updateWatch = new StopWatch();
	
    	inNormalProtocol = true;
    }
    
    // help functions
    private void debug(String s) {
    	if(debug) System.out.println(s);
    }
    
    // get and allocate (if necessary) the internal buffer
    private byte[] getBuffer(int size) {
    	if(buffer == null || buffer.length < size) {
    	    buffer = new byte[size];
    	}
    	
    	return buffer;
    }
    
    private int[] getIntBuffer(int size) {
    	if(intBuffer == null || intBuffer.length < size) {
    	    intBuffer = new int[size];
    	}
    	
    	return intBuffer;
    }
    
    public void switchFPS() {
    	fps = !fps;
    }

    public int getTime() {
    	Date date = new Date();
    	long ms = date.getTime();
    	return (int) (ms/1000);
    }
    
    public void setKbdKayout(int kbdLayout) {
    	secureLayer.setKbdKayout(kbdLayout);
    }
    
    public int getWidth() {
    	return framebufferWidth;
    }
    
    public int getHeight() {
    	return framebufferHeight;
    }
    
    public void setSize(int width, int height) {
    	framebufferWidth = width;
    	framebufferHeight = height;
    }
    
    // get the particular layers
    public SecureLayer getSecureLayer() {
    	return secureLayer;
    }
    
    public MCSLayer getMCSLayer() {
    	return secureLayer.getMCSLayer();
    }
    
    public ISOLayer getISOLayer() {
    	return secureLayer.getISOLayer();
    }
    
    public TCPLayer getTCPLayer() {
    	return secureLayer.getTCPLayer();
    }
    
    // the response function for our ping pong timer
    public void actionPerformed(ActionEvent e) {
	if(getTCPLayer().getAndClearInObserver() == 0) {
	    // no data received since last call
	    if(!killMe) {
	    	// send a ping message to the server, which should response
	    	try {
	    	    sendControlThreadsafe(RDPConstants.rdpCtlRequestControl);
	    	}
	    	catch (IOException ex) {
	    	    System.out.println("cannot send ping message, killing RDP worker thread");
	    	    // kill the thread
	    	    handler.disconnect();
	    	}
	    	killMe = true;
	    }
	    else {
	    	System.out.println("no response on ping message, killing RDP worker thread");
	    	// kill the thread
	    	handler.disconnect();
	    }
	}
	else {
	    // data received since last call, everything is fine
	    killMe = false;
	}
    }

    // wrapper function for mouse events
    public void writePointerEvent(boolean relative,
				  int x, int y, int z, int pointerMask) throws IOException {
    	if(mouseRobotMove) {
    	    // this is a robot move, don't send it
    	    setMouseRobotMove(false);	// next event should be sent;
    	}
    	else {
    	    sendInput(getTime(), RDPConstants.rdpInputMouse, pointerMask, x, y);
    	}
    }
    
    public void writeMouseSyncEvent(byte subevent) throws IOException { }
    
    // wrapper function for keyboard events which are handled
    // by the keyboard code in the applet
    public void writeKeyboardEvent(byte keycode) throws IOException {
    	// keycode is the code we get from the applet's key translator;
    	// we have to convert it into scancodes
    	boolean down;
    	int evtime = getTime();
    	
    	if((keycode & keycodePressed) != 0) {
    	    down = true;
    	    keycode &= ~keycodePressed;
    	}
    	else {
    	    down = false;
    	}
    	
    	int scancode[] = Scantable.getScancode(keycode, down);
    	
    	if(scancode == null) {
    	    logger.println(T._("RDP: unable to determine key scancode"));
    	    return;
    	}
    	
    	if(keycode == 74) {
	    /* According to MS Keyboard Scan Code
	       Specification, pressing Pause should result
	       in E1 1D 45 E1 9D C5. I'm not exactly sure
	       of how this is supposed to be sent via
	       RDP. The code below seems to work, but with
	       the side effect that Left Ctrl stays
	       down. Therefore, we release it when Pause
	       is released. */
    	    if(down) {
    		for(int i = 0; i < scancode.length; i++) {
		    sendInput(evtime, RDPConstants.rdpInputScancode, RDPConstants.rdpKeyPressed, scancode[i], 0);
	 	}
	    }
	    else {
	    	sendInput(evtime, RDPConstants.rdpInputScancode, RDPConstants.rdpKeyReleased, 0x1D, 0);
	    }
	    return;
    	}
    	
    	if(scancode[0] == 0) { //no "real" key
    	    return;
    	}
    	
    	int flags = 0;
    	
    	if(scancode.length == 1) {
    	    if(down) {
    		sendInput(evtime, RDPConstants.rdpInputScancode, RDPConstants.rdpKeyPressed, (scancode[0] & ~scancodeReleased), 0);
    	    }
    	    else {
    		sendInput(evtime, RDPConstants.rdpInputScancode, RDPConstants.rdpKeyReleased, (scancode[0] & ~scancodeReleased), 0);
    	    }    	
    	}
    	else if(scancode[0] == 0xE0) {
    	    if(down) {
    		sendInput(evtime, RDPConstants.rdpInputScancode, RDPConstants.rdpKeyPressed | RDPConstants.keyFlagExt, (scancode[1] & ~scancodeReleased), 0);
    	    }
    	    else {
    		sendInput(evtime, RDPConstants.rdpInputScancode, RDPConstants.rdpKeyReleased | RDPConstants.keyFlagExt, (scancode[1] & ~scancodeReleased), 0);
    	    }    	
    	}
    	
    }
    
    // private functions
    
    // robot mouse move
    private void robotMouseMove(int x, int y) {
    	if(rdpProfile.isJ14) {
    	    try {
    	    	// tell the mouse handler not to send the event
    	    	Robot robot = new Robot();
    	    	setMouseRobotMove(true);
    	    	Point pos = panel.getLocationOnScreen();
    	    	robot.mouseMove(pos.x + x, pos.y + y);
    	    } catch (Exception e) {
    	    	// ignore
    	    }
    	}
    }
    
    // Receive an RDP packet
    private RDPStream receive() throws IOException {
    	int len, pduType;
    	
    	if((stream == null) || (nextPacket >= stream.getEnd())) {
    	    stream = secureLayer.receive();
    	    if(stream == null) return null;
    	    
    	    nextPacket = stream.getPosition();
    	}
    	else {
    	    stream.setPosition(nextPacket);
    	}
    	
    	len = stream.inUInt16LE();
    	// 32k packets are really 8, keepalive fix
    	if(len == 0x8000) {
    	    nextPacket += 8;
    	    receivedType = 0;
    	    return stream;
    	}
    	pduType = stream.inUInt16LE();
    	// sometimes (XP) these two bytes are missing and we don't want to get an
    	// ArrayIndexOutOfBoundException here
    	if(stream.checkRemaining(2)) {
    	    stream.incrementPosition(2);	// user ID
    	}
    	else {
    	    stream.markEnd();
    	}
    	receivedType = pduType & 0xf;
    	
    	nextPacket += len;
    	return stream;
    }
    
    // Initialise an RDP data packet
    public RDPStream initData(int maxlen) {
    	RDPStream s = secureLayer.init(rdpProfile.encryption ? RDPConstants.secEncrypt : 0, maxlen + 18);
    	s.pushLayer(RDPStream.rdpHdrID, 18);
    	
    	return s;
    }
        
    public RDPStream initDataNew(int maxlen) {
    	RDPStream s = secureLayer.initNew(rdpProfile.encryption ? RDPConstants.secEncrypt : 0, maxlen + 18);
    	s.pushLayer(RDPStream.rdpHdrID, 18);
    	
    	return s;
    }
        
    // Send an RDP data packet
    private void sendData(RDPStream s, int pduType) throws IOException {
    	int len;
    	
    	s.popLayer(RDPStream.rdpHdrID);
    	len = s.getEnd() - s.getPosition();
    	
    	s.outUInt16LE(len);
    	s.outUInt16LE(RDPConstants.rdpPDUData | 0x10);
    	s.outUInt16LE(secureLayer.getMCSUserID() + 1001);
    	
    	s.outUInt32LE(shareID);
    	s.outUInt8(0);		// pad
    	s.outUInt8(1);		// stream ID
    	s.outUInt16LE(len - 14);
    	s.outUInt8(pduType);
    	s.outUInt8(0);		// compress type
    	s.outUInt16(0);		// compress length
    	
    	secureLayer.send(s, rdpProfile.encryption ? RDPConstants.secEncrypt : 0);
    }
    
    // Send a logon info packet
    private void sendLogonInfo(int flags, String domain, String user, String password, String program, String directory) throws IOException {
    	int lenDomain = 2 * domain.length();
    	int lenUser = 2 * user.length();
    	int lenPassword = 2 * password.length();
    	int lenProgram = 2 * program.length();
    	int lenDirectory = 2 * directory.length();
    	int secFlags = rdpProfile.encryption ? (RDPConstants.secLogonInfo | RDPConstants.secEncrypt) : RDPConstants.secLogonInfo;
    	
    	RDPStream s = secureLayer.init(secFlags,  18 + lenDomain + lenUser + lenPassword
			     + lenProgram + lenDirectory + 10);
	s.outUInt32(0);
	s.outUInt32LE(flags);
	s.outUInt16LE(lenDomain);
	s.outUInt16LE(lenUser);
	s.outUInt16LE(lenPassword);
	s.outUInt16LE(lenProgram);
	s.outUInt16LE(lenDirectory);
	s.outUnicodeString(domain, lenDomain);
	s.outUnicodeString(user, lenUser);
	s.outUnicodeString(password, lenPassword);
	s.outUnicodeString(program, lenProgram);
	s.outUnicodeString(directory, lenDirectory);
	
	s.markEnd();
	secureLayer.send(s, secFlags);
    }
    
    // Send a control PDU
    private void sendControl(int action) throws IOException {
    	RDPStream s = initData(8);
    	s.outUInt16LE(action);
    	s.outUInt16LE(0);	// user id
    	s.outUInt32LE(0);	// control id
    	
    	s.markEnd();
    	sendData(s, RDPConstants.rdpDataPDUControl);
    }
    
    private void sendControlThreadsafe(int action) throws IOException {
    	RDPStream s = initDataNew(8);
    	s.outUInt16LE(action);
    	s.outUInt16LE(0);	// user id
    	s.outUInt32LE(0);	// control id
    	
    	s.markEnd();
    	sendData(s, RDPConstants.rdpDataPDUControl);
    }
    
    // Send a synchronisation PDU
    private void sendSynchronize() throws IOException {
   	RDPStream s = initData(4);
    	s.outUInt16LE(1);	// type
    	s.outUInt16LE(1002);
    	
    	s.markEnd();
    	sendData(s, RDPConstants.rdpDataPDUSynchronise);
    }
    
    // Send an (empty) font information PDU
    private void sendFonts(int seq) throws IOException {
    	RDPStream s = initData(8);
    	
    	s.outUInt16(0);		// number of fonts
    	s.outUInt16LE(0x3e);	// unknown
    	s.outUInt16LE(seq);	// unknown
    	s.outUInt16LE(0x32);	// entry size
    	
    	s.markEnd();
    	sendData(s, RDPConstants.rdpDataPDUFont2);
    }
    
    // Output general capability set
    private void outGeneralCaps(RDPStream s) {
    	s.outUInt16LE(RDPConstants.rdpCapsetGeneral);
    	s.outUInt16LE(RDPConstants.rdpCaplenGeneral);
    	
    	s.outUInt16LE(1);	// OS major type
    	s.outUInt16LE(3);	// OS minor type
    	s.outUInt16LE(0x200);	// Protocol version
    	s.outUInt16(0);		// pad
    	s.outUInt16(0);		// compression type
    	s.outUInt16(0);		/* Pad, according to T.128. 0x40d seems to 
				   trigger
				   the server to start sending RDP5 packets. 
				   However, the value is 0x1d04 with W2KTSK and
				   NT4MS. Hmm.. Anyway, thankyou, Microsoft,
				   for sending such information in a padding 
				   field.. */
	s.outUInt16(0);		// update capability
	s.outUInt16(0);		// Remote unshare capability
	s.outUInt16(0);		// Compression level
	s.outUInt16(0);		// pad
    }
    
    // Output bitmap capability set
    private void outBitmapCaps(RDPStream s) {
    	s.outUInt16LE(RDPConstants.rdpCapsetBitmap);
    	s.outUInt16LE(RDPConstants.rdpCaplenBitmap);
    	
    	s.outUInt16LE(8);	// preferred bpp
    	s.outUInt16LE(1);	// receive 1 bpp
    	s.outUInt16LE(1);	// receive 4 bpp
    	s.outUInt16LE(1);	// receive 8 bpp
    	s.outUInt16LE(800);	// Desktop width
    	s.outUInt16LE(600);	// Desktop height
    	s.outUInt16(0);		// pad
    	s.outUInt16(0);		// allow resize
    	s.outUInt16LE(rdpProfile.bitmapCompression ? 1 : 0);	// support compression
    	s.outUInt16(0);		// unknown
    	s.outUInt16LE(1);	// unknown
    	s.outUInt16(0);		// pad
    }
    
    // Output order capability set
    private void outOrderCaps(RDPStream s) {
    	byte orderCaps[] = new byte[32];
    	
	orderCaps[0] = 1;	// dest blt
	orderCaps[1] = 1;	// pat blt
	orderCaps[2] = 1;	// screen blt
	orderCaps[3] = 1;	// required for memblt?
	orderCaps[8] = 1;	// line
	orderCaps[9] = 1;	// line
	orderCaps[10] = 1;	// rect
				// desksave
	orderCaps[11] = (byte)(rdpProfile.desktopSave == false ? 0 : 1);
	orderCaps[13] = 1;	// memblt
	orderCaps[14] = 1;	// triblt
	orderCaps[22] = 1;	// polyline
	orderCaps[27] = 1;	// text2
	
	// FIXME
	/*orderCaps[0] = 0;	// dest blt
	orderCaps[1] = 0;	// pat blt
	orderCaps[2] = 0;	// screen blt
	orderCaps[3] = 0;	// required for memblt?
	orderCaps[8] = 0;	// line
	orderCaps[9] = 0;	// line
	orderCaps[10] = 0;	// rect
	orderCaps[11] = 0;	// desksave
	orderCaps[13] = 0;	// memblt
	orderCaps[14] = 0;	// triblt
	orderCaps[22] = 0;	// polyline
	orderCaps[27] = 0;	// text2
	*/// end FIXME
	
	s.outUInt16LE(RDPConstants.rdpCapsetOrder);
	s.outUInt16LE(RDPConstants.rdpCaplenOrder);
	
	s.incrementPosition(20);	// Terminal desc, pad
	s.outUInt16LE(1);	// Cache X granularity
	s.outUInt16LE(20);	// Cache Y granularity
	s.outUInt16(0);		// pad
	s.outUInt16LE(1);	// max order level
	s.outUInt16LE(0x147);	// number of fonts
	s.outUInt16LE(0x2a);	// capability flags
	s.outUInt8Array(orderCaps, 0, 32);	// orders supported
	s.outUInt16LE(0x6a1);	// Text capability flags
	s.incrementPosition(6);	// pad
	s.outUInt32LE(rdpProfile.desktopSave == false ? 0 : 0x38400);	// Desktop cache size
	s.outUInt32(0);		// unknown
	s.outUInt32LE(0x4e4);	// unknown
    }

    
    // Output bitmap cache capability set
    private void outBmpCacheCaps(RDPStream s) {
    	s.outUInt16LE(RDPConstants.rdpCapsetBmpCache);
    	s.outUInt16LE(RDPConstants.rdpCaplenBmpCache);
    	
    	int Bpp = (rdpProfile.hostBpp + 7) / 8;
    	
    	s.incrementPosition(24);	// unused
    	s.outUInt16LE(0x258);		// entries
	s.outUInt16LE(0x100 * Bpp);	// max cell size
	s.outUInt16LE(0x12c);		// entries
	s.outUInt16LE(0x400 * Bpp);	// max cell size
	s.outUInt16LE(0x106);		// entries
	s.outUInt16LE(0x1000 * Bpp);	// max cell size
    }
    
    // Output control capability set
    private void outControlCaps(RDPStream s) {
    	s.outUInt16LE(RDPConstants.rdpCapsetControl);
    	s.outUInt16LE(RDPConstants.rdpCaplenControl);
    	
    	s.outUInt16(0);		// Control capabilities
	s.outUInt16(0);		// Remote detach
	s.outUInt16LE(2);	// Control interest
	s.outUInt16LE(2);	// Detach interest
    }
    
    // Output activation capability set
    private void outActivateCaps(RDPStream s) {
    	s.outUInt16LE(RDPConstants.rdpCapsetActivate);
    	s.outUInt16LE(RDPConstants.rdpCaplenActivate);
    	
    	s.outUInt16(0);		// Help key
	s.outUInt16(0);		// Help index key
	s.outUInt16(0);		// Extended help key
	s.outUInt16(0);		// Window activate
    }
    
    // Output pointer capability set
    private void outPointerCaps(RDPStream s) {
    	s.outUInt16LE(RDPConstants.rdpCapsetPointer);
    	s.outUInt16LE(RDPConstants.rdpCaplenPointer);
    	
    	s.outUInt16(0);		// Colour pointer
	s.outUInt16LE(20);	// Cache size
    }
    
    // Output share capability set
    private void outShareCaps(RDPStream s) {
    	s.outUInt16LE(RDPConstants.rdpCapsetShare);
    	s.outUInt16LE(RDPConstants.rdpCaplenShare);
    	
    	s.outUInt16(0);		// userid
	s.outUInt16(0);		// pad
    }
    
    // Output colour cache capability set
    private void outColCacheCaps(RDPStream s) {
    	s.outUInt16LE(RDPConstants.rdpCapsetColCache);
    	s.outUInt16LE(RDPConstants.rdpCaplenColCache);
    	
    	s.outUInt16LE(6);	// cache size
	s.outUInt16(0);		// pad
    }
    
    // array of unknown capabilities
    private byte cannedCaps[] = {
	(byte)0x01, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x09, (byte)0x04, (byte)0x00, (byte)0x00,
	(byte)0x04, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x0C, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x0C, (byte)0x00, (byte)0x08, (byte)0x00,
	(byte)0x01, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x0E, (byte)0x00, (byte)0x08, (byte)0x00,
	(byte)0x01, (byte)0x00, (byte)0x00, (byte)0x00,
	(byte)0x10, (byte)0x00, (byte)0x34, (byte)0x00,
	(byte)0xFE, (byte)0x00, (byte)0x04, (byte)0x00,
	(byte)0xFE, (byte)0x00, (byte)0x04, (byte)0x00,
	(byte)0xFE, (byte)0x00, (byte)0x08, (byte)0x00,
	(byte)0xFE, (byte)0x00, (byte)0x08, (byte)0x00,
	(byte)0xFE, (byte)0x00, (byte)0x10, (byte)0x00,
	(byte)0xFE, (byte)0x00, (byte)0x20, (byte)0x00,
	(byte)0xFE, (byte)0x00, (byte)0x40, (byte)0x00,
	(byte)0xFE, (byte)0x00, (byte)0x80, (byte)0x00,
	(byte)0xFE, (byte)0x00, (byte)0x00, (byte)0x01,
	(byte)0x40, (byte)0x00, (byte)0x00, (byte)0x08,
	(byte)0x00, (byte)0x01, (byte)0x00, (byte)0x01,
	(byte)0x02, (byte)0x00, (byte)0x00, (byte)0x00
    };
    
    // Output unknown capability sets (number 13, 12, 14 and 16)
    private void outUnknownCaps(RDPStream s) {
    	s.outUInt16LE(RDPConstants.rdpCapsetUnknown);
    	s.outUInt16LE(0x58);
    	s.outUInt8Array(cannedCaps, 0, RDPConstants.rdpCaplenUnknown - 4);
    }
    
    // Send a confirm active PDU
    private void sendConfirmActive() throws IOException {
    	RDPStream s;
    	
    	int secFlags = rdpProfile.encryption ? RDPConstants.secEncrypt : 0;
    	int capLen =
		RDPConstants.rdpCaplenGeneral +
		RDPConstants.rdpCaplenBitmap +
		RDPConstants.rdpCaplenOrder +
		RDPConstants.rdpCaplenBmpCache +
		RDPConstants.rdpCaplenColCache +
		RDPConstants.rdpCaplenActivate +
		RDPConstants.rdpCaplenControl +
		RDPConstants.rdpCaplenPointer +
		RDPConstants.rdpCaplenShare +
		RDPConstants.rdpCaplenUnknown +
		4 /* w2k fix, why? */ ;
	
	s = secureLayer.init(secFlags, 6 + 14 + capLen + RDPConstants.rdpSource.length());
	
	s.outUInt16LE(2 + 14 + capLen + RDPConstants.rdpSource.length());
	s.outUInt16LE(RDPConstants.rdpPDUConfirmActive | 0x10);	// version 1
	s.outUInt16LE(secureLayer.getMCSUserID() + 1001);
	
	s.outUInt32LE(shareID);
	s.outUInt16LE(0x3ea);	// userid
	s.outUInt16LE(RDPConstants.rdpSource.length());
	s.outUInt16LE(capLen);
	
	s.outByteStringNoTerm(RDPConstants.rdpSource);
	s.outUInt16LE(0x0d);	// num caps
	s.incrementPosition(2);	// pad
	
	outGeneralCaps(s);
	outBitmapCaps(s);
	outOrderCaps(s);
	outBmpCacheCaps(s);
	outColCacheCaps(s);
	outActivateCaps(s);
	outControlCaps(s);
	outPointerCaps(s);
	outShareCaps(s);
	outUnknownCaps(s);
	
	s.markEnd();
	secureLayer.send(s, secFlags);
    }
    
    // Respond to a demand active PDU
    private void processDemandActive(RDPStream s) throws IOException {
    	int type;
    	
    	shareID = s.inUInt32LE();
    	
    	sendConfirmActive();
    	sendSynchronize();
    	sendControl(RDPConstants.rdpCtlCooperate);
    	sendControl(RDPConstants.rdpCtlRequestControl);
    	receive();	// RDPConstants.rdpPDUSynchronize
    	receive();	// RDPConstants.rdpCtlCooperate
    	receive();	// RDPConstants.rdpGrantControl
    	sendInput(0, RDPConstants.rdpInputSynchronize, 0, 0, 0);
    	sendFonts(1);
    	sendFonts(2);
    	receive();	// rdpPDUUnknown 0x28
    	orders.resetOrderState();
    }
    
    // process a pointer clear PDU
    private void processPointerClear(RDPStream s) {
    	s.incrementPosition(4);
    	RDPCursor clearCursor = RDPCursor.getClearCursor();
    	rdr.setCursor(clearCursor);
    }
    
    // Process a colour pointer PDU
    private void processColorPointerPDU(RDPStream s) {
    	int x, y, width, height, cacheIndex, maskLen, dataLen;
    	
    	cacheIndex = s.inUInt16LE();
    	x = s.inUInt16LE();
    	y = s.inUInt16LE();
    	width = s.inUInt16LE();
    	height = s.inUInt16LE();
    	maskLen = s.inUInt16LE();
    	dataLen = s.inUInt16LE();
    	
    	byte[] data = new byte[dataLen];
    	s.inUInt8Array(data, 0, dataLen);
    	byte[] mask = new byte[maskLen];
    	s.inUInt8Array(mask, 0, maskLen);
    	
    	RDPCursor cursor = new RDPCursor(x, y, width, height, mask, data);
    	rdr.setCursor(cursor);
    	cache.putCursor(cacheIndex, cursor);
    	
    }
    
    // Process a cached pointer PDU
    private void processCachedPointerPDU(RDPStream s) {
    	int cacheIndex = s.inUInt16LE();
    	RDPCursor cursor = cache.getCursor(cacheIndex);
    	if(cursor != null) {
    	    rdr.setCursor(cursor);
    	}
    }
    
    // Process a pointer PDU
    private void processPointerPDU(RDPStream s) {
    	int messageType;
    	int x, y;
    	
    	messageType = s.inUInt16LE();
    	s.incrementPosition(2);		// pad
    	
    	switch(messageType) {
    	    case RDPConstants.rdpPointerClear:	// 1
    	    	processPointerClear(s);
    	    	break;
    	    case RDPConstants.rdpPointerMove:	// 3
    	    	x = s.inUInt16LE();
    	    	y = s.inUInt16LE();
    	    	if(s.checkSize()) {
    	    	    robotMouseMove(x, y);
    	    	}
    	    	break;
    	    case RDPConstants.rdpPointerColor:	// 6
    	    	processColorPointerPDU(s);
    	    	break;
    	    case RDPConstants.rdpPointerCached:	// 7
    	    	processCachedPointerPDU(s);
    	    	break;
    	    default:
    	    	// this happens too often to leave System.out.println in here
    	    	debug("RDP processPointerPDU: unknown PDU received: " + messageType);
    	}
    }
    
    // Process bitmap updates
    private void processBitmapUpdates(RDPStream s) {
	int numUpdates;
	int left, top, right, bottom, width, height;
	int cx, cy, bpp, Bpp, compress, bufSize, size;
	byte data[];
	int bmpData[];
	
	numUpdates = s.inUInt16LE();
	
	for(int i = 0; i < numUpdates; i++) {
	    left = s.inUInt16LE();
	    top = s.inUInt16LE();
	    right = s.inUInt16LE();
	    bottom = s.inUInt16LE();
	    width = s.inUInt16LE();
	    height = s.inUInt16LE();
	    bpp = s.inUInt16LE();
	    Bpp = (bpp + 7) / 8;
	    compress = s.inUInt16LE();
	    bufSize = s.inUInt16LE();
	    
	    cx = right - left + 1;
	    cy = bottom - top + 1;
	    
	    if(compress == 0) {
	    	int y;
	    	data = getBuffer(width * height * Bpp);
	    	bmpData = getIntBuffer(width * height);
	    	for(y = 0; y < height; y++) {
	    	    s.inUInt8Array(data, (height - y - 1) * (width * Bpp), width * Bpp);
	    	}
	    	rdr.translateColor(bmpData, 0, data, 0, width * height);
	    	rdr.drawBitmap(left, top, cx, cy, width, height, bmpData);
	    	continue;
	    }
	    
	    if((compress & 0x400) != 0) {
	    	size = bufSize;
	    }
	    else {
	    	s.incrementPosition(2);		// pad
	    	size = s.inUInt16LE();
	    	s.incrementPosition(4);		// line_size, final_size
	    }
	    data = getBuffer(size);
	    s.inUInt8Array(data, 0, size);
	    bmpData = bitmapDecompressor.decompress(data, width, height, size, Bpp);
	    if(bmpData != null) {
	    	rdr.drawBitmap(left, top, cx, cy, width, height, bmpData);
	    }
	    else {
	    	System.out.println("RDP processBitmapUpdates: error decompressing bitmap");
	    }
	    data = null;
	    bmpData = null;
	}
    }
    
    // Process a palette update
    private void processPalette(RDPStream s) {
    	int colors[];
    	int nColors;
    	
    	s.incrementPosition(2);		// pad
    	nColors = s.inUInt16LE();
    	s.incrementPosition(2);		// pad
    	
    	colors = new int[nColors];
    	
    	for(int i = 0; i < nColors; i++) {
    	    colors[i] = 0xff000000 | (s.inUInt8() << 16) | (s.inUInt8() << 8) | s.inUInt8();
    	}
    	
    	rdr.setColorMap(colors, nColors);
    }
    
    // Process an update PDU
    private void processUpdatePDU(RDPStream s) {
    	int updateType, count;
    	
    	updateType = s.inUInt16LE();
    	
    	switch(updateType) {
    	    case RDPConstants.rdpUpdateOrders:		// 0
    	    	s.incrementPosition(2);	//pad
    	    	count = s.inUInt16LE();
    	    	s.incrementPosition(2);	//pad
    	    	orders.processOrders(s, count, nextPacket);
    	    	break;
    	    case RDPConstants.rdpUpdateBitmap:		// 1
    	    	processBitmapUpdates(s);
    	    	break;
    	    case RDPConstants.rdpUpdatePalette:		// 2
    	    	processPalette(s);
    	    	break;
    	    case RDPConstants.rdpUpdateSynchronize:	// 3
    	    	break;
    	    default:
    	    	System.out.println("RDP processUpdatePDU: unknown PDU type received");
    	}
    	
    	rdr.doRepaint();
    	
    	if(fps) {
	    if(++updateCount == 10) {
	    	long t = updateWatch.stop();
	    	if(t == 0) t = 1;	// prevent division by 0!
	    	logger.println(" fps=" + 1000.0
	                       * (float)updateCount / (float)t
	                       + " t=" + t/updateCount + "ms"
	                       + " c=" + updateCount);
	    	updateWatch.start();
	    	updateCount = 0;
	    }
	}
    }
    
    // Process data PDU
    private void processDataPDU(RDPStream s) {
    	int dataPDUType;
    	
    	s.incrementPosition(8);		// shareid, pad, streamid, length
    	dataPDUType = s.inUInt8();
    	s.incrementPosition(3);		// compress_type, compress_len
    	
    	switch(dataPDUType) {
    	    case RDPConstants.rdpDataPDUUpdate:		// 2
    	    	processUpdatePDU(s);
    	    	break;
    	    case RDPConstants.rdpDataPDUPointer:	// 27
    	    	processPointerPDU(s);
    	    	break;
    	    case RDPConstants.rdpDataPDUBell:		// 34
    	    	Toolkit.getDefaultToolkit().beep();
    	    	break;
    	    case RDPConstants.rdpDataPDULogon:		// 38
    	    	// user logged on
    	    	break;
    	    case RDPConstants.rdpDataPDUControl:	// 20
    	        // we do have the control ;-)
    	        // this is used as a "ping pong" message
    	        
    	        break;
    	    default:
    	    	System.out.println("RDP processDataPDU: unknown PDU type received: " + dataPDUType);
    	}
    }
    
    // Send a single input event
    private void sendInput(int time, int type, int flags, int param1, int param2)  throws IOException {
    	if(!isActive) return;

    	RDPStream s = initData(16);
    	s.outUInt16LE(1);	// number of events
    	s.outUInt16(0);		// pad
    	s.outUInt32LE(time);
    	s.outUInt16LE(type);
    	s.outUInt16LE(flags);
    	s.outUInt16LE(param1);
    	s.outUInt16LE(param2);
    	
    	s.markEnd();
    	sendData(s, RDPConstants.rdpDataPDUInput);
    }
    
    // public functions
    
    // main protocol loop
    public void mainLoop() throws IOException {
    	debug("RDP: mainLoop() started");

    	RDPStream s;
    	
    	while((s = receive()) != null) {
    	    switch(receivedType) {
    	    	case RDPConstants.rdpPDUDemandActive:	// 1
    	    	    processDemandActive(s);
    	    	    break;
    	    	case RDPConstants.rdpPDUDeactivate:	// 6
    	    	    break;
    	    	case RDPConstants.rdpPDUData:		// 7
    	    	    processDataPDU(s);
    	    	    break;
    	    	case 0:
    	    	    break;
    	    	default:
    	    	    System.out.println("RDP mainLoop: unknown PDU type received: " + receivedType);
    	    }
    	}
    	
    	throw new IOException(T._("RDP Connection terminated"));
    }
    
    // Establish a connection up to the RDP layer
    public void connect(String server, int port, int flags, String domain, String password, String command, String directory) throws IOException {
    	// FIXME
    	this.hostname = server;
    	secureLayer.connect(server, port, username, hostname);
    	sendLogonInfo(flags, domain, username, password, command, directory);
    	observeTimer.start();
    	isActive = true;
    }
    
    // If the socket was killed by the other side,
    // we have to clean up some things also
    public void dispose() {
    	if(isActive) {
    	    isActive = false;
    	    observeTimer.stop();
    	}
    }
    
    // Disconnect from the RDP layer
    public void close() {
    	isActive = false;
    	observeTimer.stop();
    	secureLayer.disconnect();
    }
    
    // is the connection alive?
    public boolean connected() {
    	return isActive;
    }
    
    // set the next mouse mode as an robot move
    //   (when indicated through the RDP protocol)
    public void setMouseRobotMove(boolean m) {
    	mouseRobotMove = m;
    }
}	
