package nn.pp.sara;

import java.awt.*;
import java.util.*;
import java.io.*;
import java.text.*;
import java.net.*;
import java.awt.event.*;
import nn.pp.rc.*;

class ReplayPanelFrame extends Frame {
    ReplayPanel replayPanel;
    
    ReplayPanelFrame(ReplayPanel _replayPanel) {
    	super("Replay Control");
    	
    	this.replayPanel = _replayPanel;
    	setLayout(new BorderLayout());

	addWindowListener(new WindowAdapter() {
	    public void windowClosing(WindowEvent e) {
	    	replayPanel.switchInFrame();
	    }
	});
    }
}

class ReplayPanel extends FramePanel {
    private static final int STATE_PLAYING = 0;
    private static final int STATE_PAUSED = 1;
    private static final int STATE_STOPPED = 2;
    
    int replayState = STATE_STOPPED;
    
    SliderPanel sliderPanel;
    ButtonsPanel buttonsPanel;
    SpeedPanel speedPanel;
    SasForensicPanel forensicPanel;
    SessionInfoPanel infoPanel;

    SessionInfoFrame infoFrame = new SessionInfoFrame();
    
    String title = "Replay Control";
    
    public ReplayPanel(URL codeBase, SasForensicPanel parentPanel) {
    	super(codeBase, parentPanel);

	inFrame = false;
	
	forensicPanel = parentPanel;
	
	sliderPanel = new SliderPanel(this);
	buttonsPanel = new ButtonsPanel(codeBase, this);
	speedPanel = new SpeedPanel(this);
	infoPanel = new SessionInfoPanel(this);
    	
    	addLayout();
    }

    private void addLayout() {
    	setLayout(new BorderLayout());
    	
    	mainPanel = new Panel(new BorderLayout());
    	mainPanel.add(getFrameButtonPanel(), BorderLayout.WEST);
    	
    	Panel p = new Panel(new BorderLayout());
    	
    	p.add(sliderPanel, BorderLayout.CENTER);
    	
    	Panel p1 = new Panel(new BorderLayout());
    	p1.add(buttonsPanel, BorderLayout.WEST);
    	p1.add(speedPanel, BorderLayout.CENTER);
    	p1.add(infoPanel, BorderLayout.EAST);
    	
    	p.add(p1, BorderLayout.SOUTH);
    	
    	mainPanel.add(p, BorderLayout.CENTER);
    	
    	showInitial();
    }
    
    public void setEnabled(boolean b) {
    	sliderPanel.setEnabled(b);
    	buttonsPanel.setEnabled(b);
    }

    protected String getFrameName() {
    	return title;
    }


    /* helper functions for sub-panels callbacks */
    private void doPlay() {
    	if (forensicPanel.rcHandler != null) try {
    	    forensicPanel.rcHandler.sendPlay();
    	} catch (IOException ignore) {
    	}
    }
    
    private void doPause() {
    	if (forensicPanel.rcHandler != null) try {
    	    forensicPanel.rcHandler.sendPause();
    	} catch (IOException ignore) {
    	}
    }
    
    private void doStop() {
    	if (forensicPanel.rcHandler != null) try {
    	    forensicPanel.rcHandler.sendStop();
    	} catch (IOException ignore) {
    	}
    }
    
    private void doSeek(Date pos) {
    	if (forensicPanel.rcHandler != null) try {
    	    forensicPanel.rcHandler.sendSeek(pos);
    	} catch (IOException ignore) {
    	}
    }
    
    private void doSetSpeed(int speed) {
    	if (forensicPanel.rcHandler != null) try {
    	    forensicPanel.rcHandler.sendSpeed(speed);
    	} catch (IOException ignore) {
    	}
    }
    
    /* callbacks from RFB layer */
    private void setReplayTime(Date time) {
    	sliderPanel.setTime(time);
    }
    
    private void setReplayTimes(Date time1, Date time2) {
    	sliderPanel.setReplayTimes(time1, time2);
    	infoFrame.setReplayTimes(time1, time2);
    }
    
    private void played() {
    	replayState = STATE_PLAYING;
    	forensicPanel.rcHandler.setBlank(false);
    	buttonsPanel.play();
    	sliderPanel.play();
    }

    private void paused() {
    	replayState = STATE_PAUSED;
    	buttonsPanel.pause();
    	sliderPanel.pause();
    }
    
    private void stopped() {
    	replayState = STATE_STOPPED;
    	forensicPanel.rcHandler.setBlank(true);
    	buttonsPanel.stop();
    	sliderPanel.stop();
    }
    
    public void handleSpEvent(SpEvent evt) {
    	switch (evt.type) {
    	    case SpEvent.SessionTimes:
    	    	setReplayTimes(evt.sessionTimes.start, evt.sessionTimes.end);
    	    	break;
    	    case SpEvent.ReplayTime:
    	    	setReplayTime(evt.replayTime.time);
    	    	break;
    	    case SpEvent.Played:
    	    	played();
    	    	break;
    	    case SpEvent.Paused:
    	    	paused();
    	    	break;
    	    case SpEvent.Stopped:
    	    	stopped();
    	    	break;
    	}
    }

    void setFramebufferFormat(int width, int height, int bitsPerPixel) {
    	infoFrame.setFramebufferFormat(width, height, bitsPerPixel);
    }
    
    void setSessionInfo(RecordedSessionInfo info) {
    	infoFrame.setSessionInfo(info);
    	title = "Replay Control - " + info.hostName + " (" + info.hostAddress + ")";
    	setTitle(title);
    }
    
    /* callbacks from the sub-panels */
    void play() {
    	if (replayState != STATE_PLAYING) {
    	    doPlay();
    	}
    }
    
    void pause() {
    	if (replayState == STATE_PLAYING) {
    	    doPause();
    	} else if (replayState == STATE_PAUSED) {
    	    doPlay();
    	}
    }
    
    void stop() {
    	if (replayState != STATE_STOPPED) {
    	    doStop();
    	}
    }
    
    void seek(Date pos) {
    	if (replayState == STATE_PLAYING) {
    	    doSeek(pos);
    	}
    }
    
    void setSpeed(int speed) {
    	doSetSpeed(speed);
    }
    
    void showInfo() {
        infoFrame.show();
    }
}
