package nn.pp.sara;

import java.awt.*;
import java.util.*;
import java.text.*;
import java.awt.event.*;
import nn.pp.rc.*;

class FramebufferFormat {
    int width;
    int height;
    int bitsPerPixel;
    
    FramebufferFormat(int width, int height, int bitsPerPixel) {
        this.width = width;
        this.height = height;
        this.bitsPerPixel = bitsPerPixel;
    }
    
    FramebufferFormat() {
        this(640, 480, 16);
    }
    
    String getResolution() {
        return "" + width + " x " + height;
    }
}

class ReplayTimes {
    Date time1;
    Date time2;
    SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    
    ReplayTimes(Date time1, Date time2) {
        this.time1 = time1;
        this.time2 = time2;
    }
    
    ReplayTimes() {
        time1 = new Date();
        time2 = new Date();
    }
    
    private String getTime(Date t) {
        return dateFormat.format(t);
    }
    
    String getStartTime() {
        return getTime(time1);
    }
    
    String getEndTime() {
        return getTime(time2);
    }
    
    private String getTimeString(long t, int digits) {
        String ret = "";
        while (--digits > 0) {
            int d = 1;
            for (int i = 0; i < digits; i++) d *= 10;
            if (t < d) ret += "0";
        }
        ret += t;
        return ret;
    }
    
    String getTimeDiff() {
        long t1 = time1.getTime();
        long t2 = time2.getTime();
        long d = Math.abs(t2 - t1);
        
        long h = d / 3600000;
        d %= 3600000;
        long m = d / 60000;
        d %= 60000;
        long s = d / 1000;
        d %= 1000;
        long ms = d;
        
        return getTimeString(h, 2) + ":" + getTimeString(m, 2) + ":" + getTimeString(s, 2) + "." + getTimeString(ms, 3);
    }
}

public class SessionInfoFrame extends Frame {
    FramebufferFormat format = new FramebufferFormat();
    RecordedSessionInfo info = new RecordedSessionInfo();
    ReplayTimes times = new ReplayTimes();

    boolean shown = false;
    
    private Label hostName;
    private Label hostAddress;
    private Label colorDepth;
    private Label keyboardMode;
    private Label useJpeg;
    private Label jpegLevel;
    private Label useZlib;
    private Label zlibLevel;
    private Label keyframeInterval;
    private Label resolutionLabel;
    private Label startTime;
    private Label endTime;
    private Label timeDiff;
    
    public SessionInfoFrame() {
        this("Session Info");
    }
    
    public SessionInfoFrame(String title) {
        super(title);
        enableEvents(WindowEvent.WINDOW_CLOSING);
        
        addLayout();
        update();
    }

    private Label createLabel(String name, GridBagLayout gridbag, GridBagConstraints c, Container container) {
        Label l = new Label(name);
        gridbag.setConstraints(l, c);
        container.add(l);
        return l;
    }

    private Label createLabel(GridBagLayout gridbag, GridBagConstraints c, Container container) {
        Label l = new Label();
        gridbag.setConstraints(l, c);
        container.add(l);
        return l;
    }

    private void createHeader(String name, GridBagLayout gridbag, GridBagConstraints c, Container container) {
        c.fill = GridBagConstraints.BOTH;
        c.weightx = 0.0;
        c.gridwidth = GridBagConstraints.REMAINDER;
        Label l = createLabel(name, gridbag, c, container);
        l.setAlignment(Label.CENTER);
        l.setFont(new Font("Helvetica", Font.BOLD, 14));
    }

    private Label createRow(String name, GridBagLayout gridbag, GridBagConstraints c, Container container) {
        c.gridwidth = GridBagConstraints.RELATIVE; //next-to-last in row
        createLabel(name, gridbag, c, container);
        c.gridwidth = GridBagConstraints.REMAINDER; //end row
        return createLabel(gridbag, c, container);
    }

    private void addLayout() {
        GridBagLayout gridbag = new GridBagLayout();
        GridBagConstraints c = new GridBagConstraints();
        Panel p = new Panel(gridbag);
        
        createHeader(T._("Host information"), gridbag, c, p);
        hostName = createRow(T._("Host name"), gridbag, c, p);
        hostAddress = createRow(T._("Host IP address"), gridbag, c, p);

        createHeader(T._("Compression settings"), gridbag, c, p);
        colorDepth = createRow(T._("Color depth"), gridbag, c, p);
        useJpeg = createRow(T._("JPEG compression"), gridbag, c, p);
        jpegLevel = createRow(T._("JPEG quality level"), gridbag, c, p);
        useZlib = createRow(T._("ZLIB compression"), gridbag, c, p);
        zlibLevel = createRow(T._("ZLIB level"), gridbag, c, p);

        createHeader(T._("Session information"), gridbag, c, p);
        keyboardMode = createRow(T._("Keyboard mode"), gridbag, c, p);
        keyframeInterval = createRow(T._("Keyframe interval"), gridbag, c, p);
        resolutionLabel = createRow(T._("Screen resolution"), gridbag, c, p);
        
        createHeader(T._("Session times"), gridbag, c, p);
        startTime = createRow(T._("Session start"), gridbag, c, p);
        endTime = createRow(T._("Session end"), gridbag, c, p);
        timeDiff = createRow(T._("Duration"), gridbag, c, p);
        
        setLayout(new FlowLayout());
        add(p);
    }

    private void update() {
        setTitle("Session Info - " + info.hostName + " (" + info.hostAddress + ")");
        
        hostName.setText(info.hostName);
        hostAddress.setText(info.hostAddress);
        colorDepth.setText(info.getColorDepth());
        keyboardMode.setText(info.getKeyboardMode());
        useJpeg.setText(info.getUseJpeg());
        jpegLevel.setText(info.getJpegLevel());
        useZlib.setText(info.getUseZlib());
        zlibLevel.setText(info.getZlibLevel());
        keyframeInterval.setText(info.getKeyframeInterval());
        resolutionLabel.setText(format.getResolution());
        startTime.setText(times.getStartTime());
        endTime.setText(times.getEndTime());
        timeDiff.setText(times.getTimeDiff());
        
        pack();
    }

    public void setFramebufferFormat(int width, int height, int bitsPerPixel) {
    	format = new FramebufferFormat(width, height, bitsPerPixel);
    	update();
    }
    
    public void setSessionInfo(RecordedSessionInfo info) {
    	this.info = info;
    	update();
    }
    
    public void setReplayTimes(Date time1, Date time2) {
        times = new ReplayTimes(time1, time2);
        update();
    }

    public void show() {
        if (!shown) {
            super.show();
            shown = true;
        } else {
            toFront();
        }
    }
    
    public void hide() {
        if (shown) {
            super.hide();
            shown = false;
        }
    }

    protected void processWindowEvent(WindowEvent event) {
	if (event.getID() == WindowEvent.WINDOW_CLOSING) {
	    hide();
	}
    }
}
