#include <arpa/inet.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/reboot.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/um.h>
#include <liberic_config.h>
#if defined(PP_FEAT_CONFIGMODE)
#include <lara.h>
#endif /* PP_FEAT_CONFIGMODE */
#include <pp_kernel_common.h>

#if defined(PP_FEAT_RPCCFG)
# include <pp/rpc_ripc.h>
#endif

#ifdef PRODUCT_PDU
#include <pp/hal_rpc.h>
#include <pp/bmc/bmc_dev_oem_pp_rpc_isens_nr.h>
#include <pp/xdefs.h>
#endif /* PRODUCT_PDU */

#include "unblock.h"
#include "cmdedit.h"

/* BUFSIZE from mini readline of busybox */
#define MAXINPUTSIZE BUFSIZ
#define MAXPROMPTSIZE 255

/* FIXME: it seems, the nobody has cared about freeing memory here */

static void do_lanif_config(void);
static void reconfigure_device(void);
static void exit_handler(void);
static void conf_at_layer(pp_profile_type_t type, 
                          const char * displaystr, const char * key);
static void strtolower(char *);
static int g_defaults_set = 0;
#ifdef PRODUCT_PDU
static int pdutype_set(int);	  
#endif
static int umcfg_initialized = 0;

extern int cfg_conv_main(int mode);

inline static void conf(const char * displaystr, const char * key) {
    conf_at_layer(PP_PROFILE_LOCAL, displaystr, key);
}

/* prototype to avoid warning */
int eric_config_main(int argc, char ** argv);

int
eric_config_main(int argc, char ** argv)
{
    char * ip_auto_config_proto;
    //int  num_outlets;
    char input[MAXINPUTSIZE]; char* fret;
    int check_serial_debug = 0;
    int check_config_mode = 0;
    int check_power_cim = 0;
    int check_defaults = 0;
    int do_flush = 0;
    int do_defaults = 0;
    int do_jumper_defaults = 0;
    int do_serial_config = 0;
#if defined(PP_FEAT_CONFIGMODE)
    int serial_debug = 0;
    int power_cim = 0;
#endif /* PP_FEAT_CONFIGMODE */
    int do_sn_change = 0;
    int do_mac_change = 0;
#ifdef PRODUCT_PDU
    int do_pdu_change = 0;
#endif
    int exit_handler_cleanups = 0;
    int config_mode = 0;
    int lanif_config = 0;
    int do_unblock_dialog = 0;
    int do_reboot = 0;
    int check_eth_phy = 0;
    int send_property = 0;
    int call_with_flush_lock = 0;
    
    /* cfg conversions */
    int do_cfg_conv = 0;
    int do_raasip_um_conv = 0;
    int do_common_um_conv = 0;
    int check_pw_change = 0;
    int check_login_upper = 0;
    int split_sec_s_acl = 0;
    int do_sys_profile_conv = 0;
    
    char * endptr;
    int ret = 0;

    if (argc >= 2) {
	check_serial_debug = !strcmp(argv[1], "-csd");
	check_config_mode = !strcmp(argv[1], "-ccm");
	check_power_cim = !strcmp(argv[1], "-cpcim");
	check_defaults = !strcmp(argv[1], "-cdefaults");
	do_flush    = !strcmp(argv[1], "-flush");
	do_defaults = !strcmp(argv[1], "-defaults");
	do_jumper_defaults = !strcmp(argv[1], "-j");
	do_serial_config = !strcmp(argv[1], "-serial");
	lanif_config =  !strcmp(argv[1], "-lanif");
	do_sn_change = !strcmp(argv[1], "-changesn");
	do_mac_change = !strcmp(argv[1], "-changemac");
#ifdef PRODUCT_PDU
	do_pdu_change = !strcmp(argv[1], "-changepdu");
#endif
	send_property = !strcmp(argv[1], "-sendprop");
	do_reboot = !strcmp(argv[1], "-reboot");
	check_eth_phy = !strcmp(argv[1], "-checkethphy");
	call_with_flush_lock = !strcmp(argv[1], "-call_with_flush_lock");
#ifdef PP_FEAT_USER_BLOCKING
	do_unblock_dialog = !strcmp(argv[1], "-unblock");
#endif

        /* cfg conversions */
        do_cfg_conv = !strcmp(argv[1], "-cfg_conv");
        do_raasip_um_conv = !strcmp(argv[1], "-raasip_um_conv");
        do_common_um_conv = !strcmp(argv[1], "-common_um_conv");
        check_pw_change = !strcmp(argv[1], "-check_pw_change");
        check_login_upper = !strcmp(argv[1], "-check_login_upper");
        split_sec_s_acl = !strcmp(argv[1], "-split_sec_s_acl");
        do_sys_profile_conv = !strcmp(argv[1], "-sys_profile_conv");
    }

    // FIXME: Geez, use symbolic constants here ...
    if(do_cfg_conv) {
        return cfg_conv_main(1); // CONVERT_CONFIG_FS
    }
    if(do_raasip_um_conv) {
        return cfg_conv_main(2); // CONVERT_RAASIP_UM
    }
    if(do_common_um_conv) {
        return cfg_conv_main(3); // CONVERT_COMMON_UM
    }
    if(check_pw_change) {
        return cfg_conv_main(4); // CHECK_PW_CHANGE
    }
    if(check_login_upper) {
#if defined(PP_FEAT_CASE_INSENSITIVE_LOGINS)
        return cfg_conv_main(5); // CHECK_LOGIN_UPPER
#else /* PP_FEAT_CASE_INSENSITIVE_LOGINS */
        return PP_SUC;
#endif /* PP_FEAT_CASE_INSENSITIVE_LOGINS */
    }
    if(split_sec_s_acl) {
        return cfg_conv_main(6); // SPLIT_SEC_S_ACL
    }
    if(do_sys_profile_conv) {
        return cfg_conv_main(7); // CONVERT_SYS_PROFILE
    }
    
    if (pp_base_init("eric_config",
		     (do_unblock_dialog) ? LOG_IS_SILENT : LOG_NOT_SILENT) != 0) {
	pp_log("Initializing base-library failed.\n");
	ret = -1;
	goto bail;
    }
    
    if (do_reboot) {
	char prompt[MAXPROMPTSIZE];
	snprintf(prompt, sizeof(prompt), "To reset the device enter \"yes\": ");
	if (cmdedit_read_input(prompt, input)) {
	    if (!strcmp(input, "yes\n")) {
		sync();
		reboot(RB_AUTOBOOT);
	    }
	}
	return 0;
    }

    if (check_eth_phy) {
	if (argc == 3) {
	    return pp_eth_have_phy(argv[2]) ? 0 : 1;
	}
	return 1;
    }
    
    if (send_property) {
	union sigval sv;
	int eric_pid;
	if (argc == 3) {
	    sv.sival_int = strtol(argv[2], &endptr, 10);
	    if (*endptr != '\0' || sv.sival_int <= 0) {
		fprintf(stderr, "parameter to -sendprop must be a positive numeric value!\n");
		ret = -1;
	    } else if ((eric_pid = pp_base_get_pid_of_program_by_pidfile(ERIC_PIDFILE, ERIC_PROGRAM_PATH)) <= 0) {
		fprintf(stderr, "Cannot open pidfile - maybe the 'eric' process is not running.\n");
		ret = -1;
	    } else if (sigqueue(eric_pid, SIGPROPCHG, sv) == -1) {
		pp_log_err("sigqueue() failed");
	    }
	} else {
	    fprintf(stderr, "-sendprop requires a positive numeric parameter!\n");
	    ret = -1;
	}
	goto bail;
    }

    if (eric_config_init(FLUSH_IN_FOREGROUND) != 0) {
	pp_log("Initializing config-library failed.\n");
	ret = -1;
	goto bail;
    }

    if (call_with_flush_lock) {
	eric_config_set_flush_lock();
	pid_t pid = fork();
	if (pid == 0) {
	    if (execvp(argv[2], &argv[2]) < 0) {
		pp_log_err("Executing %s failed", argv[2]);
		exit(1);
	    }
	    exit(1); /* sanity */
	} else if (pid > 0) {
	    TEMP_FAILURE_RETRY(waitpid (pid, NULL, 0));
	} else {
	    pp_log_err("fork() failed");
	    ret = -1;
	}
	eric_config_release_flush_lock();
	goto bail;
    }
    
#if defined(PP_FEAT_CONFIGMODE)
    /* this must be here since we use config_mode later */
    ioctl(eric_fd, ERICIOCCHECKCONFIGMODE, &config_mode);

    if (check_config_mode) {
	ret = !config_mode;
	goto bail;
    }

    if (check_serial_debug) {
	ioctl(eric_fd, ERICIOCCHECKSERIALDEBUG, &serial_debug);
	ret = !serial_debug;
	goto bail;
    }

    if (check_power_cim) {
	ioctl(eric_fd, ERICIOCCHECKPOWERCIM, &power_cim);
	ret = !power_cim;
	goto bail;
    }
#endif /* PP_FEAT_CONFIGMODE */

    if (lanif_config) {
#if defined(PP_FEAT_RPCCFG)
	if (PP_FAILED(pp_rpc_init())) {
	    pp_log("Initializing rpc library failed .. \n");
	    goto bail;
	}
#endif /* PP_FEAT_CONFIGMODE */
	if (PP_ERR == pp_cfg_init(PP_CD_FNAME_DEFAULT,
                                  PP_CD_OEM_SKEL_FNAME_DEFAULT)
	    || PP_ERR == pp_um_init()) {
	    pp_log("Initializing cfg/um-library failed.\n");
	    ret = -1;
	    goto bail;
	}
	umcfg_initialized = 1;
	do_lanif_config();
	goto bail;
    }

#if !defined(PRODUCT_FLASHX4) && defined(PP_FEAT_CONFIGMODE)
    if (check_defaults) {
	ioctl(eric_fd, ERICIOCCHECKDEFAULTS, &g_defaults_set);
	ret = !g_defaults_set;
	goto bail;
    }
    
    if (do_jumper_defaults) {
	ioctl(eric_fd, ERICIOCCHECKDEFAULTS, &g_defaults_set);
	do_defaults |= g_defaults_set;
    }
#endif /* !PRODUCT_FLASHX4 && PP_FEAT_CONFIGMODE */
    
    if (do_defaults) {
	printf("Installing default configuration ...\n");
	eric_config_install_defaults();
	pp_system("/etc/rc.net > /dev/null 2>&1");
    }

    if (do_flush || do_defaults) {
	printf("Writing configuration data to flash ...\n");
	eric_config_flush();
	printf("Done.\n");
    }

    if (g_defaults_set) {
	/*
	 * Since eric_config_flush() returns immediately we use
	 * an exit handler.
	 */
	atexit(exit_handler);
	exit_handler_cleanups = 1;
    }

    /* exit now if we were called with an argument other than serial, unblock , sn, or mac change */
#ifdef PRODUCT_PDU
    if (argc >= 2 && !do_serial_config && !do_unblock_dialog && !do_mac_change && !do_sn_change && !do_pdu_change) goto bail;
#else
    if (argc >= 2 && !do_serial_config && !do_unblock_dialog && !do_mac_change && !do_sn_change) goto bail;
#endif

#if defined(PP_FEAT_RPCCFG)
    if (PP_FAILED(pp_rpc_init())) {
	pp_log("Initializing rpc library failed .. \n");
	goto bail;
    }
#endif /* PP_FEAT_CONFIGMODE */
    /* init um library */
    if (PP_ERR == pp_cfg_init(PP_CD_FNAME_DEFAULT,
                              PP_CD_OEM_SKEL_FNAME_DEFAULT)) {
	pp_log_err("cfg/um init failed\n");
	goto bail;
    }
    if (PP_ERR == pp_um_init()) {
	pp_log_err("cfg/um init failed\n");
	goto bail;
    }
    umcfg_initialized = 1;

#ifdef PP_FEAT_USER_BLOCKING
    if (do_unblock_dialog) {
	unblock_users_dialog();
	goto bail;
    }
#endif
    
 start_config:

    if (do_serial_config) {
	conf("Line Speed (bits/s)", "serialport[0].configlogin.speed");
	conf("Data Bits (7/8)", "serialport[0].configlogin.data");
	conf("Parity (none/odd/even)", "serialport[0].configlogin.parity");
	conf("Stop Bits (1/2)", "serialport[0].configlogin.stop");
	conf("Handshake (none/hw/sw)", "serialport[0].configlogin.handshake");

	pp_cfg_save(DO_FLUSH);
	printf("Done.\n");
    } else if (do_sn_change) {
	conf_at_layer(PP_PROFILE_SYSTEM, "Serial Number", "serial");
	pp_cfg_save_layer(PP_PROFILE_SYSTEM, DO_FLUSH);
	printf("Done.\n");
    } else if (do_mac_change) {
	conf_at_layer(PP_PROFILE_SYSTEM, "MAC address", "network.mac");
        pp_cfg_save_layer(PP_PROFILE_SYSTEM, DO_FLUSH);
	reconfigure_device();
#ifdef PRODUCT_PDU
    } else if (do_pdu_change) {
	int i, no;
	for (i = 0, no = 0; i < pp_hal_rpc_models_sz; ++i) {
	    if (no == 0 || pp_hal_rpc_models[i].no_outlets != no) {
		no = pp_hal_rpc_models[i].no_outlets;
		printf("+++ Num of outlets = %d  +++\n", no);
	    }
	    printf(" %d.\t%s\n", i+1,  pp_hal_rpc_models[i].model);
	};
        conf("Model Type (1~n)", "device.model");
	conf_at_layer(PP_PROFILE_SYSTEM, "Serial Number", "serial");
	conf_at_layer(PP_PROFILE_SYSTEM, "MAC address", "network.mac");
	pp_cfg_save(DO_FLUSH);
        pp_cfg_save_layer(PP_PROFILE_SYSTEM, DO_FLUSH);
        printf("\n\nNOTE\nyou will need to reset the device to apply new "
               "model configuration\n");
#endif
    } else {
	conf("IP autoconfiguration (none/dhcp/bootp)", "network.ip_auto_config_proto");
	pp_cfg_get(&ip_auto_config_proto, "network.ip_auto_config_proto");
	if (strcmp(ip_auto_config_proto, "dhcp")
	    && strcmp(ip_auto_config_proto, "bootp")) {
	    conf("IP", "network.ipaddr");
	    conf("NetMask", "network.netmask");
	    conf("Gateway (0.0.0.0 for none)", "network.gateway");
	}
	/* MAC config disabled for now since we have no default */
	/* conf("MAC","mac"); */
#ifdef PP_FEAT_NET_LANIF
# ifdef PP_FEAT_FIREWALL
	conf("Enable IP Access Control (yes/no)","security.ip_fw.enabled");
# endif
	conf("LAN interface speed (auto/10/100)", "network.eth_speed");
	conf("LAN interface duplex mode (auto/half/full)", "network.eth_duplex");
#endif
#ifdef PP_FEAT_WLAN
	{
	    char * wlan_security_type = NULL;
	    char * wlan_wpa_type = NULL;
	    conf("WLAN accesspoint's ESSID: ", "wlan.essid");
	    conf("WLAN Security Type (\n"
		 "    0 = none\n"
		 "    1 = WEP\n"
		 "    2 = WPA)", "wlan.security_type");
	    pp_cfg_get(&wlan_security_type, "wlan.security_type");
	    if (!pp_strcmp_safe(wlan_security_type, "WEP")) {
		conf("WLAN WEP key (10 hex digits, e.g. AA11BB22CC or 5 ASCII characters)", "wlan.wep_key");
	    } else if (!pp_strcmp_safe(wlan_security_type, "WPA")) {
		conf("WLAN WPA Type (\n"
		     "    0 = WPA-PSK\n"
		     "    1 = WPA-EAP-PEAP-MSCHAPv2)", "wlan.wpa.type");
		pp_cfg_get(&wlan_wpa_type, "wlan_wpa_type");
		if (!pp_strcmp_safe(wlan_wpa_type, "WPA-PSK")) {
		    conf("WLAN WPA-PSK Key (8-63 ASCII characters)", "wlan_wpa_psk_key");
		} else if (!pp_strcmp_type(wlan_wpa_type, "WPA-EAP-PEAP-MSCHAPv2")) {
		    printf("WLAN WPA-EAP (PEAP MSCHAPv2) configuration:\n");
		    conf("    Identity", "wlan_wpa_eap_peap_mschapv2_identity");
		    conf("    Password", "wlan_wpa_eap_peap_mschapv2_password");
		}
	    }
	}
#if 0
	/* disable until driver is capable of adhoc mode */
	conf("WLAN mode (adhoc, managed)", "wlan.mode");
#endif
#endif
	free(ip_auto_config_proto);

	if (config_mode) {
	    do {
		printf("Are the entered values correct? (y/n) ");	    
		fret = fgets((char*)&input, MAXINPUTSIZE, stdin);
	    } while (!fret || (strcmp(input, "y\n") && strcmp(input, "n\n")));
	    if (!strcmp(input, "y\n")) {
		printf("\nWriting configuration data to flash ...\n");
		pp_cfg_save(DO_FLUSH);
		printf("Rebooting...\n");
		sync();
		usleep(500000);
		reboot(RB_AUTOBOOT);
	    } else {
		goto start_config;
	    }
	
	} else {
	    char prompt[MAXPROMPTSIZE];
	    int rv;
	    do {
		snprintf(prompt, sizeof(prompt), "Are the entered values correct? Enter y for Yes, n for No or c to Cancel ");
		rv = cmdedit_read_input(prompt, input);
	    } while (!rv || (strcmp(input, "y\n") && strcmp(input, "n\n") && strcmp(input, "c\n") && strcmp(input, "cancel\n")));
	    if (!strcmp(input, "y\n")) {
		reconfigure_device();
	    } else if (!strcmp(input, "c\n") || !strcmp(input, "cancel\n")) {
		printf("\nConfiguration cancelled ...\n");
	    } else {
		printf("\n");
		goto start_config;
	    }
	}
    }

 bail:
    if (!exit_handler_cleanups) {
	if (umcfg_initialized) {
	    pp_um_cleanup();
	    pp_cfg_cleanup();
#if defined(PP_FEAT_RPCCFG)
	    pp_rpc_cleanup();
#endif
	}
	eric_config_cleanup();
	pp_base_cleanup();
    }

    return ret;
}

static void
do_lanif_config()
{
#if !defined(PP_FEAT_WLAN)
    /* set link speed and duplex mode */
    char *speed_str, *duplex_str, *if_str, *state_str;

    pp_eth_speed_t eth_speed;
    pp_eth_duplex_t eth_duplex;

    pp_cfg_get(&if_str, "network.eth_interface");
    pp_cfg_get(&speed_str, "network.eth_speed");
    if (!strcmp(speed_str, "10")) {
	eth_speed = PP_ETH_SPEED_10;
    } else if (!strcmp(speed_str, "100")) {
	eth_speed = PP_ETH_SPEED_100;
    } else {
	eth_speed = PP_ETH_SPEED_AUTO;
    }

    
    pp_cfg_get(&duplex_str, "network.eth_duplex");
    if (!strcmp(duplex_str, "half")) {
	eth_duplex = PP_ETH_DUPLEX_HALF;
    } else if (!strcmp(duplex_str, "full")) {
	eth_duplex = PP_ETH_DUPLEX_FULL;
    } else {
	eth_duplex = PP_ETH_DUPLEX_AUTO;
    }
    
    pp_eth_set_parameters(if_str, eth_speed, eth_duplex);
    sleep(3);
    if ((state_str = pp_eth_get_link_state(if_str)) != NULL) {
	printf("LAN interface: %s\n", state_str);
    }
    free(if_str);
    free(speed_str);
    free(duplex_str);
    free(state_str);
#endif
}

static void
reconfigure_device(void)
{
    printf("\nConfiguring device ...\n");
    pp_cfg_save(DO_FLUSH);
    if (pp_base_get_pid_of_program_by_pidfile(ERIC_PIDFILE, ERIC_PROGRAM_PATH) <= 0) {
	/* eric not running - reconfiguring by our own */
	do_lanif_config();
	pp_system("/etc/rc.net > /dev/null 2>&1");
    }
    printf("Done.\n");
}

static void
exit_handler(void)
{
    int wd_fd;

    if (umcfg_initialized) {
	pp_um_cleanup();
	pp_cfg_cleanup();
    }
    eric_config_cleanup();
    pp_base_cleanup();
    if (g_defaults_set == DEFAULTS_JUMPER) {
	printf("You should remove the defaults-jumper now and restart the device.\n");
    } else {	
	printf("Rebooting...\n");
	sync();
	usleep(500000);
	reboot(RB_AUTOBOOT);
    }
    if ((wd_fd = open("/dev/watchdog", O_WRONLY)) == -1) {
	while(1) { pause(); }
    }

    while (1) {
	write(wd_fd, "\0", 1);
	sleep(1);
    }
}

static void
conf_at_layer(pp_profile_type_t type, 
              const char * displaystr, const char * key) {
    char * value = NULL;
    char input[MAXINPUTSIZE];
    char prompt[MAXPROMPTSIZE];
    int error, i=0, saved = 0;
    const char *serial_speeds[] = { "115200", "57600", "38400", "19200", "9600", "4800", "2400", "1200", NULL };
    

    /* get data from config */
    pp_cfg_get_nodflt(&value, key);

    /* display and ask user */
    error = 1;
    while (error) {
	error = 0;

	if (value != NULL) {
	    snprintf(prompt, MAXPROMPTSIZE, "%s [%s]: ",displaystr, value);
	} else {
	    snprintf(prompt, MAXPROMPTSIZE, "%s: ", displaystr);
	}

	if (cmdedit_read_input(prompt, input)) {
	    /* check data */
	    if (!strcmp(input,"\n") && value != NULL) {
		strncpy(input, value, MAXINPUTSIZE-1);
		input[MAXINPUTSIZE - 1] = '\0';
	    } else {
		input[strlen(input) - 1] = '\0';
	    }
	    if (strstr(input, "\010") || strstr(input, "\027")) {
		/* this should fix #625 (backspaces leaked into config data */
		error++;
	    }
	    if (!strcmp(key, "network.ip_auto_config_proto")) {
		strtolower(input);
		if (strcmp(input, "none") && strcmp(input, "dhcp") && strcmp(input, "bootp")) {
		    error++;
		}
	    } else if (!strcmp(key, "network.ipaddr") || !strcmp(key, "network.netmask") || !strcmp(key, "network.gateway")) {
		if (!strcmp(key, "network.gateway") && (!strcmp(input, "0.0.0.0") || input[0] == '\0')) {

		    input[0] = '\0';
		} else {
		    error += pp_check_ip_syntax(input) == -1;
		}
		} else if (!strcmp(key, "serial")) {
			
	    } else if (!strcmp(key, "network.mac")) {
		error += pp_check_mac_syntax(input) == -1;
	    } else if (!strcmp(key, "security.ip_fw.enabled")) {
		strtolower(input);
		if (strcmp(input, "yes") && strcmp(input, "no")) {
		    error++;
		}
	    } else if (!strcmp(key, "network.eth_speed")) {
		strtolower(input);
		if (strcmp(input, "auto") && strcmp(input, "10") && strcmp(input, "100")) {
		    error++;
		}
	    } else if (!strcmp(key, "network.eth_duplex")) {
		strtolower(input);
		if (strcmp(input, "auto") && strcmp(input, "half") && strcmp(input, "full")) {
		    error++;
		}
	    } else if (!strcmp(key, "serialport[0].configlogin.speed")) {
		error = 1; i = 0;
		while (serial_speeds[i]) {
		    if (!strcmp(input, serial_speeds[i])) { error = 0; break; }		    
		    i++;
		}
	    } else if (!strcmp(key, "serialport[0].configlogin.data")) {
		if (strcmp(input, "8") && strcmp(input, "7")) {
		    error++;
		}
	    } else if (!strcmp(key, "serialport[0].configlogin.parity")) {
		strtolower(input);
		if (strcmp(input, "none") && strcmp(input, "odd") && strcmp(input, "even")) {
		    error++;
		}
	    } else if (!strcmp(key, "serialport[0].configlogin.stop")) {
		if (strcmp(input, "1") && strcmp(input, "2")) {
		    error++;
		}	
	    } else if (!strcmp(key, "serialport[0].configlogin.handshake")) {
		strtolower(input);
		if (strcmp(input, "none") && strcmp(input, "hw")  && strcmp(input, "sw")) {
		    error++;
		}
	    } else if (!strcmp(key, "wlan.essid")) {
		// no check necessary
	    } else if (!strcmp(key, "wlan.wep_key")) {
		if (!strcmp(input, "no") || !strcmp(input, "")) {
		    pp_cfg_set_enabled_at_layer(type, 0, "wlan.wep_enabled");
		} else {
		    pp_cfg_set_enabled_at_layer(type, 1, "wlan.wep_enabled");
		}
	    } else if (!strcmp(key, "wlan.mode")) {
		if (strcmp(input, "managed") && strcmp(input, "adhoc")) {
		    error++;
		}
	    } else if (!strcmp(key, "wlan_security_type")) {
		if (strcmp(input, "none") && strcmp(input, "WEP")  && strcmp(input, "WPA")) {
		    long input_long;
		    char * ep;
		    input_long = strtol(input, &ep, 10);
		    if (*input != '\0' && *ep == '\0') {
			/* input is numeric */
			switch (input_long) {
			    case 0: strncpy(input, "none", MAXINPUTSIZE);
				    break;
			    case 1: strncpy(input, "WEP", MAXINPUTSIZE);
				    break;
			    case 2: strncpy(input, "WPA", MAXINPUTSIZE);
				    break;
			    default:
				    error++;
			}
		    } else {
			error++;
		    }
		}
	    } else if (!strcmp(key, "wlan_wpa_type")) {
		if (strcmp(input, "WPA-PSK") && strcmp(input, "WPA-EAP-PEAP-MSCHAPv2")) {
		    long input_long;
		    char * ep;
		    input_long = strtol(input, &ep, 10);
		    if (*input != '\0' && *ep == '\0') {
			/* input is numeric */
			switch (input_long) {
			    case 0: strncpy(input, "WPA-PSK", MAXINPUTSIZE);
				    break;
			    case 1: strncpy(input, "WPA-EAP-PEAP-MSCHAPv2", MAXINPUTSIZE);
				    break;
			    default:
				    error++;
			}
		    } else {
			error++;
		    }
		}
	    } else if (!strcmp(key, "wlan_wpa_psk_key")) {
		if (strlen(input) < 8 || strlen(input) > 63) {
		    error++;
		}
	    } else if (!strcmp(key, "wlan_wpa_eap_peap_mschapv2_identity")) {
		// currently no check
	    } else if (!strcmp(key, "wlan_wpa_eap_peap_mschapv2_password")) {
		// currently no check
#ifdef PRODUCT_PDU
	    } else if (!strcmp(key, "device.model")) {
		int no;
		if ((no = pp_hal_rpc_model_idx_by_name(input)) >= 0) no++;
		else no = atoi(input);
		if (0 > pdutype_set(no)) error++;
		else saved = 1;
#endif
	    } else {
		/* insert check here! */
		error++;
	    }
	} else {
	    /* not read something, or reading error - we should ask our question again
	       otherwise the uninitialised buffer "input" is saved! */
	    error++;
	}
    }
    /* set data to config */
    if (!saved) pp_cfg_set_at_layer(type, input, key);
    free(value);
}

static void
strtolower(char *s)
{
    while (*s) { *s = tolower((unsigned char)*s); s++; }
}

#ifdef PRODUCT_PDU
static const char *bmc_sensors_thresh_up_crit =
    "bmc.sensors[%u].thresh.up_crit";
static const char *bmc_sensors_thresh_up_noncrit =
    "bmc.sensors[%u].thresh.up_noncrit";

#define INIT_MAX_RMS_CURRENT_PER_OUTLET(__outlet__, __current__) \
    pp_cfg_set_int_at_layer(PP_PROFILE_SYSTEM, (__current__), \
                            bmc_sensors_thresh_up_noncrit, \
                            RPC_RECEPTACLE_SENSOR_RMS_CURRENT((__outlet__))); \
    pp_cfg_set_int_at_layer(PP_PROFILE_SYSTEM, (__current__), \
                            bmc_sensors_thresh_up_crit, \
                            RPC_RECEPTACLE_SENSOR_RMS_CURRENT((__outlet__)));

int pdutype_set(int model) {
    int idx = model - 1, i, max_rms_curr, max_unit, max_board;
    
    if (idx < 0 || idx >= pp_hal_rpc_models_sz)
	return -1;

    pp_cfg_set_at_layer(PP_PROFILE_SYSTEM, 
                        pp_hal_rpc_models[idx].model, "device.model");
    
    /* remove all predefined sensor thresholds */
    pp_cfg_remove_at_layer(PP_PROFILE_SYSTEM, "bmc.sensors");
    pp_cfg_remove_at_layer(PP_PROFILE_LOCAL, "bmc.sensors");
    
    /* calculate max rms current per outlet min(max(unit), max(board)) */
    max_unit = pp_hal_rpc_models[idx].curr_max /
               pp_hal_rpc_models[idx].no_outlets;
    max_board = pp_hal_rpc_models[idx].curr_max_board /
                pp_hal_rpc_models[idx].no_outlets_per_board;
    max_rms_curr = max_unit < max_board ? max_unit : max_board;
    
    /* set system defaults */
    for(i = 0; i < pp_hal_rpc_models[idx].no_outlets; ++i) {
        INIT_MAX_RMS_CURRENT_PER_OUTLET(i, max_rms_curr);
    }

    /* if there was some model reconfiguration, shrink ol-vector to new size */
    pp_cfg_set_int(pp_hal_rpc_models[idx].no_outlets, "bmc.rpc.outlet._s_");

    return 0;
}
#endif	  
