#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <pp/base.h>
#include <pp/cfg.h>
#include <liberic_config.h>
#if defined(PP_FEAT_RPCCFG)
# include <pp/rpc_ripc.h>
#endif

int get_config_main(int argc, char ** argv); /* prototype to avoid warning */

static void
parse_error(u_int line)
{
    fprintf(stderr, "parse error in line %u\n", line + 1);
}

static int need_config_key(const char *key) {
    char *trans_key;
    int ret;

    trans_key = pp_cfg_key_transform(key, 1);
    ret = getenv(trans_key) ? 0 : 1;

    free(trans_key);
    return ret;
}

static int need_config(vector_t *keys) {
    u_int keys_sz, u;
    
    keys_sz = vector_size(keys);
    for(u = 0; u < keys_sz; ++u) 
        if(need_config_key((char*)vector_get(keys, u)))
            return 1;
        
    return 0;
}

static void
do_config_line(const char * key, const char * def_value,
	       int set_mode, int remove_mode, int preserve_keyidx, int preserve_env)
{
    char * value = NULL;
    char * esc_value, *trans_key;

    if (set_mode) {
	pp_cfg_set(def_value, key);
    } else if (remove_mode) {
        pp_cfg_remove(key);
    } else if(!preserve_env || need_config_key(key)) {
	pp_cfg_get_nodflt(&value, key);
	esc_value = value ? pp_cfg_value_shellesc(value)
                          : pp_cfg_value_shellesc(def_value);
	trans_key = pp_cfg_key_transform(key, !preserve_keyidx);
	printf("%s=\"%s\"\n", trans_key, esc_value ? esc_value : "");
	free(trans_key);
	free(esc_value);
	free(value);
    }
}

static void
do_config(int set_mode, int remove_mode, int preserve_keyidx, int do_flush, int preserve_env,
          vector_t *keys, vector_t *values)
{
    u_int line, v_sz;
    char *key;
    char *def_value;
    pp_cfg_flush_t flush_config;
    
    v_sz = vector_size(keys);
    assert(v_sz == vector_size(values));
    for (line = 0; line < v_sz; line++) {
        key = (char*)vector_get(keys, line);
        def_value = (char*)vector_get(values, line);
	do_config_line(key, def_value, set_mode, remove_mode, preserve_keyidx, preserve_env);
    }
    if (set_mode || remove_mode) {
	flush_config = do_flush ? DO_FLUSH : DONT_FLUSH;
	pp_cfg_save(flush_config);
    }
}

static void get_stdin(vector_t *keys, vector_t *values) {
    u_int line;
    char line_buf[1024];
    char key[64];
    char def_value[256];
    char nl[2];
    
    vector_new(keys, 20, free);
    vector_new(values, 20, free);
    for (line = 0; fgets(line_buf, sizeof(line_buf), stdin); line++) {
        int n = sscanf(line_buf, "%63[^= \t\n]=%255[^\n]%1[\n]",
		       key, def_value, nl);
	if (n >= 0 && n < 3) {
	    n = sscanf(line_buf, "%63[^= \t\n]=%1[\n]", key, nl);
	    if (n >= 0 && n < 2) {
		parse_error(line);
		while (!feof(stdin) && !ferror(stdin) && fgetc(stdin) != '\n');
	    } else if (n == 2) {
                vector_add(keys, strdup(key));
                vector_add(values, strdup(""));
	    }
	} else if (n == 3) {
            vector_add(keys, strdup(key));
            vector_add(values, strdup(def_value));
	}
    }
}

int
get_config_main(int argc, char ** argv)
{
    int command;
    int ret = -1;
    int set_mode=0, remove_mode=0, do_flush=0, preserve_keyidx=0, preserve_env = 0;
    vector_t keys, values;
    
    get_stdin(&keys, &values);
    
    while ((command = getopt(argc, argv, "cefirs")) != -1) {
	switch(command) {
	  case 's': /* set the config keys */
	      set_mode++;
	      break;
          case 'r': /* remove the config keys */
              remove_mode++;
              break;
	  case 'f': /* flush configuration, only in conjunction with -s or -r */
	      do_flush++;
	      break;
	  case 'i':
	      preserve_keyidx++;
	      break;
	  case 'c':
	      pp_cfg_destroy();
	      return 0;
          case 'e': /* parse environment */
              if(!need_config(&keys))
                  goto exit;
              preserve_env++;
	      break;
	}
    }
    
    if (pp_base_init("get_config", LOG_NOT_SILENT) != 0) {
	fprintf(stderr, "Initializing base-library failed.\n");
	goto fail;
    }

#if defined(PP_FEAT_RPCCFG)
    /* pp_rpc_am_i_master() can be called even if libpp_rpc is not intialized! */
    if (set_mode || !pp_rpc_am_i_master()) {
	if (PP_FAILED(pp_rpc_init())) {
	    fprintf(stderr, "Initializing rpc library failed .. \n");
	    goto fail;
	}
    }
#endif /* PP_FEAT_RPCCFG */

    if (eric_config_init(FLUSH_IN_FOREGROUND) != 0) {
	fprintf(stderr, "Initializing config-library failed.\n");
	goto fail;
    }

    if (PP_ERR == pp_cfg_init(PP_CD_FNAME_DEFAULT, 
                              PP_CD_OEM_SKEL_FNAME_DEFAULT)) abort();

    do_config(set_mode, remove_mode, preserve_keyidx, do_flush, preserve_env, 
              &keys, &values);
    
    ret = 0;

 fail:

    pp_cfg_cleanup();
    eric_config_cleanup();
#if defined(PP_FEAT_RPCCFG)
    pp_rpc_cleanup();
#endif /* PP_FEAT_RPCCFG */
    pp_base_cleanup();

 exit:
 
    vector_delete(&keys);
    vector_delete(&values);
    return ret;
}
