/*******************************************************************************
********************************************************************************
***                                                                           **
***                    COPYRIGHT 1999 INTEL CORPORATION                       **
***                                                                           **
***                INTEL CORPORATION PROPRIETARY INFORMATION                  **
***                                                                           **
***      This software is supplied under the terms of a license agreement     **
***      or non-disclosure agreement with Intel Corporation and may not be    **
***      copied or disclosed except in accordance with the terms of that      **
***      agreement.                                                           **
***                                                                           **
***      This Source Code is provided "as is" without any warranty of any     **
***      kind, express, implied, statutory or otherwise, and Intel            **
***      specifically disclaims any implied warranties for any particular     **
***      purpose.                                                             **
***                                                                           **
********************************************************************************
******** 10 ****** 20 ****** 30 ****** 40 ****** 50 ****** 60 ****** 70 *******/

/*******************************************************************************

    Source Control Information:

    $Workfile:   common.c  $

    $Revision: 1.1 $

    $Date: 2002/03/23 08:09:18Z $

    $Modtime:   23 Nov 1999 09:22:36  $

    $Archive:   R:/FW/NVSTLTS.PRJ/COMMON.C_V  $

*******************************************************************************/ 

/*******************************************************************************

   Module description: 

   This is the source code for the common interface code,
   it makes use of the ISA or SMIC interface code based on which is present.

  There is an API that provides access to any available interface and another built
  on top that allows you to configure one and then not have to provide intf info on
  each call.

  The low level API is:

	InterfaceHandle	InterfaceInit( char * interfaceName )

			Given an interface name, this function will try to find and initialize
			the requested interface.  If the interface name NULL is passed
			in, the first interface found will be initialized and returned.
			It will return a handle to the interface if	successfull, NULL otherwise.

	int				SendRequest( InterfaceHandle intf, MicroReq_t * cmd )
			
			This function will send the command indicated by 'cmd' to the micro whose
			slave address is in the MicroReq_t.
			NOTE: This implementation is NOT multithreaded and the response to a request
			must be read before a request is made on this interface.

	int				ReadResponse( InterfaceHandle intf, MicroResp_t *response )

			This function is used to collect the response to a command sent by the
			SendRequest() funtion.

	The high level API used my most utilities that don't care about accessing
	multiple interfaces is:

	BOOL			FindInterface( char * interfaceName )

			This function calls InterfaceInit() with the provided interface type and
			if successfull, the returned handle is saved in a static variable to be
			used by the next two functions.

	int				SendRequestToMicro( MicroReq_t * cmd )
	int				ReadResponseFromMicro( MicroResp_t *response )

			These two functions provide the exact same functionality as SendRequest()
			and ReadResponse() except they use the interface handle saved by
			FindInterface().
	
*******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>

#define INTERFACE_PRIVATE

typedef	struct _InterfaceApi * InterfaceHandle;

#include "common.h"
#include "isa.h"
// #include "smic.h"

static BYTE	mSeqNum;
static BYTE	mPrevSeqNum;

WORD	gIPMBRespTimeInMS;
ULONG	gIPMBTryCount;
ULONG	gSMSTOCount;
ULONG	gNAKWCount;
ULONG	gNAKACount;
ULONG	gLARBCount;
ULONG	gBERRCount;
BYTE	Debug = 0;
BYTE	gIpmiVersion = IPMI_1_0;	// figure this out later
BYTE Verbose = 0;


static MicroReq_t	LastReq;	// used for re-transmissions

clock_t clock_start_time;
//
// This points to the HW interface routines
//
static InterfaceHandle Interface = NULL;

//
// table of known interfaces, currently only ISA (KCS). NOT SMIC
//
static InterfaceHandle ( * Interfaces[] )() =
{
	IsaIntfPresent,
	NULL
};

static void FlushSMSBuffer( InterfaceHandle);

/*******************************************************************************
*  Routine: PrintStandardDisclaimer
*
*  Arguments:
*     N/A
*
*  Returns:
*     N/A
*
*  Description:
*     Prints the standard disclaimer message.
*
*******************************************************************************/

#define YEAR	(__DATE__+7)
#define TIME	(__TIMESTAMP__)

void PrintStandardDisclaimer(void)
{
   printf("Copyright (c) Intel Corporation %s\n", YEAR);
   printf("\n");
   printf("Disclaimer:\n");
   printf("Software is provided \"AS IS\" with no warranties whatsoever. The hardware\n");
   printf("vendor remains solely responsible for the design, sale and functionality\n");
   printf("of its product including any liability arising therefrom.\n");
   printf("\n");
   printf("Version: %s\n", REVISION);
   printf("Date: %s\n", TIME);
}


//
// delay for the given number of microseconds 
// 
void usDelay( WORD usecs )
{
   WORD ticks=(WORD) (0.596590*(++usecs));

   if (ticks==0)
      return;

   _asm {
      mov  cx,ticks
      mov  bx,02h
   waitfortrig:
      in   al,41h
      and  al,02h
      cmp  al,bl
      jz   waitfortrig
      xor  bl,02h
      loop waitfortrig
      }
}


void msDelay(WORD msecs)
{
   if (msecs==0)
      return;

   _asm {
      mov  dx,msecs
   start:
      mov  cx,42h
      mov  bx,10h
   waitonbit:
      in   al,61h
      and  al,10h
      cmp  al,bl
      jz   waitonbit
      xor  bl,10h
      loop waitonbit
      dec  dx
      jnz  start
   }
}


/*******************************************************************************
*  Routine: SleepMs
*
*  Arguments:
*     milliseconds - the number of milliseconds to sleep
*
*  Returns:
*     N/A
*
*  Description:
*     Pauses for a specified number of milliseconds.
*
*******************************************************************************/
void SleepMs(UINT milliseconds)
{
   msDelay((WORD) milliseconds);
}


/*******************************************************************************
*  Routine: checkMsTimeout
*
*  Arguments:
*     goal - 'goal' value for the 'tick clock'
*
*  Returns:
*     TRUE if the current 'tick clock' value exceeds the specified tick clock
*        'goal' value
*     FALSE otherwise
*
*  Description:
*     This routine returns true if the current 'tick clock' value exceeds
*     the specified tick clock 'goal' value.
*
*******************************************************************************/
BOOL CheckMsTimeout(clock_t goal)
{
   clock_t clock_reading;

   clock_reading = clock();
   
   if (clock_reading >= goal)
   {
#ifdef DEBUG
      printf("clock start time:   %lX\n", clock_start_time);
      printf("clock current time: %lX\n", clock_reading);
      printf("goal time:          %lX\n", goal);
#endif
      return TRUE;
   }
   else
      return FALSE;
}

/*******************************************************************************
*  Routine: setMsTimeout
*
*  Arguments:
*     goal - gets set to the minimum 'tick clock' value
*     timeoutInMs - timeout in ms for which to set "goal"
*
*  Returns:
*     N/A (modifies "goal")
*
*  Description:
*     This routines sets the goal parameter to the minimum 'tick clock' value
*     required to achieve a timeout that is <= the specified timeout in ms.
*     The companion routine 'checkMsTimeout' compares the current 'tick clock'
*     value to the goal, and returns TRUE if the timeout has occurred.
*
*******************************************************************************/
void SetMsTimeout( clock_t *goal, UINT timeoutInMs )
{
   clock_t wait;

   wait = ((clock_t) timeoutInMs * CLOCKS_PER_SEC) / 1000L;

   /* round up to next higher tick if non-zero remainder */
   if ((wait % 1000L) != 0L)
      wait++;

   clock_start_time = clock();
   *goal = clock_start_time + wait;
}

#define MAX_PER_LINE	  16

void PrintBuffer( char * name, BYTE *buf, UINT length )
{
	UINT count = 0;

	printf( "%s:\n", name);
		
	for ( ; count < length ; count++ )
	{
		if( count && (count % MAX_PER_LINE) == 0 )
		{
			putchar('\n');
		}
		printf( "%02x ", buf[count] );
	}
	putchar( '\n' );
}

/*******************************************************************************
*  Routine: ChecksumRecord
*
*  Arguments:
*     Buffer - buffer to checksum
*     DataLength - length of buffer, includes to-be-added checksum
*
*  Returns:
*     N/A
*
*  Description:
*     Adds a checksum to the end of the buffer.  The checksum is a
*     two's complement of the sum of the buffer (in a BYTE).
*
*******************************************************************************/
void ChecksumRecord( BYTE * Buffer, UINT DataLength )
{
   UINT i;
   UINT Sum;
   
   Sum = 0;
   DataLength--;
   
   for(i=0; i < DataLength; i++)
   {
      Sum += Buffer[i];
   }
   Buffer[DataLength] = (BYTE)(((~Sum) + 1) & 0xFF);
}


/*******************************************************************************
*  Routine: CheckCompletionCode
*
*  Arguments:
*     CompletionCode - the completion code to check
*
*  Returns:
*     N/A
*
*  Description:
*     Checks the completion code, if the code is not "NORMAL" the
*     routine prints (if verbose enabled) an error message.
*
*******************************************************************************/

struct Assoc
{
	BYTE value;
	char *	str;
} CompCodes[] =
{
	{ COMPCODE_NORMAL,		"Ok" },
	{ COMPCODE_NODEBUSY,	"NODE BUSY" },
	{ COMPCODE_INVALIDCMD,	"INVALID COMMAND" },
	{ COMPCODE_INVALIDLUN,	"INVALID LUN" },
	{ COMPCODE_TIMEOUT,		"TIMEOUT" },
	{ COMPCODE_OUTOFSPACE,	"OUT OF SPACE" },
	{ COMPCODE_RESVTNCNCLD,	"RESERVATION CANCELED" },
	{ COMPCODE_REQDATATRUNC,"REQUEST DATA TRUNCATED" },
	{ COMPCODE_REQDATABADLEN,	"REQUEST DATA BAD LENGTH" },
	{ COMPCODE_REQDATATOOLONG,	"REQUEST DATA TOO LONG" },
	{ COMPCODE_OFFSETOUTOFRANGE,"OFFSET OUT OF RANGE" },
	{ COMPCODE_RESPDATATRUNC,	"RESPONSE DATA TRUNCATED" },
	{ COMPCODE_NOTFOUND,		"NOT FOUND" },
	{ COMPCODE_INVALIDFIELD,	"INVALID FIELD" },
	{ COMPCODE_COMMANDILLEGAL,	"COMMAND ILLEGAL" },
	{ COMPCODE_NORESPONSE,		"NO RESPONSE" },
	{ COMPCODE_UNSPECIFIED,		"UNSPECIFIED" },
	{ I2C_WRITE_STATUS_LOST_ARB,	"I2C WRITE - LOST ARBITRATION" },
	{ I2C_WRITE_STATUS_BUS_ERROR,	"I2C WRITE - BUS ERROR" },
	{ I2C_WRITE_STATUS_NAK_WRITE,	"I2C WRITE - NAK ON WRITE" },
	{ I2C_WRITE_STATUS_TRUNC_READ,	"I2C WRITE - TRUNCATED READ" },
	{ I2C_WRITE_STATUS_NAK_ADDRESS,	"I2C WRITE - NAK ON ADDRESS" },
	{ I2C_WRITE_STATUS_BAD_LENGTH,	"I2C WRITE - BAD LENGTHS" },
	{ 0x00,	NULL },
};

UINT CheckCompletionCode( BYTE cCode )
{ 
	char *str = NULL;
	
	if( cCode != COMPCODE_NORMAL )
	{
		struct Assoc *a = CompCodes;
		
		while( a->str != NULL )
		{
			if( cCode == a->value )
			{
				str = a->str;
				break;
			}
			a++;         
		}
		if( str )
			vprintf( stderr, "Error from micro: %s\n", str);
		else
			vprintf( stderr, "Error from micro: unrecognized completion code %02Xh\n", cCode);
	}
   return (cCode != COMPCODE_NORMAL);
}

/*******************************************************************************
*  Routine: InterfaceInit
*
*  Arguments:
*     ifType - requested interface type, or Unknown
*
*  Returns:
*     TRUE if interface found, FALSE otherwise
*
*  Description:
*     Initializes interface related state
*
*******************************************************************************/

InterfaceHandle InterfaceInit( char * intfName )
{
	int				i;
	InterfaceHandle h = NULL;
	//
	// initialize interface related varibles
	//
	gIPMBTryCount = 0;
	gNAKWCount = 0;
	gNAKACount = 0;
	gLARBCount = 0;
	gBERRCount = 0;
	gSMSTOCount = 0;

	srand( (unsigned)time( NULL ) );

	mSeqNum = (BYTE)(rand() & 0xFF);
	//
	// look for the interface
	//
	for ( i = 0 ; Interfaces[i] ; i++ )
	{
		if( (h = Interfaces[i]( intfName )) != NULL )
		{
			break;
		}
	}
	return h;
}

BOOL FindInterface( char * intfName )
{
	Interface = InterfaceInit( intfName );

	return Interface != NULL;
}

void SetIPMIversion( byte vers )
{
	gIpmiVersion = vers;
}
/*******************************************************************************
*  Routine: wait_for_SMS_message_ready
*
*  Arguments:
*     interface to check
*
*  Returns:
*     TRUE - if message ready
*     FALSE - if timed out with no message ready
*
*  Description:
*     Waits for the interface to signal that SMS message is ready to be read
*
*******************************************************************************/

static BOOL wait_for_SMS_message_ready( InterfaceHandle intf )
{
   clock_t overallTimeout;  

   SetMsTimeout ( & overallTimeout, SMS_TIMEOUT );

   gIPMBRespTimeInMS = 0;

   while ( ! intf->smsReady( intf ) )
   {
      msDelay(1);
      gIPMBRespTimeInMS++;
      
      if ( CheckMsTimeout(overallTimeout) ) 
      {
         return FALSE; /* timed out with no message ready */
      }              
   }
   return TRUE; /* message ready */
} 

/*******************************************************************************
*  Routine: BusExists
*
*  Arguments:
*     Bus - public or private bus
*
*  Returns:
*     N/A
*
*  Description:
*     Returns TRUE if bus exists, false otherwise
*
*******************************************************************************/
BOOL BusExists(BYTE Bus)
{
   BYTE  CmdBuffer[MAX_PACKET_LEN + 2];
   BYTE  ResponseBuffer[MAX_PACKET_LEN];
   int   BytesRead;
   int   DataBytesRead;
   BYTE  I2CAddress;
   BOOL  BusIsPresent;  
   MicroReq_t *		cmd = (MicroReq_t *) CmdBuffer;
   MicroResp_t *	resp = (MicroResp_t *) ResponseBuffer;

   cmd->ipmbAddr = 0x00;
   cmd->length	 = 4;
   
   cmd->bmcReq.netFnLn	= APP_REQUEST;
   cmd->bmcReq.cmd		= MASTER_WRITE_I2C;
   cmd->bmcReq.data[0]	= Bus;
   cmd->bmcReq.data[1]	= I2CAddress;

   if( SendRequestToMicro( cmd, resp ) != 0)
   {
      fprintf(stderr,"Error scanning I2C bus at address %04X.\n",I2CAddress);
      return 0;
   }
   resp->ipmbAddr = BMC_DEV_ADDR;
   
   BytesRead = ReadResponseFromMicro( resp );

   DataBytesRead = BytesRead - RSP_OFFSET_DATA;
   
   if(DataBytesRead != 0)
   {
      fprintf(stderr,"\nError - expected 0 data bytes, read %d.\n", DataBytesRead);
      return 0;
   }

   if(ResponseBuffer[RSP_OFFSET_COMPCODE] == COMPCODE_OFFSETOUTOFRANGE)
   {
      BusIsPresent = FALSE;
   }
   else
   {
      BusIsPresent = TRUE;
   }
   return BusIsPresent;
}


/*******************************************************************************
*  Routine: IsI2CRetryNeeded
*
*  Arguments:
*     CompCode - the completion code from an I2C operation
*
*  Returns:
*     TRUE if retry needed
*     FALSE otherwise
*
*  Description:
*     This routine returns TRUE if an I2C retry is needed
*     
*******************************************************************************/
static BOOL IsI2CRetryNeeded(BYTE CompCode)
{
   BOOL Retry;

   gIPMBTryCount++;   
   switch(CompCode)
   {
      case I2C_WRITE_STATUS_NAK_WRITE:
         Retry = TRUE;
         fprintf(stderr, "NAKW ");
         gNAKWCount++;
         break;
      case I2C_WRITE_STATUS_NAK_ADDRESS:
         Retry = TRUE;
         fprintf(stderr, "NAKA ");
         gNAKACount++;
         break;
      case I2C_WRITE_STATUS_LOST_ARB:
         Retry = TRUE;
         fprintf(stderr, "LARB ");
         gLARBCount++;
         break;
      case I2C_WRITE_STATUS_BUS_ERROR:
         Retry = TRUE;
         fprintf(stderr, "BERR ");
         gBERRCount++;
         break;
      default:
         Retry = FALSE;
         CheckCompletionCode(CompCode);
         break;
   }
   return Retry;
}

/*******************************************************************************
*  Routine: BuildIpmbReq
*
*  Arguments:
*	  IpmbMsg_t * 	msg 	- where to build the request
*     MicroReq_t *	req		- request providing the cmd/data
*
*  Returns:
*     length of the built request
*
*  Description:
*     This routine builds an IPMB request from a BMC request.
*     
*******************************************************************************/

UINT BuildIpmbReq ( IpmbMsg_t * msg, MicroReq_t * req )
{
	//
	// build an IPMB message
	//
	msg->rsSa		= req->ipmbAddr;		/* slave addr */
	msg->netFnRsLn	= req->bmcReq.netFnLn;	/* NetFn + LUN (LUN = 0) */
	msg->cSum1		= -(msg->rsSa + msg->netFnRsLn);

	msg->rqSa		= BMC_DEV_ADDR;       /* BMC's slave addr       */

	mPrevSeqNum		= (BYTE)(mSeqNum & 0x3F);

	msg->seqRqLn	= (BYTE)(((mSeqNum) << 2) | SMS_LUN);  /* return to ISA */
	msg->cmd		= req->bmcReq.cmd;

	memcpy( msg->data, req->bmcReq.data, req->length );

	ChecksumRecord( & msg->rqSa, req->length + 4);     // add rqsa, seq, cmd, checksum 

	mSeqNum++;

	return req->length + 7;		// add in IPMB overhead (7) 
}

/*******************************************************************************
*  Routine: SendRequest
*
*  Arguments:
*	  InterfaceHandle intf
*     MicroReq_t *	  cmd
*
*  Returns:
*     0 success, -1 failure
*
*  Description:
*     This routine sends commands the BMC or micros on the IPMB.
*     
*******************************************************************************/

int SendRequest( InterfaceHandle intf, MicroReq_t * req, MicroResp_t * rsp, BYTE broadcast )
{
	int			ReturnValue;
	BYTE		RetryCount, respLen, i;

	BmcResp_t	bResp;

	struct BmcSendReq
	{
		BYTE	dest;		// bus/channel
		BYTE	data[2];	// broadcast reqs build IPMB req at data[1]
		
	}* bsReq = (struct BmcSendReq *) LastReq.bmcReq.data;
    
    dprintf( stderr,
		"SendRequest: intf 0x%x, req->ipmbAddr %xh, req->length 0x%x\n",
		(DWORD) intf, req->ipmbAddr, req->length);
    
	if ( req->ipmbAddr == intf->bmcAddr( intf ) )
	{
		if( Debug )
			PrintMicroReq( "Send Req", req );
		
		return intf->sendRequest(intf, & req->bmcReq, req->length);
	}
	//
	// This is to a different controller. Send it through the BMC
	//
	FlushSMSBuffer( intf);

	//
	// build the IPMB message in the last request packet structure
	//
	
	LastReq.ipmbAddr		= intf->bmcAddr( intf );

			LastReq.bmcReq.netFnLn	= APP_REQUEST;

			if ( gIpmiVersion == 0x09 )
			{
				LastReq.bmcReq.cmd	= MASTER_WRITE_I2C;
				bsReq->dest			= PUBLIC_I2C_BUS;
			}
			else
			{
				LastReq.bmcReq.cmd	= SEND_MESSAGE;
				bsReq->dest			= IPMB_CHANNEL_ID;
			}
	if( broadcast )
	{
		LastReq.length = BuildIpmbReq( (IpmbMsg_t *) & bsReq->data[1], req ) + 1;
		bsReq->data[0] = 0;
	}
	else
	{
		LastReq.length = BuildIpmbReq( (IpmbMsg_t *) & bsReq->data[0], req );
	}
	LastReq.length++;	// account for bus/channel byte

	if( Debug )
		PrintMicroReq( "SendMsg Req", &LastReq );
	//
	// Now send it to the BMC, collecting the response to the SendMessage cmd.
	//
	RetryCount = 0;

	do
	{
		if( ReturnValue = intf->sendRequest( intf, & LastReq.bmcReq, LastReq.length) )
			return ReturnValue;			
	
		if( (respLen = intf->readResponse( intf, &bResp)) < MIN_BMC_RESP_SIZE )
			return -1;			

		if( Debug )
			PrintBuffer( "SendMsg Resp", (byte *) & bResp, respLen );

	} 

    while ( IsI2CRetryNeeded( bResp.cCode ) && ++RetryCount < 3 );

	if ( RetryCount >= 3 && bResp.cCode != COMPCODE_NORMAL )
	{
		fprintf( stderr,"ERROR on I2C access, tried three times.\n");
		return -1;
	}

    rsp->respLen  = respLen-3;
    for(i=0; i<sizeof(BmcResp_t); i++)
        *(byte *)((byte *)&(rsp->bmcResp) + i) = *(byte *)((byte *)&bResp + i);
	
	return ReturnValue;
}

int SendRequestToMicro( MicroReq_t * req, MicroResp_t * resp )
{
	return SendRequest( Interface, req, resp, 0 /* no broadcast */ );
}

int BroadcastRequestToMicro( MicroReq_t * req, MicroResp_t * resp )
{
	return SendRequest( Interface, req, resp, 1 /* broadcast */ );
}

/*******************************************************************************
*  Routine: ResendLastRequest
*
*  Arguments:
*     N/A
*
*  Returns:
*     0 success, -1 failure
*
*  Description:
*     This routine resends the last request packet.
*     
*******************************************************************************/

static int ResendLastRequest( InterfaceHandle intf)
{
	int			ReturnValue;
	BYTE		RetryCount;
	BmcResp_t	bResp;
	byte		respLength;

	if( Debug )
		PrintMicroReq( "ReSend Req", & LastReq );

	RetryCount = 0;

	do
	{
		if( ReturnValue = intf->sendRequest( intf, & LastReq.bmcReq, LastReq.length) )
			return ReturnValue;

		if( (respLength = intf->readResponse( intf, & bResp)) < MIN_BMC_RESP_SIZE )
			return -1;

		if( Debug )
			PrintBuffer( "ReSend Resp", (byte *) & bResp, respLength );

	} while ( IsI2CRetryNeeded( bResp.cCode ) && ++RetryCount < 3 );

	if ( RetryCount >= 3 )
	{
		fprintf( stderr, "ERROR on I2C access, tried three times.\n");
		return -1;
	}
	return ReturnValue;
}

/*******************************************************************************
*  Routine: ReadSmsBuffer
*
*  Arguments:
*     interface handle
*	  response structure
*
*  Returns:
*     length of response data, not including the SA, netFnLn, cmd, or cCode
*
*  Description:
*     This routine reads the Sms buffer and parses the IPMB response.
*     
*******************************************************************************/

int ReadSmsBuffer ( InterfaceHandle intf, MicroResp_t * response )
{
	BmcReq_t	bReq;
	BmcResp_t	bResp;
	byte		reqLength;
	int			respLength;
	BYTE		RetryCount;
	IpmbMsg_t *	iRsp;

	RetryCount = 0;

	do
	{
		if( wait_for_SMS_message_ready( intf ) == TRUE )
			break;

		RetryCount++;
		gSMSTOCount++;
		fprintf( stderr, "SMS_MsgRdy_TO%d ", RetryCount);

		ResendLastRequest( intf);

	} while( RetryCount < 3);

	if ( RetryCount >= 3 )
	{
		response->ipmbAddr = intf->bmcAddr( intf );
		fprintf(stderr, "Error - SMS MESSAGE READY timeout, tried three times.\n" );
		return -1;
	}      
	//
	// There seems to be a response for us. Collect it
	//

	bReq.netFnLn = APP_REQUEST;
			
	bReq.cmd = ( gIpmiVersion == 0x09 ) ? READ_SMS_MESSAGE_BUFFER : GET_MESSAGE;

	reqLength = 2;

	if( Debug )
		PrintBmcReq( "ReadSmsReq", & bReq, reqLength );

	if( intf->sendRequest( intf, &bReq, reqLength) != 0 )
	{
		response->ipmbAddr = intf->bmcAddr(intf);
		printf("Error requesting SMS message containing response.\n");
		return -1;
	}
	
	if( (respLength = intf->readResponse( intf, &bResp)) < MIN_BMC_RESP_SIZE )
	{
		response->ipmbAddr = intf->bmcAddr(intf);
		printf("Error getting response to Get Message.(%d)\n", respLength );
		return -1;
	}

	if( Debug )
		PrintBuffer( "ReadSmsResp", (byte *) & bResp, respLength );
	//
	// Make sure we got the response from the BMC ok
	//
	if( bResp.cCode != COMPCODE_NORMAL )
	{
		//
		// bad completion code, just copy the response and bail
		//
		response->ipmbAddr	= intf->bmcAddr(intf);
		response->respLen	= respLength - MIN_BMC_RESP_SIZE;

		memcpy( & response->bmcResp, & bResp, respLength );

		return 0;
	}
	//
	// good response, unpack the message
	//
	if ( gIpmiVersion == 0x09 )
	{
		//
		// overlay the (missing) SA on the cCode
		//
		iRsp = (IpmbMsg_t *) & bResp.cCode;
	}
	else
	{
		//
		// overlay the (missing) SA on the channel number
		//
		iRsp = (IpmbMsg_t *) bResp.data;
	}

	if( SEQ_OF(iRsp->seqRqLn) != mPrevSeqNum )
	{
		printf("ERROR: IPMB resp. packet's seq#  does not match req. packet seq#.\n");
		return -1;
	}
	response->ipmbAddr			= iRsp->rqSa;
	response->bmcResp.netFnLn	= iRsp->netFnRsLn;
	response->bmcResp.cmd		= iRsp->cmd;
	response->bmcResp.cCode		= iRsp->data[0];
	//
	// subtract off the bmc rsp header (2), bmc cCode, channel no
	// nfln, cSum1, rqSa, seq, cmd, ctlr cCode, and csum2
	//
	respLength -= 11;

	memcpy( response->bmcResp.data, & iRsp->data[1], respLength );

	response->respLen = respLength;

	if( Debug )
		PrintBuffer( "MicroResp", (BYTE *) response, respLength + 4 );

	return 0;   
}


/*******************************************************************************
*  Routine: ReadResponseFromBroadcast
*
*  Arguments:
*     IPMBAddress - IPMB address of micro from which to receive response
*     ResponseBuffer - buffer containing response
*
*  Returns:
*     length of ResponseBuffer
*     -1 if failure
*
*  Description:
*     This routine receives responses from the BMC or micros on the IPMB.
*     
****** REWRITE THIS ***********
*
*******************************************************************************/
int ReadResponseFromBroadcast(BYTE ResponseBuffer[])
{
   BYTE ResponsePacket[MAX_PACKET_LEN];
   int ResponsePacketLength;
   int ResponseLength;
   BYTE CmdBuffer[MAX_PACKET_LEN];
   BYTE CmdLength;
   BYTE i;
   BYTE currSeqNum; 
   BOOL Retry;
   BYTE RetryCount;

   RetryCount = 0;
   do
   {
      if( wait_for_SMS_message_ready( Interface ))
      {
         Retry = FALSE;
      }
      else
      {
         ResendLastRequest( Interface);
         RetryCount++;
         /* fprintf(stderr, "SMSMRTO%d ", RetryCount); */
         Retry = TRUE;
      }

   } while( (RetryCount < 3) && (Retry == TRUE));
   
   if ((RetryCount >= 3) && (Retry == TRUE))
   {
      return(-1);
   }      

   CmdBuffer[0] = APP_REQUEST;
   CmdBuffer[1] = READ_SMS_MESSAGE_BUFFER;
   CmdLength = 2;

   if( Interface->sendRequest( Interface, (BmcReq_t *) CmdBuffer, CmdLength) != 0 )
   {
      printf("Error reading SMS message containing response.\n");
      exit(1);
   }

   ResponsePacketLength = Interface->readResponse( Interface, (BmcResp_t *) ResponsePacket);
   
   /* DEBUG++ */
/*
   printf("DEBUG: ");
   for (i = 0; i <= ResponsePacketLength; i++)
   {
      if(i != 0)
         printf(" ");
      printf("%02X", ResponsePacket[i]);
   }
   printf("\n");
*/
   /* DEBUG-- */
   
   if(ResponsePacket[2] != 0x00)
   {
      ResponseBuffer[0] = BMC_DEV_ADDR;
      for(i = 0; i < ResponsePacketLength; i++)
      {
         ResponseBuffer[1 + i] = ResponsePacket[i];
      }
      ResponseLength = 1 + ResponsePacketLength;
   }
   else
   {
      currSeqNum = (BYTE)((ResponsePacket[6] >> 2) & 0x3F);
      if(currSeqNum != mPrevSeqNum)
      {
         printf("ERROR: IPMB resp. packet's seq#  does not match req. packet seq#.\n");
         exit(1);
      }
      
      ResponseBuffer[RSP_OFFSET_ADDR]  = ResponsePacket[5];
      ResponseBuffer[RSP_OFFSET_NETFN] = ResponsePacket[3];
      ResponseBuffer[RSP_OFFSET_CMD]   = ResponsePacket[7];
      for(i=0; i <= (ResponsePacketLength - 8); i++)
      {
         ResponseBuffer[RSP_OFFSET_COMPCODE + i] = ResponsePacket[8 + i];
      }        
      ResponseLength = ResponsePacketLength - 6;
   }

   return(ResponseLength);   
}

/*******************************************************************************
*  Routine: ReadResponse
*
*  Arguments:
*     Interface handle
*     response structure pointer
*
*  Returns:
*     length of response data, not including the SA, netFnLn, cmd, or cCode
*
*  Description:
*     This routine receives responses from the BMC or micros on the IPMB.
*     
*******************************************************************************/

int ReadResponse( InterfaceHandle intf, MicroResp_t *resp )
{
	int respLen;

	if ( resp->ipmbAddr == intf->bmcAddr(intf) )
	{
		respLen = intf->readResponse( intf, & resp->bmcResp);

		if( Debug )
		{
			if( respLen < 0 )
				printf( "Resp: error %d\n");
			else
				PrintBuffer( "Resp", (byte *) & resp->bmcResp, respLen );
		}
		if( respLen < MIN_BMC_RESP_SIZE )
			return -1;

		resp->respLen = (byte)(respLen - MIN_BMC_RESP_SIZE);
		
		return 0;
	}
	//
	// This was to a micro other than the BMC. Get the response
	//
    if (intf->smsReady != NULL)
    	return ReadSmsBuffer( intf, resp );
    else
        // SMS interface not supported
        return 0;
}

int ReadResponseFromMicro( MicroResp_t *response )
{
	return ReadResponse( Interface, response );
}

/*******************************************************************************
*  Routine: FlushSMSBuffer
*
*  Arguments:
*     N/A
*
*  Returns:
*     N/A
*
*  Description:
*     This routine flushes the SMS Buffer.
*     
*******************************************************************************/

static void FlushSMSBuffer( InterfaceHandle intf)
{
	BmcResp_t	bResp;
	BmcReq_t	bReq;
	BYTE		reqLength;

	if( Debug )
		printf( "FlushSMSBuffer\n" );

    if (intf->smsReady != NULL) {
        while( intf->smsReady(intf) )
        {
            bReq.netFnLn = APP_REQUEST;

            bReq.cmd = ( gIpmiVersion == 0x09 ) ? READ_SMS_MESSAGE_BUFFER : GET_MESSAGE;
            reqLength = 0;

            if( Debug )
                printf( "Flushing SMSBuffer\n" );

            if( intf->sendRequest( intf, & bReq, reqLength) != 0 )
            {
                fprintf( stderr, "Error sending Read SMS Message Buffer command while flushing SMS Buffer.\n");
                return;
            }
            //
            // Read and throw away the contents
            //
            (void) intf->readResponse( intf, & bResp);
        }
    }
}

clock_t LastReqTime( void )
{
	return Interface->lastReqTime( Interface );
}



