/*
 * Copyright (c) 2006 Raritan, All Rights Reserved.
 * Author: Ralf Guenther ralf.guenther@raritan.com
 */

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h> 
#include <sys/stat.h>
#include <unistd.h>

#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_raritanoem.h>
#include <ipmitool/log.h>

static int
ipmi_raritanoem_usage(void)
{
    lprintf(LOG_NOTICE, "usage: kiraoem <command> [option...]");
    lprintf(LOG_NOTICE, "");
    lprintf(LOG_NOTICE, "   fw ver");
    lprintf(LOG_NOTICE, "      Get firmware version information.");
    lprintf(LOG_NOTICE, "");
    lprintf(LOG_NOTICE, "   fw upgrade <file>");
    lprintf(LOG_NOTICE, "      Upgrade firmware using binary file.");
    lprintf(LOG_NOTICE, "");
    lprintf(LOG_NOTICE, "   sn get");
    lprintf(LOG_NOTICE, "      Get serial number.");
    lprintf(LOG_NOTICE, "");
    lprintf(LOG_NOTICE, "   sn set <sn>");
    lprintf(LOG_NOTICE, "      Set serial number.");
    lprintf(LOG_NOTICE, "");
    lprintf(LOG_NOTICE, "   resetconfig");
    lprintf(LOG_NOTICE, "      Reset to factory defaults.");
    lprintf(LOG_NOTICE, "");
    return -1;
}

static const struct valstr raritanoem_cmd_strs[] = {
    { IPMI_RARITANOEM_GET_FIRMWARE_VERSION,      "Get Firmware Version" },
    { IPMI_RARITANOEM_START_FIRMWARE_UPGRADE,    "Start Firmware Upgrade" },
    { IPMI_RARITANOEM_UPLOAD_FIRMWARE,           "Upload Firmware" },
    { IPMI_RARITANOEM_FLASH_FIRMWARE,            "Flash Firmware" },
    { IPMI_RARITANOEM_CANCEL_FIRMWARE_UPGRADE,   "Cancel Firmware Upgrade" },
    { IPMI_RARITANOEM_FINALIZE_FIRMWARE_UPGRADE, "Finalize Firmware Upgrade" },
    { IPMI_RARITANOEM_RESET_TO_FACTORY_DEFAULTS, "Reset To Factory Defaults" },
    { IPMI_RARITANOEM_GET_SERIAL_NUMBER,         "Set Serial Number" },
    { IPMI_RARITANOEM_SET_SERIAL_NUMBER,         "Get Serial Number" },
    { IPMI_RARITANOEM_GET_VMEDIA_STATUS,         "Get Virtual Media Status" },
    { IPMI_RARITANOEM_CLOSE_VMEDIA_SESSION,      "Close Virtual Media Session" },
    { IPMI_RARITANOEM_START_FLOPPY_IMG_UPLOAD,   "Start Floppy Image Upload" },
    { IPMI_RARITANOEM_UPLOAD_FLOPPY_IMG,         "Upload Floppy Image" },
    { IPMI_RARITANOEM_FINALIZE_FLOPPY_IMG_UPLOAD,"Finalize FLoappy Image Upload" },
    { IPMI_RARITANOEM_START_SMB_IMG_MOUNT,       "Start SMB Image Mount" },
    { IPMI_RARITANOEM_SET_SMB_IMG_PARAM,         "Set SMB Image Parameter" },
    { IPMI_RARITANOEM_FINALIZE_SMB_IMG_MOUNT,    "Finalize SMB Image Mount" },
    { 0xffff, NULL }
};

static const struct errstr {
    uint8_t cmd;
    uint8_t err;
    const char *str;
} raritanoem_err_strs[] = {
    { IPMI_RARITANOEM_FLASH_FIRMWARE,   0x80, "incomplete upload" },
    { IPMI_RARITANOEM_FLASH_FIRMWARE,   0x81, "bad CRC" },
    { IPMI_RARITANOEM_FLASH_FIRMWARE,   0x82, "firmware invalid" },
    { IPMI_RARITANOEM_FLASH_FIRMWARE,   0x83, "upgrade failed" },
    { IPMI_RARITANOEM_GET_SERIAL_NUMBER,    0x80, "no serial number set" },
    { IPMI_RARITANOEM_SET_SERIAL_NUMBER,    0x80, "serial number already set" },
    { IPMI_RARITANOEM_START_FLOPPY_IMG_UPLOAD,  0x80, "no empty device" },
    { IPMI_RARITANOEM_START_FLOPPY_IMG_UPLOAD,  0x81, "invalid file size" },
    { IPMI_RARITANOEM_START_FLOPPY_IMG_UPLOAD,  0x82, "SMB mount in progress" },
    { IPMI_RARITANOEM_FINALIZE_FLOPPY_IMG_UPLOAD,  0x82, "SMB mount in progress" },
    { IPMI_RARITANOEM_FINALIZE_FLOPPY_IMG_UPLOAD,  0x83, "incomplete upload" },
    { IPMI_RARITANOEM_START_SMB_IMG_MOUNT,  0x80, "no empty device" },
    { IPMI_RARITANOEM_FINALIZE_SMB_IMG_MOUNT,   0x82, "SMB mount in progress" },
    { IPMI_RARITANOEM_FINALIZE_SMB_IMG_MOUNT,   0x84, "image access failed" },
    { 0xff, 0xff, NULL }
};

const char *cmd_str(unsigned char netfn, unsigned char cmd)
{
    return netfn == IPMI_NETFN_RARITANOEM ?
        val2str(cmd, raritanoem_cmd_strs) : "";
}

const char *err_str(unsigned char err, unsigned char netfn, unsigned char cmd)
{
    const char *str = NULL;

    if (netfn == IPMI_NETFN_RARITANOEM) {
        const struct errstr *errs = raritanoem_err_strs;
        for (errs = raritanoem_err_strs; errs->str != NULL; errs++) {
            if (errs->cmd == cmd && errs->err == err) {
                str = errs->str;
                break;
            }
        }
    }

    return str ? str : val2str(err, completion_code_vals);
}

#define SENDRECV(intf, req, rsp) \
    rsp = intf->sendrecv(intf, &req); \
    if (rsp == NULL) { \
        lprintf(LOG_ERR, "%s command failed", \
                cmd_str(req.msg.netfn, req.msg.cmd)); \
        goto exit; \
    } \
    if (rsp->ccode > 0) { \
        lprintf(LOG_ERR, "%s command failed: %s", \
                cmd_str(req.msg.netfn, req.msg.cmd), \
                err_str(rsp->ccode, req.msg.netfn, req.msg.cmd)); \
        goto exit; \
    }

static int
ipmi_raritanoem_fw_ver(struct ipmi_intf * intf)
{
    struct ipmi_rq req;
    struct ipmi_rs * rsp;

    memset(&req, 0, sizeof(struct ipmi_rq));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_GET_FIRMWARE_VERSION;

    SENDRECV(intf, req, rsp);

    int *ver = (int*)rsp->data;
    int hwid = *(rsp->data + 16);
    char *tag = rsp->data + 17;
    char *oem = tag + strlen(tag) + 1;
    int oem_len = rsp->data_len - (oem - (char*)rsp->data);

    printf("Version=%d.%d.%d BuildNo=%d HWID=%02x Tag=%s OEM=%.*s\n",
        ver[0], ver[1], ver[2], ver[3], hwid, tag, oem_len, oem);

exit:
    return 0;
}

static int
ipmi_raritanoem_fw_upgrade(struct ipmi_intf * intf, const char* fw_file)
{
    struct ipmi_rq req;
    struct ipmi_rs * rsp;
    unsigned char buf[256];
    int ret = -1;

    FILE *fp = ipmi_open_file_read(fw_file);
    if (!fp) {
        lprintf(LOG_ERR, "Could not open file '%s'", fw_file);
        goto exit;
    }
    struct stat fs;
    if (fstat(fileno(fp), &fs) < 0) {
        lprintf(LOG_ERR, "Could not determine file size of '%s'", fw_file);
        goto exit;
    }
    int size = (int)fs.st_size;

    printf("Starting Firmware Upgrade\n");
    memset(&req, 0, sizeof(struct ipmi_rq));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_START_FIRMWARE_UPGRADE;
    req.msg.data = (unsigned char*)&size;
    req.msg.data_len = sizeof(size);

    SENDRECV(intf, req, rsp);

    unsigned short reserv = *(unsigned short*)rsp->data;
    unsigned int chunk_size = *(unsigned int*)(rsp->data + 2);


    printf("Uploading Firmware File\n");
    printf("0%%---------------------50%%--------------------100%%\n");

    int offs, p, pold = 0;
#define MAX_CHUNK_SIZE 70
    if (chunk_size > MAX_CHUNK_SIZE) chunk_size = MAX_CHUNK_SIZE;
    for (offs = 0; offs < size; offs += chunk_size) {
        const int hdr_len = sizeof(reserv) + sizeof(offs);

        // fill args
        char buf[hdr_len + MAX_CHUNK_SIZE];
        *(unsigned short*)buf = reserv;
        *(int*)(buf + sizeof(reserv)) = offs;
        int chunk_len = fread(buf + hdr_len, 1, chunk_size, fp);

        p = offs * 50 / size + 1;
        if (p != pold) { printf("*"); fflush(stdout); pold = p; }

        //printf("Upload Firmware (offs=%d size=%d)\n", offs, chunk_len);
        memset(&req, 0, sizeof(struct ipmi_rq));
        req.msg.netfn = IPMI_NETFN_RARITANOEM;
        req.msg.cmd = IPMI_RARITANOEM_UPLOAD_FIRMWARE;
        req.msg.data = buf;
        req.msg.data_len = hdr_len + chunk_len;

        SENDRECV(intf, req, rsp);
    }

    printf("\n");

    // fill args
    *(unsigned short*)buf = reserv;
    buf[2] = 0x01; // validate only!

    printf("Flashing Firmware (takes about 1min)\n");
    memset(&req, 0, sizeof(struct ipmi_rq));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_FLASH_FIRMWARE;
    req.msg.data = buf;
    req.msg.data_len = 3;

    SENDRECV(intf, req, rsp);

    // fill args
    *(unsigned short*)buf = reserv;

    memset(&req, 0, sizeof(struct ipmi_rq));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_FINALIZE_FIRMWARE_UPGRADE;
    req.msg.data = buf;
    req.msg.data_len = 2;

retry:
    rsp = intf->sendrecv(intf, &req);
    if (rsp == NULL || rsp->ccode == 0xff) {
        // retry if Finalize timed out
        sleep(10);
        goto retry;
    }
    if (rsp->ccode > 0) {
        lprintf(LOG_ERR, "%s command failed: %s",
                cmd_str(req.msg.netfn, req.msg.cmd),
                err_str(rsp->ccode, req.msg.netfn, req.msg.cmd));
        goto exit;
    }

    printf("Firmware successfully upgraded, device is rebooting now\n");
    ret = 0;
exit:
    if (fp) fclose(fp);
    return ret;
}

static int
ipmi_raritanoem_sn_get(struct ipmi_intf * intf)
{
    struct ipmi_rq req;
    struct ipmi_rs * rsp;

    memset(&req, 0, sizeof(struct ipmi_rq));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_GET_SERIAL_NUMBER;

    SENDRECV(intf, req, rsp);

    printf("serial number = %.*s\n", rsp->data_len, rsp->data);

exit:
    return 0;
}

static int
ipmi_raritanoem_sn_set(struct ipmi_intf * intf, const char * val)
{
    struct ipmi_rq req;
    struct ipmi_rs * rsp;

    memset(&req, 0, sizeof(struct ipmi_rq));
    req.msg.netfn = IPMI_NETFN_RARITANOEM;
    req.msg.cmd = IPMI_RARITANOEM_SET_SERIAL_NUMBER;
    req.msg.data = (void*)val;
    req.msg.data_len = strlen(val);

    SENDRECV(intf, req, rsp);

    printf("serial number = %s\n", val);

exit:
    return 0;
}

int
ipmi_raritanoem_main(struct ipmi_intf * intf, int argc, char ** argv)
{
    int rc = 0;

    if (argc == 0 || strncmp(argv[0], "help", 4) == 0) {
        ipmi_raritanoem_usage();
        return 0;
    }

    if (strcmp(argv[0], "fw") == 0) {
        uint8_t pct;
        if (argc < 2) return ipmi_raritanoem_usage();
        else if (strcmp(argv[1], "ver") == 0) {
            rc = ipmi_raritanoem_fw_ver(intf);
        }
        else if (strcmp(argv[1], "upgrade") == 0) {
            rc = ipmi_raritanoem_fw_upgrade(intf, argv[2]);
        }
        else return ipmi_raritanoem_usage();
    }

    if (strcmp(argv[0], "sn") == 0) {
        if (argc < 2) return ipmi_raritanoem_usage();
        if (strcmp(argv[1], "get") == 0) {
            if (argc != 2) return ipmi_raritanoem_usage();
            rc = ipmi_raritanoem_sn_get(intf);
        }
        else if (strcmp(argv[1], "set") == 0) {
            if (argc != 3) return ipmi_raritanoem_usage();
            rc = ipmi_raritanoem_sn_set(intf, argv[2]);
        }
        else return ipmi_raritanoem_usage();
    }

    return rc;
}
