#include <stdio.h>
#include <netdb.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>
#include <iptables.h>
#include <sys/types.h>
#include <linux/netfilter_ipv4/ip_tables.h>
#include <linux/netfilter_ipv4/ipt_FASTNAT.h>

/* Function which prints out usage message. */
static void
help(void)
{
    printf("FASTNAT v%s options:\n"
	   " --to-source <ipaddr>         Address to map source to.\n"
	   " --to-destination <ipaddr>    Address to map destination to.\n"
	   " --to-sport <port>            Port to map source to.\n"
	   " --to-dport <port>            Port to map destination to.\n\n",
	   IPTABLES_VERSION);
}

static struct option opts[] = {
    { "to-source", 1, 0, '1' },
    { "to-destination", 1, 0, '2' },
    { "to-sport", 1, 0, '3' },
    { "to-dport", 1, 0, '4' },
    { 0 }
};

/* Initialize the target. */
static void
init(struct ipt_entry_target *t, unsigned int *nfcache)
{
    struct ipt_fastnat_info *info = (struct ipt_fastnat_info *)t->data;

    memset(info, 0, sizeof(*info));

    /* Can't cache this */
    *nfcache |= NFC_UNKNOWN;
}

/* Function which parses command options; returns true if it ate an option */
static int
parse(int c, char **argv, int invert, unsigned int *flags,
      const struct ipt_entry *entry,
      struct ipt_entry_target **target)
{
    struct ipt_fastnat_info *info = (struct ipt_fastnat_info *)(*target)->data;
    struct in_addr * ip;
    u_int port;

    switch (c) {
      case '1':
	  if (check_inverse(optarg, &invert, NULL, 0)) {
	      exit_error(PARAMETER_PROBLEM,
			 "Unexpected `!' after --to-source");
	  }

	  ip = dotted_to_addr(optarg);
	  if (!ip) {
	      exit_error(PARAMETER_PROBLEM, "Bad IP address `%s'\n", optarg);
	  }

	  info->type |= IPT_FASTNAT_SRC_ADDR;
	  memcpy(&info->saddr, ip, sizeof(info->saddr));
	  *flags |= IPT_FASTNAT_SRC_ADDR;
	  return 1;

      case '2':
	  if (check_inverse(optarg, &invert, NULL, 0)) {
	      exit_error(PARAMETER_PROBLEM,
			 "Unexpected `!' after --to-destination");
	  }

	  ip = dotted_to_addr(optarg);
	  if (!ip) {
	      exit_error(PARAMETER_PROBLEM, "Bad IP address `%s'\n", optarg);
	  }

	  info->type |= IPT_FASTNAT_DST_ADDR;
	  memcpy(&info->daddr, ip, sizeof(info->daddr));
	  *flags |= IPT_FASTNAT_DST_ADDR;
	  return 1;

      case '3':
	  if (check_inverse(optarg, &invert, NULL, 0)) {
	      exit_error(PARAMETER_PROBLEM,
			 "Unexpected `!' after --to-sport");
	  }

	  if (string_to_number(optarg, 0, 65535, &port) == -1) {
	      exit_error(PARAMETER_PROBLEM, "Bad port number `%s'\n", optarg);
	  }

	  info->type |= IPT_FASTNAT_SRC_PORT;
	  info->sport = port;
	  *flags |= IPT_FASTNAT_SRC_PORT;
	  return 1;

      case '4':
	  if (check_inverse(optarg, &invert, NULL, 0)) {
	      exit_error(PARAMETER_PROBLEM,
			 "Unexpected `!' after --to-dport");
	  }

	  if (string_to_number(optarg, 0, 65535, &port) == -1) {
	      exit_error(PARAMETER_PROBLEM, "Bad port number `%s'\n", optarg);
	  }

	  info->type |= IPT_FASTNAT_DST_PORT;
	  info->dport = port;
	  *flags |= IPT_FASTNAT_DST_PORT;
	  return 1;

      default:
	  return 0;
    }
}

/* Final check */
static void final_check(unsigned int flags)
{
    if (!flags) {
	exit_error(PARAMETER_PROBLEM,
		   "You must specify one of --to-source, --to-sport, --to-destination and --to-dport.");
    }
}

/* Prints out the targinfo. */
static void
print(const struct ipt_ip *ip,
      const struct ipt_entry_target *target,
      int numeric)
{
    struct ipt_fastnat_info *info = (struct ipt_fastnat_info *)target->data;

    printf("FASTNAT ");
    if (info->type & IPT_FASTNAT_SRC_ADDR) {
	printf("source to %s ", addr_to_dotted(&info->saddr));
    }
    if (info->type & IPT_FASTNAT_SRC_PORT) {
	printf("source port to %hu ", info->sport);
    }
    if (info->type & IPT_FASTNAT_DST_ADDR) {
	printf("destination to %s ", addr_to_dotted(&info->daddr));
    }
    if (info->type & IPT_FASTNAT_DST_PORT) {
	printf("destination port to %hu ", info->dport);
    }
}

/* Saves the union ipt_targinfo in parsable form to stdout. */
static void
save(const struct ipt_ip *ip, const struct ipt_entry_target *target)
{
    struct ipt_fastnat_info *info = (struct ipt_fastnat_info *)target->data;

    if (info->type & IPT_FASTNAT_SRC_ADDR) {
	printf("--to-source %s ", addr_to_dotted(&info->saddr));
    }
    if (info->type & IPT_FASTNAT_SRC_PORT) {
	printf("--to-sport %hu ", info->sport);
    }
    if (info->type & IPT_FASTNAT_DST_ADDR) {
	printf("--to-destination %s ", addr_to_dotted(&info->daddr));
    }
    if (info->type & IPT_FASTNAT_DST_PORT) {
	printf("--to-dport %hu ", info->dport);
    }
}

static
struct iptables_target fastnat
= { NULL,
    "FASTNAT",
    IPTABLES_VERSION,
    IPT_ALIGN(sizeof(struct ipt_fastnat_info)),
    IPT_ALIGN(sizeof(struct ipt_fastnat_info)),
    &help,
    &init,
    &parse,
    &final_check,
    &print,
    &save,
    opts
};

void _init(void)
{
    register_target(&fastnat);
}
