/******************************************************************/
/*                                                                */
/* Module:       jblaster.c                                       */
/*                                                                */
/* Descriptions: Main source file that manages the configuration  */
/*               processes.                                       */
/*                                                                */
/* Revisions:    1.0 02/22/02                                     */
/*                                                                */
/******************************************************************/

/* $Id: jblaster.c,v 1.9 2006-09-20 08:12:30 daba Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>

#include <lara.h>
#include <pp/base.h>

#include "jb_io.h"
#include "jb_jtag.h"
#include "jb_const.h"
#include "jb_device.h"

/* JBlaster Controller Functions */
int  VerifyChain      (void);
void DriveSignal      (int signal,int data,int clk,int test);
void Configure        (int file_id,int dev_seq,int action);
void ProcessFileInput (int finputid, int idcode);
int  CheckStatus      (int dev_seq);
void Startup          (int dev_seq);
void Help             (void);

/* CDF Parser Functions */
int  ReadCDF          (int file_id);
int  GetData          (const char* charset,char* buffer,char* data);
void SearchKeyword    (char* buffer,char* data);
int  CheckActionCode  (char* data);
void CheckAltDev      (int dev_seq);

/* JTAG instruction lengths of all devices */
int  ji_info[MAX_DEVICE_ALLOW] = {0};

#define IS_CYCLONE(idcode) (((idcode > 0x2080000) && (idcode < 0x2086000)) || ((idcode >= 0x20b10dd) && (idcode <= 0x20b60dd)))

/* extern configuration */
#ifndef LINUX_PARPORT
extern unsigned short	 lpt_addr;
#else
extern char 	devicepath[256];
#endif


int main(int argc, char* argv[])
{
	int i=0, c=0, ret=0;
	int file_id=0;
	int config_count=0, fail_count=0;
	int do_conf_check = 0;
	char* filename;
		
	D(D_ALWAYS, "JTAGBlaster (JBlaster) Version %s\n", VERSION);

	if (pp_base_init("atmel", LOG_NOT_SILENT) == -1) {
	    fprintf(stderr, "Initializing base-library failed.\n");
	    ret = 1;
	    goto bail;
	}

	/* set default values */
#ifdef LINUX_PARPORT
	snprintf(devicepath,255,"/dev/parport0");
#else
	lpt_addr = 0x378;
#endif

	while ((c = getopt(argc, argv, "d:c")) != -1) {
	    switch(c) {
	      case 'c':
		  do_conf_check = 1;
		  break;
	      case 'd':
#ifndef LINUX_PARPORT
		  { int x; sscanf(optarg,"%x",&x); lpt_addr= (unsigned short) x;}
		  D(D_NOTICE, "using parport address 0x%x.\n", lpt_addr);
#else
		  snprintf(devicepath, 255, "%s", optarg);
		  D(D_NOTICE, "using parport device: '%s'.\n", devicepath);
#endif	  
		  break;
	    }
	}
	
	
	if (optind == argc) {
	    D(D_ERROR, "No chain description file specified\n");
	    ret = 1;
	    goto bail;
	}

	filename = argv[optind];

	file_id = jb_fopen(filename, "rb");
	if (!file_id) {
	    D(D_ERROR, "Could not open Chain Description File \"%s\"!\n", filename);
	    ret = 2;
	    goto bail;
	}

	/**********Hardware Setup**********/

        InitLinuxDriver();
	/* Check if hardware is properly installed */
	if(VerifyHardware())
	{
		jb_fclose(file_id);
		ret = 2;
		goto bail;
	}

	/**********CDF Parsing**********/

	if (!ReadCDF(file_id)) {
		jb_fclose(file_id);
		ret = 2;
		goto bail;
	}

	jb_fclose(file_id);

	if (!device_count) {
	    ret = 2;
	    goto bail;
	}

	for(i=0;i<device_count;i++) {
	    if(!ji_info[i])	{
		D(D_ERROR, "Error: JTAG instruction length of device #%d NOT specified!\n", i);
		ret = 2;
		goto bail;
	    }
	}

	/* Verify the JTAG-compatible devices chain */
	if (VerifyChain()) {
	    ret = 2;
	    goto bail;
	}
		
	/**********Show Info***********/
	D(D_VERBOSE, "(1)jseq_max (2)jseq_conf_done (3)action (4)partname (5)path (6)file (7)inst_len\n");
	for(i=0;i<device_count;i++) {
	    D(D_VERBOSE, "(1)%d (2)%d (3)%c (4)%s (5)%s (6)%s (7)%d\n",
	      device_list[i].jseq_max,device_list[i].jseq_conf_done,device_list[i].action,device_list[i].partname,
	      device_list[i].path,device_list[i].file,device_list[i].inst_len);
	}

	if (do_conf_check) {
	    if (CheckStatus(1)) {
		D(D_ALWAYS, "Device not configured\n");
		ret = 1;
	    } else {
		D(D_ALWAYS, "Device is configured\n");
	    }
	    goto bail;
	}

	/**********Configuration**********/
	file_id=0;

	for(i=1;i<device_count+1;i++)
	{
		char fullpath[CDF_PATH_LEN+CDF_FILE_LEN];

		config_count=0;

		if(device_list[i-1].action != 'P')
			continue;

		while(config_count<MAX_CONFIG_COUNT)
		{
		    if (!config_count) {
			D(D_NOTICE, "Configuration setup device #%d\n", i);
		    } else {
			D(D_NOTICE, "Configuration setup device #%d (Retry)\n",i);
		    }
			
			config_count++;
			
			/* Open programming file as READ and in BINARY */
			jb_strcpy(fullpath,device_list[i-1].path);
			jb_strcat(fullpath,device_list[i-1].file);

			file_id = jb_fopen( fullpath, "rb" );
		
			if ( file_id ) {
			    D(D_NOTICE, "Programming file #%d: \"%s\" opened...\n", i, fullpath);
			} else {
			    D(D_ERROR, "Could not open programming file #%d: \"%s\"\n", i, fullpath);
			    ret = 2;
			    goto bail;
			}

			/* Start configuration */
			Configure(file_id,i,device_list[i-1].idcode);

			jb_fclose(file_id);

			/*If device is a Cyclone device, condone will go high after initialization*/
			if (IS_CYCLONE(device_list[i-1].idcode))
				Startup(i);

			if(CheckStatus(i)) {
			    D(D_ERROR, "Configuration of device #%d NOT successful!\n",i );
			    fail_count++;
			}
			else {
			    /*If device is a non-Cyclone device, initialize after condone go high*/
			    if (!IS_CYCLONE(device_list[i-1].idcode))
				Startup(i);
			    config_count=MAX_CONFIG_COUNT;
			    D(D_NOTICE, "Configuration of device #%d successful...\n", i);
			}
		}
	}
	
	// reset device
 	set_tdi(0);
	set_tdi(1);

	if (fail_count > 0) {
	    D(D_NOTICE, "Some devices could not be programmed\n");
	    ret = 1;
	}

 bail:
	pp_base_cleanup();
	return ret;
}

/******************************************************************/
/* Name:         ReadCDF                                          */
/*                                                                */
/* Parameters:   file_id                                          */
/*               -file handler                                    */
/*                                                                */
/* Return Value: The number of devices in chain found in CDF      */
/*                                                                */
/* Descriptions: ReadCDF parses through the CDF and keeps the     */
/*               device records. It looks for 'P' or 'N' when     */
/*               mark=0. Once found, it searches for "ActionCode" */
/*               when mark=1 and once mark=2, it starts to        */
/*               retrieve the data in device record declarations. */
/*                                                                */
/******************************************************************/
int ReadCDF(int file_id)
{
	char  buffer[160];    /* line buffer */
/* [chtong,10/24/02,jrunner.c,ver1.2] Changed the following lines */
/*	char  data[50];												  
    The length of data array must be same with the longest data between 
	'(' and ')'	which is usually the path of the cdf file.
	Therefore, CDF_PATH_LEN is used								  */

	char  data[CDF_PATH_LEN];       /* device record data between '(' and ')' */
	int   mark= 0;
	
	while(jb_fgets(buffer,file_id))
	{
		if (mark==1)
		{
			mark =2;

			if(GetData("ACTIONCODE",buffer,data))
			{
				if(!CheckActionCode(data))
					return 0;
			}

			device_count++;

			SearchKeyword(buffer,data);

			/* End of device record and reset flag */
			if(jb_str_cmp(";",buffer))
				mark=0;
		}
		else if (mark==2)
		{
			SearchKeyword(buffer,data);

			/* End of device record and reset flag */
			if(jb_str_cmp(";",buffer))
				mark=0;
		}
/* [sbng,4/12/02,jrunner.c,ver 1.1] The following line has been removed */
/*		else if(jb_str_cmp("P",buffer) || jb_str_cmp("N",buffer)) */
		else
		{
/* [sbng,4/12/02,jrunner.c,ver1.1] Use jb_grabdata function instead of */
/*      jb_str_cmp */
			char c_temp[50];
			
			jb_grabdata(buffer,0,1,c_temp);
						
			if(!jb_str_cmp(c_temp,"P") && !jb_str_cmp(c_temp,"N"))
				continue;
/***********************************/

			mark++;
			
			if(GetData("ACTIONCODE",buffer,data))
			{
				if(!CheckActionCode(data))
					return 0;

				/* End of device record and reset flag */
				if(jb_str_cmp(";",buffer))
				{
					device_count++;

					SearchKeyword(buffer,data);
				
					mark=0;
				}
			}
		}
/* [sbng,4/12/02,jrunner.c,ver1.1] Removed the following lines */
/*		else
		{
			continue;
		}
*/	}

	D(D_NOTICE, "Parsed %i Entrys.\n", device_count);
	return device_count;
}

/******************************************************************/
/* Name:         GetData                                          */
/*                                                                */
/* Parameters:   charset, buffer, data                            */
/*               -charset is the character string or keyword to   */
/*                look for.                                       */
/*               -buffer is the line buffer stored when reading   */
/*                the CDF.                                        */
/*               -data is the string between brackets, '(' and ')'*/
/*                                                                */
/* Return Value: The position of the first character of charset   */
/*               found in buffer.                                 */
/*             		                                          */
/* Descriptions: The function copies the string between brackets  */
/*               right after charset into data. If charset is not */
/*               found in buffer, '0' is returned.                */
/*                                                                */
/******************************************************************/
int GetData(const char* charset,char* buffer,char* data)
{
	int   char_count=0,i;
	char* buff_pointer;
	char* data_pointer;
	int   mark=0;
	char * charset_upcase = strdupa(charset);
	char * tmpbuf = malloc(sizeof(char)*jb_strlen(buffer));
	strncpy (tmpbuf,buffer,jb_strlen(buffer));

	jb_toupper(charset_upcase);
	jb_toupper(tmpbuf);
	
	/* looking for charset in buffer */
	char_count= jb_str_cmp(charset_upcase,tmpbuf);

	/* charset not found in buffer */
	if(!char_count)
		return 0;

	data_pointer= data;
	buff_pointer= buffer;
	buff_pointer+= char_count-1+jb_strlen(charset_upcase);

	for(i=0;i<jb_strlen(buffer)-1;i++)
	{
		if(*buff_pointer=='(')
		{
			mark++;
		}
		else if(*buff_pointer==')')
		{
			if(mark==1)
			{
			    D(D_ERROR, "Error: Invalid Action Code!\n");
			    return 0;
			}

			/* put a null-zero to indicate the end of string to data */
			*data_pointer= '\0';
			break;
		}
		else if(mark)
		{
			mark=2;
			/* ignore '"' character */
			if(*buff_pointer!='"')
			{
				*data_pointer = *buff_pointer;
				data_pointer++;
			}
		}
		else
			return 0;

		buff_pointer++;
	}

//	jb_toupper(data);

	return char_count;
}

/******************************************************************/
/* Name:         CheckActionCode                                  */
/*                                                                */
/* Parameters:   data                                             */
/*               -The 3 character string indicating the action    */
/*                                                                */
/* Return Value: '0' if valid action code is detected, '1' if not */
/*                                                                */
/* Descriptions: Update the action to take in device list.        */
/*               A 'B' or a 'P' is stored for BYPASS and PORGRAM/ */
/*               CONFIGURE respectively.                          */
/*                                                                */
/******************************************************************/
int CheckActionCode(char* data)
{
	if(!jb_strcmp(data,"IGN"))
		device_list[device_count].action= 'B';
	else if(!jb_strcmp(data,"CFG"))
		device_list[device_count].action= 'P';
	else
	{
	    D(D_ERROR, "Invalid ActionCode: %s\n",data);
	    return 0;
	}

	return 1;
}

/******************************************************************/
/* Name:         SearchKeyword                                    */
/*                                                                */
/* Parameters:   buffer, data                                     */
/*               -buffer is the line buffer stored when reading   */
/*                the CDF.                                        */
/*               -data is the string between brackets, '(' and ')'*/
/*                found in buffer.                                */
/*                                                                */
/* Return Value: None.                                            */
/*               		                                          */
/* Descriptions: The function search for device records corres-   */
/*               pond to part name, path, file name and           */
/*               instruction length.                              */
/*                                                                */
/******************************************************************/
void SearchKeyword(char* buffer,char* data)
{
	char  Info_name[4][20] = { "PARTNAME","PATH","FILE","INSTRUCTIONREG" };
	int   i;

	for(i=0;i<4;i++)
	{
		if(GetData(Info_name[i],buffer,data))
		{
			switch(i)
			{
			case 0:
				jb_strcpy(device_list[device_count-1].partname,data);
				CheckAltDev(device_count);
				break;
			case 1:
				jb_strcpy(device_list[device_count-1].path,data);
				break;
			case 2:
				jb_strcpy(device_list[device_count-1].file,data);
				break;
			case 3:
				device_list[device_count-1].inst_len= jb_atoi(data);
				D(D_NOTICE, "parsed %s to %i\n",data, device_list[device_count-1].inst_len);
				ji_info[device_count-1]=device_list[device_count-1].inst_len;
				break;
			default:
				break;
			}
		}
	}
}

/******************************************************************/
/* Name:         CheckAltDev                                      */
/*                                                                */
/* Parameters:   dev_seq                                          */
/*               -dev_seq is the device sequence in JTAG chain.   */
/*                                                                */
/* Return Value: None.                                            */
/*               		                                          */
/* Descriptions: The function matches the partname specified in   */
/*               CDF with Altera devices list. If matches, the    */
/*               JTAG chain information will be updated in device */
/*               _list. The action code is updated by ActionCode. */
/*               If the partname is not recognized, the device    */
/*               will be bypassed.                                */
/*                                                                */
/******************************************************************/
void CheckAltDev(int dev_seq)
{
	int i,j,altera=0;
	dev_seq--;

	for(i=0;i<MAX_DEV_LIST;i++)
	{
		if(!jb_strcmp(device_list[dev_seq].partname,device_name[i]))
		{
			if(!(device_info[i][0] && device_info[i][1] && device_info[i][2]))
			{
				device_list[dev_seq].inst_len = device_info[i][3];
				ji_info[dev_seq]                    = device_list[dev_seq].inst_len;
				D(D_ERROR, "Device #%d not supported! Bypassed!\n", dev_seq+1);
			}
			else
			{
				device_list[dev_seq].idcode         = device_info[i][0];
				device_list[dev_seq].jseq_max       = device_info[i][1];
				device_list[dev_seq].jseq_conf_done = device_info[i][2];
				device_list[dev_seq].inst_len       = device_info[i][3];
				ji_info[dev_seq]                    = device_list[dev_seq].inst_len;
				altera = 1;
				break;
			}
		}
	}
/* [chtong,1/6/03,jrunner.c,ver 1.3] The following section has been modified*/
/* [chtong,6/2/03,jrunner.c,ver 1.4] The following section has been modified*/
/* Source code before version 1.2 cannot print family device if the device to be configured is the last family member*/
	for(j=0;j<MAX_DEV_FAMILY;j++)
	{
		if(i<start_of_device_family[j])
		{
			device_family = j-1;
			D(D_NOTICE, "family: %s(%d)\n",family_name[j-1],j);
			break;
		}
		/* Source code before version 1.2 cannot print family device if the device to be configured is the last family member*/
		/* add-in && altera ==1 in version 1.4*/
		else if (j==MAX_DEV_FAMILY-1 && i>=start_of_device_family[MAX_DEV_FAMILY-1] && altera ==1)
		{
			device_family = MAX_DEV_FAMILY-1;
			D(D_NOTICE, "family: %s(%d)\n",family_name[MAX_DEV_FAMILY-1],MAX_DEV_FAMILY);
		}
	}

	if(!altera)
	{
		device_list[dev_seq].idcode=0;
		device_list[dev_seq].jseq_max=0;
		device_list[dev_seq].jseq_conf_done=0;
		device_list[dev_seq].action='B';
	}
}

/******************************************************************/
/* Name:         VerifyChain                                      */
/*                                                                */
/* Parameters:   None.                                            */
/*                                                                */
/* Return Value: None.                                            */
/*               		                                          */
/* Descriptions: Putting all devices in BYPASS mode, a 8-bit      */
/*               vector is driven to TDI, the number of '0'       */
/*               detected indicates the number of devices in      */
/*               chain. The 8-bit vector must follows the zeroes. */
/*                                                                */
/******************************************************************/
int VerifyChain()
{
	unsigned int data=0,temp=0,test_vect=0x55;
	int i,num=0,error=0;
	
	Js_Reset();
	
	/* Load BYPASS instruction and test JTAG chain with a few vectors */
	if(Ji_Bypass(device_count,ji_info))
		return (1);
	Js_Shiftdr();

	/* Drive a 8-bit vector of "10101010" (right to left) to test */
	data = ReadTDO(8+device_count,test_vect,0);
	/* The number of leading '0' detected must equal to the number of devices specified */
	temp = data;
	
	for(i=0;i<device_count;i++)
	{
		temp = temp&1;
		if(temp)
			break;		
		else
			num++;
		temp = data>>(i+1);
	}

	if (temp==test_vect) {
	    D(D_NOTICE, "Detected %d device(s) in chain...\n", num);
	} else {
	    D(D_ERROR, "JTAG chain broken or #device in chain unmatch!\n");
	    return 1;
	}

	Js_Updatedr();
	
	/* Read device IDCODE */
	Ji_Idcode(device_count,ji_info);
	Js_Shiftdr();

	for(i=device_count-1;i>=0;i--)
	{
		data = ReadTDO(CDF_IDCODE_LEN,0,0);

		if(device_list[i].idcode)
		{
			/* The partname specified in CDF must match with its ID Code */
			if((unsigned)device_list[i].idcode != data)
			{
				D(D_ERROR, "Error: Expected 0x%X but detected 0x%X!\n",device_list[i].idcode,data);
				error=1;
			}
			else
				D(D_NOTICE, "Dev%d: Altera: 0x%X\n",i+1,data);
		}
		else
		{
			D(D_NOTICE, "Info: Dev%d: Non-Altera: 0x%X\n",i+1,data);
		}
	}

	Js_Updatedr();
	Js_Runidle();
	
	return error;
}

/******************************************************************/
/* Name:         Configure                                        */
/*                                                                */
/* Parameters:   file_id,dev_seq,action                           */
/*               -file_id is the ID of the file.                  */
/*               -dev_seq is the device sequence in chains.       */
/*               -action is the action to take:BYPASS or PROGRAM  */
/*                                                                */
/* Return Value: None.                                            */
/*               		                                          */
/* Descriptions: Issue PROGRAM instruction to the device to be    */
/*               configured and BYPASS for the rest of the devices*/
/*               Call function that processes the source file.    */
/*                                                                */
/******************************************************************/
void Configure(int file_id,int dev_seq,int idcode)
{
	int i,j;
	int action = idcode? JI_PROGRAM:JI_BYPASS;

	/* Load PROGRAM instruction */
	SetupChain(device_count,dev_seq,ji_info,action);

	if(action==JI_PROGRAM)
	{
		/* Drive TDI HIGH while moving JSM to SHIFTDR */
		DriveSignal(SIG_TDI,1,0,1);
		Js_Shiftdr();
		/* Issue MAX_JTAG_INIT_CLOCK clocks in SHIFTDR state */
		for(i=0;i<MAX_JTAG_INIT_CLOCK[device_family];i++)
		{
			DriveSignal(SIG_TDI,1,1,0);
		}
		/* Start dumping configuration bits into TDI and clock with TCK */
		ProcessFileInput(file_id,idcode);
		
		for (j=0;j<128;j++ )
		{
			DriveSignal(SIG_TDI,1,1,1);
		}


		/* Move JSM to RUNIDLE */
		Js_Updatedr();
		Js_Runidle();
	}
}

/******************************************************************/
/* Name:         CheckStatus                                      */
/*                                                                */
/* Parameters:   dev_seq                                          */
/*               -dev_seq is the device sequence in chains.       */
/*                                                                */
/* Return Value: '0' if CONF_DONE is HIGH;'1' if it is LOW.       */
/*               		                                          */
/* Descriptions: Issue CHECK_STATUS instruction to the device to  */
/*               be configured and BYPASS for the rest of the     */
/*               devices.                                         */
/*                                                                */
/*               <conf_done_bit> =                                */
/*                  ((<Maximum JTAG sequence> -                   */
/*                    <JTAG sequence for CONF_DONE pin>)*3) + 1   */
/*                                                                */
/*               The formula calculates the number of bits        */
/*               to be shifted out from the device, excluding the */
/*               1-bit register for each device in BYPASS mode.   */
/*                                                                */
/******************************************************************/
int CheckStatus(int dev_seq)
{
	int bit,data=0,error=0;
	int jseq_max=0,jseq_conf_done=0,conf_done_bit=0;

	D(D_VERBOSE, "Checking Status\n" );

	/* Load CHECK_STATUS instruction */
	SetupChain(device_count,dev_seq,ji_info,JI_CHECK_STATUS);

	Js_Shiftdr();

	/* Maximum JTAG sequence of the device in chain */
	jseq_max= device_list[dev_seq-1].jseq_max;

	jseq_conf_done= device_list[dev_seq-1].jseq_conf_done;

	conf_done_bit = ((jseq_max-jseq_conf_done)*3)+1;

	/* Compensate for 1 bit unloaded from every Bypass register */
	conf_done_bit+= (device_count-dev_seq);
	
	set_tdi(0);
	for(bit=0;bit<conf_done_bit;bit++)
	{
		DriveSignal(SIG_TDI,0,1,1);
	}

	data = ReadTDO(1,0,0);

	if(!data)
		error++;

	/* Move JSM to RUNIDLE */
	Js_Updatedr();
	Js_Runidle();

	return (error);	
}

/******************************************************************/
/* Name:         Startup                                          */
/*                                                                */
/* Parameters:   dev_seq                                          */
/*               -the device sequence in the chain.               */
/*                                                                */
/* Return Value: None.                                            */
/*               		                                          */
/* Descriptions: Issue STARTUP instruction to the device to       */
/*               be configured and BYPASS for the rest of the     */
/*               devices.                                         */
/*                                                                */
/******************************************************************/
void Startup(int dev_seq)
{
	int i;

	/* Load STARTUP instruction to move the device to USER mode */
	SetupChain(device_count,dev_seq,ji_info,JI_STARTUP);

	Js_Runidle();

	for(i=0;i<INIT_COUNT;i++)
	{
		DriveSignal(SIG_TCK,0,0,0);
		DriveSignal(SIG_TCK,1,0,0);
	}

	/* Reset JSM after the device is in USER mode */
	Js_Reset();
}

/******************************************************************/
/* Name:         ProcessFileInput                                 */
/*                                                                */
/* Parameters:   finputid                                         */           
/*               -programming file pointer.                       */
/*                                                                */
/* Return Value: None.                                            */
/*                                                                */
/* Descriptions: Get programming file size, parse through every   */
/*               single byte and dump to parallel port.           */
/*                                                                */
/******************************************************************/
void ProcessFileInput(int finputid, int idcode)
{
	int seek_position=0,one_byte=0;
	long int file_size=0,i=0;

	/* Get file size */
	seek_position = jb_fseek(finputid,0,S_END);

	if(seek_position)
	{
	    D(D_ERROR, "End of file could not be located!" );
	      return;
	}

	file_size = jb_ftell(finputid);
		
	/* Start configuration */
	/* Reset file pointer */
	jb_fseek(finputid,0,S_SET);

	D(D_NOTICE, "Starting configuration process...");

	/* Loop through every single byte */
	for(i=0;i<file_size;i++)
	{
		/*Ignore first 44 bytes in the rbf file for Cyclone device*/
		if(i<44 && IS_CYCLONE(idcode))
			one_byte = jb_fgetc(finputid);
		else
		{
//			int	bit = 0,j;
	
			one_byte = jb_fgetc(finputid);
	
			/* Program a byte,from LSb to MSb */
#if 0
			for (j=0;j<8;j++ )
			{
				bit = one_byte >> j;
				bit = bit & 0x1;
				
				/* Dump to TDI and drive a positive edge pulse at the same time */
				DriveSignal(SIG_TDI,bit,1,0);
			}
#else
			set_tdi_byte(one_byte);
#endif
		}
	}

	D(D_NOTICE, " done\n");
}

/******************************************************************/
/* Name:         DriveSignal                                      */
/*                                                                */
/* Parameters:   signal,data,clk,test                             */
/*               -the name of the signal (SIG_*).                 */
/*               -the value to be dumped to the signal,'1' or '0' */
/*               -driving a LOW to HIGH transition to SIG_TCK     */
/*                together with signal.                           */
/*               -test is used by WritePort function.             */
/*                                                                */
/* Return Value: None.                                            */
/*                                                                */
/* Descriptions: Dump data to signal. If clk is '1', a clock pulse*/
/*               is driven after the data is dumped to signal.    */
/*                                                                */
/******************************************************************/
void DriveSignal(int signal,int data,int clk, int test UNUSED)
{

	switch (signal){
        	case SIG_TCK:
        		set_tck(data);
        		break;
        	case SIG_TMS:
        		set_tms(data);
        		break;
        	case SIG_TDI:
        		set_tdi(data);
        		break;
        	default:
        		break;
        }
	if (clk) set_tck(1);
	if (clk) set_tck(0);
}

/******************************************************************/
/* Name:         Help                                             */
/*                                                                */
/* Parameters:	 None.                                            */
/*                                                                */
/* Return Value: None.                                            */
/*                                                                */
/* Descriptions: Print help to standard output.                   */
/*                                                                */
/******************************************************************/
void Help()
{
#ifdef LINUX_PARPORT
    D(D_ALWAYS, "Syntax: \"jblaster [-d <device>] [ -r ] [ -c ] <Chain Description File(.cdf)>\"\n");
#else
    D(D_ALWAYS, "Syntax: \"jblaster [-d <parport io address>] [ -r ] [ -c ] <Chain Description File(.cdf)>\"\n");
#endif
}

