/**
 * esb2_tpt_mod.c
 *
 * KIRA100 FML (slave) driver for Intel's ESB2 TCP Pass-Trough
 *
 * (c) 2005 Peppercon AG, 2005/07/05, Ralf Guenther <rgue@peppecon.de>
 */

#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/netdevice.h>
#include <linux/ioctl.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <linux/poll.h>
#include <linux/pci.h>
#include <linux/wait.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/cdev.h>
#include <asm/semaphore.h>
#include <asm/io.h>

#ifdef __arm__
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#include <asm/arch/cpe/peppercon.h>
#define cond_resched() if (current->need_resched) schedule()
#else
#include <asm/arch/platform/peppercon.h>
#define MOD_INC_USE_COUNT
#define MOD_DEC_USE_COUNT
#endif
#endif

#include "esb2_tpt.h"
#include "fml_esb2.h"
#include "fml_test.h"
#include "fmlcore.h"

#define SUCCESS 0

int debug = 0;

#define FML_IO_SIZE 16384
#define RECV_BUF_SIZE 8192
#define NAME                    "esb2_tpt"
#define LOG_COLORED
#define noLOG_BLACK_BG
#define COND_DBG
#include "log.h"

#define ESB2_TPT_MAJOR ((uint8_t)249)

// PCI dev and INTMUX not used by KIRA
#if defined(LARA_KIMMSI) || defined(LARA_KIMSMI) || defined (LARA_KIMAMD) || defined (LARA_KIMINTEL)
#  define LPC_USE_PCI
#  define LPC_USE_INTMUX
#endif

#ifdef LPC_USE_PCI
#  define KIRA100_PCI_VENDOR_ID  0x1743
#  define KIRA100_PCI_DEVICE_ID  0x1003
#  define KIRA100_PCI_BAR_NO     1
#endif

/* KIRA100 IO addresses/bits */

uint32_t kira100_io_base = 0;
//#define FML_IO_SIZE             (20 * sizeof(uint32_t)) /* regs are dword alligned */

#if defined(LARA_KIMMSI) || defined(LARA_KIMSMI) || defined (LARA_KIMAMD) || defined (LARA_KIMINTEL)
#  define FML_IO_BASE_OFFS      0x9b0000
#  define DEF_IRQ               27
#endif

#ifdef PP_BOARD_KIRA
#  define FML_IO_BASE_OFFS      CPE_FML_BASE
#  define DEF_IRQ               29
#endif

#ifdef LPC_USE_INTMUX
#  define INTMUX_IO_BASE_OFFS   0x9a0000
#  define INTMUX_IO_SIZE        (1 * sizeof(uint32_t)) /* 1 dword */
#  define FML_INT               ((uint16_t)(1 << 14)) // interrupt flag bits (INTMUX register)
#endif

#define FML_SLAVE_ADDR          0x55
#define TPT_CHAN                1

/* ------------------------------------------------------------------------- *
 * private data
 * ------------------------------------------------------------------------- */

static struct semaphore sem=__SEMAPHORE_INIT(sem,1);        /* serialize ESB2 write ops */

static uint8_t send_buf[SBUF_SIZE];
static unsigned int send_fill;
static wait_queue_head_t send_wait=__WAIT_QUEUE_HEAD_INITIALIZER(send_wait);
static unsigned int data_to_send;
static unsigned int send_thread_running;

static atomic_t refcnt=ATOMIC_INIT(0);
static void *fml_addr;
#ifdef LPC_USE_INTMUX
static void *intmux_addr;
#endif
static int isr_registered;
static int conn_lost;

static struct fmlcore_private fml;
static int fml_init;

esb2_tpt_ioctl_hk_t hk;

typedef struct {
    wait_queue_head_t   wait;       /* receivers wait queue */
    unsigned char recv_timeout; /* receive timeout in secs set by IOCTL */
    unsigned char       recv_buf[RECV_BUF_SIZE]; /* receive data buffer */
    int                 recv_size;
    spinlock_t          recv_lock;

    int chan;
 
    struct cdev cdev;
} esb2_tpt_dev_t;

static esb2_tpt_dev_t esb2_dev[2] = {
{
    .wait = __WAIT_QUEUE_HEAD_INITIALIZER(esb2_dev[0].wait),
    .recv_lock = SPIN_LOCK_UNLOCKED,
    .recv_timeout = 60,
    .chan = 1,
},
{
    .wait = __WAIT_QUEUE_HEAD_INITIALIZER(esb2_dev[1].wait),
    .recv_lock = SPIN_LOCK_UNLOCKED,
    .recv_timeout = 60,
    .chan = 3,
},
};

/* ------------------------------------------------------------------------- *
 * device driver prototypes
 * ------------------------------------------------------------------------- */

static int esb2_mod_release(void);
static int esb2_mod_open(void);

static int  esb2_tpt_mod_init(void);
static void esb2_tpt_mod_cleanup(void);

static int  esb2_tpt_mod_ioctl(struct inode *, struct file *, uint, ulong);
static int  esb2_tpt_mod_open(struct inode *, struct file *);
static int  esb2_tpt_mod_release(struct inode *, struct file *);
static int  esb2_tpt_mod_read(struct file *, char *, size_t, loff_t *);
static int  esb2_tpt_mod_write(struct file *, const char *, size_t, loff_t *);
static unsigned int esb2_tpt_mod_poll(struct file *, poll_table *);

static int  esb2_fni_mod_ioctl(struct inode *, struct file *, uint, ulong);
static int  esb2_fni_mod_open(struct inode *, struct file *);
static int  esb2_fni_mod_release(struct inode *, struct file *);
static int  esb2_fni_mod_read(struct file *, char *, size_t, loff_t *);
static int  esb2_fni_mod_write(struct file *, const char *, size_t, loff_t *);
static unsigned int esb2_fni_mod_poll(struct file *, poll_table *);

static irqreturn_t esb2_tpt_mod_isr(int irq, void *dev_id, struct pt_regs *regs);
static int recv_cb(struct fmlcore_private *fml);
static int send_collector_thread(void * arg);

/* ------------------------------------------------------------------------- *
 * linux kernel module stuff
 * ------------------------------------------------------------------------- */

MODULE_AUTHOR("Ralf Guenther <rgue@peppercon.de>");
MODULE_DESCRIPTION("KIRA100 FML (slave) driver for Intel's ESB2 TCP Pass-Trough");

module_init(esb2_tpt_mod_init);
module_exit(esb2_tpt_mod_cleanup);

/* ------------------------------------------------------------------------- *
 * structure with driver operations
 * ------------------------------------------------------------------------- */

static struct file_operations esb2_tpt_mod_ops = {
    .owner = THIS_MODULE,
    .llseek = NULL,
    .ioctl = esb2_tpt_mod_ioctl,
    .open = esb2_tpt_mod_open,
    .release = esb2_tpt_mod_release,
    .read = esb2_tpt_mod_read,
    .write = esb2_tpt_mod_write,
    .poll = esb2_tpt_mod_poll,
}, esb2_fni_mod_ops = {
    .owner = THIS_MODULE,
    .llseek = NULL,
    .ioctl = esb2_fni_mod_ioctl,
    .open = esb2_fni_mod_open,
    .release = esb2_fni_mod_release,
    .read = esb2_fni_mod_read,
    .write = esb2_fni_mod_write,
    .poll = esb2_fni_mod_poll,
};

/* ------------------------------------------------------------------------- *
 * driver initialization
 * ------------------------------------------------------------------------- */

int __init
esb2_tpt_mod_init(void)
{
    int ret = SUCCESS, r;
    uint32_t fml_io_base;
#ifdef LPC_USE_INTMUX
    uint32_t intmux_io_base;
#endif

    if (fml_test_slave_init(&fml)) {
        ERR("failed to fml test slave part.");
        ret = -ENODEV;
        goto fail;
    }

    /* ---- register the character device ------------------------------ */
    /*
    if ((r = register_chrdev(ESB2_TPT_MAJOR, NAME, &esb2_tpt_mod_ops)) < 0) {
        ERR("failed to register device (err %d)", r);
        ret = -ENODEV;
        goto fail;
    }
    */

    cdev_init(&esb2_dev[0].cdev,&esb2_tpt_mod_ops);
    esb2_dev[0].cdev.owner=THIS_MODULE;

    if((r=cdev_add(&esb2_dev[0].cdev,MKDEV(ESB2_TPT_MAJOR,0),1))<0)
    {
        ERR("failed to register device (err %d)", r);
        ret = -ENODEV;
        goto fail;
    }

    cdev_init(&esb2_dev[1].cdev,&esb2_fni_mod_ops);
    esb2_dev[1].cdev.owner=THIS_MODULE;

    if((r=cdev_add(&esb2_dev[1].cdev,MKDEV(ESB2_TPT_MAJOR,1),1))<0)
    {
        ERR("failed to register device (err %d)", r);
        ret = -ENODEV;
        goto fail;
    }

#ifdef LPC_USE_PCI
    {
        /* find hardware */
        struct pci_dev *pci = pci_find_device(KIRA100_PCI_VENDOR_ID, KIRA100_PCI_DEVICE_ID, NULL);
        if (pci == NULL) {
            ERR("failed to find PCI device (%04x:%04x) containing FML core",
                KIRA100_PCI_VENDOR_ID, KIRA100_PCI_DEVICE_ID);
            ret = -ENODEV;
            goto fail;
        }
        kira100_io_base = pci_resource_start(pci, KIRA100_PCI_BAR_NO);
    }
#endif

    fml_io_base = kira100_io_base + FML_IO_BASE_OFFS;
    if (!(fml_addr = ioremap_nocache(fml_io_base, FML_IO_SIZE))) {
        ERR("failed to map FML core IO space at %08x-%08x to memory",
            fml_io_base, fml_io_base + FML_IO_SIZE - 1);
        ret = -EIO;
        goto fail;
    }

#ifdef __arm__
    if (KIRA_MAJOR(pp_kira_get_revision()) < 2) {
        INFO("FML slave support is not available in KIRA100 R01.");
        ret = -ENODEV;
        goto fail;
    }
#endif

#ifdef LPC_USE_INTMUX
    intmux_io_base = kira100_io_base + INTMUX_IO_BASE_OFFS;
    if (!(esb2_tpt_dev.intmux_addr = ioremap_nocache(intmux_io_base, INTMUX_IO_SIZE))) {
        ERR("failed to map INTMUX IO space at %08x-%08x to memory",
            intmux_io_base, intmux_io_base + INTMUX_IO_SIZE - 1);
        ret = -EIO;
        goto fail;
    }
#endif

#ifdef LPC_USE_INTMUX
    INFO("successfully loaded. (io: %08x, intmux: %08x, irq: %d)",
        fml_io_base, intmux_io_base, DEF_IRQ);
#else // !LPC_USE_INTMUX
    INFO("successfully loaded. (io: %08x, irq: %d)",
        fml_io_base, DEF_IRQ);
#endif

    return ret;
fail:
    esb2_tpt_mod_cleanup();
    return ret;
}

static void
esb2_tpt_mod_cleanup(void)
{
    int r;
    if ((r = unregister_chrdev(ESB2_TPT_MAJOR, NAME)) < 0) {
        ERR("failed to unregister device (err %d)", r);
    }
#ifdef LPC_USE_INTMUX
    if (esb2_tpt_dev.intmux_addr) {
        iounmap(esb2_tpt_dev.intmux_addr);
        esb2_tpt_dev.intmux_addr = NULL;
    }
#endif
    if (fml_addr) {
        iounmap(fml_addr);
        fml_addr = NULL;
    }
    
    fml_test_slave_cleanup();
}


/* ------------------------------------------------------------------------- *
 * the driver operations
 * ------------------------------------------------------------------------- */

static int
esb2_fni_mod_open(struct inode *inode, struct file *file)
{
    esb2_tpt_dev_t *dev=&esb2_dev[1];
    int ret=esb2_mod_open();

    if(ret)
	return ret;

    // tell the ESB2 our in buffer space and request data
    if (esb2_fml_fni_read_req(&fml, sizeof(dev->recv_buf) - dev->recv_size) < 0) {
	ERR("failed to initially report free FNI receive buffer size");
    }
    return 0;
}

static int
esb2_tpt_mod_open(struct inode *inode, struct file *file)
{
    esb2_tpt_dev_t *dev=&esb2_dev[0];
    int ret=esb2_mod_open();

    if(ret)
	return ret;

    // tell the ESB2 our in buffer space and request data
    if (esb2_fml_tpt_read_req(&fml, sizeof(dev->recv_buf) - dev->recv_size) < 0) {
	ERR("failed to initially report free TPT receive buffer size");
    }
    return 0;
}

static int
esb2_mod_open(void)
{
    int ret = SUCCESS;
    MOD_INC_USE_COUNT;

    if (atomic_inc_return(&refcnt) == 1) {
        /* first time init */
        int r;
        uint32_t fml_io_base = kira100_io_base + FML_IO_BASE_OFFS;

        DBG("load");

        if ((r = check_mem_region(fml_io_base, FML_IO_SIZE))) {
            ERR("IO space %08x-%08x already in use (err %d)",
                fml_io_base, fml_io_base + FML_IO_SIZE - 1, r);
            ret = -EBUSY;
            goto fail;
        }
        request_mem_region(fml_io_base, FML_IO_SIZE, "fml");

        /* fill fml struct */
        fml.name     = NAME;
        fml.io_base  = (u32)fml_addr;
        fml.fml_clk  = 8000; /* 8Mhz */
#ifdef PP_BOARD_KIRA
        fml.sys_clk  = 100000; /* 100 Mhz */
#else 
	fml.sys_clk  = 66666; /* 66 MHz */
#endif
        fml.irq      = DEF_IRQ;
        fml.slave    = 1;
        fml.saddr    = FML_SLAVE_ADDR << 1;
        fml.slave_req_callback  = recv_cb;

        if (fmlcore_init(&fml) < 0) {
            ret = -ENODEV;
            goto fail;
        }
        fml_init = 1;

        if ((r = request_irq(DEF_IRQ, esb2_tpt_mod_isr, SA_SHIRQ, NAME, esb2_dev)) < 0) {
            ERR("failed to request irq %d (err %d)", DEF_IRQ, r);
            ret = -EBUSY;
            goto fail;
        }
        isr_registered = 1;

        //start the write collector thread
        send_fill = 0;
        send_thread_running = 1;
        if ((r = kernel_thread(send_collector_thread, NULL, 0)) < 0) {
            ERR("failed to initialize send collector thread (err %d)", r);
            ret = -EBUSY;
            goto fail;
        }

    }

    DBG("opened");
    return ret;
fail:
    esb2_mod_release();
    return ret;
}

static int
esb2_mod_release(void)
{
    if (atomic_dec_return(&refcnt) == 0) {
        DBG("unload");
        /* least time cleanup */
	send_fill = 0;
	send_thread_running = 0;
        wake_up(&send_wait);
        if (isr_registered) free_irq(DEF_IRQ, esb2_dev); isr_registered = 0;
        if (fml_init); fmlcore_release(&fml); fml_init = 0;
        release_mem_region(kira100_io_base + FML_IO_BASE_OFFS, FML_IO_SIZE);
    }

    DBG("closed");
    MOD_DEC_USE_COUNT;
    return SUCCESS;
}

static int
esb2_fni_mod_release(struct inode *inode, struct file *file)
{
    esb2_dev[1].recv_size = 0;
    return esb2_mod_release();
}

static int
esb2_tpt_mod_release(struct inode *inode, struct file *file)
{
    esb2_dev[0].recv_size = 0;
    return esb2_mod_release();
}

static int
esb2_fni_mod_ioctl(struct inode *inode, struct file *file, uint cmd, ulong arg)
{
    esb2_tpt_dev_t *dev = &esb2_dev[1];
    switch(cmd)
    {
	case ESB2_FNI_IOCTL_SET_CHAN:
	    dev->chan=arg;
	    return 0;
	case ESB2_FNI_IOCTL_SET_DEBUG:
	    debug=arg;
	    return 0;
	default:
	    return -EINVAL;
    }
}

static int
esb2_tpt_mod_ioctl(struct inode *inode, struct file *file, uint cmd, ulong arg)
{
    esb2_tpt_dev_t *dev = &esb2_dev[0];

    switch (cmd) {
    case ESB2_TPT_IOCTL_GET_TPT_STAT:
    {
        esb2_tpt_ioctl_tpt_stat_t d;
        if (copy_from_user(&d, (void*)arg, sizeof(d))) return -EFAULT;
        if (esb2_tpt_get_status(&fml, d.chan, &d.status, d.remote_ip) < 0) {
            ERR("failed to get TPT status");
            return -EIO;
        }
        if (copy_to_user((void*)arg, &d, sizeof(d))) return -EFAULT;
        return SUCCESS;
    }
    case ESB2_TPT_IOCTL_SET_RECV_TIMEOUT:
    {
        esb2_tpt_ioctl_recv_timeout_t d;
        if (copy_from_user(&d, (void*)arg, sizeof(d))) return -EFAULT;
        dev->recv_timeout = d.timeout; /* chan is ignored */
        return SUCCESS;
    }
    case ESB2_TPT_IOCTL_GET_REG:
    {
        esb2_tpt_ioctl_reg_t d;
        if (copy_from_user(&d, (void*)arg, sizeof(d))) return -EFAULT;
        d.val = fml.read_reg(&fml, d.reg);
        if (copy_to_user((void*)arg, &d, sizeof(d))) return -EFAULT;
        return SUCCESS;
    }
    case ESB2_TPT_IOCTL_SET_REG:
    {
        esb2_tpt_ioctl_reg_t d;
        if (copy_from_user(&d, (void*)arg, sizeof(d))) return -EFAULT;
        fml.write_reg(d.val, &fml, d.reg);
        return SUCCESS;
    }
    case ESB2_TPT_IOCTL_GET_HK:
    {
        if (copy_to_user((void*)arg, &hk, sizeof(esb2_tpt_ioctl_hk_t))) return -EFAULT;
        return SUCCESS;
    }
    case ESB2_TPT_IOCTL_SET_CONNLOST:
    {
	conn_lost = 1;
	wake_up(&dev->wait);
	return SUCCESS;
    }
    default:
        /* invalid cmd */
        return -EINVAL;
    }
}

static int
esb2_fni_mod_read(struct file* file, char *buffer, size_t size, loff_t *offset)
{
    esb2_tpt_dev_t *dev = &esb2_dev[1];
    u_long flags;
    int i, newsize;

    printk("esb2_fni_mod_read started\n");

    // blocking read, timeout set by IOCTL (default: 1 min)
    for (i = 0; dev->recv_size == 0 && i < dev->recv_timeout; i++) {
        /* after 1 sec, every 10 secs */
        if ((i % 10) == 1) {
            /* use time to check connections status */
            unsigned char stat = ESB2_TPT_STAT_CONNECTED; // optimistic point of view ;)

            if (down_interruptible(&sem) < 0) return -EINTR;
            if (esb2_fni_get_status(&fml, dev->chan, &stat, NULL) < 0)
	    {
                WARN("failed to query connection state while waiting for data");
		return -EIO;
	    }
            up(&sem);

            if (stat!=0) return -ENOTCONN;
        }

        if (wait_event_interruptible_timeout(dev->wait, dev->recv_size, HZ) < 0) return -EINTR;
    }
    if (dev->recv_size == 0) return -EAGAIN;

    spin_lock_irqsave(&dev->recv_lock, flags);

    /* we cannot read more than received */
    if (size > dev->recv_size) size = dev->recv_size;

    /* get the data */
    copy_to_user(buffer, dev->recv_buf, size);

    /* copy the rest to buffer head */
    if (size < dev->recv_size) memmove(dev->recv_buf, dev->recv_buf + size, dev->recv_size - size);

    /* reduce remaining recv buf size */
    dev->recv_size -= size;
    newsize = sizeof(dev->recv_buf) - dev->recv_size;

    spin_unlock_irqrestore(&dev->recv_lock, flags);

    DBG("(%d lost) TPT packet read (%d bytes): %.2x %.2x %.2x ...", esb2_tpt_blk_lost, size,
        (unsigned char)buffer[0], (unsigned char)buffer[1], (unsigned char)buffer[2]);

    /* inform FML master about new recv buf size */
    if (size > 0 && dev->recv_size == 0) {
        if (down_interruptible(&sem) < 0) return -EINTR;
        if (esb2_fml_fni_read_req(&fml, newsize) < 0) {
            WARN("failed to report free FNI receive buffer size");
        }
        up(&sem);
    }

    return size;
}

static int
esb2_tpt_mod_read(struct file* file, char *buffer, size_t size, loff_t *offset)
{
    esb2_tpt_dev_t *dev = &esb2_dev[0];
    u_long flags;
    int i, newsize;

    if (conn_lost && (dev->recv_size == 0)) {
	//special case: connection was lost, in this case we only send EOF
	conn_lost = 0;
	return 0;
    }

    // blocking read, timeout set by IOCTL (default: 1 min)
    for (i = 0; dev->recv_size == 0 && i < dev->recv_timeout; i++) {
        /* after 1 sec, every 10 secs */
        if ((i % 10) == 1) {
            /* use time to check connections status */
            unsigned char stat = ESB2_TPT_STAT_CONNECTED; // optimistic point of view ;)

            if (down_interruptible(&sem) < 0) return -EINTR;
            if (esb2_tpt_get_status(&fml, TPT_CHAN, &stat, NULL) < 0)
                WARN("failed to query connection state while waiting for data");
            up(&sem);

            if (!ESB2_TPT_IS_CONNECTED(stat)) return -ENOTCONN;
        }

        if (wait_event_interruptible_timeout(dev->wait, dev->recv_size, HZ) < 0) return -EINTR;
    }
    if (dev->recv_size == 0) return -EAGAIN;

    spin_lock_irqsave(&dev->recv_lock, flags);

    /* we cannot read more than received */
    if (size > dev->recv_size) size = dev->recv_size;

    /* get the data */
    copy_to_user(buffer, dev->recv_buf, size);

    /* copy the rest to buffer head */
    if (size < dev->recv_size) memmove(dev->recv_buf, dev->recv_buf + size, dev->recv_size - size);

    /* reduce remaining recv buf size */
    dev->recv_size -= size;
    newsize = sizeof(dev->recv_buf) - dev->recv_size;

    spin_unlock_irqrestore(&dev->recv_lock, flags);

    DBG("(%d lost) TPT packet read (%d bytes): %.2x %.2x %.2x ...", esb2_tpt_blk_lost, size,
        (unsigned char)buffer[0], (unsigned char)buffer[1], (unsigned char)buffer[2]);

    /* inform FML master about new recv buf size */
    if (size > 0 && dev->recv_size == 0) {
        if (down_interruptible(&sem) < 0) return -EINTR;
        if (esb2_fml_tpt_read_req(&fml, newsize) < 0) {
            WARN("failed to report free TPT receive buffer size");
        }
        up(&sem);
    }

    return size;
}


static int
esb2_fni_mod_write(struct file* file, const char *buffer, size_t size, loff_t *offset)
{
    char temp[size];
    int ret;
    copy_from_user(temp,buffer,size);
    if (down_interruptible(&sem) < 0) return -EINTR;
    ret=esb2_fni_send(&fml,esb2_dev[1].chan,3,0,temp,size);
    up(&sem);
    if(ret<0)
	return -EIO;
    else
	return size;
}

static int
esb2_tpt_mod_write(struct file* file, const char *buffer, size_t size, loff_t *offset)
{
    int len = 0, c, count;
    int retval = 0;

    DBG("(%d lost) sending TPT packet (%d bytes): %.2x %.2x %.2x ...", esb2_tpt_blk_lost, size,
        (unsigned char)buffer[0], (unsigned char)buffer[1], (unsigned char)buffer[2]);

    if (down_interruptible(&sem) < 0) return -EINTR;
    while (len < (int)size) {
        count = min(size-len, MESB2_MAX_BLK_LEN-(send_fill));
        copy_from_user(send_buf+send_fill, buffer+len, count);

	if ((send_fill+count) >= MESB2_MAX_BLK_LEN) {
            //buffer full, send it out
            c = esb2_tpt_send(&fml, TPT_CHAN, send_buf, MESB2_MAX_BLK_LEN);
            send_fill = 0;
            hk.bytes_sent += c;
        } else {
            //we have got not enough data to send it out, so wake the collector thread
            c = count;
            send_fill += count;
            data_to_send = count;
            wake_up(&send_wait);
        }
        len += count;
        if (c <= 0) {
            ERR("failed to send TPT packet (%d)", c);
            retval = -EIO;
            goto exit;
        }
   }
    retval = len;
exit:
    up(&sem);
    return retval;
}

static unsigned int
esb2_fni_mod_poll(struct file *file, poll_table *wait)
{
    esb2_tpt_dev_t *dev = &esb2_dev[1];
    unsigned int mask = POLLOUT | POLLWRNORM;
    if (dev->recv_size || conn_lost) mask |= POLLIN | POLLRDNORM;
    poll_wait(file, &dev->wait, wait);
    return mask;
}

static unsigned int
esb2_tpt_mod_poll(struct file *file, poll_table *wait)
{
    esb2_tpt_dev_t *dev = &esb2_dev[0];
    unsigned int mask = POLLOUT | POLLWRNORM;
    if (dev->recv_size || conn_lost) mask |= POLLIN | POLLRDNORM;
    poll_wait(file, &dev->wait, wait);
    return mask;
}

/* ------------------------------------------------------------------------- *
 * interrupt service routine
 * ------------------------------------------------------------------------- */

static irqreturn_t
esb2_tpt_mod_isr(int irq, void *dev_id, struct pt_regs *regs)
{
    //esb2_tpt_dev_t *dev = &esb2_tpt_dev;

    /* statistics */
    hk.intr_cnt++;
#ifdef LPC_USE_INTMUX
    {
        unsigned int intmux = readl(dev->intmux_addr);
        if (intmux & FML_INT) hk.fml_intr_cnt++;
    }
#endif

    /* call the FML core driver's IRQ handler */
//WARN("enter isr");
    fmlcore_irq_handler(irq, &fml, regs);
//WARN("leave isr");
    return IRQ_HANDLED;
}

static int
send_collector_thread(void * arg)
{
    //esb2_tpt_dev_t *dev = &esb2_tpt_dev;
    int ret, c;

    daemonize(NAME);

    while (send_thread_running) {
        ret = wait_event_interruptible_timeout(send_wait, (data_to_send > 0), HZ/20);
        if (ret <= 0) {
            //timeout, send immediately
            if (down_interruptible(&sem) < 0) return -EINTR;
            if (send_fill > 0) {
                if ((c = esb2_tpt_send(&fml, TPT_CHAN, send_buf, send_fill)) <= 0) {
                    ERR("failed to send from collector thread");
		} else {
                    send_fill = 0;
 		    hk.bytes_sent += c;
                }
            }
            up(&sem);
        } 
        data_to_send = 0;
	cond_resched();
    }
    return 0;
}

/* will be called back by FML core driver's IRQ handler in case of any notification IRQ */
static int
recv_cb(struct fmlcore_private *fml)
{
//WARN("enter cb");
    esb2_tpt_dev_t *dev;
    int size;
    unsigned char chan,op;
    char buf[SBUF_SIZE];

    hk.slave_intr_cnt++;

    size=esb2_recv(fml,&chan,&op,buf,sizeof(buf));

    if(op==ESB2_OP_TPT)
	dev=&esb2_dev[0];
    else if(op==ESB2_OP_FNI)
	dev=&esb2_dev[1];
    else
	return 0;

    // any op according to recv_buf must be locked against READ
    spin_lock(&dev->recv_lock);

    memcpy(dev->recv_buf + dev->recv_size,buf,min((size_t)size,sizeof(dev->recv_buf) - dev->recv_size));

    /*
    size = esb2_tpt_recv(fml, &chan,
                         dev->recv_buf + dev->recv_size,
                         sizeof(dev->recv_buf) - dev->recv_size);
			 */
//DBG("TPT packet received (%d bytes): %.2x %.2x %.2x %.2x %.2x %.2x %.2x %.2x %.2x ...", size,
//dev->recv_buf[dev->recv_size], dev->recv_buf[dev->recv_size + 1], dev->recv_buf[dev->recv_size + 2],
//dev->recv_buf[dev->recv_size + 3], dev->recv_buf[dev->recv_size + 4], dev->recv_buf[dev->recv_size + 5],
//dev->recv_buf[dev->recv_size + 6], dev->recv_buf[dev->recv_size + 7], dev->recv_buf[dev->recv_size + 8]);

    if (size > 0 && chan == dev->chan) {
        dev->recv_size += size;
        hk.bytes_received += size;
    }

    spin_unlock(&dev->recv_lock);

    if (size > 0 && chan == dev->chan) {
        DBG("%s packet received (%d bytes)", op==ESB2_OP_TPT?"TPT":"FNI",size);
    } else {
        if (size < 0) ERR("failed to read received packet for %s",op==ESB2_OP_TPT?"TPT":"FNI");
    }

    // wake any pending pollers
    if (dev->recv_size > 0) wake_up(&dev->wait);

//WARN("leave cb");
    return 0;
}
