/**
 * fml.h
 *
 * KIRA100 FML device IOCTLs
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#ifndef __FML_H__
#define __FML_H__

typedef struct {
    unsigned char saddr;     /* slave's SMBus address */
    unsigned char cmd;       /* 0bflcccccc f-first l-last c-cmd */
    unsigned char len;       /* number of valid bytes in data */
    unsigned char data[240]; /* max packet size = 240 */
} fml_ioctl_raw_pkt_t;

typedef struct {
    int reg;
    unsigned long val;
} fml_ioctl_reg_t;

typedef struct {
    /* irq counting */
    int intr_cnt;
    int slave_intr_cnt;
} fml_ioctl_hk_t;

#define FML_IOCTL_RAW_SEND  SIOCDEVPRIVATE + 10
#define FML_IOCTL_RAW_RECV  SIOCDEVPRIVATE + 11
#define FML_IOCTL_GET_REG   SIOCDEVPRIVATE + 12
#define FML_IOCTL_SET_REG   SIOCDEVPRIVATE + 13
#define FML_IOCTL_GET_HK    SIOCDEVPRIVATE + 15

/* this is used for our self-test module */
/* possible errno values:
   EIO: an I/O error occured
   ETIMEDOUT: device did not respond
   ENOMSG: response message was not of expected format */
#define FML_IOCTL_PERFORM_TEST  SIOCDEVPRIVATE + 16

/*
 * Helper function to cover IOCTL (only in user mode)
 */
#ifndef __KERNEL__
#include <sys/ioctl.h>
#include <linux/if.h>

static __inline__ int fml_ioctl_raw_send(int sock, const char* if_name, fml_ioctl_raw_pkt_t *pkt)
{
    struct ifreq ifr = { .ifr_ifru.ifru_data = (char*)pkt, };
    strncpy(ifr.ifr_ifrn.ifrn_name, if_name, sizeof(ifr.ifr_ifrn.ifrn_name));
    return ioctl(sock, FML_IOCTL_RAW_SEND, &ifr);
}

static __inline__ int fml_ioctl_raw_recv(int sock, const char* if_name, fml_ioctl_raw_pkt_t *pkt)
{
    struct ifreq ifr = { .ifr_ifru.ifru_data = (char*)pkt, };
    strncpy(ifr.ifr_ifrn.ifrn_name, if_name, sizeof(ifr.ifr_ifrn.ifrn_name));
    return ioctl(sock, FML_IOCTL_RAW_RECV, &ifr);
}

static __inline__ int fml_ioctl_set_reg(int sock, const char* if_name, int reg, unsigned long val)
{
    fml_ioctl_reg_t ioc = { .reg = reg, .val = val, };
    struct ifreq ifr = { .ifr_ifru.ifru_data = (char*)&ioc, };
    strncpy(ifr.ifr_ifrn.ifrn_name, if_name, sizeof(ifr.ifr_ifrn.ifrn_name));
    return ioctl(sock, FML_IOCTL_SET_REG, &ifr);
}

static __inline__ int fml_ioctl_get_reg(int sock, const char* if_name, int reg, unsigned long *val)
{
    fml_ioctl_reg_t ioc = { .reg = reg, };
    struct ifreq ifr = { .ifr_ifru.ifru_data = (char*)&ioc, };
    strncpy(ifr.ifr_ifrn.ifrn_name, if_name, sizeof(ifr.ifr_ifrn.ifrn_name));
    int r = ioctl(sock, FML_IOCTL_GET_REG, &ifr);
    *val = ioc.val;
    return r;
}

static __inline__ int fml_ioctl_get_hk(int sock, const char* if_name, fml_ioctl_hk_t *hk)
{
    struct ifreq ifr = { .ifr_ifru.ifru_data = (char*)hk, };
    strncpy(ifr.ifr_ifrn.ifrn_name, if_name, sizeof(ifr.ifr_ifrn.ifrn_name));
    return ioctl(sock, FML_IOCTL_GET_HK, &ifr);
}

#endif /* !__KERNEL__ */

#endif /* !__FML_H__ */
