#include <linux/kernel.h>
#include <linux/wait.h>
#include <linux/types.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <asm/byteorder.h>

#include "fml_i825xx.h"

#define SUCCESS 0

#define NAME "TCO"
#define LOG_COLORED
#define noLOG_DBG
#include "log.h"

// detailed debug logging
//#define DBG2 DBG
#define DBG2(...)

int i825xx_detect(tco_t *tco)
{
    uint8_t ctrl = 0x10;
    if (tco->write_block(tco, 0xc8, 1, &ctrl) < 0) {
	/* NAK: i8254x Gigabit Ethernet Controller */
	INFO("i825xx_detect(): i8254x detected.\n");
    } else {
	/* ACK: i8255x Fast Ethernet Controller */
	INFO("i825xx_detect(): i8255x detected.\n");
    }
    return 0;
}

int i825xx_xmit_packet(tco_t *tco, uint8_t seq, uint8_t len, uint8_t *data)
{
    return tco->write_block(tco, seq | I825XX_CMD_XMIT, len, data);
}

int i825xx_recv_enable(tco_t *tco, uint8_t ctrl)
{
    return tco->write_block(tco, I825XX_SEQ_SINGLE | I825XX_CMD_RECV_EN,
	    1, &ctrl);
}

int i825xx_request_status(tco_t *tco)
{
    uint8_t data[] = { 0 };
    return tco->write_block(tco, I825XX_SEQ_SINGLE | I825XX_CMD_REQ_STATUS,
	    sizeof(data), data);
    /* i825xx_recv() should now return a I825XX_RECV_STATUS packet */
}

int i825xx_force_tco(tco_t *tco, uint8_t mode)
{
    return tco->write_block(tco, I825XX_SEQ_SINGLE | I825XX_CMD_FORCE_TCO,
	    1, &mode);
}

int i825xx_recv(tco_t *tco, uint8_t *count, tco_recv_pkt_t *recv_pkt)
{
    int r;
    if ((r = tco->read_block(tco, I825XX_SEQ_SINGLE | I825XX_CMD_RECV_ANY,
		    count, (uint8_t*)recv_pkt)) < 0)
	return r;
    return SUCCESS;
}

int i825xx_read_status(tco_t *tco, uint8_t *status)
{
    int r;
    tco_recv_pkt_t recv_pkt;
    uint8_t count;

    if ((r = i825xx_request_status(tco)) < 0) {
	return r;
    }

    if ((r = i825xx_recv(tco, &count, &recv_pkt)) < 0) {
	return r;
    }

    if (recv_pkt.op_code != (I825XX_SEQ_SINGLE | I825XX_RECV_STATUS)) {
	return -EIO;
    }
    if (count != 3 || !(recv_pkt.data[1] & 0x01)) return -EIO;

    *status = recv_pkt.data[0];
    return SUCCESS;
}

