#include <linux/errno.h>

#include "fml.h"
#include "fml_mod.h"
#include "fml_i825xx.h"
#include "fml_jobs.h"
#include "mac_cache.h"

#define NAME "TCO"
#define LOG_COLORED
#define noLOG_DBG
#include "log.h"

// detailed debug logging
//#define DBG2 DBG
#define DBG2(...)

/* ------------------------------------------------------------------------- *
 * utils
 * ------------------------------------------------------------------------- */

static void log_stat(uint8_t stat, int info)
{
    if (stat & I825XX_CMD_ABORTED) {
	WARN("Status: ABRT=%d LINK=%d RU=%d INIT=%d PWR=D%d",
		(stat & I825XX_CMD_ABORTED) ? 1 : 0,
		(stat & I825XX_LINK_STATE)  ? 1 : 0,
		(stat & I825XX_RU_READY)    ? 1 : 0,
		(stat & I825XX_INIT)        ? 1 : 0,
		(stat & I825XX_POWER_STATE_MASK) >> 8);
    } else if (info) {
	INFO("Status: ABRT=%d LINK=%d RU=%d INIT=%d PWR=D%d",
		(stat & I825XX_CMD_ABORTED) ? 1 : 0,
		(stat & I825XX_LINK_STATE)  ? 1 : 0,
		(stat & I825XX_RU_READY)    ? 1 : 0,
		(stat & I825XX_INIT)        ? 1 : 0,
		(stat & I825XX_POWER_STATE_MASK) >> 8);
    } else {
	DBG("Status: ABRT=%d LINK=%d RU=%d INIT=%d PWR=D%d",
		(stat & I825XX_CMD_ABORTED) ? 1 : 0,
		(stat & I825XX_LINK_STATE)  ? 1 : 0,
		(stat & I825XX_RU_READY)    ? 1 : 0,
		(stat & I825XX_INIT)        ? 1 : 0,
		(stat & I825XX_POWER_STATE_MASK) >> 8);
    }
}

/* ------------------------------------------------------------------------- *
 * jobs
 * ------------------------------------------------------------------------- */

int send(void *ctx)
{
    struct net_device *dev = &fml_mod_dev;
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;
    struct sk_buff *skb = (struct sk_buff*)priv->send_skb;

    int first, last, len, r;
    uint8_t seq;

    if (!priv->is_init) {
	priv->stats.tx_errors++;
	goto exit;
    }

    mac_cache_lookup(skb);

    DBG2("sending fragment of frame %ld (offs=%d frag-len=%d pkt-len=%d)",
	    priv->stats.tx_packets, priv->send_offs, len, skb->len);

    first = 1;
    do {
	last = skb->len - priv->send_offs <= priv->tco->max_data_size;
	len = last ? skb->len - priv->send_offs : priv->tco->max_data_size;
	seq = (first ? I825XX_SEQ_FIRST : 0) | (last ? I825XX_SEQ_LAST : 0);
	if ((r = i825xx_xmit_packet(priv->tco, seq, len, skb->data + priv->send_offs)) < 0) {
	    ERR("send: failed to send frame - offset %d (err %d)", priv->send_offs, r);
	    priv->stats.tx_errors++;
	    goto exit;
	}
	first = 0;
	priv->send_offs += len;
    } while (!last);

    dev->trans_start = jiffies;
    /* statistics */
    DBG("ethernet frame %ld successfully sent (%d bytes)",
	    priv->stats.tx_packets, skb->len);
    //DBG("  %02x %02x %02x %02x", 
    //    skb->data[0], skb->data[1], skb->data[2], skb->data[3]);
    priv->stats.tx_packets++;
    priv->stats.tx_bytes += skb->len;

exit:
    /* free the buffer */
    dev_kfree_skb(priv->send_skb);
    priv->send_skb = NULL;

    /* switch on transmission again */
    netif_wake_queue(dev);

    return SUCCESS;
}

int recv(void *ctx)
{
    //TWARN("enter recv task");
    struct net_device *dev = &fml_mod_dev;
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;
    struct sk_buff *skb = priv->recv_skb;

    tco_recv_pkt_t recv_pkt;
    uint8_t count;
    int ret = SUCCESS;

    if (!priv->is_init) return -1;
    priv->hk.slave_intr_cnt++;

    do {
	if (i825xx_recv(priv->tco, &count, &recv_pkt) != SUCCESS) {
	    ERR("recv: TCO read failed");
	    ret = -EIO;
	    break;
	}

	switch (recv_pkt.op_code & I825XX_CMD_MASK) {
	    case I825XX_RECV_PACKET:
		{
		    if (recv_pkt.op_code & I825XX_SEQ_FIRST) {
			/* first packet fragment: prepare sbk */
			if (skb->len != 0) {
			    ERR("recv: new frame while old not completed, discard old frame (%d bytes)",
				    skb->len);
			    skb_trim(skb, 0);
			}
		    }

		    if (recv_pkt.op_code & I825XX_SEQ_LAST) {
			int ok = 1;

			/* last packet fragment: just contains status */
			if (count == 5) {
			    DBG2("receiving last fragment of frame %ld (status 0x%02x 0x%02x 0x%02x 0x%02x)",
				    priv->stats.rx_packets,
				    recv_pkt.data[0], recv_pkt.data[1], recv_pkt.data[2], recv_pkt.data[3]);
			} else {
			    ERR("recv: last fragment (status) has wrong size: %d", count);
			    ok = 0;
			}

			if (ok) {
			    DBG("ethernet frame %ld successfully received (%d bytes)",
				    priv->stats.rx_packets, skb->len);
			    //DBG("  %02x %02x %02x %02x",
			    //    skb->data[0], skb->data[1], skb->data[2], skb->data[3]);
			    mac_cache_add(skb);

			    dev->last_rx = jiffies;

			    /* strip CRC */
			    skb->len -= 4;

			    /* statistics */
			    priv->stats.rx_packets++;
			    priv->stats.rx_bytes += skb->len;

			    /* send packet upstairs */
			    skb->dev = &fml_mod_dev;
			    skb->protocol = eth_type_trans(skb, &fml_mod_dev);
			    netif_rx(skb);

			    /* get new buffer (size: payload + header + CRC) */
			    priv->recv_skb = dev_alloc_skb(dev->mtu + ETH_HLEN + 4);
			} else {
			    skb_trim(skb, 0);
			    ret = -EIO;
			    break;
			}
		    } else {
			/* process that packet fragment */
			DBG2("receiving fragment of frame %ld (offs=%d frag-len=%d)",
				priv->stats.rx_packets, priv->recv_skb->len, count - 1);

			if (priv->recv_skb->tail + count - 1 > priv->recv_skb->end) {
			    ERR("recv: frame exceeds max length (offs=%d frag-len=%d max=%d)",
				    priv->recv_skb->len, count - 1, dev->mtu + ETH_HLEN);
			} else {
			    memcpy(__skb_put(priv->recv_skb, count - 1), recv_pkt.data, count - 1);
			}
		    }
		    break;
		}

	    case I825XX_RECV_STATUS:
		{
		    uint8_t stat = recv_pkt.data[0];
		    log_stat(stat, 1);

		    /* update link status */
		    if (stat & I825XX_LINK_STATE) {
			if (!netif_carrier_ok(dev)) {
			    netif_carrier_on(dev);
			}
		    } else {
			if (netif_carrier_ok(dev)) {
			    netif_carrier_off(dev);
			}
		    }

		    if (stat & I825XX_INIT) {
			INFO("reset detected - re-init");
			add_job(priv, &priv->reinit_job);
		    }

		    break;
		}

	    default:
		ERR("recv: got unknown op_code 0x%02x", recv_pkt.op_code);
		ret = -EIO;
		break;
	}
    } while (!(recv_pkt.op_code & I825XX_SEQ_LAST));

    /* re-enable FML slave interrupt */
    priv->tco->enable_client_int(priv->tco, 1);

    //TWARN("leave recv task");
    return ret;
}

int init(void* ctx)
{
    struct net_device *dev = &fml_mod_dev;
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;
    int r = SUCCESS;
    uint8_t stat;

    /* disable alerting */
    priv->tco->enable_client_int(priv->tco, 0);

    /* read status */
    if ((r = i825xx_read_status(priv->tco, &stat)) < 0) {
	ERR("failed to read status (err %d)", r);
    } else {
	log_stat(stat, 1);
    }

    /* force TCO mode */
    if (ctx == FIRST_INIT) {
	i825xx_detect(priv->tco);
	if ((r = i825xx_recv_enable(priv->tco, 0)) < 0) {
	    ERR("failed to disable pass-thru (err %d)", r);
	    return -EIO;
	}
	if ((r = i825xx_force_tco(priv->tco, 1)) < 0) {
	    ERR("failed to force TCO mode (err %d)", r);
	    return -EIO;
	}
    }

    /* enable pass-thru */
    DBG("Enable Pass-Thru");
    if ((r = i825xx_recv_enable(priv->tco,
		    I825XX_RCV_EN | I825XX_EN_STA)) < 0) {
	ERR("failed to enable pass-thru (err %d)", r);
	return -EIO;
    }

    dev->mtu = 182; /* 200 bytes, including MAC header and CRC */
    dev->rebuild_header = NULL;
    dev->hard_header_cache = NULL;
    dev->header_cache_update = NULL;
    dev->flags |= IFF_NOARP;

    priv->tco->enable_client_int(priv->tco, 1);
    priv->is_init = 1;

    return SUCCESS;
}

int done(void* ctx)
{
    struct net_device *dev = &fml_mod_dev;
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;

    // mark driver as unloaded
    priv->is_init = 0;
    return SUCCESS;
}

int set_mac(void *ctx)
{
    struct net_device *dev = &fml_mod_dev;
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;
    uint8_t *new_mac_addr = ctx;

    // store mac addr
    memcpy(dev->dev_addr, new_mac_addr, dev->addr_len);
    priv->mac_set = 1;

    DBG("MAC addr set to %02x:%02x:%02x:%02x:%02x:%02x",
	    new_mac_addr[0], new_mac_addr[1], new_mac_addr[2],
	    new_mac_addr[3], new_mac_addr[4], new_mac_addr[5]);

    return SUCCESS;
}

int raw_send(void *ctx)
{
    struct net_device *dev = &fml_mod_dev;
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;

    fml_ioctl_raw_pkt_t *d = (fml_ioctl_raw_pkt_t*)ctx;
    return priv->tco->write_block(priv->tco, d->cmd, d->len, d->data);
}

int raw_recv(void *ctx)
{
    struct net_device *dev = &fml_mod_dev;
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;

    fml_ioctl_raw_pkt_t *d = (fml_ioctl_raw_pkt_t*)ctx;
    return priv->tco->read_block(priv->tco, d->cmd, &d->len, d->data);
}

