#include "fml.h"
#include "fml_mod.h"
#include "fml_jobs.h"
#include "fml_ophir.h"

#define NAME "TCO"
#define LOG_COLORED
#define noLOG_DBG
#include "log.h"

// detailed debug logging
//#define DBG2 DBG
#define DBG2(...)

/* ------------------------------------------------------------------------- *
 * utils
 * ------------------------------------------------------------------------- */

static void log_stat(uint16_t stat, int info)
{
    const char *power_states[] = { "Dr", "D0u", "D0", "D3" };
    if (stat & OPHIR_CMD_ABORTED) {
        WARN("Status: ABRT=%d LINK=%d INIT=%d PWR=%s DRV=%d INT=%d ICR=%d",
            (stat & OPHIR_CMD_ABORTED)    ? 1 : 0,
            (stat & OPHIR_LINK_STATE)     ? 1 : 0,
            (stat & OPHIR_INIT)           ? 1 : 0,
            power_states[(stat & OPHIR_POWER_STATE_MASK) >> 8],
            (stat & OPHIR_DRV_VALID)      ? 1 : 0,
            (stat & OPHIR_INTR_PENDING)   ? 1 : 0,
            (stat & OPHIR_ICR_REG_READ)   ? 1 : 0);
    } else if (info) {
        INFO("Status: ABRT=%d LINK=%d INIT=%d PWR=%s DRV=%d INT=%d ICR=%d",
            (stat & OPHIR_CMD_ABORTED)    ? 1 : 0,
            (stat & OPHIR_LINK_STATE)     ? 1 : 0,
            (stat & OPHIR_INIT)           ? 1 : 0,
            power_states[(stat & OPHIR_POWER_STATE_MASK) >> 8],
            (stat & OPHIR_DRV_VALID)      ? 1 : 0,
            (stat & OPHIR_INTR_PENDING)   ? 1 : 0,
            (stat & OPHIR_ICR_REG_READ)   ? 1 : 0);
    } else {
        DBG("Status: ABRT=%d LINK=%d INIT=%d PWR=%s DRV=%d INT=%d ICR=%d",
            (stat & OPHIR_CMD_ABORTED)    ? 1 : 0,
            (stat & OPHIR_LINK_STATE)     ? 1 : 0,
            (stat & OPHIR_INIT)           ? 1 : 0,
            power_states[(stat & OPHIR_POWER_STATE_MASK) >> 8],
            (stat & OPHIR_DRV_VALID)      ? 1 : 0,
            (stat & OPHIR_INTR_PENDING)   ? 1 : 0,
            (stat & OPHIR_ICR_REG_READ)   ? 1 : 0);
    }
}

static void log_cfg(uint8_t* mac_addr, uint8_t* ip_addr,
                    uint8_t bmc_smbus_addr, uint8_t ctrl)
{
    INFO("MAC address: %02x:%02x:%02x:%02x:%02x:%02x",
         mac_addr[0], mac_addr[1], mac_addr[2],
         mac_addr[3], mac_addr[4], mac_addr[5]);
    INFO("IP address: %d.%d.%d.%d",
         ip_addr[0], ip_addr[1], ip_addr[2], ip_addr[3]);
    INFO("BMC SMBus address: 0x%x", bmc_smbus_addr);
    INFO("Flags: RCV_EN=%d RCV_ALL=%d EN_STA=%d EN_ARP_RES=%d NM=%d RPSTL=%d CBDM=%d",
        (ctrl & OPHIR_RCV_EN)     ? 1 : 0,
        (ctrl & OPHIR_RCV_ALL)    ? 1 : 0,
        (ctrl & OPHIR_EN_STA)     ? 1 : 0,
        (ctrl & OPHIR_EN_ARP_RES) ? 1 : 0,
        (ctrl & OPHIR_NM_MASK) >> 4,
        (ctrl & OPHIR_RPSTL)      ? 1 : 0,
        (ctrl & OPHIR_CBDM)       ? 1 : 0);
}

/* ------------------------------------------------------------------------- *
 * jobs
 * ------------------------------------------------------------------------- */

int send(void *ctx)
{
    struct net_device *dev = &fml_mod_dev;
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;
    struct sk_buff *skb = (struct sk_buff*)priv->send_skb;

    // don't check skb->len against ETH_ZLEN since ESB2 will do the padding
    int first = priv->send_offs == 0;
    int last = skb->len - priv->send_offs <= priv->tco->max_data_size;
    int len = last ? skb->len - priv->send_offs : priv->tco->max_data_size;
    uint8_t seq = (first ? OPHIR_SEQ_FIRST : 0) | (last ? OPHIR_SEQ_LAST : 0);
    int r;

    if (!priv->is_init) {
        priv->stats.tx_errors++;
        goto exit;
    }

    /* shut down slave alerting which may give disturbing interrupts */
    priv->tco->enable_client_int(priv->tco, 0);

    DBG2("sending fragment of frame %ld (offs=%d frag-len=%d pkt-len=%d)",
        priv->stats.tx_packets, priv->send_offs, len, skb->len);

    if ((r = ophir_xmit_packet(priv->tco, seq, len, skb->data + priv->send_offs)) < 0) {
        ERR("send: failed to send frame - offset %d (err %d)", priv->send_offs, r);
        priv->stats.tx_errors++;
        goto exit;
    }

    priv->send_offs += len;

    if (first) {
        dev->trans_start = jiffies;
    }

    // more fragments to send?
    if (!last) {
        add_job(priv, &priv->send_job);
        return SUCCESS;
    } else {
        /* statistics */
        DBG("ethernet frame %ld successfully sent (%d bytes)",
            priv->stats.tx_packets, skb->len);
        //DBG("  %02x %02x %02x %02x", 
        //    skb->data[0], skb->data[1], skb->data[2], skb->data[3]);
        priv->stats.tx_packets++;
        priv->stats.tx_bytes += skb->len;
    }

exit:
    /* free the buffer */
    dev_kfree_skb(priv->send_skb);
    priv->send_skb = NULL;

    /* re-enable slave alerting */
    priv->tco->enable_client_int(priv->tco, 1);

    /* switch on transmission again */
    netif_wake_queue(dev);

    return SUCCESS;
}

int recv(void *ctx)
{
    struct net_device *dev = &fml_mod_dev;
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;
    struct sk_buff *skb = priv->recv_skb;

    tco_recv_pkt_t recv_pkt;
    uint8_t count;

    if (!priv->is_init) return -1;

    if (priv->tco->check_int_source && SUCCESS != priv->tco->check_int_source(priv->tco)) {
        // other alert source
        DBG("other alarm source");
    } else if (SUCCESS == ophir_recv_any(priv->tco, &count, &recv_pkt)) {

        priv->hk.slave_intr_cnt++;

        switch (recv_pkt.op_code & OPHIR_CMD_MASK) {
        case OPHIR_OP_RECV_PKT:
        {
            if (recv_pkt.op_code & OPHIR_SEQ_FIRST) {
                /* first packet fragment: prepare sbk */
                if (skb->len != 0) {
                    ERR("recv: new frame while old not completed, discard old frame (%d bytes)",
                        skb->len);
                    skb_trim(skb, 0);
                }
            }

            if (recv_pkt.op_code & OPHIR_SEQ_LAST) {
                int ok = 1;

                /* last packet fragment: just contains status */
                if (count == 5) {
                    DBG2("receiving last fragment of frame %ld (short status 0x%02x 0x%02x 0x%02x 0x%02x)",
                        priv->stats.rx_packets,
                        recv_pkt.data[0], recv_pkt.data[1], recv_pkt.data[2], recv_pkt.data[3]);
                } else if (count == 9) {
                    DBG2("receiving last fragment of frame %ld (long status"
                         " 0x%02x 0x%02x 0x%02x 0x%02x"
                         " 0x%02x 0x%02x 0x%02x 0x%02x)",
                        priv->stats.rx_packets,
                        recv_pkt.data[0], recv_pkt.data[1], recv_pkt.data[2], recv_pkt.data[3],
                        recv_pkt.data[4], recv_pkt.data[5], recv_pkt.data[6], recv_pkt.data[7]);
                    uint16_t len = recv_pkt.data[0] + ((uint16_t)(recv_pkt.data[1] & 0x1f) << 8);
//                    int crc_stripped = recv_pkt.data[5] & 0x10;

                    if (len != skb->len) {
                        ERR("recv: packet size mismatch (exp:%d got:%d)",
                            len, skb->len);
                        ok = 0;
                    }

#if 0 // not working!
                    if (!crc_stripped) {
                        // Ethernet CRC check
                        uint32_t crc;
                        skb->len -= 4; // remove CRC
                        crc = crc32(~0, skb->data, skb->len); // calc CRC

                        if (crc != *(uint32_t*)(skb->data + skb->len)) {
                            ERR("recv: Ethernet CRC mismatch (exp:%08x got:%08x)",
                                *(uint32_t*)(skb->data + skb->len), crc);
                            //ok = 0;
                        }
else INFO("crc test passed");
                    }
#endif
                } else {
                    ERR("recv: last fragment (status) has wrong size: %d", count);
                    ok = 0;
                }

                if (ok) {
                    DBG("ethernet frame %ld successfully received (%d bytes)",
                        priv->stats.rx_packets, skb->len);
                    //DBG("  %02x %02x %02x %02x",
                    //    skb->data[0], skb->data[1], skb->data[2], skb->data[3]);

                    dev->last_rx = jiffies;

                    /* statistics */
                    priv->stats.rx_packets++;
                    priv->stats.rx_bytes += skb->len;

                    /* send packet upstairs */
                    skb->dev = &fml_mod_dev;
                    skb->protocol = eth_type_trans(skb, &fml_mod_dev);
                    netif_rx(skb);

                    /* get new buffer (size: payload + header, without CRC!) */
                    priv->recv_skb = dev_alloc_skb(dev->mtu + ETH_HLEN);
                } else {
                    skb_trim(skb, 0);
                }
            } else {
                /* process that packet fragment */
                DBG2("receiving fragment of frame %ld (offs=%d frag-len=%d)",
                    priv->stats.rx_packets, priv->recv_skb->len, count - 1);

                if (priv->recv_skb->tail + count - 1 > priv->recv_skb->end) {
                    ERR("recv: frame exceeds max length (offs=%d frag-len=%d max=%d)",
                        priv->recv_skb->len, count - 1, dev->mtu + ETH_HLEN);
                } else {
                    memcpy(__skb_put(priv->recv_skb, count - 1), recv_pkt.data, count - 1);
                }
            }
            break;
        }

        case OPHIR_OP_READ_STAT:
        {
            uint16_t stat = be16_to_cpu(*(uint16_t*)recv_pkt.data);
            log_stat(stat, 0);

            /* update link status */
            if (stat & OPHIR_LINK_STATE) {
                if (!netif_carrier_ok(dev)) {
                    netif_carrier_on(dev);
                }
            } else {
                if (netif_carrier_ok(dev)) {
                    netif_carrier_off(dev);
                }
            }

            if (stat & OPHIR_INIT) {
                // reset detected: re-init ESB2
                INFO("reset detected - re-init");
                add_job(priv, &priv->reinit_job);
            }

            break;
        }

        default:
            ERR("recv: got unknown op_code 0x%02x", recv_pkt.op_code);
        }
    } else {
        ERR("recv: failed to get status");
    }

#if 0
    if (skb->len != 0) {
        // do we still expect fragments?
        // if yes, than manually queue the next recv job, since sometimes for large packages
        // the ESB2 seems not to generate SMBus alert for later fragments!
        add_job(priv, &priv->recv_job);

        return SUCCESS;
    }
#endif

    /* re-enable FML slave interrupt */
    priv->tco->enable_client_int(priv->tco, 1);

    return SUCCESS;
}

int init(void* ctx)
{
    struct net_device *dev = &fml_mod_dev;
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;
    int r = SUCCESS;
    uint16_t stat;
    uint8_t ctrl, mac_addr[6], ip_addr[4], bmc_smbus_addr, if_data, alert_data;
    uint32_t filter_en, mng2host;
    uint8_t len;

    if (ctx == RE_INIT) {
        // don't do re-init when has been already called
        if (!priv->is_init) return -1;
    }

    /* disable alerting */
    priv->tco->enable_client_int(priv->tco, 0);

    INFO("ESB2 original config:");

    /* read status */
    if ((r = ophir_read_status(priv->tco, &stat)) < 0) {
        ERR("failed to read status (err %d)", r);
    } else {
        log_stat(stat, 1);
    }

    /* get configuration */
    if ((r = ophir_read_recv_enable(priv->tco, &ctrl, mac_addr, ip_addr, &bmc_smbus_addr, &if_data, &alert_data)) < 0) {
        ERR("failed to read config (err %d)", r);
        r = -EIO;
        goto exit;
    } else {
        log_cfg(mac_addr, ip_addr, bmc_smbus_addr, ctrl);
    }

    /* modify filter settings */
    len = sizeof(filter_en);
    if ((r = ophir_read_filter(priv->tco, 0x01, &len, (uint8_t*)&filter_en)) < 0) {
        ERR("failed to get filter enable value (err %d)", r);
    } else {
        filter_en = be32_to_cpu(filter_en);
        INFO("filter enable: 0x%08x", filter_en);

        filter_en |= OHPIR_MANC_MNG2HOST_EN | OHPIR_MANC_BCAST_EN;
        filter_en = cpu_to_be32(filter_en);
        DBG("setting filter enable to %08x", filter_en);
        if ((r = ophir_update_filter(priv->tco, 0x01, sizeof(filter_en), (uint8_t*)&filter_en)) < 0) {
            ERR("failed to set filter enable value (err %d)", r);
            r = -EIO;
            goto exit;
        }
    }

    len = sizeof(mng2host);
    if ((r = ophir_read_filter(priv->tco, 0x0a, &len, (uint8_t*)&mng2host)) < 0) {
        ERR("failed to get mng2host value (err %d)", r);
    } else {
        mng2host = be32_to_cpu(mng2host);
        INFO("mng2host: 0x%08x", mng2host);

        mng2host |= OHPIR_MNG2HOST_ARP_REQ | OHPIR_MNG2HOST_ARP_RES | OHPIR_MNG2HOST_BCAST;
        mng2host = cpu_to_be32(mng2host);
        DBG("setting mng2host to %08x", mng2host);
        if ((r = ophir_update_filter(priv->tco, 0x0a, sizeof(mng2host), (uint8_t*)&mng2host)) < 0) {
            ERR("failed to set mng2host value (err %d)", r);
            r = -EIO;
            goto exit;
        }
    }

    /* enable pass-thru */
    DBG("Enable Pass-Thru");
#ifdef ARP_BY_ESB2
    if (priv->mac_set) {
        // MAC already set manually: write it to ESB2
        memcpy(mac_addr, dev->dev_addr, 6);
    } else {
        // MAC not set manually: get it from ESB2
        memcpy(dev->dev_addr, mac_addr, 6);
    }

    // ESB2 does the ARP
    ctrl = OPHIR_RCV_EN | OPHIR_EN_STA  | OPHIR_EN_ARP_RES | OPHIR_NM_SMB_ALERT
          | OPHIR_RPSTL | OPHIR_CBDM;
    // AMC addr already set
    // TODO: find a way to determine IP address here or to initialize later
    ip_addr[0] = 192; ip_addr[1] = 168; ip_addr[2] = 1; ip_addr[3] = 90;
    if ((r = ophir_recv_enable_ex(priv->tco, ctrl, mac_addr, ip_addr,
                                  bmc_smbus_addr, if_data, alert_data)) < 0) {
        ERR("failed to enable pass-thru (err %d)", r);
        r = -EIO;
        goto exit;
    }
#else
    // we do the ARP
    ctrl = OPHIR_RCV_EN | OPHIR_EN_STA  | OPHIR_NM_SMB_ALERT | OPHIR_RPSTL
         | OPHIR_CBDM;
    memcpy(mac_addr, dev->dev_addr, 6);
    if ((r = ophir_recv_enable_ex(priv->tco, ctrl, mac_addr, ip_addr,
                                  bmc_smbus_addr, if_data, alert_data)) < 0) {
        ERR("failed to enable pass-thru (err %d)", r);
        r = -EIO;
        goto exit;
    }
#endif

    /* get system mac addr */
    INFO("Get System MAC Addr");
    if ((r = ophir_get_sys_mac_addr(priv->tco, mac_addr)) < 0) {
        ERR("failed to get system MAC addr (err %d)", r);
    } else {
        INFO("  System MAC address: %02x:%02x:%02x:%02x:%02x:%02x",
            mac_addr[0], mac_addr[1], mac_addr[2],
            mac_addr[3], mac_addr[4], mac_addr[5]);
    }

    INFO("ESB2 initial config:");

    /* read status */
    if ((r = ophir_read_status(priv->tco, &stat)) < 0) {
        ERR("failed to read status (err %d)", r);
    } else {
        log_stat(stat, 1);
    }

    /* get configuration */
    if ((r = ophir_read_recv_enable(priv->tco, &ctrl, mac_addr, ip_addr,
                               &bmc_smbus_addr, &if_data, &alert_data)) < 0) {
        ERR("failed to read config (err %d)", r);
    } else {
        log_cfg(mac_addr, ip_addr, bmc_smbus_addr, ctrl);
    }

exit:
    if (r != SUCCESS) {
        // wait for 10 secs to not spam I2C or FML in case of ESB2 not responding
        set_current_state(TASK_INTERRUPTIBLE);
        schedule_timeout(10 * HZ);
        // TODO: just for bug hunting. remove this line later!
        if (signal_pending(current)) ERR("SIGNAL DETECTED!");

        // queues another init job (will be executed after this init has returned)
        add_job(priv, &priv->reinit_job);

        r = -EAGAIN;
    } else {
        /* enable alerting (only if init succeeded) */
        priv->tco->enable_client_int(priv->tco, 1);
    }

    // first time init passed: mark driver as loaded
    priv->is_init = 1;

    return r;
}

int done(void* ctx)
{
    struct net_device *dev = &fml_mod_dev;
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;

    // mark driver as unloaded
    priv->is_init = 0;
    return SUCCESS;
}

int set_mac(void *ctx)
{
    struct net_device *dev = &fml_mod_dev;
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;
    uint8_t *new_mac_addr = ctx;

#ifdef ARP_BY_ESB2
    // TODO: this could be called before driver is opened
    //       -> in that case, all init should be done at driver load time

    // ARP done by ESB2, it need to know the MAC addr
    int r;

    // get config
    uint8_t ctrl, mac_addr[6], ip_addr[4], bmc_smbus_addr, if_data, alert_data;
    if ((r = ophir_read_recv_enable(priv->tco, &ctrl, mac_addr, ip_addr,
                                &bmc_smbus_addr, &if_data, &alert_data)) < 0) {
        ERR("fml_mod_set_mac_addr: failed to read config (err %d)", r);
        return -EIO;
    }

    // patch mac addr
    memcpy(mac_addr, new_mac_addr, sizeof(mac_addr));

    // set config
    if ((r = ophir_recv_enable_ex(priv->tco, ctrl, mac_addr, ip_addr,
                                  bmc_smbus_addr, if_data, alert_data)) < 0) {
        ERR("fml_mod_set_mac_addr: failed to write config (err %d)", r);
        return -EIO;
    }
#endif

    // store mac addr
    memcpy(dev->dev_addr, new_mac_addr, dev->addr_len);
    priv->mac_set = 1;

    DBG("MAC addr set to %02x:%02x:%02x:%02x:%02x:%02x",
        new_mac_addr[0], new_mac_addr[1], new_mac_addr[2],
        new_mac_addr[3], new_mac_addr[4], new_mac_addr[5]);

    return SUCCESS;
}

int raw_send(void *ctx)
{
    struct net_device *dev = &fml_mod_dev;
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;

    fml_ioctl_raw_pkt_t *d = (fml_ioctl_raw_pkt_t*)ctx;
    return priv->tco->write_block(priv->tco, d->cmd, d->len, d->data);
}

int raw_recv(void *ctx)
{
    struct net_device *dev = &fml_mod_dev;
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;

    fml_ioctl_raw_pkt_t *d = (fml_ioctl_raw_pkt_t*)ctx;
    return priv->tco->read_block(priv->tco, d->cmd, &d->len, d->data);
}

