/**
 * fml_mod.c
 *
 * Linux network driver via FML Advanced Pass-Thru
 *
 * (c) 2004 Peppercon AG, 2004/12/20, Ralf Guenther <rgue@peppecon.de>
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/netdevice.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/etherdevice.h>
#include <linux/pci.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/tqueue.h>
#include <linux/crc32.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/atomic.h>

#include "tco_core.h"
#include "fml_mod.h"
#include "fml.h"
#include "fml_jobs.h"

#define SUCCESS 0
#define FAIL    -1

int debug = 0;

#define NAME "TCO"
#define LOG_COLORED
#define noLOG_DBG
#include "log.h"

// detailed debug logging
#define DBG2 DBG
//#define DBG2(...)

// either we (Linux) or the ESB2 can do the ARP
// (if ESB2 is doing it, it needs to know our IP addr, see TODO below)
//#define ARP_BY_ESB2

void dump(unsigned char* buf, int size)
{
    int i, j;
    for (j = 0; j < size; j += 16) {
        for (i = 0; i < 16; i++) {
            if (j + i >= size) break;
            if (i == 8) printk(" ");
            printk(" %02x", buf[j + i]);
        }
        printk("\n");
    }
}

/* ------------------------------------------------------------------------- *
 * Ethernet CRC stuff
 * ------------------------------------------------------------------------- */

const uint32_t crc32_table[256] = {
	0x00000000L, 0x77073096L, 0xee0e612cL, 0x990951baL, 0x076dc419L,
	0x706af48fL, 0xe963a535L, 0x9e6495a3L, 0x0edb8832L, 0x79dcb8a4L,
	0xe0d5e91eL, 0x97d2d988L, 0x09b64c2bL, 0x7eb17cbdL, 0xe7b82d07L,
	0x90bf1d91L, 0x1db71064L, 0x6ab020f2L, 0xf3b97148L, 0x84be41deL,
	0x1adad47dL, 0x6ddde4ebL, 0xf4d4b551L, 0x83d385c7L, 0x136c9856L,
	0x646ba8c0L, 0xfd62f97aL, 0x8a65c9ecL, 0x14015c4fL, 0x63066cd9L,
	0xfa0f3d63L, 0x8d080df5L, 0x3b6e20c8L, 0x4c69105eL, 0xd56041e4L,
	0xa2677172L, 0x3c03e4d1L, 0x4b04d447L, 0xd20d85fdL, 0xa50ab56bL,
	0x35b5a8faL, 0x42b2986cL, 0xdbbbc9d6L, 0xacbcf940L, 0x32d86ce3L,
	0x45df5c75L, 0xdcd60dcfL, 0xabd13d59L, 0x26d930acL, 0x51de003aL,
	0xc8d75180L, 0xbfd06116L, 0x21b4f4b5L, 0x56b3c423L, 0xcfba9599L,
	0xb8bda50fL, 0x2802b89eL, 0x5f058808L, 0xc60cd9b2L, 0xb10be924L,
	0x2f6f7c87L, 0x58684c11L, 0xc1611dabL, 0xb6662d3dL, 0x76dc4190L,
	0x01db7106L, 0x98d220bcL, 0xefd5102aL, 0x71b18589L, 0x06b6b51fL,
	0x9fbfe4a5L, 0xe8b8d433L, 0x7807c9a2L, 0x0f00f934L, 0x9609a88eL,
	0xe10e9818L, 0x7f6a0dbbL, 0x086d3d2dL, 0x91646c97L, 0xe6635c01L,
	0x6b6b51f4L, 0x1c6c6162L, 0x856530d8L, 0xf262004eL, 0x6c0695edL,
	0x1b01a57bL, 0x8208f4c1L, 0xf50fc457L, 0x65b0d9c6L, 0x12b7e950L,
	0x8bbeb8eaL, 0xfcb9887cL, 0x62dd1ddfL, 0x15da2d49L, 0x8cd37cf3L,
	0xfbd44c65L, 0x4db26158L, 0x3ab551ceL, 0xa3bc0074L, 0xd4bb30e2L,
	0x4adfa541L, 0x3dd895d7L, 0xa4d1c46dL, 0xd3d6f4fbL, 0x4369e96aL,
	0x346ed9fcL, 0xad678846L, 0xda60b8d0L, 0x44042d73L, 0x33031de5L,
	0xaa0a4c5fL, 0xdd0d7cc9L, 0x5005713cL, 0x270241aaL, 0xbe0b1010L,
	0xc90c2086L, 0x5768b525L, 0x206f85b3L, 0xb966d409L, 0xce61e49fL,
	0x5edef90eL, 0x29d9c998L, 0xb0d09822L, 0xc7d7a8b4L, 0x59b33d17L,
	0x2eb40d81L, 0xb7bd5c3bL, 0xc0ba6cadL, 0xedb88320L, 0x9abfb3b6L,
	0x03b6e20cL, 0x74b1d29aL, 0xead54739L, 0x9dd277afL, 0x04db2615L,
	0x73dc1683L, 0xe3630b12L, 0x94643b84L, 0x0d6d6a3eL, 0x7a6a5aa8L,
	0xe40ecf0bL, 0x9309ff9dL, 0x0a00ae27L, 0x7d079eb1L, 0xf00f9344L,
	0x8708a3d2L, 0x1e01f268L, 0x6906c2feL, 0xf762575dL, 0x806567cbL,
	0x196c3671L, 0x6e6b06e7L, 0xfed41b76L, 0x89d32be0L, 0x10da7a5aL,
	0x67dd4accL, 0xf9b9df6fL, 0x8ebeeff9L, 0x17b7be43L, 0x60b08ed5L,
	0xd6d6a3e8L, 0xa1d1937eL, 0x38d8c2c4L, 0x4fdff252L, 0xd1bb67f1L,
	0xa6bc5767L, 0x3fb506ddL, 0x48b2364bL, 0xd80d2bdaL, 0xaf0a1b4cL,
	0x36034af6L, 0x41047a60L, 0xdf60efc3L, 0xa867df55L, 0x316e8eefL,
	0x4669be79L, 0xcb61b38cL, 0xbc66831aL, 0x256fd2a0L, 0x5268e236L,
	0xcc0c7795L, 0xbb0b4703L, 0x220216b9L, 0x5505262fL, 0xc5ba3bbeL,
	0xb2bd0b28L, 0x2bb45a92L, 0x5cb36a04L, 0xc2d7ffa7L, 0xb5d0cf31L,
	0x2cd99e8bL, 0x5bdeae1dL, 0x9b64c2b0L, 0xec63f226L, 0x756aa39cL,
	0x026d930aL, 0x9c0906a9L, 0xeb0e363fL, 0x72076785L, 0x05005713L,
	0x95bf4a82L, 0xe2b87a14L, 0x7bb12baeL, 0x0cb61b38L, 0x92d28e9bL,
	0xe5d5be0dL, 0x7cdcefb7L, 0x0bdbdf21L, 0x86d3d2d4L, 0xf1d4e242L,
	0x68ddb3f8L, 0x1fda836eL, 0x81be16cdL, 0xf6b9265bL, 0x6fb077e1L,
	0x18b74777L, 0x88085ae6L, 0xff0f6a70L, 0x66063bcaL, 0x11010b5cL,
	0x8f659effL, 0xf862ae69L, 0x616bffd3L, 0x166ccf45L, 0xa00ae278L,
	0xd70dd2eeL, 0x4e048354L, 0x3903b3c2L, 0xa7672661L, 0xd06016f7L,
	0x4969474dL, 0x3e6e77dbL, 0xaed16a4aL, 0xd9d65adcL, 0x40df0b66L,
	0x37d83bf0L, 0xa9bcae53L, 0xdebb9ec5L, 0x47b2cf7fL, 0x30b5ffe9L,
	0xbdbdf21cL, 0xcabac28aL, 0x53b39330L, 0x24b4a3a6L, 0xbad03605L,
	0xcdd70693L, 0x54de5729L, 0x23d967bfL, 0xb3667a2eL, 0xc4614ab8L,
	0x5d681b02L, 0x2a6f2b94L, 0xb40bbe37L, 0xc30c8ea1L, 0x5a05df1bL,
	0x2d02ef8dL
};

/* ------------------------------------------------------------------------- *
 * realtime job functions
 * ------------------------------------------------------------------------- */

static int job_thread(void* arg);

// add a job to job queue for asynchron execution
void add_job(fml_mod_priv_t *priv, job_t *job)
{
    unsigned long flags;

    // add job into queue
    spin_lock_irqsave(&priv->job_queue_lock, flags);
    if (!list_empty(&job->anchor)) {
        ERR("add_job: job %s already queued", 
            job->func == send ? "SEND" :
            job->func == recv ? "RECV" :
            job->func == init ? "INIT" :
            job->func == set_mac ? "SET_MAC" : "OTHER");
    }
    else list_add_tail(&job->anchor, &priv->job_queue);
    spin_unlock_irqrestore(&priv->job_queue_lock, flags);

    // wake up job thread
    wake_up(&priv->job_queue_wait);
}

// add a highest-priority job to job queue for asynchron execution
void add_job_head(fml_mod_priv_t *priv, job_t *job)
{
    unsigned long flags;

    // add job into queue
    spin_lock_irqsave(&priv->job_queue_lock, flags);
    if (!list_empty(&job->anchor)) {
	/*FIXME: we suppress the error message here if the
                 receive job is queued twice - the reason
                 for this must be investigated (most likely
                 we need to improve slave interupt masking) */
	if (job->func != recv) 
            ERR("add_job: job %s already queued", 
                job->func == send ? "SEND" :
                job->func == recv ? "RECV" :
                job->func == init ? "INIT" :
                job->func == set_mac ? "SET_MAC" : "OTHER");
    }
    else list_add(&job->anchor, &priv->job_queue);
    spin_unlock_irqrestore(&priv->job_queue_lock, flags);

    // wake up job thread
    wake_up(&priv->job_queue_wait);
}

// syncrhronly executes a job and returns result
int exec_job(fml_mod_priv_t *priv, job_func_t func, void *ctx)
{
    DECLARE_MUTEX_LOCKED(done);
    job_t job = { .anchor = LIST_HEAD_INIT(job.anchor), .func = func, .ctx = ctx, .done = &done };

    // issue job
    add_job(priv, &job);

    // wait for job completion
    down(&done);

    // return job result
    return job.ret;
}

// checks, if there are job in the queue (called from job thread)
int jobs_avail(fml_mod_priv_t *priv)
{
    int ret;
    unsigned long flags;
    spin_lock_irqsave(&priv->job_queue_lock, flags);
    ret = !list_empty(&priv->job_queue);
    spin_unlock_irqrestore(&priv->job_queue_lock, flags);
    return ret;
}

// removes oldest job from the queue and returns it (called from job thread)
int get_next_job(fml_mod_priv_t *priv, job_t **job)
{
    int ret = FAIL;
    unsigned long flags;

    // return next job (queue head) if present
    spin_lock_irqsave(&priv->job_queue_lock, flags);
    if (!list_empty(&priv->job_queue)) {
        *job = list_entry(priv->job_queue.next, job_t, anchor);
        list_del(priv->job_queue.next);
        INIT_LIST_HEAD(&(*job)->anchor); // make job as non-queued
        ret = SUCCESS;
    }
    spin_unlock_irqrestore(&priv->job_queue_lock, flags);

    return ret;
}

// perform a job (called from job thread)
int do_job(job_t *job)
{
    // do the job
    if (job->func != term) job->ret = job->func(job->ctx);

    // signal completion (for synchron jobs only)
    if (job->done) up(job->done);

    // return FAIL to signal job thread termination
    return job->func != term ? SUCCESS : FAIL;
}

/* ------------------------------------------------------------------------- *
 * Network driver prototypes
 * ------------------------------------------------------------------------- */

static int  fml_mod_init(void);
static void fml_mod_cleanup(void);
static int  fml_mod_init_dev(struct net_device *dev);
static int  fml_mod_open(struct net_device *dev);
static int  fml_mod_release(struct net_device *dev);
static int  fml_mod_tx(struct sk_buff *skb, struct net_device *dev);
static struct net_device_stats *fml_mod_stats(struct net_device *dev);
static int  fml_mod_ioctl(struct net_device *dev, struct ifreq *ifr, int cmd);
static int  fml_mod_set_mac_addr(struct net_device *dev, void *addr);

/* ------------------------------------------------------------------------- *
 * linux kernel module stuff
 * ------------------------------------------------------------------------- */

MODULE_AUTHOR("Ralf Guenther <rgue@peppercon.de>");
MODULE_DESCRIPTION("Linux network driver for Intel's TCO "
                   "pass-thru sideband interface (via FML or I2C)");

module_init(fml_mod_init);
module_exit(fml_mod_cleanup);

struct net_device fml_mod_dev;
fml_mod_priv_t fml_mod_priv;

/* ------------------------------------------------------------------------- *
 * network driver funtions
 * ------------------------------------------------------------------------- */

static int
fml_mod_init(void)
{
    int r;
    trig_init();
    trig_fire();

    /* init net_device struct */
    memset(&fml_mod_dev, 0, sizeof(fml_mod_dev));
    strcpy(fml_mod_dev.name, tco->name);
    fml_mod_dev.init = fml_mod_init_dev;

    if ((r = register_netdev(&fml_mod_dev)) < 0) {
        ERR("register_netdev() failed (err %d)", r);
	//release hardware
        tco->cleanup(tco);
        return -EIO;
    }
    return SUCCESS;
}

static void 
fml_mod_cleanup(void)
{
    fml_mod_priv_t *priv = &fml_mod_priv;

    // stop job thread
    exec_job(priv, term, NULL);

    // release hardware
    tco->cleanup(tco);

    unregister_netdev(&fml_mod_dev);
}

static int
fml_mod_init_dev(struct net_device *dev)
{
    fml_mod_priv_t *priv;
    int r = SUCCESS;

    /* setup ethernet defaults */
    ether_setup(dev);

    /* fill other standard network_dev fields */
    dev->open               = fml_mod_open;
    dev->stop               = fml_mod_release;
    dev->hard_start_xmit    = fml_mod_tx;
    dev->get_stats          = fml_mod_stats;
    dev->do_ioctl           = fml_mod_ioctl;
    dev->set_mac_address    = fml_mod_set_mac_addr;
    dev->base_addr          = 0; /* will be set below */
    dev->irq                = 0; /* will be set below */
    SET_MODULE_OWNER(dev);

    /* init privat data */
    dev->priv = &fml_mod_priv;
    priv = (fml_mod_priv_t*)dev->priv;
    memset(priv, 0, sizeof(fml_mod_priv_t));

    INIT_LIST_HEAD(&priv->job_queue);
    spin_lock_init(&priv->job_queue_lock);
    init_waitqueue_head(&priv->job_queue_wait);

    // init async jobs (all other field remain 0)
    INIT_LIST_HEAD(&priv->recv_job.anchor);
    priv->recv_job.func = recv;
    INIT_LIST_HEAD(&priv->send_job.anchor);
    priv->send_job.func = send;
    INIT_LIST_HEAD(&priv->reinit_job.anchor);
    priv->reinit_job.func = init;
    priv->reinit_job.ctx = RE_INIT;

    /* start realtime job thread */
    r = kernel_thread(job_thread, NULL, 0);
    if (r < 0) {
        ERR("unable to start real-time job thread (err %d)", r);
        goto exit;
    }

    /* find hardware */
    priv->tco = tco;
    r = priv->tco->init(dev, priv->tco);
    if (r < 0) {
        ERR("unable to find hardware core (err %d)", r);
        goto exit;
    }

exit:
    return r;
}

static int
fml_mod_open(struct net_device *dev)
{
    int ret = SUCCESS;
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;

    MOD_INC_USE_COUNT;
    DBG("open");

    if (atomic_inc_return(&priv->refcnt) == 1) {
        /* first time init */
        priv->recv_skb = dev_alloc_skb(dev->mtu + ETH_HLEN);
        priv->send_skb = NULL;

        /* init hardware core */
        if (priv->tco->core_init && ((ret = priv->tco->core_init(dev, priv->tco)) != SUCCESS)) {
            goto fail;
        }

        // sync exec init job
        ret = exec_job(priv, init, FIRST_INIT);
        if (ret == -EAGAIN) {
            WARN("init not yet succeeded, retrying in background");
            ret = SUCCESS;
        } else if (ret != SUCCESS) {
            ERR("init failed");
            goto fail;
        }

        INFO("loaded");
    }

    /* start upper layer */
    netif_start_queue(dev);
    return ret;
fail:
    fml_mod_release(dev);
    return ret;
}

static int
fml_mod_release(struct net_device *dev)
{
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;
    DBG("release");

    /* stop upper layer */
    netif_stop_queue(dev);

    if (atomic_dec_return(&priv->refcnt) == 0) {
        /* least time cleanup */

        // job to sync with job thread and mark driver as unloaded
        exec_job(priv, done, NULL);

        if (priv->tco->core_release) {
            priv->tco->core_release(dev, priv->tco);
        }

        if (priv->recv_skb) {
            dev_kfree_skb(priv->recv_skb);
            priv->recv_skb = NULL;
        }

        if (priv->send_skb) {
            dev_kfree_skb(priv->send_skb);
            priv->send_skb = NULL;
        }

        INFO("unloaded");
    }

    MOD_DEC_USE_COUNT;
    return 0;
}

static int
fml_mod_ioctl(struct net_device *dev, struct ifreq *ifr, int cmd)
{
    int r = SUCCESS;
    fml_mod_priv_t *priv;
    if (dev == NULL) {
        ERR("ioctl failed: driver not open");
        return -EINVAL;
    }
    priv = (fml_mod_priv_t*)dev->priv;

    switch (cmd) {
    case FML_IOCTL_RAW_SEND:
    {
        fml_ioctl_raw_pkt_t d;
        copy_from_user(&d, ifr->ifr_ifru.ifru_data, sizeof(fml_ioctl_raw_pkt_t)); /* we could copy less here */
        DBG("ioctl_raw_send: saddr=0x%02x cmd=0x%02x len=%d", d.saddr, d.cmd, d.len);
        r = exec_job(priv, raw_send, &d);
        break;
    }
    case FML_IOCTL_RAW_RECV:
    {
        fml_ioctl_raw_pkt_t d;
        copy_from_user(&d, ifr->ifr_ifru.ifru_data, sizeof(fml_ioctl_raw_pkt_t)); /* we could copy less here */
        DBG("ioctl_raw_recv: saddr=0x%02x cmd=0x%02x", d.saddr, d.cmd);
        r = exec_job(priv, raw_recv, &d);
        if (r >= SUCCESS) {
            copy_to_user(ifr->ifr_ifru.ifru_data, &d, sizeof(fml_ioctl_raw_pkt_t)); /* we could copy less here */
        }
        break;
    }
    case FML_IOCTL_GET_REG:
    {
        if (priv->tco->read_reg) {
            fml_ioctl_reg_t d;
            copy_from_user(&d, ifr->ifr_ifru.ifru_data, sizeof(fml_ioctl_reg_t));
            d.val = priv->tco->read_reg(priv->tco, d.reg);
            //DBG("ioctl_get_reg: @%04x -> %08lx", d.reg, d.val);
            copy_to_user(ifr->ifr_ifru.ifru_data, &d, sizeof(fml_ioctl_reg_t));
        } else {
            r = -EINVAL;
        }
        break;
    }
    case FML_IOCTL_SET_REG:
    {
        if (priv->tco->write_reg) {
            fml_ioctl_reg_t d;
            copy_from_user(&d, ifr->ifr_ifru.ifru_data, sizeof(fml_ioctl_reg_t));
            //DBG("ioctl_set_reg: @%04x <- %08lx", d.reg, d.val);
            priv->tco->write_reg(priv->tco, d.reg, d.val);
        } else {
            r = -EINVAL;
        }
        break;
    }
    case FML_IOCTL_GET_HK:
    {
        DBG("ioctl_get_hk");
        copy_to_user(ifr->ifr_ifru.ifru_data, &priv->hk, sizeof(fml_ioctl_hk_t));
        break;
    }
    case SIOCETHTOOL:
    {
        /* we are no real ethernet device, so
           we don't need to react on SIOCETHTOOL */
        break;
    }
    default:
        /* invalid cmd */
        ERR("ioctl failed: invalid cmd %d", cmd);
        r = -EINVAL;
    }

    return r;
}

static int
fml_mod_tx(struct sk_buff *skb, struct net_device *dev)
{
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;

    /* switch off upper layer and prevent task frome being restartet */
    netif_stop_queue(dev); 

    /* start sending in non-irq mode */
    priv->send_skb = skb;
    priv->send_offs = 0;
    add_job(priv, &priv->send_job);

    return SUCCESS;
}

static struct net_device_stats *
fml_mod_stats(struct net_device *dev)
{
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;
    return &priv->stats;
}

static int
fml_mod_set_mac_addr(struct net_device *dev, void *addr)
{
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;
    struct sockaddr *sa = (struct sockaddr*) addr;

    return exec_job(priv, set_mac, &sa->sa_data);
}

/* ------------------------------------------------------------------------- *
 * job thread (high prio!)
 * ------------------------------------------------------------------------- */

static int job_thread(void * arg)
{
    struct net_device *dev = &fml_mod_dev;
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;
    int fml = strcmp(dev->name, "tco_fml") == 0;

    daemonize();
    strcpy(current->comm, tco->name);

    //  realtime priority to reduce delays between messages and message bytes!
    current->policy = SCHED_RR;
    current->rt_priority = 50;

    while (1) {
        job_t *job;
        int ret = wait_event_interruptible_timeout(
            priv->job_queue_wait, jobs_avail(priv), 10 * HZ); // 10secs timeout

        if (ret < 0) {
            ERR("signal detected: terminate");
            break;
        }

        if (ret == 0) {
            // force a re-init for safety, because we could have missed an
            //   Initialization Identication and then would never re-init the
            //   ESB2 otherwise
            DBG("no actions since 10secs, force a re-init");
            add_job(priv, &priv->reinit_job);
            // fall through
        }

        while (get_next_job(priv, &job) == SUCCESS) {
            if (fml && current->need_resched) schedule();
            // do the job (termination job returns FAIL here!)
            if (do_job(job) != SUCCESS) goto exit;
        }
    }

exit:
    return 0;
}

/* ------------------------------------------------------------------------- *
 * interrupt handling
 * ------------------------------------------------------------------------- */

int tco_recv_cb(tco_t *tco)
{
//TWARN("enter cb");
    struct net_device *dev = &fml_mod_dev;
    fml_mod_priv_t *priv = (fml_mod_priv_t*)dev->priv;

    priv->hk.intr_cnt++;

    /* prevent core from generating more interrupts */
    priv->tco->enable_client_int(priv->tco, 0);

    /* start receiving in non-irq mode */
    add_job_head(priv, &priv->recv_job);

//TWARN("leave cb");
    return 0;
}
