/*********************************************************************
 **                                                                 
 **  File          : fmlcore.h                                       
 **  Project       : FML-Core Driver for Linux 2.6.x                
 **  Purpose       : usage of the FML-Bus to access Intels 82570 MAC
 **  Contents      : defines for the register access to the         
 **                  FML-Core and declarations of the basic routines
 **  Author        : Marko R�ler <marr@infotech.tu-chemnitz.de>    
 **  Last modified : <08.02.2005 15:29:17 marr>                     
 **                                                                 
 *********************************************************************/

#ifndef _LINUX_FMLCORE_H
#define _LINUX_FMLCORE_H

#define FML_MAX_BLK_SIZE 240 //in byte!

#define FML_CORE_TEST 0 /*enables some tests during core initialization */

//#define FML_IO_SIZE 		0x2000 // == 8kbyte
//#define FML_BASE_ADDR 0x80003000
#define WB_BUFFER_ADDR  0x0  /* address of the buffer from Cores point of view */

#include <linux/ioctl.h>
#include <linux/spinlock.h>

#define FML_MINOR 201
#define FML_IOCSPREFS _IOW( FML_MINOR*MISC_MAJOR, 0x01, struct fmlcore_private )

//include needed for DMA hack on KIRA < R02
#ifdef __arm__
#include <asm/arch/ahb_dma.h>
#endif

/******************************************************/
/* private structure for each instance of the Core... */
/******************************************************/
struct fmlcore_private
{
    /* parameter that need to be filled in 
     * prior calling fml_core_init() 
     */
    char *name;           /* name the device will be registered as*/
    u32 io_base;          /* IO base address */
    u32 phys_base;        /* physical IO base address */
    unsigned int fml_clk; /* clock frequency in kHz! */
    unsigned int sys_clk; /* system frequency in kHz! */
    unsigned int irq;     /* interrupt line number */
    unsigned char slave;  /* 0..master 1..slave */
    unsigned char saddr;  /* slave address (bit7..1) */
    unsigned char finished; /* tells the wait_while_tip routine when a transfer is finished */
    spinlock_t fin_lock;
    struct semaphore slave_write_sem; /* serialize slave write operations */

    //buffer offsets (different for R01 and R02)
    u32 buf_offset;
    u32 buf_size;
    
    void (* write_reg)(u32, struct fmlcore_private *, u32);
    u32 (* read_reg)(struct fmlcore_private *, u32);
    u32 (* read_statusreg)(struct fmlcore_private *);
    void (* write_block)(u32 *vals, struct fmlcore_private * fml_priv, u32 offset, int cnt);
    void (* read_block)(u32 *buf, struct fmlcore_private * fml_priv, u32 offset, int cnt);
    uint32_t kira_rev;

    int (* slave_req_callback)(struct fmlcore_private *); /* callback function 
							      to call when fml-slave 
							      request a transaction */
//    void (* slave_rdy_callback)(struct fmlcore_private *); /* callback function 
//							      to call when fml-slave 
// write block finished */

    /* don't touch these fields... */

    wait_queue_head_t wait; /*wait queue */
};

/************************************************/
/* functions that are provided by the driver... */
/************************************************/
extern int fmlcore_probe(u32 io_base, u32 phys_base, int irq);
extern int fmlcore_init(struct fmlcore_private *fml_priv);
extern int fmlcore_release(struct fmlcore_private *fml_priv);

extern int fmlcore_write_block(struct fmlcore_private * fml_priv,
			       u8 command,
			       u8 slave_addr,
			       u8 byte_count,
			       const u8 * data);

extern int fmlcore_read_block(struct fmlcore_private *,
			      u8 command,
			      u8 slave_addr,
			      u8 *byte_count,
			      u8 *data);

extern int fmlcore_read_block_simple(struct fmlcore_private *,
			             u8 slave_addr,
			             u8 *byte_count,
			             u8 *data);

extern int fmlcore_slave_write(struct fmlcore_private * fml_priv,
                           const unsigned char *data,
                           int size);

extern int fmlcore_slave_read(struct fmlcore_private * fml_priv,
                              unsigned char *data,
                              int bufsize);

#ifdef __arm__
//typedef void irqreturn_t;
#endif

extern irqreturn_t fmlcore_irq_handler(int this_irq,
				       void *dev_id,
				       struct pt_regs *regs);


/*****************************/
/* buffer offsets...       */
/*****************************/
#define REG_OFFSET  0x0

#define MONLY_BUF_OFFSET 0x100
#define MONLY_BUF_SIZE   0x40

#define MS_BUF_OFFSET    0x400
#define MS_BUF_SIZE      0x80

/* slave buffers (2K) */
#define SBUF_WR_OFFSET 0x400
#define SBUF_RD_OFFSET 0x600
#define SBUF_SIZE      (0x200 * 4) // 2048 bytes

/*****************************/
/* register offsets...       */
/*****************************/
#define MGR_ADDR 0x0          /* general register */
#define MCR0_ADDR 0x1         /* command register */
#define MIACK_ADDR 0x2        /* interrupt acknowledge register */
#define MGSR_ADDR 0x3         /* status register */
#define MPRER_ADDR 0x4        /* devider register */
#define MTOUT_ADDR 0x5        /* FML timeout value in system clock cycles! */
#define MID1_ADDR 0x6         /* FML-Core ID1 register */
#define MID2_ADDR 0x7         /* FML-Core ID2 register */
#define SRBC_ADDR 0x8         /* slave read byte count */
#define SWBC_ADDR 0x9         /* slave write byte count */
#define SAR_ADDR  0xa         /* slave addr */
#define CORESTAT_ADDR 0xb     /* debug: core engine state */
#define BYTESTAT_ADDR 0xc     /* debug: byte engine state */
#define BITSTAT_ADDR 0xd      /* debug: bit engine state */

/*****************************/
/* define register values... */
/*****************************/

/* two ID registers to find the core */
#define FML_ID1 0x464d4c2d /* "FML-" */
#define FML_ID2 0x436f7265 /* "Core" */

/* values for the MGR-Register */
#define MGO  (0x1 << 0) // enable core
#define MRST (0x1 << 1) // reset core (auto cleared by hardware)
#define MS   (0x1 << 2) // 0..slave, 1..master
#define MALLIEN (0x1 << 4) // enable global interrupts
#define MFMLIEN (0x1 << 5) // enable FML line interrupts
#define MCOREIEN (0x1 << 6) // enable core interrupts

/* values for the MCR0-Register */
#define MSTC (0x1<<0)       // Master Start Command
#define MSPM (0x1<<1)       // Master Stop Command
#define MNC  (0x1<<2)       // Master NACK Command
#define MRDC (0x1<<4)       // Master Read Command
#define MWRC (0x1<<5)       // Master Write Command
#define MPC  (0x1<<6)       // Master PEC Command
#define BTCNT_EN (0x1<<11)
#define MBUFSEL_MASK (0x3<< 12)
#define MBUFSEL_SHIFT 12
#define MBC_MASK (0xff<<16)
#define MBC_SHIFT 16		 
		 
/* values of the MIACK-Register */
#define MCOREIACK (0x1)
#define MFMLIACK (0x2)

/* values of the MGSR-Register */
#define MERR             (0x1 << 0)
#define MAB              (0x1 << 1)
#define MTRANS		 (0x1 << 2)
#define MOP              (0x1 << 7)
#define MCORE_IRQ        (0x1 << 16)
#define MFMLSL_IRQ       (0x1 << 17)
#define MOFLW            (0x1 << 19)
#define GET_MBWD(a)      (( a & 0xff00 ) >> 8)
#define GET_RD_BNTCNT(a)      (( a & 0xff000000 ) >> 24) 

#endif //_LINUX_FMLCORE_H
