/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  CIM Protocol Diagnostic Utilities
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2005-2006 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <poll.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>
#include <sys/ioctl.h>

#include "fpd_ioctl.h"
#include "cim.h"


int  debug_on = 0;
int  max_link_if = 0;
int  max_line_cnt = 0;
int  max_host_chan = 0;
char bgnd_link_if_present = 0;
unsigned char *protocol_buffer;

static FPD_msg_t gmsg;
static line_connection_t **line_conn;
static line_connection_t *bgnd_line_conn = NULL;


int cim_init( int fd )
{
    line_connection_t *pconn;
    int link, i;
    int rc;

    memset(&gmsg, 0, sizeof(FPD_msg_t));

    /* determine Link and Line Interface count */
    rc = ioctl(fd, FPD_GET_INFO, &gmsg);
    if( rc == 0 ) {
        max_link_if = gmsg.info.link_if_cnt;
        max_line_cnt = gmsg.info.line_cnt;
        max_host_chan = gmsg.info.host_buffer_cnt;
        bgnd_link_if_present = gmsg.info.bgnd_link_if_present;
    }

    /* allocate memory for line connections */
    if( bgnd_link_if_present ) {
        bgnd_line_conn = malloc(sizeof(line_connection_t));
        if( bgnd_line_conn == NULL ) {
            fprintf(stderr, "bgnd_line_conn not allocated\n");
            return -1;
        }
    }

    line_conn = malloc(max_link_if * sizeof(line_connection_t *));
    if( line_conn ) {
        for( i = 0; i < max_link_if; i++ ) {
            line_conn[i] = malloc(sizeof(line_connection_t));
            if( line_conn[i] == NULL ) {
                fprintf(stderr, "line_conn[%d] not allocated\n", i);
                free(line_conn);
                free(bgnd_line_conn);
                return -3;
            }
        }
    }
    else {
        fprintf(stderr, "line_conn not allocated\n");
        free(bgnd_line_conn);
        return -2;
    }

    /* allocate buffer */
    protocol_buffer = (unsigned char *)malloc(MAX_PROTOCOL_BUFFER_SIZE);
    if( !protocol_buffer ) {
        for( i = 0; i < max_link_if; i++ ) {
            free(line_conn[i]);
        }
        free(line_conn);
        free(bgnd_line_conn);
        return -4;
    }

    /* find out current switch info */
    rc = ioctl(fd, FPD_GET_SWITCH_INFO, &gmsg);
    if( rc == 0 ) {
        for( link = 0; link < max_link_if; link++ ) {
            pconn = line_conn[link];
            if( gmsg.switch_info.link[link].enabled ) {
                pconn->target_port = gmsg.switch_info.link[link].target_port;
                pconn->protocol = gmsg.switch_info.link[link].protocol;
                pconn->tx_parity = gmsg.switch_info.link[link].tx_parity;
                pconn->rx_parity = gmsg.switch_info.link[link].rx_parity;
            }
            else {
                pconn->target_port = -1;
                pconn->protocol = -1;
                pconn->tx_parity = -1;
                pconn->rx_parity = -1;
            }
        }

        if( bgnd_link_if_present ) {
            if( gmsg.switch_info.bgnd_link.enabled ) {
                bgnd_line_conn->target_port = gmsg.switch_info.bgnd_link.target_port;
                bgnd_line_conn->protocol = gmsg.switch_info.bgnd_link.protocol;
                bgnd_line_conn->tx_parity = gmsg.switch_info.bgnd_link.tx_parity;
                bgnd_line_conn->rx_parity = gmsg.switch_info.bgnd_link.rx_parity;
            }
            else {
                bgnd_line_conn->target_port = -1;
                bgnd_line_conn->protocol = -1;
                bgnd_line_conn->tx_parity = -1;
                bgnd_line_conn->rx_parity = -1;
            }
        }
    }
    else {
        for( link = 0; link < max_link_if; link++ ) {
            pconn = line_conn[link];
            pconn->target_port = -1;
            pconn->protocol = -1;
            pconn->tx_parity = -1;
            pconn->rx_parity = -1;
        }

        if( bgnd_link_if_present ) {
            bgnd_line_conn->target_port = -1;
            bgnd_line_conn->protocol = -1;
            bgnd_line_conn->tx_parity = -1;
            bgnd_line_conn->rx_parity = -1;
        }
    }

    return 0;
}

int cim_cleanup( void )
{
    int i;

    /* free allocated memory */
    for( i = 0; i < max_link_if; i++ ) {
        free(line_conn[i]);
    }
    free(line_conn);
    free(bgnd_line_conn);
    free(protocol_buffer);

    return 0;
}

int cim_get_current_link_if( void )
{
    return 0;
}

int cim_get_line_connection_info( int link_id, line_connection_t **pconn )
{
    if( link_id < max_link_if ) {
        *pconn = line_conn[link_id];
    }
    else if( link_id == FPD_LINK_IF_ID_BGND ) {
        *pconn = bgnd_line_conn;
    }
    else {
        fprintf(stderr, "\nERROR: Invalid Link Interface specified\n");
        *pconn = NULL;
        return -1;
    }

    return 0;
}

int cim_set_line_connection_info( int link_id, int target, int protocol, int txpar, int rxpar )
{
    line_connection_t *pconn;

    if( link_id < max_link_if ) {
        pconn = line_conn[link_id];
    }
    else if ( link_id == FPD_LINK_IF_ID_BGND ) {
        pconn = bgnd_line_conn;
    }
    else {
        return -1;
    }

    pconn->target_port = target;
    pconn->protocol = protocol;
    pconn->tx_parity = txpar;
    pconn->rx_parity = rxpar;

    return 0;
}

int cim_get_link_if_protocol( int link_id )
{
    line_connection_t *pconn;

    if( link_id < max_link_if ) {
        pconn = line_conn[link_id];
    }
    else if ( link_id == FPD_LINK_IF_ID_BGND ) {
        pconn = bgnd_line_conn;
    }
    else {
        return -1;
    }

    return pconn->protocol;
}

int cim_get_type( int link_id )
{
    line_connection_t *property;

    if( link_id < max_link_if ) {
        property = line_conn[link_id];
    }
    else if ( link_id == FPD_LINK_IF_ID_BGND ) {
        property = bgnd_line_conn;
    }
    else {
        printf("Invalid link ID %d\n", link_id);
        return CIM_UNKNOWN;
    }

    if( property->protocol == FPD_PROTOCOL_PARAGON ) {
        if( property->tx_parity == FPD_PARAGON_PARITY_ODD &&
            property->rx_parity == FPD_PARAGON_PARITY_ODD ) {
            return PCIM;
        }
        else if( property->tx_parity == FPD_PARAGON_PARITY_EVEN &&
                 property->rx_parity == FPD_PARAGON_PARITY_EVEN ) {
            return DCIM;
        }
        if( property->tx_parity == FPD_PARAGON_PARITY_ODD &&
            property->rx_parity == FPD_PARAGON_PARITY_IGNORE ) {
            return PCIM;
        }
        else {
            printf("Invalid TX/RX parity\n");
            return CIM_UNKNOWN;
        }
    }
    else if( property->protocol == FPD_PROTOCOL_VM ) {
        return VMCIM;
    }

    printf("Invalid protocol\n");
    return CIM_UNKNOWN;
}

int cim_disconnect_link( int fd, int link_id )
{
    int rc;

    gmsg.disc.link_if = link_id;
    gmsg.disc.driver_txbuf_wait_time = 0;
    gmsg.disc.fpga_txbuf_wait_time = 0;
    rc = ioctl(fd, FPD_DISCONNECT_CHANNEL, &gmsg);

    if( rc == 0 ) {
        cim_set_line_connection_info(link_id, -1, -1, -1, -1);
    }

    return rc;
}

int cim_switch_link( int fd, int link_id, int target, int protocol, int txpar, int rxpar )
{
    int rc;

    gmsg.conn.link_if = link_id;
    gmsg.conn.target_port = target;
    gmsg.conn.protocol = protocol;
    gmsg.conn.tx_parity = txpar;
    gmsg.conn.rx_parity = rxpar;
    gmsg.conn.driver_txbuf_wait_time = 0;
    gmsg.conn.fpga_txbuf_wait_time = 0;
    rc = ioctl(fd, FPD_SWITCH_CHANNEL, &gmsg);

    if( rc == 0 ) {
        cim_set_line_connection_info(link_id, target, protocol, txpar, rxpar);
    }

    return rc;
}

static void display_link_properties( line_connection_t *plink )
{
    switch( plink->protocol ) {
        case FPD_PROTOCOL_VM:
            printf("VM\n");
            break;
        case FPD_PROTOCOL_PARAGON:
            printf("Paragon\n");
            printf("Current TX Parity selected                             : ");
            switch( plink->tx_parity ) {
                case FPD_PARAGON_PARITY_EVEN:
                    printf("Even\n");
                    break;
                case FPD_PARAGON_PARITY_ODD:
                    printf("Odd\n");
                    break;
                default:
                    printf("Unknown\n");
                    break;
            }
            printf("Current RX Parity selected                             : ");
            switch( plink->rx_parity ) {
                case FPD_PARAGON_PARITY_EVEN:
                    printf("Even\n");
                    break;
                case FPD_PARAGON_PARITY_ODD:
                    printf("Odd\n");
                    break;
                case FPD_PARAGON_PARITY_IGNORE:
                    printf("Ignored\n");
                    break;
                default:
                    printf("Unknown\n");
                    break;
            }
            break;
        default:
            printf("Unknown!\n");
            break;
    }

    return;
}

void cim_display_current_setup( int link_id )
{
    line_connection_t *pconn;

    printf("\nCurrent Link Interface selected                        : ");
    if( link_id < max_link_if ) {
        printf("%d\n", link_id);
        pconn = line_conn[link_id];
    }
    else if( link_id == FPD_LINK_IF_ID_BGND ) {
        printf("Background\n");
        pconn = bgnd_line_conn;
    }
    else {
        printf("None\n");
        pconn = NULL;
        return;
    }

    printf("Current Protocol Type selected                         : ");
    if( link_id < max_link_if || link_id == FPD_LINK_IF_ID_BGND ) {
        display_link_properties(pconn);
    }
    else {
        printf("None\n");
    }

    printf("Current Line selected [-1 = Disconnected]              : ");
    if( link_id < max_link_if || link_id == FPD_LINK_IF_ID_BGND ) {
        printf("%d\n", pconn->target_port);
    }
    else {
        printf("None\n");
    }

    return;
}

int cim_select_linkif_and_line( int fd, int link_id )
{
    unsigned int val;
    unsigned int cim_protocol, tx_parity, rx_parity;
    int line;

#if 0
    printf("\nEnter Link Interface [0-%d] or Background Link [%d]   : ",
           max_link_if - 1, FPD_LINK_IF_ID_BGND);
    scanf("%d", &val);
    if( val < max_link_if ) {
        link_id = val;
    }
    else if ( val == FPD_LINK_IF_ID_BGND ) {
        if( bgnd_link_if_present ) {
            link_id = val;
        }
        else {
            printf("\nERROR: Background Link Interface is not present\n");
            printf("Please re-do!!!\n");
            return -1;
        }
    }
    else {
        printf("\nERROR: Invalid Link Interface ID %d\n", val);
        printf("Please re-do!!!\n");
        return -1;
    }
#else
    if( link_id >= max_link_if ) {
        if ( link_id == FPD_LINK_IF_ID_BGND ) {
            if( !bgnd_link_if_present ) {
                printf("\nERROR: No BGND Link Interface found\n");
                printf("Please re-do!!!\n");
                return -1;
            }
        }
        else {
            printf("\nERROR: Invalid Link Interface ID %d\n", link_id);
            printf("Please re-do!!!\n");
            return -1;
        }
    }
#endif

    printf("Enter Protocol Type [0=VM, 1=Paragon]                  : ");
    scanf("%d", &val);
    switch( val ) {
        case FPD_PROTOCOL_VM:
            cim_protocol = FPD_PROTOCOL_VM;
            break;
        case FPD_PROTOCOL_PARAGON:
            cim_protocol = FPD_PROTOCOL_PARAGON;
            printf("Enter TX Parity [0=Even, 1=Odd]                        : ");
            scanf("%d", &tx_parity);
            switch( tx_parity ) {
                case FPD_PARAGON_PARITY_EVEN:
                case FPD_PARAGON_PARITY_ODD:
                    break;
                default:
                    printf("\nERROR: Invalid Parity\n");
                    printf("Please re-do!!!\n");
                    return -3;
            }

            printf("Enter RX Parity [0=Even, 1=Odd, 2=Ignore]              : ");
            scanf("%d", &rx_parity);
            switch( rx_parity ) {
                case FPD_PARAGON_PARITY_EVEN:
                case FPD_PARAGON_PARITY_ODD:
                case FPD_PARAGON_PARITY_IGNORE:
                    break;
                default:
                    printf("\nERROR: Invalid Parity\n");
                    printf("Please re-do!!!\n");
                    return -4;
            }
            break;
        default:
            printf("\nERROR: Invalid Protocol Type\n");
            printf("Please re-do!!!\n");
            return -2;
    }

    printf("Enter Line for connection [0-%02d] or -1 to disconnect   : ",
           max_line_cnt - 1);
    scanf("%d", &line);

    if(( line >= max_line_cnt ) || ( line < -1 )) {
        printf("\nERROR: Invalid Line ID %d\n", line);
        printf("Please re-do!!!\n");
        return -5;
    }

    /* process switch or disconnect */
    if( line == -1 ) {
        if( link_id < max_link_if ) {
            printf("\nDisconnecting Link IF %d....", link_id);
        }
        else {
            printf("\nDisconnecting Background Link IF....");
        }

        if( cim_disconnect_link(fd, link_id) == 0 ) {
            printf("passed.\n");
        }
        else {
            printf("FAILED.\n");
            if(errno) {
                printf("FPD_DISCONNECT_CHANNEL ioctl failed : %s\n", strerror(errno) );
            }
            return -6;
        }
    }
    else {
        if( link_id < max_link_if ) {
            printf("\nSwitching Link IF %d to Line %d....", link_id, line);
        }
        else {
            printf("\nSwitching Background Link IF to Line %d....", line);
        }

        if( cim_switch_link(fd, link_id, line, cim_protocol, tx_parity, rx_parity) == 0 ) {
            printf("passed.\n");
        }
        else {
            printf("FAILED.\n");
            if(errno) {
                printf("FPD_SWITCH_CHANNEL ioctl failed : %s\n", strerror(errno) );
            }

            return -7;
        }
    }

    return 0;
}

int cim_poll_once( int fd, unsigned int timeout, FPD_event_t *pevent )
{
    struct pollfd fds;
    int rc;

    fds.fd = fd;
    fds.events = POLLIN;

    if( debug_on ) {
        printf("waiting for events ... ");
    }

    rc = poll(&fds, 1, timeout);
    if ((rc == 0) || (rc < 0 && errno == EINTR)) {
        if( !debug_on ) {
            fprintf(stderr, "FAILED.\n");
            fprintf(stderr, "waiting for events ... timedout\n");
        }
        else {
            fprintf(stderr, "timedout\n");
        }
        return -1;
    }
    else if( rc < 0 ) {
        if( !debug_on ) {
            fprintf(stderr, "FAILED.\n");
            fprintf(stderr, "waiting for events ... poll() returned ERROR %d\n", rc);
        }
        else {
            fprintf(stderr, "poll() returned %d\n", rc);
        }
        return -2;
    }

    if( debug_on ) {
        printf("receive notifications\n");
        printf("getting events ... ");
    }

    rc = ioctl(fd, FPD_CHECK_EVENTS, pevent);

    if( debug_on ) {
        printf("done\n");
    }

    return rc;
}

static void display_stats( FPD_stats_t *pinfo )
{
    if( pinfo->link_if == FPD_LINK_IF_ID_BGND ) {
        printf("\nBackground Link IF:\n");
    }
    else {
        printf("\nLink IF %d:\n", pinfo->link_if);
    }

    printf("  RX Statistics\n");
    printf("\tBuffer Full   : %d\n", pinfo->rx.buf_full_cnt);
    printf("\tProtocol Error: %d\n", pinfo->rx.protocol_err_cnt);
    printf("\tCRC Error     : %d\n", pinfo->rx.crc_err_cnt);
    printf("\tInvalid Packet: %d\n", pinfo->rx.inv_pkt_cnt);
    printf("\tParity Error  : %d\n", pinfo->rx.parity_err_cnt);
    printf("\tSequence Error: %d\n", pinfo->rx.seq_err_cnt);
    printf("\tTimeout       : %d\n", pinfo->rx.timeout_cnt);
    printf("\tNoise         : %d\n", pinfo->rx.noise_cnt);

    printf("  TX Statistics\n");
    printf("\tMax Retry     : %d\n", pinfo->tx.max_retry);
    printf("\tBuffer Full   : %d\n", pinfo->tx.buf_full_cnt);
    printf("\tProtocol Error: %d\n", pinfo->tx.protocol_err_cnt);
    printf("\tCRC Error     : %d\n", pinfo->tx.crc_err_cnt);

    return;
}

int cim_get_stats( int fd, int link_id )
{
    int rc;

    gmsg.stats.link_if = link_id;
    rc = ioctl(fd, FPD_GET_STATISTICS, &gmsg);
    if (rc == 0) {
        /* display info */
        display_stats(&gmsg.stats);
    }
    else {
        if(errno) {
            fprintf(stderr, "FPD_GET_STATISTICS ioctl failed : %s\n",
                    strerror(errno) );
        }
    }

    return 0;
}
