/******************************************************************************
 *  MODULE:           FPGA PROTOCOL
 ******************************************************************************
 *
 *  CIM Protocol Diagnostics Menu
 *
 *  FILE:             $Workfile$
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential
 * proprietary information distributed solely pursuant to a confidentiality
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <string.h>

#include "fpd_ioctl.h"

#include "cim_menu.h"
#include "cim.h"
#include "paragon_cim.h"
#include "keybd.h"
#include "cim_update.h"
#include "vm_cim.h"


/* Local function declarations */
static int pcim_menu( int fd, int link );
static int pcim_eeprom_menu( int fd, int link );
static int pcim_powercim_menu( int fd, int link );
static int vmcim_menu( int fd, int link );
static int vmcim_init_menu( int fd, int link );
static int vmcim_info_menu( int fd, int link );
static int vmcim_control_menu( int fd, int link );
static int vmcim_mem_menu( int fd, int link );
static int vmcim_kbd_menu( int fd, int link );
static int vmcim_mouse_menu( int fd, int link );
static int vmcim_usbctl_menu( int fd, int link );


int cim_menu( int fd, int link )
{
    int result;
    char answer;

    cim_display_current_setup(link);

    printf("\nDo you want to change the setup (y/n)                  : ");
    scanf("%c", &answer);

    /* CIM connection */
    if((answer == 'y') || (answer == 'Y')) {
        result = cim_select_linkif_and_line(fd, link);
        if( result != 0 ) {
            cim_cleanup();
            return result;
        }
    }

    /* initialize according to protocol type */
    if( cim_get_link_if_protocol(link) == FPD_PROTOCOL_PARAGON ) {
        /* Paragon Protocol */
        if((answer == 'y') || (answer == 'Y')) {
            paragon_cim_init(fd, link, 1);
        }
        result = pcim_menu(fd, link);
    }
    else {
        /* KX2.0 Protocol */
        result = vmcim_menu(fd, link);
    }

    return result;
}

static void display_pcim_menu( void )
{
    int i = 0;

    printf("\nParagon CIM Diagnostics:");
    printf("\n========================");
    printf("\n\t %d: Exit Diagnostics", i++);
    printf("\n\t %d: Send Echo Request", i++);
    printf("\n\t %d: Send Keybd & Mouse Init Request", i++);
    printf("\n\t %d: Send Beep", i++);
    printf("\n\t %d: Send Read CIM EEPROM", i++);
    printf("\n\t %d: Send Write CIM EEPROM", i++);
    printf("\n\t %d: Send Keybd Packets", i++);
    printf("\n\t %d: Send Mouse Packets", i++);
    printf("\n\t %d: CIM EEPROM Utilities", i++);
    printf("\n\t %d: Keybd Loop Test", i++);
    printf("\n\t%d: CIM Update (Initial State)", i++);
    printf("\n\t%d: CIM Update (Recovery State)", i++);
    printf("\n\t%d: Power CIM Utilities", i++);
    printf("\n\n");

    return;
}

static int pcim_menu( int fd, int link )
{
    char running = 1;
    int result = 1;
    int diag_opt;
    char addr, lval, hval;

    while( running ) {
        display_pcim_menu();

        printf("Please choose: ");
        scanf("%d", &diag_opt);

        switch( diag_opt ) {
            case PCIM_MENU_EXIT:
                running = 0;
                break;

            case PCIM_MENU_SEND_ECHO:
                printf("\n");
                paragon_cim_send_echo(fd, link, NULL);
                break;

            case PCIM_MENU_SEND_KEYBD_MOUSE_INIT:
                paragon_cim_send_kbms_init(fd, link);
                break;

            case PCIM_MENU_SEND_BEEP:
                printf("\nEnter beep time (tenths of a second) [0=Off, FF=On]: 0x");
                scanf("%hhx", &lval);
                paragon_cim_send_beep(fd, link, lval);
                break;

            case PCIM_MENU_SEND_READ_EEPROM:
                printf("\nEnter EEPROM offset: 0x");
                scanf("%hhx", &addr);
                paragon_cim_send_eeprom_read(fd, link, addr, NULL);
                break;

            case PCIM_MENU_SEND_WRITE_EEPROM:
                printf("\nEnter EEPROM offset: 0x");
                scanf("%hhx", &addr);
                printf("Enter Data Low     : 0x");
                scanf("%hhx", &lval);
                printf("Enter Data High    : 0x");
                scanf("%hhx", &hval);
                paragon_cim_send_eeprom_write(fd,  link, addr, lval, hval);
                break;

            case PCIM_MENU_SEND_KEYBD_PKTS:
                keybd_send_char(fd, link);
                break;

            case PCIM_MENU_SEND_MOUSE_PKTS:
                printf("May the force be with you\n");
                break;

            case PCIM_MENU_CIM_EEPROM_UTILS:
                pcim_eeprom_menu(fd, link);
                break;

            case PCIM_MENU_KEYBD_LOOP_TEST:
                keybd_loop_test(fd, link);
                break;

            case PCIM_MENU_CIM_UPDATE_INIT:
                cim_update(fd, link, CIM_UPDATE_INITIAL_STATE);
                break;

            case PCIM_MENU_CIM_UPDATE_RECOVERY:
                cim_update(fd, link, CIM_UPDATE_RECOVERY_STATE);
                break;

            case PCIM_MENU_POWER_CIM_UTILS:
                pcim_powercim_menu(fd, link);
                break;

            default:
                break;
        }
    }

    printf("\n");
    cim_cleanup();

    return result;
}

static void display_pcim_eeprom_menu( void )
{
    int i = 0;

    printf("\nParagon CIM EEPROM Utilities:");
    printf("\n=============================");
    printf("\n\t %d: Exit", i++);
    printf("\n\t %d: Get Serial Number", i++);
    printf("\n\t %d: Get Timestamp", i++);
    printf("\n\t %d: Get Channel Name", i++);
    printf("\n\t %d: Get Device Type", i++);
    printf("\n\t %d: Get Firmware Version", i++);
    printf("\n\t %d: Get Hardware Version", i++);
    printf("\n\t %d: Set Timestamp", i++);
    printf("\n\t %d: Set Channel Name", i++);
    printf("\n\n");

    return;
}

static int pcim_eeprom_menu( int fd, int link )
{
    char running = 1;
    int result = 1;
    int diag_opt;

    while( running ) {
        display_pcim_eeprom_menu();

        printf("Please choose: ");
        scanf("%d", &diag_opt);
        printf("\n");

        switch( diag_opt ) {
            case PCIM_EEPROM_MENU_EXIT:
                running = 0;
                break;

            case PCIM_EEPROM_MENU_GET_SERNUM:
                paragon_cim_get_serial_number(fd, link);
                break;

            case PCIM_EEPROM_MENU_GET_TIMESTAMP:
                paragon_cim_get_timestamp(fd, link);
                break;

            case PCIM_EEPROM_MENU_GET_CHANNEL_NAME:
                paragon_cim_get_channel_name(fd, link);
                break;

            case PCIM_EEPROM_MENU_GET_DEVICE_TYPE:
                paragon_cim_get_device_type(fd, link, NULL);
                break;

            case PCIM_EEPROM_MENU_GET_FWVER:
                paragon_cim_get_fw_version(fd, link);
                break;

            case PCIM_EEPROM_MENU_GET_HWVER:
                paragon_cim_get_hw_version(fd, link);
                break;

            case PCIM_EEPROM_MENU_SET_TIMESTAMP:
                paragon_cim_set_timestamp(fd, link);
                break;

            case PCIM_EEPROM_MENU_SET_CHANNEL_NAME:
                paragon_cim_set_channel_name(fd, link);
                break;

            default:
                break;
        }
    }

    return result;
}

static void display_pcim_powercim_menu( void )
{
    int i = 0;

    printf("\nPower CIM Utilities:");
    printf("\n====================");
    printf("\n\t %d: Exit", i++);
    printf("\n\t %d: Get Info", i++);
    printf("\n\t %d: Read EEPROM", i++);
    printf("\n\t %d: Write EEPROM", i++);
    printf("\n\t %d: Enter B-SNMP Mode", i++);
    printf("\n\t %d: Turn on all outlet", i++);
    printf("\n\t %d: Turn on specified outlet", i++);
    printf("\n\t %d: Turn off all outlet", i++);
    printf("\n\t %d: Turn off specified outlet", i++);
    printf("\n\t %d: Get power strip status", i++);
    printf("\n\t%d: Send User-defined BSNMP command", i++);
    printf("\n\n");

    return;
}

static int pcim_powercim_menu( int fd, int link )
{
    char running = 1;
    int result = 1;
    int diag_opt;
    char cmd[64];
    char buf[4];
    long outlet;
    powercim_msg_t msg;
    int i;

    while( running ) {
        display_pcim_powercim_menu();

        printf("Please choose: ");
        scanf("%d", &diag_opt);
        printf("\n");

        switch( diag_opt ) {
            case PCIM_POWER_MENU_EXIT:
                running = 0;
                break;

            case PCIM_POWER_MENU_GET_INFO:
                paragon_cim_send_power_cmd(fd, link, POWER_REQUEST_INFO, &msg);
                break;

            case PCIM_POWER_MENU_READ_EEPROM:
                printf("Enter EEPROM offset  : 0x");
                scanf("%hx", &msg.eeprom_rd.addr);
                printf("Enter # bytes to read: ");
                scanf("%hhd", &msg.eeprom_rd.len);
                paragon_cim_send_power_cmd(fd, link, POWER_READ_EEPROM, &msg);
                break;

            case PCIM_POWER_MENU_WRITE_EEPROM:
                printf("Enter EEPROM offset   : 0x");
                scanf("%hx", &msg.eeprom_wr.addr);
                printf("Enter # bytes to write: ");
                scanf("%hhd", &msg.eeprom_wr.len);
                if( msg.eeprom_wr.len > 64 ) {
                    printf("\nERROR: Maximum length is 64.\n");
                }
                else {
                    for( i = 0; i < msg.eeprom_wr.len; i++ ) {
                        printf("Data byte %03d         : 0x", i+1);
                        scanf("%hhx", &msg.eeprom_wr.data[i]);
                    }
                    paragon_cim_send_power_cmd(fd, link, POWER_WRITE_EEPROM, &msg);
                }
                break;

            case PCIM_POWER_MENU_ENTER_BSNMP:
                paragon_cim_send_power_cmd(fd, link, POWER_ENTER_BSNMP, &msg);
                break;

            case PCIM_POWER_MENU_SNMPON:
                paragon_cim_send_power_strip_cmd(fd, link, SNMPON, NULL);
                break;

            case PCIM_POWER_MENU_SNMPON_OUTLET:
                memset(&buf[0], 0, sizeof(buf));
                getchar(); /* swallow the previous carriage return */
                printf("Enter outlet number: ");
                if( fgets(&buf[0], sizeof(buf), stdin) != NULL ) {
                    outlet = strtol(&buf[0], NULL, 10);
                    if( outlet > 0 && outlet <= 20 ) {
                        paragon_cim_send_power_strip_cmd(fd, link, SNMPON_OUTLET, &buf[0]);
                    }
                }
                break;

            case PCIM_POWER_MENU_SPOFF:
                paragon_cim_send_power_strip_cmd(fd, link, SPOFF, NULL);
                break;

            case PCIM_POWER_MENU_SPOFF_OUTLET:
                memset(&buf[0], 0, sizeof(buf));
                getchar(); /* swallow the previous carriage return */
                printf("Enter outlet number: ");
                if( fgets(&buf[0], sizeof(buf), stdin) != NULL ) {
                    outlet = strtol(&buf[0], NULL, 10);
                    if( outlet > 0 && outlet <= 20 ) {
                        paragon_cim_send_power_strip_cmd(fd, link, SPOFF_OUTLET, &buf[0]);
                    }
                }
                break;

            case PCIM_POWER_MENU_SNMPS:
                paragon_cim_send_power_strip_cmd(fd, link, SNMPS, NULL);
                break;

            case PCIM_POWER_MENU_USER_DEFINED:
                memset(&cmd[0], 0, sizeof(cmd));
                getchar(); /* swallow the previous carriage return */
                printf("Enter BSNMP command: ");
                if( fgets(&cmd[0], sizeof(cmd), stdin) != NULL ) {
                    paragon_cim_send_power_strip_cmd(fd, link, USER_DEFINED, &cmd[0]);
                }
                break;

            default:
                break;
        }
    }

    return result;
}

static void display_vmcim_menu( void )
{
    int i = 0;

    printf("\nVM CIM Diagnostics:");
    printf("\n===================");
    printf("\n\t %d: Exit", i++);
    printf("\n\t %d: Initialization packets", i++);
    printf("\n\t %d: Information packets", i++);
    printf("\n\t %d: Control packets", i++);
    printf("\n\t %d: Memory packets", i++);
    printf("\n\t %d: Keyboard packets", i++);
    printf("\n\t %d: Mouse packets", i++);
    printf("\n\t %d: USB Control packets", i++);
    printf("\n\n");

    return;
}

static int vmcim_menu( int fd, int link )
{
    char running = 1;
    int result = 1;
    int diag_opt;

    while( running ) {
        display_vmcim_menu();

        printf("Please choose: ");
        scanf("%d", &diag_opt);

        switch( diag_opt ) {
            case VMCIM_MENU_EXIT:
                running = 0;
                break;

            case VMCIM_MENU_INIT_PKTS:
                vmcim_init_menu(fd, link);
                break;

            case VMCIM_MENU_INFO_PKTS:
                vmcim_info_menu(fd, link);
                break;

            case VMCIM_MENU_CNTL_PKTS:
                vmcim_control_menu(fd, link);
                break;

            case VMCIM_MENU_MEM_PKTS:
                vmcim_mem_menu(fd, link);
                break;

            case VMCIM_MENU_KBD_PKTS:
                vmcim_kbd_menu(fd, link);
                break;

            case VMCIM_MENU_MOUSE_PKTS:
                vmcim_mouse_menu(fd, link);
                break;

            case VMCIM_MENU_USBCTL_PKTS:
                vmcim_usbctl_menu(fd, link);
                break;

            default:
                break;
        }
    }

    printf("\n");
    cim_cleanup();

    return result;
}

static void display_vmcim_init_menu( void )
{
    int i = 0;

    printf("\nVM CIM Init Packets:");
    printf("\n====================");
    printf("\n\t %d: Exit", i++);
    printf("\n\t %d: Send CIM Reset", i++);
    printf("\n\t %d: Send CIM Init", i++);
    printf("\n\n");

    return;
}

static int vmcim_init_menu( int fd, int link )
{
    char running = 1;
    int result = 1;
    int diag_opt;

    while( running ) {
        display_vmcim_init_menu();

        printf("Please choose: ");
        scanf("%d", &diag_opt);

        switch( diag_opt ) {
            case VMCIM_INIT_MENU_EXIT:
                running = 0;
                break;

            case VMCIM_INIT_MENU_RESET:
                vm_cim_send_protocol(fd, link, INITIALIZATION, RESET);
                break;

            case VMCIM_INIT_MENU_INIT:
                vm_cim_send_protocol(fd, link, INITIALIZATION, INIT);
                break;

            default:
                break;
        }
    }

    return result;
}

static void display_vmcim_info_menu( void )
{
    int i = 0;

    printf("\nVM CIM Info Packets:");
    printf("\n====================");
    printf("\n\t %d: Exit", i++);
    printf("\n\t %d: Get CIM FW Version", i++);
    printf("\n\t %d: Get CIM HW Version", i++);
    printf("\n\t %d: Get CIM FPGA Version", i++);
    printf("\n\t %d: Get CIM Protocol Version", i++);
    printf("\n\t %d: Get CIM Bootloader Version", i++);
    printf("\n\t %d: Send CIM Heartbeat", i++);
    printf("\n\t %d: Get Link Stats", i++);
    printf("\n\t %d: Get FPGA Errors", i++);
    printf("\n\t %d: Read CIM FPGA Register", i++);
    printf("\n\n");

    return;
}

static int vmcim_info_menu( int fd, int link )
{
    char running = 1;
    int result = 1;
    int diag_opt;
    unsigned char addr;

    while( running ) {
        display_vmcim_info_menu();

        printf("Please choose: ");
        scanf("%d", &diag_opt);

        switch( diag_opt ) {
            case VMCIM_INFO_MENU_EXIT:
                running = 0;
                break;

            case VMCIM_INFO_MENU_FW_VERSION:
                vm_cim_setup_version(CIM, FW, 0);
                vm_cim_send_protocol(fd, link, INFORMATION, VERSION);
                break;

            case VMCIM_INFO_MENU_HW_VERSION:
                vm_cim_setup_version(CIM, HW, 0);
                vm_cim_send_protocol(fd, link, INFORMATION, VERSION);
                break;

            case VMCIM_INFO_MENU_FPGA_VERSION:
                vm_cim_setup_version(CIM, FPGA, 0);
                vm_cim_send_protocol(fd, link, INFORMATION, VERSION);
                break;

            case VMCIM_INFO_MENU_PROTOCOL_VERSION:
                vm_cim_setup_version(CIM, PR, 0);
                vm_cim_send_protocol(fd, link, INFORMATION, VERSION);
                break;

            case VMCIM_INFO_MENU_BTLDR_VERSION:
                vm_cim_setup_version(CIM, BL, 0);
                vm_cim_send_protocol(fd, link, INFORMATION, VERSION);
                break;

            case VMCIM_INFO_MENU_CIM_HEARTBEAT:
                vm_cim_send_protocol(fd, link, INFORMATION, CIM_HEARTBEAT);
                break;

            case VMCIM_INFO_MENU_LINK_STATS:
                vm_cim_send_protocol(fd, link, INFORMATION, LINK_STATS);
                break;

            case VMCIM_INFO_MENU_CIM_FPGA_ERROR:
                vm_cim_send_protocol(fd, link, INFORMATION, CIM_FPGA_ERROR);
                break;

            case VMCIM_INFO_MENU_CIM_FPGA_READ:
                printf("\nEnter FPGA address: 0x");
                scanf("%hhx", &addr);
                if( vm_cim_setup_fpga_read(addr) == 0 ) {
                    vm_cim_send_protocol(fd, link, INFORMATION, CIM_FPGA_READ);
                }
                else {
                    printf("\nERROR: Invalid parameter specified.\n");
                }
                break;

            default:
                break;
        }
    }

    return result;
}

static void display_vmcim_control_menu( void )
{
    int i = 0;

    printf("\nVM CIM Control Packets:");
    printf("\n=======================");
    printf("\n\t %d: Exit", i++);
    printf("\n\t %d: CIM Set Color Calibration Voltage On", i++);
    printf("\n\t %d: CIM Set Color Calibration Voltage Off", i++);
    printf("\n\t %d: CIM Set Transmit Timeout Value", i++);
    printf("\n\t %d: CIM Set LED Off", i++);
    printf("\n\t %d: CIM Set LED On", i++);
    printf("\n\t %d: CIM Set LED Toggle", i++);
    printf("\n\t %d: CIM Enter CIM Buffer Loopback Test", i++);
    printf("\n\t %d: CIM Enter Priority Buffer Loopback Test", i++);
    printf("\n\t %d: CIM Exit Loopback Test", i++);
    printf("\n\t%d: CIM Exit Test", i++);
    printf("\n\t%d: Write CIM FPGA Register", i++);
    printf("\n\n");

    return;
}

static int vmcim_control_menu( int fd, int link )
{
    char running = 1;
    int result = 1;
    int diag_opt;
    unsigned long arg;
    unsigned char opt[8];
    unsigned char addr, data;

    while( running ) {
        display_vmcim_control_menu();

        printf("Please choose: ");
        scanf("%d", &diag_opt);

        switch( diag_opt ) {
            case VMCIM_CNTL_MENU_EXIT:
                running = 0;
                break;

            case VMCIM_CNTL_MENU_SET_CCV_ON:
                if( vm_cim_setup_ccv(1) == 0 ) {
                    vm_cim_send_protocol(fd, link, CONTROL, CIM_SET_COLOR_CAL_VOLTAGE);
                }
                else {
                    printf("\nERROR: Invalid reference value\n");
                }
                break;

            case VMCIM_CNTL_MENU_SET_CCV_OFF:
                if( vm_cim_setup_ccv(0) == 0 ) {
                    vm_cim_send_protocol(fd, link, CONTROL, CIM_SET_COLOR_CAL_VOLTAGE);
                }
                else {
                    printf("\nERROR: Invalid reference value\n");
                }
                break;

            case VMCIM_CNTL_MENU_SET_TXTIMEOUT:
                printf("\nEnter timeout value: ");
                scanf("%ld", &arg);
                if( vm_cim_setup_txtimeout(arg) == 0 ) {
                    vm_cim_send_protocol(fd, link, CONTROL, CIM_SET_TX_TIMEOUT_VALUE);
                }
                else {
                    printf("\nERROR: Invalid timeout value. Valid range is 1 to 5.\n");
                }
                break;

            case VMCIM_CNTL_MENU_SET_LED_OFF:
                if( vm_cim_setup_led(LED_OFF) == 0 ) {
                    vm_cim_send_protocol(fd, link, CONTROL, CIM_SET_LED);
                }
                else {
                    printf("\nERROR: Invalid LED setting.\n");
                }
                break;

            case VMCIM_CNTL_MENU_SET_LED_ON:
                if( vm_cim_setup_led(LED_ON) == 0 ) {
                    vm_cim_send_protocol(fd, link, CONTROL, CIM_SET_LED);
                }
                else {
                    printf("\nERROR: Invalid LED setting.\n");
                }
                break;

            case VMCIM_CNTL_MENU_SET_LED_TOGGLE:
                if( vm_cim_setup_led(LED_TOGGLE) == 0 ) {
                    vm_cim_send_protocol(fd, link, CONTROL, CIM_SET_LED);
                }
                else {
                    printf("\nERROR: Invalid LED setting.\n");
                }
                break;

            case VMCIM_CNTL_MENU_ENTER_CIM_BUFFER_LPBK_TEST:
                opt[0] = LPBK_TEST_CIM_BUFFER;
                if( vm_cim_setup_test_mode(TEST_LOOPBACK, 1, &opt[0]) == 0 ) {
                    vm_cim_send_protocol(fd, link, CONTROL, CIM_SET_TEST_MODE);
                }
                else {
                    printf("\nERROR: Invalid Test.\n");
                }
                break;
            case VMCIM_CNTL_MENU_ENTER_PRI_BUFFER_LPBK_TEST:
                opt[0] = LPBK_TEST_PRIORITY_BUFFER;
                if( vm_cim_setup_test_mode(TEST_LOOPBACK, 1, &opt[0]) == 0 ) {
                    vm_cim_send_protocol(fd, link, CONTROL, CIM_SET_TEST_MODE);
                }
                else {
                    printf("\nERROR: Invalid Test.\n");
                }
                break;
            case VMCIM_CNTL_MENU_EXIT_LPBK_TEST:
                opt[0] = LPBK_TEST_DISABLED;
                if( vm_cim_setup_test_mode(TEST_LOOPBACK, 1, &opt[0]) == 0 ) {
                    vm_cim_setup_buffer(FPD_PRIORITY_BUFFER);
                    vm_cim_send_protocol(fd, link, CONTROL, CIM_SET_TEST_MODE);
                    vm_cim_setup_buffer(FPD_CIM_BUFFER);
                }
                else {
                    printf("\nERROR: Invalid Test.\n");
                }
                break;
            case VMCIM_CNTL_MENU_EXIT_TEST:
                if( vm_cim_setup_test_mode(TEST_EXIT, 0, NULL) == 0 ) {
                    vm_cim_setup_buffer(FPD_PRIORITY_BUFFER);
                    vm_cim_send_protocol(fd, link, CONTROL, CIM_SET_TEST_MODE);
                    vm_cim_setup_buffer(FPD_CIM_BUFFER);
                }
                else {
                    printf("\nERROR: Invalid Test.\n");
                }
                break;

            case VMCIM_CNTL_MENU_CIM_FPGA_WRITE:
                printf("\nEnter FPGA address : 0x");
                scanf("%hhx", &addr);
                printf("\nEnter Byte to write: 0x");
                scanf("%hhx", &data);
                if( vm_cim_setup_fpga_write(addr, data) == 0 ) {
                    vm_cim_send_protocol(fd, link, CONTROL, CIM_FPGA_WRITE);
                }
                else {
                    printf("\nERROR: Invalid parameter specified.\n");
                }
                break;

            default:
                break;
        }
    }

    return result;
}

static void display_vmcim_mem_menu( void )
{
    int i = 0;

    printf("\nVM CIM Memory Packets:");
    printf("\n======================");
    printf("\n\t %d: Exit", i++);
    printf("\n\t %d: Write EEPROM", i++);
    printf("\n\t %d: Read EEPROM", i++);
    printf("\n\n");

    return;
}

static int vmcim_mem_menu( int fd, int link )
{
    char running = 1;
    int result = 1;
    int diag_opt;
    unsigned short addr, cnt, i;
    unsigned char data[128];

    while( running ) {
        display_vmcim_mem_menu();

        printf("Please choose: ");
        scanf("%d", &diag_opt);

        switch( diag_opt ) {
            case VMCIM_MEM_MENU_EXIT:
                running = 0;
                break;

            case VMCIM_MEM_MENU_WRITE_EEPROM:
                printf("\nEnter EEPROM offset: 0x");
                scanf("%hx", &addr);
                printf("Enter Byte count   : ");
                scanf("%hd", &cnt);
                if( cnt < 128 ) {
                    for( i = 0; i < cnt; i++ ) {
                        printf("Enter byte %03d     : 0x", i+1);
                        scanf("%hhx", &data[i]);
                    }
                    if( vm_cim_setup_write_eeprom(addr, cnt, data) == 0 ) {
                        vm_cim_send_protocol(fd, link, MEMORY, CIM_EEPROM_WRITE);
                    }
                }
                else {
                    printf("\nERROR: Single write limit is 128 bytes\n");
                }
                break;

            case VMCIM_MEM_MENU_READ_EEPROM:
                printf("\nEnter EEPROM offset: 0x");
                scanf("%hx", &addr);
                printf("Enter Byte count   : ");
                scanf("%hd", &cnt);
                if( vm_cim_setup_read_eeprom(addr, cnt) == 0 ) {
                    vm_cim_send_protocol(fd, link, MEMORY, CIM_EEPROM_READ);
                }
                else {
                    printf("\nERROR: Invalid parameter specified.\n");
                }
                break;

            default:
                break;
        }
    }

    return result;
}

static void display_vmcim_kbd_menu( void )
{
    int i = 0;

    printf("\nVM CIM Keyboard Packets:");
    printf("\n========================");
    printf("\n\t %d: Exit", i++);
    printf("\n\t %d: Set Keybd Model", i++);
    printf("\n\t %d: Set Keybd Language", i++);
    printf("\n\t %d: Get Target Keybd LED Settings", i++);
    printf("\n\t %d: Send Keybd Data", i++);
    printf("\n\n");

    return;
}

static void display_vmcim_kbd_model( void )
{
    printf("\nSupported Keyboard Models:");
    printf("\n==========================");
    printf("\n\t%02xh: Standard 101 Key", STD_101_KEY);
    printf("\n\t%02xh: Pinnacle Deko Fast Action", PINNACLE_DEKO);
    printf("\n\t%02xh: Bella DV", BELLA_DV);
    printf("\n\n");

    return;
}

static void display_vmcim_kbd_language( void )
{
    printf("\nSupported Keyboard Language:");
    printf("\n============================");
    printf("\n\t%02x: US English", US_ENGLISH);
    printf("\n\t%02x: British English", BRITISH_ENGLISH);
    printf("\n\t%02x: French", FRENCH);
    printf("\n\t%02x: German", GERMAN);
    printf("\n\t%02x: Italian", ITALIAN);
    printf("\n\t%02x: Spanish", SPANISH);
    printf("\n\t%02x: Norway", NORWAY);
    printf("\n\t%02x: Japanese", JAPANESE);
    printf("\n\t%02x: Korean", KOREAN);
    printf("\n\t%02x: Chinese (Traditional)", CHINESE_TRADITIONAL);
    printf("\n\t%02x: Chinese (Simplified)", CHINESE_SIMPLIFIED);
    printf("\n\n");

    return;
}

static int vmcim_kbd_menu( int fd, int link )
{
    char running = 1;
    int result = 1;
    int diag_opt;
    unsigned char model, lang;

    while( running ) {
        display_vmcim_kbd_menu();

        printf("Please choose: ");
        scanf("%d", &diag_opt);

        switch( diag_opt ) {
            case VMCIM_KBD_MENU_EXIT:
                running = 0;
                break;

            case VMCIM_KBD_MENU_SET_MODEL:
                display_vmcim_kbd_model();
                printf("Please choose: 0x");
                scanf("%hhx", &model);
                if( vm_cim_setup_kbd_model(model) == 0 ) {
                    vm_cim_send_protocol(fd, link, KEYBOARD, KBD_MODEL_SET);
                }
                else {
                    printf("\nERROR: Invalid parameter specified.\n");
                }
                break;

            case VMCIM_KBD_MENU_SET_LANG:
                display_vmcim_kbd_language();
                printf("Please choose: 0x");
                scanf("%hhx", &lang);
                if( vm_cim_setup_kbd_language(lang) == 0 ) {
                    vm_cim_send_protocol(fd, link, KEYBOARD, KBD_LANGUAGE_SET);
                }
                else {
                    printf("\nERROR: Invalid parameter specified.\n");
                }
                break;

            case VMCIM_KBD_MENU_GET_LED_SETTINGS:
                vm_cim_send_protocol(fd, link, KEYBOARD, TARGET_KBD_LED_SETTINGS);
                break;

            case VMCIM_KBD_MENU_SEND_DATA:
                vm_cim_send_protocol(fd, link, KEYBOARD, KBD_DATA);
                break;

            default:
                break;
        }
    }

    return result;
}

static void display_vmcim_mouse_menu( void )
{
    int i = 0;

    printf("\nVM CIM Mouse Packets:");
    printf("\n=====================");
    printf("\n\t %d: Exit", i++);
    printf("\n\t %d: Set Mouse Mode", i++);
    printf("\n\t %d: Send Mouse Data", i++);
    printf("\n\n");

    return;
}

static void display_vmcim_mouse_mode( void )
{
    int i = 0;

    printf("\nSupported Mouse Mode:");
    printf("\n=====================");
    printf("\n\t %d: Absolute Positioning", i++);
    printf("\n\t %d: Relative Positioning", i++);
    printf("\n\n");

    return;
}

static int vmcim_mouse_menu( int fd, int link )
{
    char running = 1;
    int result = 1;
    int diag_opt;
    unsigned char mode;

    while( running ) {
        display_vmcim_mouse_menu();

        printf("Please choose: ");
        scanf("%d", &diag_opt);

        switch( diag_opt ) {
            case VMCIM_MOUSE_MENU_EXIT:
                running = 0;
                break;

            case VMCIM_MOUSE_MENU_SET_MODE:
                display_vmcim_mouse_mode();
                printf("Please choose: ");
                scanf("%hhd", &mode);
                if( vm_cim_setup_mouse_mode(mode) == 0 ) {
                    vm_cim_send_protocol(fd, link, MOUSE, MOUSE_MODE_SET);
                }
                else {
                    printf("\nERROR: Invalid parameter specified.\n");
                }
                break;

            case VMCIM_MOUSE_MENU_SEND_DATA:
                vm_cim_send_protocol(fd, link, MOUSE, MOUSE_DATA);
                break;

            default:
                break;
        }
    }

    return result;
}

static void display_vmcim_usbctl_menu( void )
{
    int i = 0;

    printf("\nVM CIM USB Control Packets:");
    printf("\n===========================");
    printf("\n\t %d: Exit", i++);
    printf("\n\t %d: USB CIM Disconnect", i++);
    printf("\n\t %d: USB CIM Connect", i++);
    printf("\n\t %d: VM USB Device Configuration", i++);
    printf("\n\t %d: Stall Bulk Endpoint", i++);
    printf("\n\t %d: Mass Storage Reset Rsp", i++);
    printf("\n\t %d: Clear Feature Halt Rsp", i++);
    printf("\n\n");

    return;
}

static int vmcim_usbctl_menu( int fd, int link )
{
    char running = 1;
    int result = 1;
    int diag_opt;
    unsigned char settings;
    unsigned char mass_storage_id;
    char val;

    while( running ) {
        display_vmcim_usbctl_menu();

        printf("Please choose: ");
        scanf("%d", &diag_opt);

        switch( diag_opt ) {
            case VMCIM_USBCTL_MENU_EXIT:
                running = 0;
                break;

            case VMCIM_USBCTL_MENU_DISCONNECT:
                vm_cim_send_protocol(fd, link, USB, USB_CIM_DISCONNECT);
                break;

            case VMCIM_USBCTL_MENU_CONNECT:
                vm_cim_send_protocol(fd, link, USB, USB_CIM_CONNECT);
                break;

            case VMCIM_USBCTL_MENU_CONFIGURATION:
                vm_cim_send_protocol(fd, link, USB, VM_USB_DEVICE_CONFIG);
                break;

            case VMCIM_USBCTL_MENU_STALL_BULKEP:
                printf("\nEnter Mass Storage ID (0-based, %d-channel(s)): ",
                       max_host_chan);
                scanf("%hhd", &val);
                if( val >= max_host_chan ) {
                    printf("\nERROR: Invalid ID\n");
                    return -1;
                }
                mass_storage_id = val;

                settings = 0;
                printf("\nStall Bulk-IN (0=no, 1=yes)   : ");
                scanf("%hhd", &val);
                if( val == 1 ) {
                    settings |= STALL_BULK_ENDPOINT_IN;
                }
                else if( val != 0 ) {
                    printf("\nERROR: Invalid answer\n");
                    return -1;
                }

                printf("\nUnstall Bulk-IN (0=no, 1=yes) : ");
                scanf("%hhd", &val);
                if( val == 1 ) {
                    settings |= UNSTALL_BULK_ENDPOINT_IN;
                }
                else if( val != 0 ) {
                    printf("\nERROR: Invalid answer\n");
                    return -1;
                }

                printf("\nStall Bulk-OUT (0=no, 1=yes)  : ");
                scanf("%hhd", &val);
                if( val == 1 ) {
                    settings |= STALL_BULK_ENDPOINT_OUT;
                }
                else if( val != 0 ) {
                    printf("\nERROR: Invalid answer\n");
                    return -1;
                }

                printf("\nUnstall Bulk-OUT (0=no, 1=yes): ");
                scanf("%hhd", &val);
                if( val == 1 ) {
                    settings |= UNSTALL_BULK_ENDPOINT_OUT;
                }
                else if( val != 0 ) {
                    printf("\nERROR: Invalid answer\n");
                    return -1;
                }

                printf("\nMake Permanent (0=no, 1=yes)  : ");
                scanf("%hhd", &val);
                if( val == 1 ) {
                    settings |= BULK_ENDPOINT_SETTING_PERMANENT;
                }
                else if( val != 0 ) {
                    printf("\nERROR: Invalid answer\n");
                    return -1;
                }

                if( vm_cim_setup_stall_bulkep(mass_storage_id, settings) == 0 ) {
                    vm_cim_send_protocol(fd, link, USB, STALL_BULK_EP);
                }
                else {
                    printf("\nERROR: Invalid parameter specified.\n");
                }
                break;

            case VMCIM_USBCTL_MENU_MS_RESET_RSP:
                vm_cim_send_protocol(fd, link, USB, MASS_STORAGE_RESET);
                break;

            case VMCIM_USBCTL_MENU_CLR_FEAT_HALT_RSP:
                vm_cim_send_protocol(fd, link, USB, CLR_FEATURE_HALT);
                break;

            default:
                break;
        }
    }

    return result;
}
